/**
 * Class that searches water trade routes after game load.
 */
class RestoreCruiseLinesAction extends InstantAction
{
/* public */
	function GetName()
	{
		return "Seraching cruise lines";
	}

	function GetPriority()
	{
		/* We need this action AFTER those transport nodes will be created */
		/* And this is the simplest(yet bad) way to do so */
		return APriority.MAX - 1;
	}

/* protected */
	function Execute()
	{
		local c_name = CruiseCenter.GetClassName();
		CodeUtils.Log("Restoring [" + c_name + "] cruise lines", 2);

		local good_name = CruiseLine.special_ship_name_prefix;
		local sell_name = SellLostVehiclesTask.special_sell_name;
		local vehicles = AIVehicleList();

		vehicles.Valuate(AIVehicle.GetVehicleType);
		vehicles.KeepValue(AIVehicle.VT_WATER);
		foreach (v, dummy in vehicles) {
			if (AIVehicle.GetName(v) == (good_name + v)) {
				/*
				 * Now all ships assigned to cruise lines before game save
				 *  will be in to-sell list.
				 * When we restore trade route hovewer, it'll rename own
				 *  vehicles back into (good_name + v).
				 * Ships that remain unrenamed must be sold because there is
				 *  no trade routes for them.
				 * Such situation can occur because of different water quadtrees
				 *  before game save and after game load(nasty terraforming).
				 * (different qt => different sea regions => different routes =>
				 *  => possible "lost" routes => "lost" trade routes)
				 */
				AIVehicle.SetName(v, (sell_name + v));
			}
		}

		local structured_nodes = GlobalTransportMap.Get().structured_transport_nodes;

		foreach (dummy_type_id, type_nodes in structured_nodes[c_name]) {
			foreach (dummy_node_id, node in type_nodes) {
				foreach (dummy_id, link in node.links_out) {
					foreach (dummy_id, route in link.routes) {
						this.TryToRestoreTrade(route); 
					}
				}
			}
		}

		CodeUtils.Log("Water trade routes restoration complete", 2);
		return 0;
	}

/* private */
	/**
	 * Creates a new cruise line from the given route.
	 * @param water_route Water route.
	 * @return True if and only if new trade route has been created.
	 */
	function TryToRestoreTrade(water_route)
	{
		if (water_route.GetStart().is_closed) return false;
		if (water_route.GetEnd().is_closed) return false;

		local s_docks = Dock.GetDocks(water_route.GetStart());
		if (s_docks.len() == 0) return false;
		local e_docks = Dock.GetDocks(water_route.GetEnd());
		if (e_docks.len() == 0) return false;

		local result = false;
		local buoymap = ShipDepartment.Get().water_paths_register;
		foreach (dummy_s_id, s in s_docks) {
			if (s.depot.location == -1) continue;
			foreach (dummy_e_id, e in e_docks) {
				if (e.depot.location == -1) continue;

				local buoys = buoymap.GetBuoySequence(s.GetStationID(), e.GetStationID());
				if (buoys != null) {
					local new_trade_route = CruiseLine(water_route, s, e, buoys);
					CodeUtils.Log("[" + new_trade_route.GetName() + "] restored", 2);
					result = true;
				}
			}
		}

		return result;
	}
}
