/**
 * Class that represent ship related AI settings.
 */
class WaterSettings extends Terron_Settings
{
/* public */
	static function GetClassName()
	{
		return "WaterSettings";
	}

	static function Restore(memento)
	{
		local restored = WaterSettings();
		restored.max_route_length = memento.max_route_length;
		restored.min_route_length = memento.min_route_length;
		restored.region_to_operate = memento.region_to_operate;
		restored.ships_limit = memento.ships_limit;
		return restored;
	}

	/**
	 * Get approximate cost of the dock.
	 * @return Approximate construction cost for the dock.
	 */
	static function GetDockCost()
	{
		local c = AIMarine.GetBuildCost(AIMarine.BT_DOCK);
		c += AITile.GetBuildCost(AITile.BT_TERRAFORM);
		c += AIMarine.GetBuildCost(AIMarine.BT_DEPOT);
		return c;
	}

	/** Defines whether AI can use slow ships. */
	allow_slow_ships = false;

	/** Defines how close to each other AI will build buoys. */
	max_buoys_interval = 26;

	/** Defines max ship line length allowed */
	</ must_save = true />
	max_route_length = null;

	/** Defines min ship line length allowed */
	</ must_save = true />
	min_route_length = null;

	/** ID of the map region where AI will be able to use ships. */
	</ must_save = true />
	region_to_operate = -1;

	/** The amount of ships when AI must cease new docks construction */
	</ must_save = true />
	ships_limit = null;

	/** Minimal allowed acceptance level in a point of new dock placing. */
	min_acceptance_for_dock = 20;

	/** Amount of vehicles AI must have to start sea expansion. */
	min_vehicles_to_enable_ships = 0;

	/** 1 <=> northeast map border is water, 0 otherwise */
	ne_water = 0;

	/** 1 <=> northwest map border is water, 0 otherwise */
	nw_water = 0;

	/** 1 <=> southeast map border is water, 0 otherwise */
	se_water = 0;

	/** 1 <=> southwest map border is water, 0 otherwise */
	sw_water = 0;

	/** Ships with max speed lesser than this value are considered slow. */
	slow_ship_speed = 50;

	/** Ships with capacity greater than this value are considered big. */
	big_ship_capacity = 300;

/* protected */
	/**
	 * Set water settings up.
	 */
	constructor()
	{
		::Terron_Settings.constructor();

		this.max_route_length = AIController.GetSetting("max_water_route_length");
		if (this.max_route_length == -1) this.max_route_length = 320;
		this.min_route_length = (15 + this.max_route_length / 8).tointeger()

		if (AIGameSettings.IsDisabledVehicleType(AIVehicle.VT_WATER)
		|| AIController.GetSetting("ShipModule") <= 0
		|| !AICargo.IsValidCargo(CorporationUtils.pass_cargo_id.value)) {
			this.ships_limit = 0;
			return;
		}

		if (AIController.GetSetting("use_slow_ships") > 0) {
			this.allow_slow_ships = true;
		}

		if (AIController.GetSetting("BusModule") >= 2) {
			this.min_acceptance_for_dock = 12;
		}

		this.min_vehicles_to_enable_ships = 0;
		if (AIController.GetSetting("ShipModule") == 1) {
			this.min_vehicles_to_enable_ships = 30;
		}

		local water = AIGameSettings.IsValid("game_creation.water_borders") ?
			AIGameSettings.GetValue("game_creation.water_borders") : 15;

		// x == 1
		this.nw_water = (8 == (water & 8)) ? 1 : 0;

		// y == 1
		this.se_water = (2 == (water & 2)) ? 1 : 0;

		// x == MapSizeX - 2
		this.ne_water = (1 == (water & 1)) ? 1 : 0;

		// y == MapSizeY - 2
		this.sw_water = (4 == (water & 4)) ? 1 : 0;

		local n = AIGameSettings.IsValid("vehicle.max_ships") ?
			9 * AIGameSettings.GetValue("vehicle.max_ships") / 20 : 50;
		this.ships_limit = min(n, 50);
	}

/* private */
}

Terron_ClassTable.RegisterClass(WaterSettings);
