/**
 * Class that utilizes a task of water quadtree creation.
 */
class WaterformScaner
{
/* public */
	/**
	 * Create a quadtree object that indexes water space in the given region.
	 * @param region_id ID of the region to map.
	 * @return Created quadtree, or null if fails.
	 */
	function GetRegionWaterQuadTree(region_id)
	{
		this.workspace = PlayableRegions.Get().GetRegionBorders(region_id);
		if (this.workspace == null) return null;

		local size_x = AIMap.GetMapSizeX();
		local size_y = AIMap.GetMapSizeY();
		local r_size = PlayableRegions.REGION_SIZE;
		this.workspace.x_max <- min(this.workspace.x_min + r_size, size_x - 1);
		this.workspace.y_max <- min(this.workspace.y_min + r_size, size_y - 1);

		local cfg = WaterSettings.Get();
		this.workspace.border_water_level <- {};
		local wl = this.workspace.border_water_level;
		wl.x_min <- this.workspace.x_min == 0 ? 100 * cfg.nw_water : 0;
		wl.y_min <- this.workspace.y_min == 0 ? 100 * cfg.se_water : 0;
		wl.x_max <- this.workspace.x_max == size_x - 1 ? 100 * cfg.ne_water : 0;
		wl.y_max <- this.workspace.y_max == size_y - 1 ? 100 * cfg.sw_water : 0;

		local dx = this.workspace.x_max - this.workspace.x_min;
		local dy = this.workspace.y_max - this.workspace.y_min;
		local n = CodeUtils.iLog2(1 + max(dx, dy));
		local r = this.good_min_resolution;
		local f = this.GetWaterLevel.bindenv(this);
		local result = QuadTree(r, n - r, f, this.good_black_precision, this.good_white_precision);
		result.SetShift(this.workspace.x_min, this.workspace.y_min);
		this.workspace = null;
		return result;
	}

/* private */
	/** Setting value I found 'good' */
	good_min_resolution = 3;

	/** Setting value I found 'good' */
	good_black_precision = 0.65;

	/** Setting value I found 'good' */
	good_white_precision = 0.55;

	/** Keep borders of target rectangle. */
	workspace = null;

	/**
	 * Returns water level in the given square.
	 * @param x The square "left" border - lesser x coordinate.
	 * @param y The square "bottom" border - lesser y coordinate.
	 * @param sq_size Square edge length.
	 * @return Percentage of water tiles in the given square.
	 */
	function GetWaterLevel(x, y, sq_size)
	{
		x = x + this.workspace.x_min;
		y = y + this.workspace.y_min;

		local list = AITileList();
		local corner1 = AIMap.GetTileIndex(x, y);
		if (!AIMap.IsValidTile(corner1)) {
			local result = 0;
			if (x <= this.workspace.x_min) {
				result += this.workspace.border_water_level.x_min;
			}
			if (y <= this.workspace.y_min) {
				result += this.workspace.border_water_level.y_min;
			}
			if (x + sq_size >= this.workspace.x_max) {
				result += this.workspace.border_water_level.x_max;
			}
			if (y + sq_size >= this.workspace.y_max) {
				result += this.workspace.border_water_level.y_max;
			}
			return min(result, 100);
		}

		local corner2 = AIMap.GetTileIndex(x + sq_size - 1, y + sq_size - 1);
		if (!AIMap.IsValidTile(corner2)) {
			local result = 0;
			if (x <= this.workspace.x_min) {
				result += this.workspace.border_water_level.x_min;
			}
			if (y <= this.workspace.y_min) {
				result += this.workspace.border_water_level.y_min;
			}
			if (x + sq_size >= this.workspace.x_max) {
				result += this.workspace.border_water_level.x_max;
			}
			if (y + sq_size >= this.workspace.y_max) {
				result += this.workspace.border_water_level.y_max;
			}
			return min(result, 100);
		}
		list.AddRectangle(corner1, corner2);
		local z = list.Count();
		if (z == 0) z = 1;
		list.Valuate(AITile.IsWaterTile);
		list.RemoveValue(0);
		local p = (list.Count() * 100.0 / z).tointeger();
		if (p >= 80) {
			p = 100;
		} else if (p <= 20) {
			p = 0;
		} else {
			local n = this.GetSpecialWater(x, y, sq_size);
			p = (n * 100.0 / z).tointeger();
		}
		return p;
	}

	/**
	 * Returns number of water tiles in the given square.
	 * @param x The square "left" border - lesser x coordinate.
	 * @param y The square "bottom" border - lesser y coordinate.
	 * @param size Square edge length.
	 * @return Number of water tiles in the given square.
	 */
	function GetSpecialWater(x, y, size)
	{
		if (size == 2) {
			local list = AITileList();
			local corner1 = AIMap.GetTileIndex(x, y);
			local corner2 = AIMap.GetTileIndex(x + 1, y + 1);
			list.AddRectangle(corner1, corner2);
			list.Valuate(AITile.IsWaterTile);
			list.RemoveValue(0);
			local n = list.Count();
			return n > 2 ? 4 : (n < 2 ? 0 : 0);
		} else if (size > 2) {
			local new_size = size / 2;
			local new_x = x + new_size;
			local new_y = y + new_size;
			local n = 0;
			n += this.GetSpecialWater(x, y, new_size);
			n += this.GetSpecialWater(new_x, y, new_size);
			n += this.GetSpecialWater(x, new_y, new_size);
			n += this.GetSpecialWater(new_x, new_y, new_size);
			return n;
		} else {
			assert(null);
		}
	}
}
