/**
 * Class that provides utility service functions for AI water section.
 */
class WaterlineUtils
{
/* public */
	/**
	 * Account buoy path length.
	 * @param start_tile Path start tile ID.
	 * @param start_tile Path end tile ID.
	 * @param buoys Array with buoys connecting start and end.
	 * @return Summ of distances between adjacent points in path from start to
	 *  end.
	 */
	static function GetBuoyPathLength(start_tile, end_tile, buoys)
	{
		local n = buoys.len();
		if (n == 0) return AIMap.DistanceManhattan(start_tile, end_tile);

		local result = AIMap.DistanceManhattan(start_tile, buoys[0]);
		result += AIMap.DistanceManhattan(end_tile, buoys[n - 1]);
		for (local i = 1; i < n; i++) {
			result += TileUtils.DiagDistance(buoys[i - 1], buoys[i]);
		}

		return result;
	}

	/**
	 * Aux function to select best ship type for this route.
	 * @param cargo_id Cargo ID which the desired engine must transport.
	 * @param ignore_big_ships When true:
	 *  big(> WaterSettings.big_ship_capacity cargo units) ships'll be ignored.
	 * @param min_reliability Minimum allowed reliability(int from 0 to 99)
	 *  for the result engine, or 0 if the minimum reliability doesn't matter.
	 * @return Engine ID of the ship type considered to be the best for
	 *  the given params or -1 if no such engines available.
	 */
	static function ChooseBestEngine(cargo_id, ignore_big_ships, min_reliability)
	{
		/* Scan engine list and remove futile engines */
		local list = AIEngineList(AIVehicle.VT_WATER);
		/* Can't handle route cargo => futile */
		list.Valuate(AIEngine.CanRefitCargo, cargo_id);
		list.RemoveValue(0);
		/* Breaks often => futile */
		if (0 < min_reliability && min_reliability < 100) {
			list.Valuate(AIEngine.GetReliability);
			list.KeepAboveValue(min_reliability - 1);
		}

		local cfg = WaterSettings.Get();
		if (!cfg.allow_slow_ships) {
			/* Slow ships do not work well so far, it's better to ignore them */
			list.Valuate(AIEngine.GetMaxSpeed);
			list.KeepAboveValue(cfg.slow_ship_speed);
		}

		if (ignore_big_ships) {
			list.Valuate(AIEngine.GetCapacity);
			list.RemoveAboveValue(cfg.big_ship_capacity);
		}

		/* Select engine with max (capacity * speed) */
		local f = function (e) {
			local result = AIEngine.GetReliability(e);
			result = result * AIEngine.GetCapacity(e) * 1.0 / 100;
			result = result * AIEngine.GetMaxSpeed(e) / 10;
			result = result * AIEngine.GetMaxSpeed(e) / 10;
			result = result * AIEngine.GetMaxSpeed(e) / 10;
			return result;
		}
		local e = CorporationUtils.ChooseBestEngineFromList(list, f);
		return e;
	}
}
