/**
 * Base class that describes "simple route".
 * Simple means: vehicle should pick cargo at one(load, or start) node
 * and without ony transit stops delivere it to other(unload, end) node,
 * then it must go again to start.
 * It's A => B => A, not A => B => C => => D ... => A, where A, B, ... - nodes.
 */
class SimpleRoute extends SingleEngineRoute
{
/* public */
	/**
	 * SimpleRoute constructor.
	 * @param start_node Start node.
	 * @param end_node End node.
	 * @param cargo_id ID of the cargo to transport.
	 */
	constructor(start_node, end_node, cargo_id, is_transit)
	{
		local label = " (" + AICargo.GetCargoLabel(cargo_id) + ")";
		this.one_way = !(cargo_id in start_node.GetType().consumption
			&& cargo_id in end_node.GetType().production);
		local d = this.one_way ? " -> " : " <-> ";
		local name = start_node.GetName() + d + end_node.GetName() + label;
		::SingleEngineRoute.constructor(name, cargo_id);

		this.start_node = start_node;
		this.end_node = end_node;
		this.length = AIMap.DistanceManhattan(start_node.GetLocation(), end_node.GetLocation());
		this.is_transit = is_transit;

		//this.one_way = !(start_node.IsAccepting(c) && end_node.IsProducing(c));

		this.free_production = Property(this.doGetFreeProduction.bindenv(this), 16);
		this.free_production.SetName("Get free production for route [" + this.name + "]");
	}

	/**
	 * Get free to transport route production.
	 * @return Amount of cargo ready to transported via this route per month.
	 */
	function GetFreeProduction()
	{
		return this.free_production.Get();
	}

	/**
	 * Get "free" route production.
	 * @return Amount of cargo produced(per month) by route nodes,
	 *  but not transported yet.
	 */
	function doGetFreeProduction()
	{
		local start = this.start_node;
		local end = this.end_node;
		local c = this.cargo_id;
		/* Industries can stop cargo acception */
		if (this.one_way && !end.IsAccepting(c)) return 0;

		local s_production = start.GetFreeProduction(c);
		local n = start.GetOpponentsCount();
		if (n > 0) s_production = 110 * s_production / (100 * (n + 1));
		if (this.one_way) return s_production;

		local e_production = end.GetFreeProduction(c);
		local n = end.GetOpponentsCount();
		if (n > 0) e_production = 110 * e_production / (100 * (n + 1));
		return min(s_production, e_production);
	}

	/**
	 * Get this route start.
	 * @return This route start node.
	 */
	function GetStart()
	{
		return this.start_node;
	}

	/**
	 * Get this route end.
	 * @return This route end node.
	 */
	function GetEnd()
	{
		return this.end_node;
	}

	/**
	 * Check if this route is one way route.
	 * @return True when route has single cargo transport direction (from start to end),
	 * false when end node produces cargo as start node, start node accepts cargo as end node,
	 * and cargo can be delivered from end node to start node as well as in opposite direction.
	 * e.g. one way route: coal from coal mine to power plant,
	 * e.g. not one way route: mail from town A to town B (and from B to A).
	 */
	function IsOneWay()
	{
		return this.one_way;
	}

	/**
	 * Check if this route is transit way route.
	 * @return True when cargo must be transfered somewhere else after it arrive
	 *  to the this route's endpoit.
	 */
	function IsTransit()
	{
		return this.is_transit;
		//return this.end_node.type_id == NodeTypeID.NT_TRANSIT_DROP;
	}

/* protected */
	/** Holds route's [free to transport per month production amount] property */
	free_production = null;

	/** Route start node */
	start_node = null;

	/** Route end node */
	end_node = null;

	/** Route length */
	length = null;

	/** Flag showing if there is no way to delivere cargo from end node to start node */
	one_way  = null;

	is_transit = false;
}
