/**
 * Helpers that can be useful for the entire AI corporation.
 */
class CorporationUtils
{
/* public */
	/**
	 * Select best engine from given list.
	 * @param engine_list list of engines.
	 * @param rank_callback callback function to compare engines.
	 * @return best engine id or -1(empty list).
	 */
	static function ChooseBestEngineFromList(engine_list, rank_callback)
	{
		if (engine_list == null) return -1;

		local best_engine_efficiency = 0;
		local best_engine = -1;
		foreach (e, dummy in engine_list) {
			local current_engine_efficiency = rank_callback(e);
			if (current_engine_efficiency <= best_engine_efficiency) continue;
			best_engine_efficiency = current_engine_efficiency;
			best_engine = e;
		}
		return best_engine;
	}

	/**
	 * Routine operations to close AI parts.
	 * @param system_to_close System to close.
	 * @return Not used.
	 */
	static function CloseSystem(system_to_close)
	{
		local corporation = Corporation.Get();

		// think it's needed due to save/load
		corporation.closing_systems.AddItem(system_to_close);

		Terron_Event.WipeOutListener(system_to_close);

		system_to_close.Disable(5 * GameTime.YEAR - GameTime.DAY);

		// (!) order is important.
		// x.Close() must be called before x will be removed from parents
		local is_closed = system_to_close.Close();

		local id = system_to_close.GetID();
		foreach (dummy_id, _parent in CorporationUtils.GetOwners(system_to_close)) {
			_parent.GetSubsystems().RemoveItem(id);
		}

		if (!is_closed) {
			AIController.Sleep(50);
			is_closed = system_to_close.Close();
		}
		if (!is_closed) {
			AIController.Sleep(100);
			is_closed = system_to_close.Close();
		}

		if (is_closed) {
			corporation.closing_systems.RemoveItem(id);
			TransportSystemClosedEvent.Fire(system_to_close);
		} else {
			Terron.UpdateMemento();
		}
	}

	/**
	 * Make sure that AI has requestet money amount.
	 * @param money_amount Wanted money amount.
	 * @return True if AI has requested money amount(ready to spend right now),
	 *  false else.
	 */
	static function GetMoney(money_amount)
	{
		local balance = AICompany.GetBankBalance(AICompany.COMPANY_SELF);
		local max_loan = AICompany.GetMaxLoanAmount();
		/* to prevent bankruptcy */
		local safe_amount = 12000;
		balance = balance - balance / 100 - 15000;

		if (money_amount < balance) return true;

		local max_loan = AICompany.GetMaxLoanAmount();
		return (money_amount < balance + max_loan - AICompany.GetLoanAmount()) ?
			AICompany.SetLoanAmount(max_loan) : false;
	}

	/**
	 * Get owners(parents in corporation hierarchy) of a transport system.
	 * @param transport_system Object, the transport system to check.
	 * @return Container with parent objects of given object.
	 */
	static function GetOwners(transport_system)
	{
		local RecoverOwnersVisitor = VGetOwners(transport_system);
		RecoverOwnersVisitor.Visit(Corporation.Get());
		return RecoverOwnersVisitor.GetResult();
	}

	/**
	 * Get month income.
	 * @return AI company's month income.
	 */
	static function GetMonthIncome()
	{
		return CorporationUtils.month_income.value;
	}

	/**
	 * Set month income.
	 * @param new_value New value of AI company's month income.
	 */
	static function SetMonthIncome(new_value)
	{
		CorporationUtils.month_income.value = new_value;
	}

	/**
	 * Get month income.
	 * @param new_value New value of the inflation(float from 1.0 to infinite).
	 */
	static function SetInflation(new_value)
	{
		CorporationUtils.inflation_factor.value = new_value;
	}

	/**
	 * Get the inflation factor.
	 * @return Inflation rate since start of the game.
	 */
	static function GetInflation()
	{
		return CorporationUtils.inflation_factor.value;
	}

/* private */
	/** Passengers cargo id */
	static pass_cargo_id = {value = -1};

	/** Mail cargo id */
	static mail_cargo_id = {value = -1};

	/** Month income */
	static month_income = {value = 0};

	/** Inflation rate since start of the game */
	static inflation_factor = {value = 1.0};

/* public */
	/**
	 * Find and save for futute use pass cargo ID.
	 */
	static function Init()
	{
		local i = 0, j = 0;
		foreach (c, dummy in AICargoList()) {
			if (AICargo.GetCargoLabel(c) == "PASS" && i == 0) {
				CorporationUtils.pass_cargo_id.value = c;
				i++;
			}
			if (AICargo.HasCargoClass(c, AICargo.CC_MAIL) && j == 0) {
				CorporationUtils.mail_cargo_id.value = c;
				j++;
			}
			if (i > 0 && j > 0) break;
		}
	}
}
