/**
 * Helper class to work with tiles.
 */
class TileUtils
{
/* public */
	/**
	 * A safe implementation of AITileList.AddRectangle. Only valid tiles are
	 *  added to the tile list.
	 * @param tile_list The AITileList to add the tiles to.
	 * @param center_tile The center of the rectangle.
	 * @param x_min The amount of tiles to the north-east, relative to center_tile.
	 * @param y_min The amount of tiles to the north-west, relative to center_tile.
	 * @param x_plus The amount of tiles to the south-west, relative to center_tile.
	 * @param y_plus The amount of tiles to the south-east, relative to center_tile.
	 * @note Taken from the Admiral AI.
	 */ 
	/* static */ function AddRectangleSafe(tile_list, center_tile, x_min, y_min, x_plus, y_plus)
	{
		local tx = AIMap.GetTileX(center_tile);
		local ty = AIMap.GetTileY(center_tile);

		local safe_corner1_x = max(1, tx - x_min);
		local safe_corner1_y = max(1, ty - y_min);
		local safe_corner1 = AIMap.GetTileIndex(safe_corner1_x, safe_corner1_y);

		local safe_corner2_x = min(AIMap.GetMapSizeX() - 2, tx + x_plus);
		local safe_corner2_y = min(AIMap.GetMapSizeY() - 2, ty + y_plus);
		local safe_corner2 = AIMap.GetTileIndex(safe_corner2_x, safe_corner2_y);

		tile_list.AddRectangle(safe_corner1, safe_corner2);
		return tile_list;
	}

	/**
	 * A safe implementation of AITileList.AddRectangle. Only valid tiles are
	 *  added to the tile list.
	 * @param tile_list The AITileList to add the tiles to.
	 * @param center_tile The center of the rectangle.
	 * @param x_shift The amount of tiles to the north-east,
	 *  and the amount of tiles to the south-west, relative to center_tile.
	 * @param y_min The amount of tiles to the north-west,
	 *  amd the amount of tiles to the south-east, relative to center_tile.
	 */
	/* static */ function AddSimmetricRectangleSafe(tile_list, center_tile, x_shift, y_shift)
	{
		return TileUtils.AddRectangleSafe(tile_list, center_tile, x_shift, y_shift, x_shift, y_shift);
	} 

	/**
	 * This function will destroy road "dead ends".
	 * Mean if there is road piece connected only with one other road piece =>
	 * => it will be removed.
	 * @param list List with tiles to check the "dead ends".
	 * @param number_of_trees_to_plant Number of trees to plant in place
	 *  of destroyed road. Used to remove town authority unrest.
	 */
	/* static */ function CutRoads(list, number_of_trees_to_plant)
	{
		local old_road_type = AIRoad.GetCurrentRoadType();
		AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_ROAD);

		local offsets = [AIMap.GetTileIndex(0, 1), AIMap.GetTileIndex(0, -1),
	                 AIMap.GetTileIndex(1, 0), AIMap.GetTileIndex(-1, 0)];
		foreach (t, dummy in list) {
			if (!AIRoad.IsRoadTile(t)) continue;
			if (AIRoad.GetNeighbourRoadCount(t) != 1) continue;
			if (AITile.IsStationTile(t + offsets[0])) continue;
			if (AIRoad.IsRoadDepotTile(t + offsets[0])) continue;
			if (AITile.IsStationTile(t + offsets[1])) continue;
			if (AIRoad.IsRoadDepotTile(t + offsets[1])) continue;
			if (AITile.IsStationTile(t + offsets[2])) continue;
			if (AIRoad.IsRoadDepotTile(t + offsets[2])) continue;
			if (AITile.IsStationTile(t + offsets[3])) continue;
			if (AIRoad.IsRoadDepotTile(t + offsets[3])) continue;
			if (!AITile.DemolishTile(t)) continue;
			for (local j = 0; j < number_of_trees_to_plant; j++) {
				AITile.PlantTree(t);
			}
		}
		AIRoad.SetCurrentRoadType(old_road_type);
	}

	/**
	 * Get the list of tiles near given node, where desired(see params) station can be build.
	 * @note It is not the same as ::Node.AITileList_NodeProducing function
	 * @param node The node where we want to build.
	 * @param min_cargo_production Minimal wanted cargo production.
	 * @param station_x The "x size" of desired station.
	 * @param station_y The "y size" of desired station.
	 * @param coverage_radius The coverage radius of desired station.
	 * @param cargo_id The cargo we want to transport.
	 * @return List with suitable tiles, or empty list if there is no such tiles.
	 */
	/* static */ function GetProductionNodeFreeStationLocations(node, min_cargo_production, station_x, station_y, coverage_radius, cargo_id)
	{
		local list = node.IsTown() ?
			node.AITileList_NodeAccepting(coverage_radius, cargo_id) :
			node.AITileList_NodeProducing(coverage_radius, cargo_id);

		list.Valuate(AITile.GetCargoProduction, cargo_id, station_x, station_y, coverage_radius);
		list.RemoveBelowValue(min_cargo_production);

		local helper_list = AITileList();
		helper_list.AddList(list);
		/* this isn't very efficient */
		foreach (t, dummy in helper_list) {
			TileUtils.AddRectangleSafe(list, t, station_x - 1, station_y - 1, 0, 0);
		}

		return list;
	}

	/**
	 * Get amount of flat tiles near the given tile.
	 * @param tile_to_check Tile to check.
	 * @return -1 if the given tile not valid, amount of tiles around with
	 *  slope in [SLOPE_FLAT, SLOPE_NWS, SLOPE_WSE, SLOPE_SEN, SLOPE_ENW] else.
	 */
	/* static */ function GetAmountOfFlatTilesAround(tile_to_check)
	{
		local result = 0;
		local x = AIMap.GetTileX(tile_to_check);
		local y = AIMap.GetTileY(tile_to_check);
		if (x <= 1 || x >= AIMap.GetMapSizeX() - 2) return -1;
		if (y <= 1 || y >= AIMap.GetMapSizeY() - 2) return -1;
		for (local i = -1; i <= 1; i++) {
			for (local j = -1; j <= 1; j++) {
				local t = AIMap.GetTileIndex(x + i, y + j);
				if (AITile.IsWaterTile(t)) continue;
				if (!AITile.IsBuildable(t)) continue;
				local s = AITile.GetSlope(t);
				if (TileUtils.IsSlopeFlatOrWithThreeCornersRaised(s)) result++;
			}
		}
		return result;
	}

	/**
	 * Check the tile and define: is it road or not.
	 * @param t Tile to check.
	 * @return True only when the tile is road, false else.
	 * @note Drive through stations are not roads!
	 */
	/* static */ function IsRoad(t)
	{
		if (!AIRoad.IsRoadTile(t)) return false;
		return !AIRoad.IsDriveThroughRoadStationTile(t);
	}

	/**
	 * Check if the slope is almost flat, but with one peak.
	 * @param s Slope to check.
	 * @return True if the slope in [SLOPE_N, SLOPE_S, SLOPE_E, SLOPE_W].
	 */
	/* static */ function IsSlopeWithOneCornerRaised(s)
	{
		return s == AITile.SLOPE_N || s == AITile.SLOPE_S ||
				s == AITile.SLOPE_E || s == AITile.SLOPE_W;
	}

	/**
	 * Check if the slope is almost flat.
	 * @param s Slope to check.
	 * @return True if the slope flat or with 3 corners raised.
	 */
	/* static */ function IsSlopeFlatOrWithThreeCornersRaised(s)
	{
		if (s == AITile.SLOPE_FLAT) return true;
		if (AITile.IsSteepSlope(s) || AITile.IsHalftileSlope(s)) return false;
		return s == AITile.SLOPE_NWS || s == AITile.SLOPE_WSE ||
				s == AITile.SLOPE_SEN || s == AITile.SLOPE_ENW;
	}

	/**
	 * Check if the given tile is almost flat.
	 * @param t Tile to check.
	 * @return True if the tile is flat or with 3 corners raised.
	 */
	/* static */ function IsTileFlatOrWithThreeCornersRaised(t)
	{
		return TileUtils.IsSlopeFlatOrWithThreeCornersRaised(AITile.GetSlope(t));
	}

	/**
	 * Make the given tile flat.
	 * @param t Tile to level.
	 * @param h Desired height.
	 * @return true if succeed, false else.
	 */
	/* static */ function LevelTile(t, h)
	{
		local s = AITile.GetSlope(t);
		local max = AITile.GetMaxHeight(t);
		if (s == AITile.SLOPE_FLAT && max == h) return true;
		if (max >= h + 2) return false;
		if (max == h + 1) {
			if (s == AITile.SLOPE_FLAT) s = AITile.SLOPE_ELEVATED;
			return AITile.LowerTile(t, s);
		}
		if (AITile.GetMinHeight(t) <= h - 2) return false;
		return AITile.RaiseTile(t, AITile.GetComplementSlope(s));
	}

	/**
	 * Terraform land and make station(or other thing) construction possible.
	 * @param tile The tile where to build station.
	 * @param pure_flat True means that land will be made flat. False means that
	 *  result can contain sloped tiles but foundation will be "flat enough"
	 *  to place station with respect to build_on_slopes game parameter.
	 * @param station_x_size Desired flat area x-size.
	 * @param station_y_size Desired flat area y-size.
	 * @param test Flag showing that function must run in test mode.
	 * @return true If land was made flat, false otherwise.
	 */
	/* static */ function MakeStationFoundation(tile, pure_flat, station_x_size, station_y_size, test)
	{
		if (station_x_size == 0 || station_y_size == 0) return false;

		station_x_size -= station_x_size / abs(station_x_size);
		station_y_size -= station_y_size / abs(station_y_size);
		/*if (station_x_size > 0) {
			station_x_size--;
		} else if (station_x_size < 0) {
			station_x_size++;
		} else {
			return false;
		}*/

		/*if (station_y_size > 0) {
			station_y_size--;
		} else if (station_y_size < 0) {
			station_y_size++;
		} else {
			return false;
		}*/

		local list = AITileList();

		list.AddRectangle(tile, tile + AIMap.GetTileIndex(station_x_size, station_y_size));
		list.Valuate(AITile.GetMaxHeight);

		/* h_list: list of pairs [height, number of flat tiles in list with that height] */
		local h_list = AIList();
		local total_flat = 0;
		foreach (t, h in list) {
			if (AITile.GetSlope(t) == AITile.SLOPE_FLAT) {
				total_flat++;
				local count = h_list.GetValue(h);
				count == 0 ? h_list.AddItem(h, 1) : h_list.SetValue(h, count + 1);
			}
		}
		if (total_flat == 0) return false;

		/* base_height: height of max number of flat tiles in list */
		local base_height = h_list.Begin();
		local tiles_with_base_height = 0;
		foreach (h, tiles_with_height_h in h_list) {
			if (tiles_with_height_h > tiles_with_base_height) {
				tiles_with_base_height = tiles_with_height_h;
				base_height = h;
			}
		}
		if (total_flat - tiles_with_base_height > 10) return false;

		foreach (t, h in list) {
			if (AITile.GetSlope(t) == AITile.SLOPE_FLAT) {
				if (h > base_height + 1 || h < base_height - 1) return false;
			} else {
				local top_h = AITile.GetMaxHeight(t);
				local bottom_h = AITile.GetMinHeight(t);
				if (top_h > base_height + 1 || bottom_h < base_height - 1) return false;
			}
		}

		local build_on_slopes = pure_flat ? false :
			AIGameSettings.GetValue("construction.build_on_slopes") != 0;
		local mode = test ? AITestMode() : null;
		foreach (t, dummy in list) {
			local s = AITile.GetSlope(t);
			local h = AITile.GetMaxHeight(t);

			if (s == AITile.SLOPE_FLAT) {
				if (h == base_height - 1) {
					if (AITile.RaiseTile(t, AITile.SLOPE_ELEVATED)) continue;
					if (!build_on_slopes) return false;
					/* With build on slopes and raised tile corner(s) we can build */
					/* So we failed only when tile remained flat */
					if (AITile.GetSlope(t) == AITile.SLOPE_FLAT) return false;
				}
				if (h == base_height + 1) {
					if (!AITile.LowerTile(t, s)) return false;
				}
			} else {
				local top_h = h;
				if (top_h == base_height + 1) {
					if (!AITile.LowerTile(t, s)) return false;
				} else if (!build_on_slopes) {
					/* Else mean top_h <= base_height, and tile not flat =>
					/* Tile bottom height < base_height => we need to raise this tile */
					if (!AITile.RaiseTile(t, AITile.GetComplementSlope(s))) return false;
				}
			}
		}

		return true;
	}

	/* static */ function DiagDistance(t1, t2)
	{
		local dx = abs(AIMap.GetTileX(t1) - AIMap.GetTileX(t2));
		local dy = abs(AIMap.GetTileY(t1) - AIMap.GetTileY(t2));
		if (dx < dy) return dy + (0.41 * dx).tointeger();
		return dx + (0.41 * dy).tointeger();
	}

	/**
	 * Bresenham's line algorithm.
	 */
	/* static */ function GetLine(x1, y1, x2, y2)
	{
		local dx = x2 - x1, dy = y2 - y1; 
		local sign_x = 1, sign_y = 1;
		if (x1 >= x2) {
			sign_x = -1;
			dx = -dx;
		}
		if (y1 >= y2) {
			sign_y = -1;
			dy = -dy;
		}

		local error = dx - dy;

		local result = [];
		do {
			result.append({x = x1, y = y1});

			local error2 = error * 2;

			if(error2 > -dy) {
				error -= dy;
				x1 += sign_x;
			}
			if(error2 < dx) {
				error += dx;
				y1 += sign_y;
			}
		} while (x1 != x2 || y1 != y2);

		return result;
	}

	static function GetBaseDirectionToTile(tile_from, tile_to)
	{
		local dx = AIMap.GetTileX(tile_from) - AIMap.GetTileX(tile_to);
		local dy = AIMap.GetTileY(tile_from) - AIMap.GetTileY(tile_to);

		if (dx >= dy) {
			return dx > -dy ? BaseDirection.BD_NE : BaseDirection.BD_SE;
		} else {
			return dx < -dy ? BaseDirection.BD_SW : BaseDirection.BD_NW;
		}
	}
}
