require("util.nut");

class AirManager {
	cargo_passengers = null;
	lastBlacklistClearYear = null;
	static airportTypes = [AIAirport.AT_INTERCON, AIAirport.AT_INTERNATIONAL, AIAirport.AT_METROPOLITAN,
						  AIAirport.AT_LARGE, AIAirport.AT_COMMUTER, AIAirport.AT_SMALL,
						  AIAirport.AT_HELISTATION, AIAirport.AT_HELIDEPOT, AIAirport.AT_HELIPORT
	                      ];
	static smallAirportTypes = [AIAirport.AT_COMMUTER, AIAirport.AT_SMALL];
	static helicopterAirportTypes = [AIAirport.AT_HELISTATION, AIAirport.AT_HELIDEPOT, AIAirport.AT_HELIPORT];
	
	static airportTypeMaxVehicles = {
		[AIAirport.AT_HELIPORT]=3,
		[AIAirport.AT_HELIDEPOT]=3,
		[AIAirport.AT_HELISTATION]=5,
		[AIAirport.AT_SMALL]=2,
		[AIAirport.AT_COMMUTER]=3,
		[AIAirport.AT_LARGE]=4,
		[AIAirport.AT_METROPOLITAN]=6,
		[AIAirport.AT_INTERNATIONAL]=8,
		[AIAirport.AT_INTERCON]=10
	}
	townBlacklist = [];
	airportUpgradeBlacklist = [];
	static function GetNumberOfPlanes(stationID);
	static function SupportsExtraPlanes(stationID);
	static function IsClearOrAirport(tile, station);
	static function CanEngineLand(engineID, airportType);
	static function CanVehicleLand(vehicleID, airportType);
	static function GetNumHangarsStation(stationID);
	function BuildAirRoute();
	function UpgradePlanes();
	function BuildNewAirport();
	function BuildAirport(townID);
	function UpgradeAirports();
	function ConnectAirports(air1, air2);
	function BuildAirportType(townID, type);
	function ReplacePlane(planeID);
	function Save();
	function Load(data);
	
	constructor() {
		lastBlacklistClearYear = Util.GetYear();
		local cargos = AICargoList();
		for (local cargo = cargos.Begin(); !cargos.IsEnd(); cargo = cargos.Next()) {
			if (AICargo.HasCargoClass(cargo, AICargo.CC_PASSENGERS)) {
				if (cargo_passengers != null){
					AILog.Warning("multiple passenger cargos detected, airplanes will not work correctly");
				}
				cargo_passengers = cargo;
			}
		}
	}
}

function AirManager::GetNumberOfPlanes(stationID){
	local vehicles = AIVehicleList_Station(stationID);
	vehicles.Valuate(AIVehicle.GetVehicleType);
	vehicles.KeepValue(AIVehicle.VT_AIR);
	return vehicles.Count();
}

function AirManager::CanEngineLand(engineID, airportType){
	local planeType = AIEngine.GetPlaneType(engineID);
	switch (planeType){
		case AIAirport.PT_BIG_PLANE:
			foreach (at in AirManager.smallAirportTypes){
				if (at==airportType){
					return false;
				}
			}
			//small plane conditions apply to big lanes as well
		case AIAirport.PT_SMALL_PLANE:
			foreach (at in AirManager.helicopterAirportTypes){
				if (at == airportType){
					return false;
				}
			}
		case AIAirport.PT_HELICOPTER:
			return true;
		default:
			return false;
	}
}

function AirManager::CanVehicleLand(vehicleID, airportType){
	return AirManager.CanEngineLand(AIVehicle.GetEngineType(vehicleID), airportType);
}

function AirManager::SupportsExtraPlanes(stationID){
	local numberOfPlanes = AirManager.GetNumberOfPlanes(stationID);
	local maxNumberOfPlanes = AirManager.airportTypeMaxVehicles[AIAirport.GetAirportType(AIBaseStation.GetLocation(stationID))];
	return numberOfPlanes < maxNumberOfPlanes;
}

function AirManager::IsClearOrAirport(tile, station){
	return (AITile.IsBuildable(tile) /*&& AITile.GetSlope(tile)==AITile.SLOPE_FLAT*/) ||
	       (station != null && AIAirport.IsAirportTile(tile) && AIStation.GetStationID(tile) == station)
}

function AirManager::GetNumHangarsStation(station){
	return AIAirport.GetNumHangars(AIStation.GetLocation(station));
}

function AirManager::FindDepotNear(tile){
	local airports = AIStationList(AIStation.STATION_AIRPORT);
	airports.Valuate(GetNumHangarsStation);
	airports.RemoveValue(0);
	if (airports.IsEmpty()){
		AILog.Warning("no airports found");
	}
	airports.Valuate(AIStation.GetDistanceManhattanToTile,tile);
	airports.Sort(AIList.SORT_BY_VALUE, true);
	local depot = null;
	for (local airport = airports.Begin(); !airports.IsEnd(); airport = airports.Next()){
		depot = AIAirport.GetHangarOfAirport(AIStation.GetLocation(airport));
		if (depot != null && AIAirport.IsHangarTile(depot)){
			break;
		}
	}
	return depot;
}

function AirManager::RenamePlane(plane, from, to){
	if (!AIVehicle.IsValidVehicle(plane)){
		return false;
	}
	if (!AIVehicle.SetName(plane, Util.CapLength(AIBaseStation.GetName(from),9)+" to "+
		                            Util.CapLength(AIBaseStation.GetName(to),9))){
		for (local i=1; true; i++){
			if (AIVehicle.SetName(plane, Util.CapLength(AIBaseStation.GetName(from),8)+" to "+
		                                   Util.CapLength(AIBaseStation.GetName(to),8)+" "+i)){
				return true;
			}
		}
	}
	else {
		return true;
	}
}

function AirManager::BuildAirRoute(){
	if (UpgradeAirports()){
		return true;
	}
	
	if (Util.GetYear() > lastBlacklistClearYear + 5){
		AILog.Info("clearing blacklist...");
		lastBlacklistClearYear = Util.GetYear();
		townBlacklist=[];
		airportUpgradeBlacklist=[];
	}
	local stations = AIStationList(AIStation.STATION_AIRPORT);
	local numStations = stations.Count();
	stations.Valuate(Util.StationWaitingCargo);
	stations.KeepAboveValue(50);
	local stationsWithCargo = stations.Count();
	stations.Valuate(SupportsExtraPlanes);
	stations.KeepValue(1);
	local stations2 = AIStationList(AIStation.STATION_AIRPORT);
	stations2.Valuate(GetNumberOfPlanes);
	stations2.KeepValue(0);
	AILog.Info("numStations: "+numStations+" stations with cargo: "+stationsWithCargo + " stations without planes: "+stations2.Count());
	if (numStations>=3 && stationsWithCargo<=2 && stations2.Count()==0){
		//wait
		return false;
	}
	stations.AddList(stations2);
	if (stations.Count()>=2 && numStations >= 3){
		stations.Valuate(AIBase.RandItem); //sort randomly
		stations.Sort(AIList.SORT_BY_VALUE, true);
		local station1=stations.Begin();
		local station2=stations.Next();
		if (!ConnectAirports(AIBaseStation.GetLocation(station1), AIBaseStation.GetLocation(station2))){
			BuildNewAirport();
		}
	}
	else {
		AILog.Info("not enough stations: "+stations.Count()+"/"+numStations);
		return BuildNewAirport();
	}
}

function AirManager::UpgradePlanes(){
	local engines = AIEngineList(AIVehicle.VT_AIR);
	engines.Valuate(AIEngine.GetCapacity);
	engines.Sort(AIList.SORT_BY_VALUE, false);
	
	local planes = AIVehicleList();
	planes.Valuate(AIVehicle.GetVehicleType);
	planes.KeepValue(AIVehicle.VT_AIR);
	
	local airports = AIStationList(AIStation.STATION_AIRPORT);
	
	foreach (plane, _ in planes){
		if (Util.CapLength(AIVehicle.GetName(plane),6)=="delete"){
			continue;
		}
		foreach (en, _ in engines){
			if ((AIVehicle.GetCapacity(plane, cargo_passengers)<AIEngine.GetCapacity(en) ||
			     (AIVehicle.GetAgeLeft(plane)<AIBase.RandRange(300)))
			    && Money.ReserveMoney(AIEngine.GetPrice(en))
				&& AIEngine.GetPlaneType(AIVehicle.GetEngineType(plane))==AIEngine.GetPlaneType(en)){

				local depot = FindDepotNear(AIVehicle.GetLocation(plane));
				
				if (depot != null && AIAirport.IsHangarTile(depot)){
					local newplane = AIVehicle.BuildVehicle(depot, en);
					if (newplane != null && AIVehicle.IsValidVehicle(newplane)){
						local name = AIVehicle.GetName(plane);
						AIOrder.ShareOrders(newplane, plane);
						AIVehicle.StartStopVehicle(newplane);
						
						AIVehicle.SendVehicleToDepot(plane);
						local i=0;
						while (!AIVehicle.SetName(plane, "delete"+i)){
							i++;
						}
						
						AIVehicle.SetName(newplane, name);
						AILog.Info("Upgraded \""+name+"\" from \""+AIEngine.GetName(AIVehicle.GetEngineType(plane))+"\" to \""+AIEngine.GetName(en)+"\"");
						break;
					}
				}
			}
		}
	}
}

function AirManager::BuildNewAirport() {
	local towns = AITownList();
	foreach (town in townBlacklist){
		towns.RemoveItem(town);
	}
	//towns.Valuate(AITown.GetAllowedNoise);
	//towns.KeepAboveValue(AIAirport.GetNoiseLevelIncrease(AIAirport.AT_INTERCON));
	local stations = AIStationList(AIStation.STATION_AIRPORT);
	//stations.Valuate(AIStation.HasStationType, AIStation.STATION_AIRPORT);
	//stations.KeepValue(1);
	stations.Valuate(AIStation.GetNearestTown);
	foreach (station, town in stations){
		towns.RemoveItem(town);
		towns.Valuate(AITown.GetDistanceManhattanToTile, AIStation.GetLocation(station));
		towns.KeepAboveValue(40);
	}
	towns.Valuate(Util.GetRandomizedPopulation);
	towns.KeepAboveValue(1000);
	towns.Sort(AIList.SORT_BY_VALUE, false);
	
	local totalStations = stations.Count();
	
	stations.Valuate(Util.StationWaitingCargo);
	stations.KeepAboveValue(100);
	
	local stations2=AIStationList(AIStation.STATION_AIRPORT);
	stations2.Valuate(GetNumberOfPlanes);
	stations2.KeepValue(0);
	stations2.Valuate(Util.ConstValue, 1000);
	stations.AddList(stations2);
	
	if (totalStations > 0 && (stations.Count()<2||stations.Count()<totalStations/3)){
		AILog.Info("not building, valid stations: "+stations.Count()+", total stations: "+totalStations);
		return false;
	}
	
	local biggestTown = towns.Begin();
	for (local biggestTown = towns.Begin(); !towns.IsEnd(); biggestTown=towns.Next()){
		AILog.Info("Building airport in "+AITown.GetName(biggestTown));
		local air1 = BuildAirport(biggestTown);
		if (air1 != null && AIMap.IsValidTile(air1)){
			if (totalStations==0){
				local towns2 = AITownList();
				towns2.KeepList(towns);
				towns2.RemoveItem(biggestTown);
				towns2.Valuate(AITown.GetDistanceManhattanToTile, air1);
				towns2.KeepAboveValue(min(300, (AIMap.GetMapSizeX() + AIMap.GetMapSizeY())/3));
				towns2.Valuate(Util.GetRandomizedPopulation);
				towns2.Sort(AIList.SORT_BY_VALUE, false);
				if (towns2.IsEmpty()){
					AILog.Info("no towns found");
				}
				for (local secondTown = towns2.Begin(); !towns2.IsEnd(); secondTown = towns2.Next()){
					AILog.Info("attempting secondary airport in "+AITown.GetName(secondTown));
					local air2 = BuildAirport(secondTown);
					if (air2!=null && AIMap.IsValidTile(air2)){
						AILog.Info("Connecting "+AITown.GetName(biggestTown)+" to "+AITown.GetName(secondTown));
						if (ConnectAirports(air1, air2)){
							return true;
						}
					}
				}
			
			}
			else {
				stations.Sort(AIList.SORT_BY_VALUE, false);
				stations.Valuate(AIStation.GetDistanceManhattanToTile, air1);
				stations.KeepAboveValue(30);
				
				stations.Valuate(SupportsExtraPlanes);
				stations.KeepValue(1);
				if (stations.IsEmpty()){
					local capacity = airportTypeMaxVehicles[AIAirport.GetAirportType(air1)];
					for (local i=0; i<capacity - 1; i++){ //leave one free
						RedirectPlane(air1);
					}
					return true;
				}
				else {
					for (local biggest = stations.Begin(); !stations.IsEnd(); biggest=stations.Next()){
						if (ConnectAirports(air1, AIStation.GetLocation(biggest))){
							local capacity = airportTypeMaxVehicles[AIAirport.GetAirportType(air1)];
							for (local i=0; i<capacity -2; i++){
								RedirectPlane(air1);
							}
							return true;
						} //end if
					} //end for
				} //end else
			} //end else
		} //end if
	} //end for
	return false;
}

function AirManager::RedirectPlane(newAirportTile){
	AILog.Info("trying to redirect plane to "+AIStation.GetName(AIStation.GetStationID(newAirportTile)));
	local planes = AIVehicleList();
	planes.Valuate(AIVehicle.GetVehicleType);
	planes.KeepValue(AIVehicle.VT_AIR);
	local number1 = planes.Count();
	
	local airportType = AIAirport.GetAirportType(newAirportTile);
	
	planes.Valuate(CanVehicleLand, airportType);
	planes.KeepValue(1);
	local number2 = planes.Count();
	
	local existingPlanes = AIVehicleList_Station(AIStation.GetStationID(newAirportTile));
	planes.RemoveList(existingPlanes);
	
	if (planes.IsEmpty()){
		//No empty stations but also no planes, strange...
		AILog.Warning("no planes found, total "+number1+" correct type: "+number2);
		return false;
	}
	for (local plane = planes.Begin(); !planes.IsEnd(); plane = planes.Next()){
		local orderPos = AIBase.Chance(1,2)?0:1;
		local distance = AIOrder.GetOrderDistance(AIVehicle.VT_AIR, AIOrder.GetStopLocation(plane, 1 - orderPos), newAirportTile);
		if (AIEngine.GetMaximumOrderDistance(AIVehicle.GetEngineType(plane))==0 ||
			AIEngine.GetMaximumOrderDistance(AIVehicle.GetEngineType(plane))>distance){ //check the range
			AIOrder.RemoveOrder(plane, orderPos);
			AIOrder.InsertOrder(plane, orderPos, newAirportTile, AIOrder.OF_NONE);
			local name = AIVehicle.GetName(plane);
			RenamePlane(plane, AIStation.GetStationID(AIOrder.GetOrderDestination(plane,0)), AIStation.GetStationID(AIOrder.GetOrderDestination(plane, 1)));
			AILog.Info("redirecting plane \""+name+"\" to \""+AIVehicle.GetName(plane)+"\"");
			return true;
		}
	}
}

function AirManager::UpgradeAirports(){
	local airports = AIStationList(AIStation.STATION_AIRPORT);
	airports.Valuate(SupportsExtraPlanes);
	airports.KeepValue(0); //only full airports
	airports.Valuate(AIStation.GetCargoWaiting, cargo_passengers);
	airports.KeepAboveValue(250); //Low value for testing, ideally should probably be 500-750
	foreach (port in airportUpgradeBlacklist){
		airports.RemoveItem(port);
	}
	
	if (airports.Count() > 0){
		return UpgradeAirport(airports.Begin())
	}
}

function AirManager::ConnectAirports(air1, air2){
	AILog.Info("connecting "+AIBaseStation.GetName(AIStation.GetStationID(air1))+" to "+AIBaseStation.GetName(AIStation.GetStationID(air2)));
	local engines = AIEngineList(AIVehicle.VT_AIR);
	engines.Valuate(CanEngineLand, AIAirport.GetAirportType(air1));
	engines.KeepValue(1);
	engines.Valuate(CanEngineLand, AIAirport.GetAirportType(air2));
	engines.KeepValue(1);
	engines.Valuate(AIEngine.GetMaximumOrderDistance);
	engines.RemoveBetweenValue(5, AIOrder.GetOrderDistance(AIVehicle.VT_AIR, air1, air2));
	engines.Valuate(AIEngine.GetCapacity);
	engines.Sort(AIList.SORT_BY_VALUE, false);
	if (engines.IsEmpty()){
		AILog.Warning("no engine found");
		return false;
	}
	AILog.Info("number of engines: "+engines.Count());
	
	local depot;
	if (AIAirport.GetNumHangars(air1) >= 1){
		depot = AIAirport.GetHangarOfAirport(air1);
	}
	else if (AIAirport.GetNumHangars(air2) >= 1){
		depot = AIAirport.GetHangarOfAirport(air2);
	}
	else {
		depot = FindDepotNear(air1);
	}
	if (depot == null || !AIMap.IsValidTile(depot) || !AIAirport.IsHangarTile(depot)){
		AILog.Warning("no depot found");
		return false;
	}
	
	local vehicle = null;
	for (local engine = engines.Begin(); !engines.IsEnd(); engine = engines.Next()){
		if (Money.ReserveMoney(AIEngine.GetPrice(engine))){
			AILog.Info("Building engine: "+AIEngine.GetName(engine));
			vehicle = AIVehicle.BuildVehicle(depot, engine);
			if (!AIVehicle.IsValidVehicle(vehicle)){
				AILog.Warning("failed to build vehicle: "+AIError.GetLastErrorString());
			}
			else {
				break;
			}
		}
	}
	
	if (vehicle!=null){
		AIOrder.AppendOrder(vehicle, air1, AIOrder.OF_NONE);
		AIOrder.AppendOrder(vehicle, air2, AIOrder.OF_NONE);
		
		AIVehicle.StartStopVehicle(vehicle);
		RenamePlane(vehicle, AIStation.GetStationID(air1), AIStation.GetStationID(air2));
		AILog.Info("end");
		return true;
	}
}

function AirManager::BuildAirport(townID){
	foreach (a in airportTypes){
		if (AIAirport.IsValidAirportType(a) && Money.ReserveMoney(AIAirport.GetPrice(a))){
			local tile=BuildAirportType(townID, a, null)
			if (tile != null && AIMap.IsValidTile(tile)){
				return tile;
			}
		}
	}
	townBlacklist.push(townID);
	return null;
}

function AirManager::UpgradeAirport(airportID){
	foreach (type in airportTypes){
		if (type == AIAirport.GetAirportType(AIStation.GetLocation(airportID))){
			if (AIStation.IsAirportClosed(airportID)){
				AIStation.OpenCloseAirport(airportID);
			}
			return false;
		}
		
		else if (AIAirport.IsValidAirportType(type) && Money.ReserveMoney(AIAirport.GetPrice(1)+400)){
			
		
			if (!AIStation.IsAirportClosed(airportID)){
				AIStation.OpenCloseAirport(airportID);
				AIController.Sleep(10);
			}
			
			local newAirport = BuildAirportType(AITile.GetClosestTown(AIStation.GetLocation(airportID)), type, airportID);
			if (newAirport!=null && AIMap.IsValidTile(newAirport) && AIAirport.IsAirportTile(newAirport)){
				if (AIStation.GetStationID(newAirport)==airportID){
					if (AIStation.IsAirportClosed(airportID)){
						AILog.Info("airport closed, opening");
						AIStation.OpenCloseAirport(airportID);
					}
					return true;
					AILog.Info("upgraded airport "+AIStation.GetName(airportID));
				}
				else {
					local planes = AIVehicleList_Station(airportID);
					planes.Valuate(AIVehicle.GetVehicleType);
					planes.KeepValue(AIVehicle.VT_AIR);
					foreach (plane, _ in planes){
						for (local i=0; AIOrder.IsValidVehicleOrder(plane, i); i++){
							if (AIStation.GetStationID(AIOrder.GetOrderDestination(plane, i))==airportID){
								if (!AIOrder.RemoveOrder(plane, i) || !AIOrder.InsertOrder(plane, i, newAirport, AIOrder.OF_NONE)){
									AILog.Warning("Failed to fix order: "+AIError.GetLastErrorString());
								}
								else {
									AILog.Info("fixed order");
								}
								//Don't break in case multiple orders are to this station
							}
							else {
								//AILog.Info("ignored "+AIStation.GetName(AIStation.GetStationID(AIOrder.GetOrderDestination(plane, i))));
							}
						}
					}
					local name = AIStation.GetName(airportID);
					AIStation.SetName(airportID, "obsolete airport");
					AIStation.SetName(AIStation.GetStationID(newAirport), name);
					while (!AITile.DemolishTile(AIStation.GetLocation(airportID))) {
						if (AIError.GetLastError() == AIError.ERR_VEHICLE_IN_THE_WAY || AIError.GetLastError() ==  AIError.ERR_NOT_ENOUGH_CASH){
							AIController.Sleep(10);
						}
						else {
							AILog.Warning("Failed to demolish: "+AIError.GetLastErrorString());
						}
					}
					
					/*if (AIStation.IsAirportClosed(airportID)){
						AILog.Info("airport closed, opening");
						AIStation.OpenCloseAirport(airportID);
					}*/
					AILog.Info("upgraded airport "+name);
					
					return true;
				}
			}
			else {
				AILog.Warning("airport upgrade of "+AIStation.GetName(airportID)+" failed");
				airportUpgradeBlacklist.push(airportID);
			}
		}
	}
	if (AIStation.IsAirportClosed(airportID)){
		AIStation.OpenCloseAirport(airportID);
	}
}



function AirManager::BuildAirportType(townID, airportType, stationID){
	AILog.Info("Trying to build "+airportType);
	local location = AITown.GetLocation(townID);
	//AISign.BuildSign(location, "town");
	
	local width = AIAirport.GetAirportWidth(airportType);
	local height = AIAirport.GetAirportHeight(airportType);
	
	local tiles = Util.GetTileListRadius(location - AIMap.GetTileIndex(width/2, height/2), 20);
	tiles.Valuate(AITile.GetClosestTown);
	tiles.KeepValue(townID);
	tiles.Valuate(IsClearOrAirport, stationID);;
	tiles.KeepValue(1);
	
	local tilesBefore = tiles.Count();
	
	local cargos = AICargoList();
	cargos.Valuate(AICargo.HasCargoClass, AICargo.CC_PASSENGERS);
	cargos.KeepValue(1);
	
	foreach (cargo, value in cargos){
		if (AICargo.IsValidCargo(cargo)){
			tiles.Valuate(AITile.GetCargoAcceptance, cargo, width, height, AIAirport.GetAirportCoverageRadius(airportType));
			tiles.KeepAboveValue(80); //reject tiles with little acceptance
			//typical range seems to be 50-200 ish
		}
	}
	
	tiles.Valuate(AITile.GetCargoAcceptance, cargos.Begin(), width, height, AIAirport.GetAirportCoverageRadius(airportType));
	tiles.Sort(AIList.SORT_BY_VALUE, false);
	AILog.Info("number of tiles before: "+tilesBefore+" after: "+tiles.Count()+
				" closest: "+AITile.GetDistanceManhattanToTile(tiles.Begin(), location - AIMap.GetTileIndex(width / 2, height / 2)));
	//if (!tiles.IsEmpty()){
	//	AISign.BuildSign(tiles.Begin(), "first");
	//}
	foreach (tile, value in tiles){
	    local tiles2=AITileList();
		tiles2.AddRectangle(tile, tile + AIMap.GetTileIndex(width - 1, height - 1));
		tiles2.Valuate(IsClearOrAirport, stationID);
		tiles2.KeepValue(1);
		
		if (tiles2.Count()==(width*height)){
			local maxHeight = AITile.GetMaxHeight(tiles2.Begin());
			foreach (tile, _ in tiles2){
				if (AITile.GetMaxHeight(tile)!=maxHeight){
					maxHeight = -1;
					break;
				}
			}
			if (maxHeight == -1){
				continue;
			}
			
			foreach (tile, _ in tiles2){
				if (!AITile.IsBuildable(tile)){
					AILog.Warning("attempting to demolish "+tile);
					while (!AITile.DemolishTile(tile)){
						AIController.Sleep(10);
						if (AIError.GetLastError()!=AIError.ERR_VEHICLE_IN_THE_WAY){
							AILog.Warning("clearing tile failed: "+AIError.GetLastErrorString());
						}
					}
				}
			}
			if (AIAirport.BuildAirport(tile, airportType, stationID!=null?stationID:AIStation.STATION_NEW) ||
				(AIError.GetLastError()==AIError.ERR_STATION_TOO_SPREAD_OUT && AIAirport.BuildAirport(tile, airportType,
				AIStation.STATION_NEW))){
				AIBaseStation.SetName(AIStation.GetStationID(tile), AITown.GetName(townID));
				//AISign.BuildSign(tile, "topleft");
				AILog.Info("Airport built");
				return tile;
			}
			else if (AIError.GetLastError() != AIError.ERR_NONE) {
				AISign.BuildSign(tile, AIError.GetLastErrorString());
				AISign.BuildSign(tile + AIMap.GetTileIndex(width, height), "invalid bottom");
				return;
				AILog.Warning("failed to build airport: "+AIError.GetLastErrorString());
			}
		}
		else {
			//AISign.BuildSign(tile, number1+", "+tiles2.Count()+" ("
			//	+AITile.GetCargoAcceptance(tile, cargos.Begin(), width, height, AIAirport.GetAirportCoverageRadius(airportType))+")");
		}
	}
	AILog.Info("failed");
	return null;
}

function AirManager::ReplacePlane(planeID){
	if (AIVehicle.GetProfitLastYear(planeID)>10000 && AIOrder.IsValidVehicleOrder(planeID, 1)){
		ConnectAirports(AIOrder.GetOrderDestination(planeID, 0), AIOrder.GetOrderDestination(planeID, 1));
	}
}

function AirManager::Save() {
	return {lastBlacklistClearYear = lastBlacklistClearYear,
	        townBlacklist = townBlacklist,
			airportUpgradeBlacklist = airportUpgradeBlacklist};
}

function AirManager::Load(data) {
	if ("lastBlacklistClearYear" in data){
		lastBlacklistClearYear = data.lastBlacklistClearYear;
	}
	else {
		AILog.Warning("AirLoad: Missing field \"air.lastBlacklistClearYear\"");
	}
	
	if ("townBlacklist" in data){
		townBlacklist = data.townBlacklist;
	}
	else {
		AILog.Warning("AirLoad: Missing field \"air.townBlacklist\"");
	}
	
	if ("airportUpgradeBlacklist" in data){
		airportUpgradeBlacklist = data.airportUpgradeBlacklist;
	}
	else {
		AILog.Warning("AirLoad: Missing field \"air.airportUpgradeBlacklist\"");
	}
}