﻿require("constants.nut");
require("utils/log.nut");
require("utils/functional.nut");
require("utils/debug.nut");
require("utils/misc.nut");
require("utils/random.nut");
require("utils/route.nut");
require("utils/route_plan.nut");
require("utils/scheduler.nut");
require("utils/string.nut");

require("cargo.nut");
require("depot.nut");
require("engine.nut");
require("industry.nut");
require("town.nut");
require("transport/all.nut");
require("stations/base.nut");

require("utils/test.nut");
require("utils/pathfinder.nut");
require("utils/balance.nut");

class BorkAI extends AIController 
{
    routes = [];
    industries = [];
    
    _logger = Log.GetLogger("main");

    constructor()
    {
        routes = [];
        industries = [];
    }
    
    function Start();
    function ChooseCompanyName();
    function ManageTransports();

    function _findIndustryRoutes(cargo);
    function _findTownRoutes(cargo);
}




function BorkAI::Save()
{
    return {routes = map(routes,routeToTable)};
}



function BorkAI::Load(version,data)
{
    this.routes = map(data.routes,tableToRoute);
}


// const KMISH_PER_TILE = 664;


function GetProducingTiles(industry,cargo,stationType)
{
    Log.GetLogger("main").trace("GetProducingTiles");
    local radius = AIStation.GetCoverageRadius(stationType);
    local tiles = AITileList_IndustryProducing(industry,radius);
    tiles.Valuate(AITile.GetCargoProduction, cargo, 1, 1, radius);
    tiles.RemoveBelowValue(1);
    return tiles;
}



function GetAcceptingTiles(industry,cargo,stationType)
{
    Log.GetLogger("main").trace("GetAcceptingTiles");
    local radius = AIStation.GetCoverageRadius(stationType);
    local tiles = AITileList_IndustryAccepting(industry.Id(),radius);
    tiles.Valuate(AITile.GetCargoAcceptance, cargo, 1, 1, radius);
    tiles.RemoveBelowValue(8);
    return tiles;
}






function BorkAI::TownData(town)
{
    return Town.TownTable()[town];
    
}


function hasStationNearby(location, otherDistance, ownDistance, cargo)
{
    local radius = max(otherDistance,ownDistance);
    for(local i=-radius ; i<=radius ; ++i )
    {
        for(local j=-radius+abs(i) ; j<=radius-abs(i) ; ++j )
        {
            if (i==0 && j==0)
                continue;
            local place = location + AIMap.GetTileIndex(i,j);
	    local distance = AITile.GetDistanceManhattanToTile(location,place);
            if (AITile.IsStationTile(place))
            {
                if (distance <= ownDistance && AIStation.IsValidStation(AIStation.GetStationID(place)) && AIStation.GetCargoRating(place,cargo) > 0)
		{
                    return true;
                }
		if (distance <= otherDistance)
		{
		    return true;
		}
            }
        }
    }
    return false;
}

function TileSum(tile,delta)
{
    return AIMap.GetTileIndex( AIMap.GetTileX(tile) + delta[0],
			       AIMap.GetTileY(tile) + delta[1] );
}


function TileDiff(tile1,tile2)
{
    return [AIMap.GetTileX(tile1) - AIMap.GetTileX(tile2),
	    AIMap.GetTileY(tile1) - AIMap.GetTileY(tile2)];
}


      



function BorkAI::AlreadyRoute(from,to,cargo)
{
    foreach(route in this.routes)
    {
        if ( route.fromTown == false && route.industryFrom == from && route.cargo == cargo )
            return true;
        if ( route.fromTown && (route.cargo == cargo ) && (route.industryFrom == from && route.industryTo == to) || (route.industryFrom == to && route.industryTo==from) )
        {
            return true;
        }
    }

    return false;
}


function BorkAI::_findIndustryRoutes(cargo)
{
    _logger.debug("Finding industry routes for cargo", cargo );
    local producers = Industry.Producing(cargo);
    local consumers = Industry.Accepting(cargo);
    producers = random_subset(producers,INDUSTRY_LIMIT);
    foreach( from in producers )
    {
        if ( from.Production(cargo) == 0 )
            continue;
	foreach( to in consumers )
	{
	    if ( from == to || AlreadyRoute(from,to,cargo))
                continue;
	    foreach( transport in enabledTransports() )
	    {
		local gain = transport.EstimateGain( from, to, cargo );
		assert( from != null );
		assert( to != null);
		if ( gain > 0 ) 
		{
                    local requiredMoney = transport.MinimumMoneyRequired();
                    if(!CanSpend(requiredMoney))
                    {
                        RequireBalance(requiredMoney);
                    }
                    else 
                    {
                        yield RoutePlan(gain, from, to, false, false, cargo, transport);
                    }
		}
	    }
	}
    }
}

function findTownToTownRoutes(cargo)
{
    if (AICargo.GetTownEffect(cargo) != AICargo.TE_PASSENGERS &&
        AICargo.GetTownEffect(cargo) != AICargo.TE_MAIL)
    {
        return;
    }
    
    local towns = random_subset(Town.All(),TOWNS_LIMIT);

    foreach (id1,town1 in towns)
    {
	
	// prune out small towns.
        if (town1.LastMonthProduction(cargo) < 25)
	    continue;

        foreach(id2,town2 in Town.All())
        {
            if (id1 >= id2 )
                continue;
	    // prune out small towns.
            if (town2.LastMonthProduction(cargo) < 25)
		continue;
            
	    if ( AlreadyRoute(Town.Get(id1),Town.Get(id2),cargo))
                continue;

	    if ( town2.Acceptance(cargo).len() == 0 )
		continue;
	    foreach( transport in enabledTransports() )
	    {
		local dist = distance(town1.Location(),town2.Location());
		if ( dist > transport.MaximumDistance() || dist < transport.MinimumDistance() )
		{
                    continue;
		}            


		local fromMap = town1.Production(cargo);
		local toMap = town2.Production(cargo);
		local gain = transport.EstimateGain( town1, town2, cargo );
            
		if ( gain <= 0 )
                    continue;

                local requiredMoney = transport.MinimumMoneyRequired();
                if(!CanSpend(requiredMoney)) 
                {
                    RequireBalance(requiredMoney);
                } 
                else  
                {
		    yield RoutePlan(gain, town1, town2, true, true, cargo, transport);
                }
	    }
        }
    }
}

            
function BorkAI::_findTownRoutes(cargo)
{
    if (AICargo.GetTownEffect(cargo) == AICargo.TE_PASSENGERS )
    {
        _logger.debug("We don't send passengers from industries (for now)");
        return [];
    }

    if (AICargo.GetTownEffect(cargo) == AICargo.TE_NONE )
    {
        _logger.trace( "Cannot send cargo", AICargo.GetCargoLabel(cargo), "to a town");
        return [];
    }

    if (AICargo.GetTownEffect(cargo) == AICargo.TE_WATER )
    {
	// We treat water as cargo to an industry
        return [];
    }
    local producers = random_subset(Industry.Producing(cargo), INDUSTRY_LIMIT);
    
    foreach( from in producers )
    {
        if ( from.Production(cargo) == 0 )
            continue;
        local towns = Town.All();
        towns = filter(towns, function(to):(from,cargo){ return AlreadyRoute(from,to,cargo) == false; });
        towns = filter(towns, function(to):(cargo){ return to.Acceptance(cargo).len() > 0; });
	foreach( transport in enabledTransports() ) {
            foreach( cost in map(towns, function(to):(from,cargo,transport){
			return {gain = transport.EstimateGain(from, to, cargo),
                            from = from,
                            to = to,
                            toTown = true,
                            fromTown = false,
                            cargo = cargo,
			    transport = transport }}) )
		yield cost;
	}
    }
}




function BorkAI::FindPossibleRoute(cargo)
{
    _logger.debug("Finding routes for cargo", AICargo.GetCargoLabel(cargo));
    foreach( cost in _findIndustryRoutes(cargo) )
    {
        yield cost;
    }
    foreach( cost in _findTownRoutes(cargo) )
    {
        yield cost;
    }
    if (PASSENGERS_AND_MAIL)
    {
        foreach( cost in findTownToTownRoutes(cargo) )
            yield cost;
    }
}


function BorkAI::AddToRouteList(info)
{
    routes.append(info);
}




function BorkAI::BuildBestRoute(routePlan)
{
    local route = routePlan.transport.BuildRoute( routePlan );
    local transport = routePlan.transport;

    _logger.trace("Will transport", AICargo.GetCargoLabel(routePlan.cargo), "from", routePlan.from, " to ", routePlan.to, " via ", transport);
    
    if (route)
    {
	local nVehicles;
	if ( routePlan.fromTown )
	{
	    nVehicles = transport.EstimateNVehicles(routePlan.from, routePlan.to, routePlan.cargo );
	    local opposite = transport.EstimateNVehicles(routePlan.to, routePlan.from, routePlan.cargo );
	    if (opposite > nVehicles)
            {
                local tmp = routePlan.from;
                routePlan.from = routePlan.to;
                routePlan.to = tmp;
                nVehicles = opposite;
            }
                    
	}
	else
            nVehicles = transport.EstimateNVehicles(routePlan.from, routePlan.to, routePlan.cargo);
        route.cargo = routePlan.cargo;
        route.AddVehicle();
        AddToRouteList(route);
        return true;
    }		   
    else
    {
	_logger.trace("Unable to build route!");
        return false;
    }
}

function BorkAI::ManageTransports()
{
    CheckCurrentRoutes();
    
    local created = CreateNewRoutes();
}    


/** Return the number of vehicles present at the station
*/
function VehiclesAtStation(station)
{
    local vehicles = AIVehicleList_Station(station);

    local count = 0;
    foreach (v,i in vehicles)
    {
	if ( AIVehicle.GetLocation(v) == AIStation.GetLocation(station) )
	    count += 1;
    }
    return count;
}


function sendOldestVehicleToDepot(route)
{
    local vehicles = route.Vehicles();
    vehicles.sort( function(a,b)
                 { 
                     local age1 = AIVehicle.GetAge(a);
                     local age2 =  AIVehicle.GetAge(b);
                     if (age1==age2) return 0;
                     if (age1> age2) return -1;
                     return 1;} );
    foreach( vehicle in vehicles )
    {
        _logger.debug("Sending vehicle ", AIVehicle.GetName(vehicle), " of route ", route, " to a depot...");
	route.RetireVehicle(vehicle);
        _logger.debug("Now ",  vehicles.len() , " vehicles remain");
        return;
    }
}




function BorkAI::CheckCurrentRoutes()
{
    foreach( route in routes )
    {
        local vehiclesNeeded = route.NeededVehicles();
        local maxVehicles = route.FullCapacity();
        if (maxVehicles == 0 )
	{
            log("WARN",1,"Warning, got 0 max vehicles for route " + route );
	}
        if (vehiclesNeeded == 0 )
	{
            log("WARN",1,"Warning, got 0 vehicles needed for route " + route );
	}
        assert(maxVehicles != null);
        assert(vehiclesNeeded != null );
        
        
        if (!maxVehicles )
        {
            log("WARN",1,"Route " + route + " has no engines for cargo " + AICargo.GetCargoLabel(route.cargo));
        }
        

        if ( vehiclesNeeded > maxVehicles )
        {
            route.ExpandStations();
            log("INFO",6,"Capped route " + route + ": " + vehiclesNeeded + " > " + maxVehicles );
            vehiclesNeeded = maxVehicles;
        }
	local transport = route._transport;
        if ( route.Vehicles().len() < vehiclesNeeded  && transport.CanBuildVehicleFor( route.depot, route.cargo, route.from, route.to))
            route.AddVehicle();
        if ( route.Vehicles().len() > vehiclesNeeded+4 && transport.CanBuildVehicleFor( route.depot, route.cargo, route.from, route.to))
        {
            // AILog.Warning("Vehicles: " +route + ":" + route.Vehicles().len() + "/" + vehiclesNeeded+4 );
            sendOldestVehicleToDepot(route);
        }
    }

}

function BorkAI::CreateNewRoutes()
{
    ResetDesiredBalance();
    local cargoList = shuffle(Cargo.All());

    local routes = []
    foreach( cargo in cargoList )
    { 
	foreach (route in FindPossibleRoute(cargo._id))
	{
	    routes.append(route);
            if (routes.len() > ROUTE_SEARCH_LIMIT )
                break;

	}
        if (routes.len() > ROUTE_SEARCH_LIMIT )
            break;
	
    }
    _logger.info("Looking through ", routes.len(), " routes to find a good one");
    routes.sort( function(a,b) { if (a.gain > b.gain) return -1;
				 if (a.gain < b.gain) return 1;
				 return 0; } );
    for ( local i=0; i<routes.len(); i++ )
    {
        local selected = chanceRoulette(RANDOMNESS/100.0,routes.len());
        assert( selected < routes.len());
        if(!CanSpend(routes[selected].transport.MinimumMoneyRequired()))
	{
            RequireBalance(routes[selected].transport.MinimumMoneyRequired());
	    continue;
	}
        if (BuildBestRoute(routes[selected]))
        {
	    _logger.info("Route built");
            return true;
        }
        routes.remove(selected);
    }
    _logger.info("No routes built this time");
    return false;
}




function BorkAI::ChooseCompanyName()
{
    if (!AICompany.SetName("BorkAI")) 
    {
	local i = 2;
	while (!AICompany.SetName("BorkAI #" + i)) {
	    i = i + 1;
    }
  }
}



function BorkAI::CloseRoute(i,route)
{
    _logger.info("Closing route", route);
    foreach( vehicle in routeTrucks(route) )
    {
	_logger.trace("Retiring vehicle", vehicle);
        route.RetireVehicle(vehicle);
    }
    if (route.Vehicles().len() == 0 )
    {
	route.from.Destroy();
	route.to.Destroy();
	routes.remove(i);
    }
}

function BorkAI::DeleteBrokenRoutes(industry)
{
    foreach(i,route in routes)
    {
        if (route.industryFrom == industry || 
            (route.toTown == false && route.industryTo == industry ) ||
            (route.Vehicles() == []))
        {
            CloseRoute(i,route);
       }
    }

}


function reason(validFrom,validTo,closeFrom,closeTo)
{
    if (!validFrom)
	return "Start Industry/Town no longer valid";
    if (!validTo)
	return "End Industry/Town no longer valid";
    if (!closeFrom)
	return "Start Industry/Town production stopped";
    if (!closeTo)
	return "End Industry/Town acceptance stopped";
}

function BorkAI::DeleteAllBrokenRoutes()
{
    _logger.info("Removing all broken routes");
    foreach(i,route in routes)
    {
	local radius = AIStation.GetCoverageRadius(AIStation.STATION_TRUCK_STOP);
	
	// FIXME We should add check for town routes too...
	local validFrom = route.industryFrom.IsValid();
	local validTo = route.industryTo.IsValid();
	local closeFrom = AITile.GetCargoProduction(route.from.Location(), route.cargo, 1, radius, radius) > 0;
        local closeTo = AITile.GetCargoAcceptance(route.to.Location(), route.cargo, 1, radius, radius ) >= 8;
                                                    
        if (!validFrom || !validTo || !closeFrom || !closeTo)
        {
	    route.brokenCount += 1;
	    if (route.brokenCount >= 10 )
	    {
		_logger.warning("Closing broken route from", route.industryFrom, "to", route.industryTo, ":", reason(validFrom,validTo,closeFrom,closeTo));
		_logger.warning("Route was transporting", AICargo.GetCargoLabel(route.cargo));
		CloseRoute(i,route);
            
	    }
	    else 
	    {
		_logger.warning("Yellow flag", route.brokenCount,"for route",route,":",reason(validFrom,validTo,closeFrom,closeTo));
	    }
	}
    }
    
    foreach(vehicle,dummy in AIVehicleList())
    {
        if (AIVehicle.IsStoppedInDepot(vehicle))
            AIVehicle.SellVehicle(vehicle);
    }
}


function BorkAI::HandleEvents()
{
    while (AIEventController.IsEventWaiting()) 
    {
	local e = AIEventController.GetNextEvent();
	switch (e.GetEventType()) 
	{
	case AIEvent.ET_VEHICLE_CRASHED:
	    local ec = AIEventVehicleCrashed.Convert(e);
	    local v  = ec.GetVehicleID();
	    _logger.info("We have a crashed vehicle (" + v + ")");
            handleVehicleCrash(v);
	    break;
	case AIEvent.ET_VEHICLE_WAITING_IN_DEPOT:
	    local event = AIEventVehicleWaitingInDepot.Convert(e);
	    local vehicle = event.GetVehicleID();
	    log("INFO",5,"Selling vehicle " + vehicle);
	    AIVehicle.SellVehicle(vehicle);
	    break;
        case AIEvent.ET_INDUSTRY_CLOSE:
            local event = AIEventIndustryClose.Convert(e);
            local industry = event.GetIndustryID();
            DeleteBrokenRoutes(industry);
            break;
	default:
	}
    }
}


function BorkAI::CheckVehicleAge()
{
    _logger.info("Checking vehicle age");
    foreach(i,route in routes)
    {
        foreach( vehicle in routeTrucks(route) )
        {
            // if ( (AIVehicle.GetAge(vehicle) >= 1000) && route.AddVehicle())
            if (AIVehicle.GetAge(vehicle) >= AIVehicle.GetMaxAge(vehicle))
            {
                route.RetireVehicle(vehicle);
            }
        }
    }

}



function cleanupVehicles()
{
    local vehicles = AIVehicleList();
    foreach( v,dummy in vehicles )
    {
        if (!hasOrders(v))
        {
            log("WARN",3,"Killing lost vehicle " + v);
            sendToDepot(v);
        }
    }
}

DEBUG <- false;
ROADINESS <- 30;
RANDOMNESS <- 30;
TEST_MODE <- 1;
PASSENGERS_AND_MAIL <- 0;

function BorkAI::ReadConfiguration()
{
    DEBUG = GetSetting("Debug");
    ROADINESS = GetSetting("Roadiness");
    RANDOMNESS = GetSetting("Randomness");
    PASSENGERS_AND_MAIL = GetSetting("Passengers and mail");
    Log.GetLogger("main").SetLogLevel(GetSetting("log_level"));
    Log.GetLogger("AirTransport").SetLogLevel(GetSetting("log_level_aircrafts"));
    Log.GetLogger("RoadTransport").SetLogLevel(GetSetting("log_level_trucks"));
    Log.GetLogger("TramTransport").SetLogLevel(GetSetting("log_level_trams"));
    TRANSPORT_ENABLED[1] = GetSetting("enable_aircrafts");
    TRANSPORT_ENABLED[2] = GetSetting("enable_trams");
}


function integrityCheck()
{
    Route.IntegrityCheck();
}

function BorkAI::Start()
{
    ::Sleep <- this.Sleep;
    if(TEST_MODE)
    {
        runTests();
    }
    log("INFO",1,"Starting AI");
    ChooseCompanyName();

    local monthly = MonthlyScheduler();
    local me = this;
    monthly.SetActions([function():(me){ me.CheckVehicleAge();},
                        function():(me){ me.DeleteAllBrokenRoutes();},
                        Industry.UpdateAllStatistics,
                        integrityCheck,
                        cleanupVehicles]);
    while (true) 
    {
	ReadConfiguration();
        local balance = AICompany.GetBankBalance(AICompany.COMPANY_SELF);
        GetLoan();

        monthly.Update();

        ManageTransports();
	HandleEvents();
        RepayLoan();
	this.Sleep(10);
    }
}

       
