require("utils/map.nut");
class Town
{
    _id = null;
    _acceptance = null;
    _production = null;
    
    _acceptanceUpdated = null;
    _productionUpdated = null;
    _airports = null;
    
    _townList = {};
    _logger = null;
    
    constructor(townId)
    {
	this._id = townId;
        _acceptance = {};
        _production = {};
        
        _acceptanceUpdated = {};
        _productionUpdated = {};
        _airports = [];
	_logger = Log.GetLogger("Town");
    }

    function Airports();
};

function Town::_tostring()
{
    return AITown.GetName(this._id);
}



function Town::Population()
{
    return AITown.GetPopulation(_id);

}


function Town::Location()
{
    return AITown.GetLocation(_id);
}


function Town::All()
{
    local towns = AITownList();
    foreach(t,x in towns)
        Town.Get(t);
    return Town._townList;
}


function Town::Id()
{
    return _id;
}


function Town::IsValid()
{
    return AITown.IsValidTown(_id);

}



function Town::TownTable()
{
    return Town.All();
}



function Town::Get(id)
{
    if ( !(id in Town._townList) )
    {
        local t = Town(id);
        Town._townList[id] <- t;
    }
    return Town._townList[id];
}


const OLD_DATA_THRESHOLD = 1000;



function Town::Acceptance(cargo)
{
    if (AICargo.GetTownEffect(cargo) == AICargo.TE_NONE)
    {
        log("DEBUG",7,"Ignoring acceptance of cargo " + AICargo.GetCargoLabel(cargo));
        return {};
    }
    local now = AIDate.GetCurrentDate();
    local prev = cargo in _acceptanceUpdated ? _acceptanceUpdated[cargo] : 0;
    
    if ( now - prev > OLD_DATA_THRESHOLD && (prev == 0 || AIBase.RandRange(5) == 0 ) )
    {
        local tiles = ComputeAcceptanceMap(cargo);
        _acceptanceUpdated[cargo] <- now;
        this._acceptance[cargo] <- tiles;
    }
    return _acceptance[cargo];
}



function Town::LastMonthProduction(cargo)
{
    return AITown.GetLastMonthProduction(_id,cargo);
}



function Town::EstimateProduction(cargo)
{
    local prod = Production(cargo);
    // 49 is the number of tiles a cargo station covers
    local totalProduction = sum(prod);
    return 49.0*sum(prod)/prod.len();
}

function Town::Production(cargo)
{
    if (AICargo.GetTownEffect(cargo) != AICargo.TE_PASSENGERS &&
        AICargo.GetTownEffect(cargo) != AICargo.TE_MAIL)
    {
        return {};
    }   
    local now = AIDate.GetCurrentDate();
    local prev = cargo in _productionUpdated ? _productionUpdated[cargo] : 0;
    
    if ( now - prev > OLD_DATA_THRESHOLD && (prev == 0 || AIBase.RandRange(5) == 0 ) )
    {
        local tiles = ComputeProductionMap(cargo);
        _productionUpdated[cargo] <- now;
        this._production[cargo] <- tiles;
	
    }
    return _production[cargo];
}



function Town::ProductionAt(cargo,location)
{
    if ( location in Production(cargo) )
	return Production(cargo)[location];
    return 0;
}    

function Town::ComputeProductionMap(cargo)
{
    _logger.debug("Computing production map for town " + this + "/" + AICargo.GetCargoLabel(cargo));
    local radius = AIStation.GetCoverageRadius(AIStation.STATION_BUS_STOP);

    local eval = function(tile):(cargo,radius)
    {
        local value = AITile.GetCargoProduction(tile,cargo,1,radius, radius);
        return value;
    }

    local cond = function(tile):(cargo,radius,eval)
    {
        return AITown.IsWithinTownInfluence(this._id,tile)
//        && eval(tile) > 0; // FIXME This is isn't completely right. Maybe we should check if we have a road or a building
    }
        
    local tiles = scanArea(
        AITown.GetLocation(this._id), 
        cond,
        eval );
    
    return tiles;
}




function Town::ComputeAcceptanceMap(cargo)
{
    log("CACHE",7,"Computing acceptance for town " + this + "/" + AICargo.GetCargoLabel(cargo));
    
    local radius = AIStation.GetCoverageRadius(AIStation.STATION_BUS_STOP);

    local eval = function(tile):(cargo,radius)
    {
        local value = AITile.GetCargoAcceptance(tile,cargo,1,radius, radius);
        if (DEBUG)
        {
	    //AILog.Warning("Production of " + AICargo.GetCargoLabel(cargo) + " at " + location(tile) + " = " + value );
            AISign.BuildSign(tile,AICargo.GetCargoLabel(cargo) + " : " + value );
        }
        return value;
    }

    local cond = function(tile):(cargo,radius,eval)
    {
        return AITown.IsWithinTownInfluence(this._id,tile)
        && eval(tile) >= 8;
    }
        
    local tiles = scanArea(
        AITown.GetLocation(this._id), 
        cond,
        eval );
    
    return tiles;
}


function Town::Airports()
{
    return _airports;
}