class AirTransport extends Transport
{
    static _airportTypes = [
	    AIAirport.AT_INTERNATIONAL,
	    AIAirport.AT_METROPOLITAN,
	    AIAirport.AT_LARGE,
	    AIAirport.AT_COMMUTER,
	    AIAirport.AT_SMALL 
	];

    constructor()
    {
	_logger = Log.GetLogger("AirTransport");
    }

    function Depot();
    function MaximumDistance();
    function MinimumDistance();
    function MinimumMoneyRequired();
    function GetEngineFor(cargo, from, to);
    function EstimateDistance(from, to);
    function FindPath(from, to);
    function BuildPath(path);
    function TransportType();
    function FindOrBuildDepot(station);
    function GetDefaultFlags(vehicle);
    function GetDescription();
    function DaysAtStation();
    function Id();

    function _BuildBestAirport(town, tiles);
    function _BuildStations(routePlan, path);
    function _FindPossibleStationLocations(tiles);
    function _FindPossibleIndustryStationLocations(industry, cargo, producing);
    function _FindPossibleTownStationLocations(town, cargo, producing);
};

function AirTransport::MaximumDistance()
{
    return AIR_MAXIMUM_DISTANCE;
}


function AirTransport::MinimumDistance()
{
    // This will compensate for the fact that we don't properly handle 
    // the cost of building and the servicing time
    return AIR_MINIMUM_DISTANCE;
}


function AirTransport::MinimumMoneyRequired()
{
    return AIR_MINIMUM_MONEY_REQUIRED;
}


function AirTransport::EstimateDistance(fromTile, toTile)
{
    local fromX = AIMap.GetTileX(fromTile);
    local fromY = AIMap.GetTileY(fromTile);
    local toX = AIMap.GetTileX(toTile);
    local toY = AIMap.GetTileY(toTile);
    local dx = abs(fromX-toX);
    local dy = abs(fromY-toY);
    local dist1 = minf(dx,dy)*1.41421; // sqrt(2*min(dx,dy)^2)
    local dist2 = abs(dx-dy);
    return dist1 + dist2;
//    return AITile.GetDistanceSquareToTile(fromTile,toTile);
}


function AirTransport::FindPath(from,to)
{
    return [from, to];
}


function AirTransport::BuildPath(path)
{
    return true;
}


function planesHandled(airport, engine) 
{
    local planeType = AIEngine.GetPlaneType(engine);
    local airportType = AIAirport.GetAirportType(airport.Location());

    switch(planeType) 
    {
    case AIAirport.PT_HELICOPTER:
	switch(airportType) 
	{
	case AIAirport.AT_SMALL:
	case AIAirport.AT_LARGE:
	case AIAirport.AT_METROPOLITAN:
	case AIAirport.AT_HELIPORT:
	case AIAirport.AT_HELIDEPOT:
	    return 1;
	case AIAirport.AT_INTERNATIONAL:
	case AIAirport.AT_COMMUTER:
	case AIAirport.AT_INTERCON:
	    return 2;
	case AIAirport.AT_HELISTATION:
	    return 3;
	case AIAirport.AT_INVALID:
	    assert(false);
	    log("ERROR",1,"canHandle: invalid airport:" + airportType);
	default:
	    log("ERROR",1,"canHandle: unknown airport:" + airportType);
	}
    case AIAirport.PT_SMALL_PLANE:
	switch(airportType) 
	{
	case AIAirport.AT_HELIPORT:
	case AIAirport.AT_HELIDEPOT:
	case AIAirport.AT_HELISTATION:
	    return 0;
	case AIAirport.AT_SMALL:
	    return 2;
	case AIAirport.AT_COMMUTER:
	case AIAirport.AT_LARGE:
	case AIAirport.AT_METROPOLITAN:
	    return 3;
	case AIAirport.AT_INTERNATIONAL:
	    return 6;
	case AIAirport.AT_INTERCON:
	    return 8;
	case AIAirport.AT_INVALID:
	    log("ERROR",1,"canHandle: invalid airport:" + airportType);
	    assert(false);
	default:
	    log("ERROR",1,"canHandle: unknown airport:" + airportType);
	    return 0;
	}
	break;
    case AIAirport.PT_BIG_PLANE:
	switch(airportType) 
	{
	case AIAirport.AT_HELIPORT:
	case AIAirport.AT_HELIDEPOT:
	case AIAirport.AT_HELISTATION:
	case AIAirport.AT_SMALL:
	case AIAirport.AT_COMMUTER:
	    return 0;
	case AIAirport.AT_LARGE:
	case AIAirport.AT_METROPOLITAN:
	    return 3;
	case AIAirport.AT_INTERNATIONAL:
	    return 6;
	case AIAirport.AT_INTERCON:
	    return 8;
	case AIAirport.AT_INVALID:
	    assert(false);
	    log("ERROR",1,"canHandle: invalid airport:" + airportType);
	default:
	    log("ERROR",1,"canHandle: unknown airport:" + airportType);
	    return 0;
	}
    case AIAirport.PT_INVALID:
	log("ERROR",1,"canHandle: invalid engine:" + engine);
	return 0;
    default:
	log("ERROR",1,"canHandle: unknown engine:" + engine);
	return 0;
    }
}
function AirTransport::GetEngineFor(cargo, from, to)
{
    local canHandle = function(x):(from, to) {
	return (from == null ||planesHandled(from, x) > 0) && 
	( to == null || planesHandled(to, x) > 0 );
    };

    local engineList = AIEngineList(AIVehicle.VT_AIR);
    engineList.Valuate(AIEngine.GetCargoType);
    engineList.KeepValue(cargo);

    engineList.Valuate(canHandle);
    engineList.KeepValue(1);

    engineList.Valuate(AIEngine.GetMaxSpeed);
    engineList.Sort(AIList.SORT_BY_VALUE, false);
    

    if (engineList.Count()>0) 
    {
	return Engine(engineList.Begin(), null);
    }
    else 
    { 
	return null;
    }
}


function AirTransport::FindOrBuildDepot(station)
{
    return AIAirport.GetHangarOfAirport(station.Location());
}

function AirTransport::TransportType()
{
    return AITile.TRANSPORT_AIR;
}


function AirTransport::GetDefaultFlags(vehicle)
{
    return [];
}

function AirTransport::GetDescription() 
{
    return "air";
}


function AirTransport::DaysAtStation()
{
    return AIR_DAYS_AT_STATION;
}


function AirTransport::Id()
{
    return TransportId.air;
}


function AirTransport::_FindPossibleStationLocations(tiles)
{

    foreach (tile, value in tiles)
    {
	foreach ( type in _airportTypes )
	{
	    if (AIAirport.BuildAirport(tile,type, AIStation.STATION_NEW))
	    {
	        yield { 
                    tile= tile,
                    type= type,
                    value= value
                };
		break;
	    }
	}
    }
}

function AirTransport::_FindPossibleIndustryStationLocations(industry, cargo, producing)
{
    // FIXME: The one in RoadTransport is the same, barring the station type...
    // should we merge ?
    local test = AITestMode();
    local location = industry.Location();

    _logger.debug("Looking for a suitable station for industry", industry, "at", ::location(industry.Location()));

    local tiles = producing ? GetProducingTiles(industry.Id(),cargo,AIStation.STATION_AIRPORT) 
        : GetAcceptingTiles(industry,cargo,AIStation.STATION_AIRPORT);
    
    local result = [];
    foreach( x in _FindPossibleStationLocations( tiles )) 
    {
	result.push(x);
    }
    return result;

}

function AirTransport::_FindPossibleTownStationLocations(town, cargo, producing)
{
    _logger.info("Looking for a suitable station for town", town, "at", ::location(town.Location()));
    local test = AITestMode();
    local location = town.Location();

    local buildable = function(tile):(cargo, _airportTypes)
    {
 	foreach ( type in _airportTypes )
	{
	    if (AIAirport.BuildAirport(tile,type, AIStation.STATION_NEW))
	    {
		return [tile, type];
	    }
	}
	return null;
    }

    local tiles = producing ? town.Production(cargo) : town.Acceptance(cargo);
    _logger.trace("Got", tiles.len(), "tiles.");
    local result = {};
    foreach( tile, value in tiles)
    {
        local b = buildable(tile);
        if (b)
        {
            result[tile] <- {
                tile = tile,
                type = b[1],
                value = value
            }
        }
    }

    _logger.trace("Got", tiles.len(), "buildable tiles.");
    result = filterValues(result, function(x){ return x.value > 8; });
    _logger.trace("Got", tiles.len(), "tiles giving profit.");
    local result = makeArray(tableValues(result));
    result.sort(function(x,y) { return y.value - x.value;} );
    // FIXME: Order by type ?
    return result;

}


function AirTransport::_BuildBestAirport(town,fitnesses)
{
    foreach (f in fitnesses)
    {
	local ok = AIAirport.BuildAirport(f.tile,f.type, AIStation.STATION_NEW);
        if (ok) 
        {
	    local station = Airport(f.tile, f.tile, AIStation.STATION_AIRPORT, true);
	    town.Airports().push(station);
	    return station;
        }
    }
    return null;
}

function AirTransport::_BuildStations(routePlan, path)
{
    local vehicleType = AICargo.HasCargoClass(routePlan.cargo,AICargo.CC_PASSENGERS) ? AIRoad.ROADVEHTYPE_BUS : AIRoad.ROADVEHTYPE_TRUCK;

    local stationFrom = _BuildBestAirport(routePlan.from, path[0]);
    local stationTo = _BuildBestAirport(routePlan.to, path[1]);
    if ( stationFrom == null)
    {
        if (stationTo == null ) 
        {
            stationTo.Destroy();
        }
        return null;
    }
    if ( stationTo == null)
    {
        stationFrom.Destroy();
        return null;
    }

    return [stationFrom, stationTo];

}

