class RoadTransport extends Transport
{
    static SEARCH_HISTORY_MAX_AGE=10000;

    _recentSearches = 0;

    constructor() 
    {
	::Transport.constructor();
	_recentSearches = {};
	_logger = Log.GetLogger("RoadTransport");
    }
    function MaximumDistance();
    function MinimumDistance();
    function MinimumMoneyRequired();
    function GetEngineFor(cargo, from, to);
    function EstimateDistance(from, to);
    function FindPath(from, to);
    function BuildPath(path);
    function FindTownStation(town, cargo, producing);
    function FindOrBuildStation(industry, cargo, producing);
    function TransportType();
    function FindOrBuildDepot(station);
    function VehiclesLimit();
    function GetDefaultFlags(vehicle);
    function GetDescription();
    function DaysAtStation();
    function Id();

    function RoadType();
    function BuildStation(orig, dest, vehicleType, stationType);

    function _GetPathFinder(from, to);
    function _DeletePathFinder(from, to);
    function _SetPathFinderResult(from, to, result);
    function _FindPossibleStationLocations(tiles);
    function _FindPossibleIndustryStationLocations(industry, cargo, producing);
    function _FindPossibleTownStationLocations(town, cargo, producing);
    function _BuildStations(routePlan, path);
    function _DepotsNeedsFinalRoad();
};


function tryRoad(orig,dest)
{
    local mode = AITestMode();
    local ok = AIRoad.BuildRoad(orig,dest);
    if (!ok && AIError.GetLastError() == AIError.ERR_ALREADY_BUILT)
	ok = true;
    return ok
}

function RoadTransport::_FindPossibleStationLocations( tiles )
{
    _logger.trace("FindPossibleStationLocations");
    foreach( orig,value in tiles )
    {
	foreach (dir in shuffle(DIRECTIONS) )
	{
	    local dest = orig + dir;
	    AIRoad.SetCurrentRoadType(RoadType());
	    
	    if (tryRoad(orig,dest) &&
		BuildStation(orig, dest, AIRoad.ROADVEHTYPE_TRUCK,  AIStation.STATION_NEW ))
	    {
                AIRoad.BuildRoad(orig,dest);
		yield orig;
		break;
	    }
	}
    }
}

function RoadTransport::_FindPossibleIndustryStationLocations(industry, cargo, producing)
{
    local test = AITestMode();
    local location = industry.Location();

    _logger.debug("Looking for a suitable station for industry", industry, "at", ::location(industry.Location()));

    local tiles = producing ? GetProducingTiles(industry.Id(),cargo,AIStation.STATION_TRUCK_STOP) 
        : GetAcceptingTiles(industry,cargo,AIStation.STATION_TRUCK_STOP);
    
    local result = [];
    foreach( tile in _FindPossibleStationLocations( tiles )) 
    {
	result.push(tile);
    }
    _logger.debug("Found", result.len(),"tiles");
    return result;
}


function RoadTransport::_FindPossibleTownStationLocations(town, cargo, producing)
{
    _logger.debug("Looking for a suitable station for town", town, "at", ::location(town.Location()));
    local test = AITestMode();
    AIRoad.SetCurrentRoadType(RoadType());
    local location = town.Location();
    local roadVehicleType = AICargo.HasCargoClass(cargo,AICargo.CC_PASSENGERS) ? AIRoad.ROADVEHTYPE_BUS : AIRoad.ROADVEHTYPE_TRUCK;
    local prod = roadVehicleType == AIRoad.ROADVEHTYPE_BUS ? true : producing;
    
    local buildable = function(tile):(roadVehicleType,prod, cargo)
    {
	if (!AIRoad.HasRoadType(tile,RoadType()))
	{
	    return null;
	}
	
        if (prod && hasStationNearby(tile,2,4,cargo))
	{
            return null;
	}
        foreach (dir in shuffle(DIRECTIONS))
        {
            local result = AIRoad.BuildDriveThroughRoadStation(tile, tile+dir,roadVehicleType, AIStation.STATION_NEW);
            if (result)
            {
		return tile;
	    }
        }
        return null;
    };
    
    
    local tiles = producing ? town.Production(cargo) : town.Acceptance(cargo);
    tiles = filterKeys(tiles, buildable); 
    tiles = filterValues(tiles, function(value){ return value > 20; });
    _logger.trace("Got", tiles.len(), "tiles giving profit.");
    return tableKeys(tiles);
}


function RoadTransport::FindOrBuildDepot(station)
{
    // Find a depot near the given tile, or build it
    local pos = station.frontTile;
    _logger.info("FindOrBuildDepot: looking for a depot near", location(pos));
    local existing = DepotIds();
    local f = function(tile):(pos) { return AITile.GetDistanceManhattanToTile(tile,pos) < 10; }

    local closeOnes = makeArray(filter(existing, f)); 
    local closeFronts = map(closeOnes, function(x){ return AIRoad.GetRoadDepotFrontTile(x); });
    local alreadyOne = closeOnes.len() > 0;
    if (!alreadyOne )
    {
        _logger.info("FindOrBuildDepot: no depots close, I'll build one");
        local depot = FindPossibleDepotPosition(pos,5);
        if (!depot)
            depot = FindPossibleDepotPosition(pos,15);
        if (!depot)
            return null;
        _logger.trace("Found depot position:", ::location(depot[0]), "with front", ::location(depot[1]));
        local ok = AIRoad.BuildRoadDepot(depot[0], depot[1]);
	if (!ok) 
	{
	    _logger.debug("Unable to build depot at:", AIError.GetLastErrorString());
	    return null;
	}
        _logger.trace("Depot built");
        closeOnes = [depot[0]];
        closeFronts = [depot[1]];
    } else {
	_logger.info("FindOrBuildDepot: found", closeOnes.len());
	_logger.info(closeOnes[0]);
    }
    
    
    local path =  BuildRoad( [pos], closeFronts, ROADINESS);
    if (!path) 
    {
	if (!alreadyOne)
	    AIRoad.RemoveRoadDepot(closeOnes[0]);
        return null;
    }
    local depot = findKey(path[0], closeFronts, closeOnes)
    AIRoad.BuildRoad(path[0],depot);
    return depot;
}


function RoadTransport::MinimumMoneyRequired()
{
    return ROAD_MINIMUM_MONEY_REQUIRED;
}


function RoadTransport::MinimumDistance()
{
    return ROAD_MINIMUM_DISTANCE;
}


function RoadTransport::MaximumDistance()
{
    return ROAD_MAXIMUM_DISTANCE;
}




function routeTrucks(route)
{
    return tableKeys(AIVehicleList_Station(AIStation.GetStationID(route.from.Location())));
}


function sendToDepot(vehicle)
{
    // We remove all the order. We may lose a cargo, but then we don't risk keeping the truck in some strange loop
    // local count = AIOrder.GetOrderCount(vehicle);
    // for (local i = 0; i < count; i++)
    // {
    local result = AIOrder.RemoveOrder(vehicle, 0);
    if (!result)
    {
        AILog.Warning(AIError.GetLastErrorString());
    }
    result = AIOrder.RemoveOrder(vehicle, 0);
    if (!result)
    {
        AILog.Warning(AIError.GetLastErrorString());
    }
    
    if (!AIOrder.AppendOrder(vehicle, AIMap.TILE_INVALID, AIOrder.OF_GOTO_NEAREST_DEPOT| AIOrder.OF_STOP_IN_DEPOT))
    {

        AILog.Warning(AIError.GetLastErrorString());
    }
    

}

function hasOrders(vehicle)
{
    return AIOrder.GetOrderCount(vehicle) > 0 ;
}


function RoadTransport::GetEngineFor(cargo, from, to)
{
    local engineList = Engine.GetList(AIVehicle.VT_ROAD, RoadType());
    local isTruck = function(e) { return e.GetRoadType() == RoadType(); };
    local canTransport = function (e):(cargo) { return e.CanRefitCargo(cargo); }
    local isNotArticulated = function(e) { return !e.IsArticulated(); }
    engineList = filter( engineList, isTruck );
    engineList = filter( engineList, canTransport );
    // We can't handle articulated vehicles until we use
    // passthrough stations instead of loading bays
    engineList = makeArray(filter( engineList, isNotArticulated ));

    local cmp = function(x,y):(cargo)
    {
	local xv = x.GetCapacity(cargo)*x.GetMaxSpeed();
	local yv = y.GetCapacity(cargo)*y.GetMaxSpeed();
	return yv - xv;
    }   
    if ( engineList.len() > 0 )
    {
	foreach( e in engineList )
	{
	    e.GetCapacity(cargo);
	}
	engineList.sort(cmp)
	return engineList[0];
    }
    else
    {
	return null;
    }
}


function RoadTransport::EstimateDistance(fromTile, toTile)
{
    return AITile.GetDistanceManhattanToTile(fromTile,toTile);
}

function makeKey(from, to) 
{
    local result = "";
    foreach(i,value in from) 
    {
	result = result + value + " ";
    }
    result = result + " - ";
    foreach(i,value in to) 
    {
	result = result + value + " ";
    }
    return result;
}

function makeKey2(from, to) {
    local l1 = join(" ", map(from, location));
    local l2 = join(" ", map(to, location));

    return l1 + " - " + l2;
}

function RoadTransport::_GetPathFinder(from, to) {
    local key = makeKey(from, to);
    if (!(key in _recentSearches)) 
    {
	_logger.debug("making a new finder for", makeKey2(from, to));
	local roadiness = ROADINESS;
	local pathfinder = MyRoadPF      ()
	pathfinder.cost.no_existing_road = 120 - roadiness;
	pathfinder.InitializePath(from, to);
	local value = {
	    age = 0,
	    pf = pathfinder,
	    result = null
	};
	local searches = _recentSearches;
	_recentSearches[key] <- value;
    }
    else 
    {
	_logger.debug("Using old planner for", makeKey2(from, to));
    }
	
    local value = _recentSearches[key];
    value.age = 0;

    foreach( key, value in _recentSearches ) 
    {
	value.age += 1;
	if (value.age >= SEARCH_HISTORY_MAX_AGE) 
	{
	    _logger.debug("removing finder for path", key);
	    delete _recentSearches[key];
	}
    }
    return value;
}

function RoadTransport::_DeletePathFinder(from, to) 
{
    _logger.debug("Path finder", makeKey2(from,to), "is not good");
    local key = makeKey(from, to);
    delete _recentSearches[key];
}

function RoadTransport::_SetPathFinderResult(from, to, result)
{
    local key= makeKey(from, to);
    _recentSearches[key].result = result;
}


function RoadTransport::FindPath(from, to)
{
    local pathFinderInfo = _GetPathFinder(from, to);
    if (pathFinderInfo.result != null )
    {
	_logger.debug("Found existing result");
	return pathFinderInfo.result;
    }
    local pathfinder = pathFinderInfo.pf;

    AIRoad.SetCurrentRoadType(RoadType());

    local path = pathfinder.FindPath(ROAD_PLANNER_LIMIT);
    if (path == false )
    {
        return null;
    }
    if (path == null)
    {
	_DeletePathFinder(from, to);
    }
    else 
    {
	_SetPathFinderResult(from, to, path);
    }
    return path;
}


function RoadTransport::BuildPath(path)
{
    local start = path.GetTile();
    local end = path.GetTile();;
    while (path != null) 
    {
	if (!CanSpend(path.GetCost()))
	{
	    _logger.debug("Nice path, but can't really build it, cost is " + path.GetCost() + " and budget is " + AICompany.GetBankBalance(AICompany.COMPANY_SELF));

	    return null;
	}
	local par = path.GetParent();
	if (par != null) {
            end = par.GetTile();
	    local last_node = path.GetTile();
	    if (AIMap.DistanceManhattan(path.GetTile(), par.GetTile()) == 1 ) 
	    {
                local ok = true;
                local msg;
		if (!AIRoad.BuildRoad(path.GetTile(), par.GetTile())) 
		{
                    ok = false;
		    local err=AIError.GetLastError();
                    msg = AIError.GetLastErrorString();
                    if (err == AIError.ERR_ALREADY_BUILT)
		    {
                        ok = true;
		    }
		    else if (err == AIError.ERR_VEHICLE_IN_THE_WAY )
                    {
			for ( local i=0; i<100; i+=1)
                        {
                            Sleep(1);
                            if (AIRoad.BuildRoad(path.GetTile(), par.GetTile()))
                            {
                                ok = true;
                                break;
                            }
                        }
                    }
                    if (!ok)
                    {
                        _logger.debug("BuildPath: Unable to build path from " + location(path.GetTile()) + " to " + location(par.GetTile()) + ": " + msg);
                        return null;
                    }
                }
	    }
	    else
	    {
		/* Build a bridge or tunnel. */
		if (!AIBridge.IsBridgeTile(path.GetTile()) && !AITunnel.IsTunnelTile(path.GetTile())) {
		    /* If it was a road tile, demolish it first. Do this to work around expended roadbits. */
		    if (AIRoad.IsRoadTile(path.GetTile())) AITile.DemolishTile(path.GetTile());
		    if (AITunnel.GetOtherTunnelEnd(path.GetTile()) == par.GetTile()) {
			if (!AITunnel.BuildTunnel(AIVehicle.VT_ROAD, path.GetTile())) {
			    /* An error occured while building a tunnel. TODO: handle it. */
			}
		    } else {
			local bridge_list = AIBridgeList_Length(AIMap.DistanceManhattan(path.GetTile(), par.GetTile()) + 1);
			bridge_list.Valuate(AIBridge.GetMaxSpeed);
			bridge_list.Sort(AIList.SORT_BY_VALUE, false);
			if (!AIBridge.BuildBridge(AIVehicle.VT_ROAD, bridge_list.Begin(), path.GetTile(), par.GetTile())) {
			    /* An error occured while building a bridge. TODO: handle it. */
			}
		    }
		}
	    }
	}
	path = par;
    }

    return [start,end];
}


function RoadTransport::FindPossibleDepotPosition(tile, radius)
{
    local front = tile;
    local test = function(x):(tile,radius) { local dist = distance(x,tile); return dist < radius && dist > 1; }
    local tiles = areaAround(tile, 2, radius);
    local isBuildable = function(tile) { return AITile.IsBuildable(tile); }
    local buildableTiles = makeArray(filter(tiles, isBuildable));
    _logger.debug("FindPossibleDepotPosition: got", buildableTiles.len(),"tiles for building a depot");
    test = null;
    if (buildableTiles.len() < 1 )
    {
	return null;
    }
    local path = FindPath([front],buildableTiles);
    if (!path)
    {
        _logger.debug("Cannot find path between", tile, dumpArray(buildableTiles));
        return null;
    }
    else
    {
	_logger.trace("FindPossibleDepotPosition: found a path");
    }
    path = pathToList(path);
    return [path[0],path[1]];

}


function RoadTransport::TransportType()
{
    return AITile.TRANSPORT_ROAD;
}


function RoadTransport::BuildRoad(from,to, roadiness)
{
    _logger.debug("Building road from", dumpArray(map(from,location)), " to " + dumpArray(map(to,location)));
    local path = FindPath(from, to);
    if ( path )
    {
        local startEnd = BuildPath(path);
        if (!startEnd) 
	{
	    // Since we can't build the path, the pathfinder was using stale data, remove it
	    _DeletePathFinder(from, to);
            return null;
	}
	else 
	{
            _logger.debug("Found path from " + location(startEnd[0]) + " to " + location(startEnd[1]));
            return pathToList(path);
	}
    }
    _logger.debug("Unable to build road");
    return null;
}

function RoadTransport::VehiclesLimit()
{
    return AIGameSettings.GetValue("max_roadveh")-1;
}

function RoadTransport::GetDefaultFlags(vehicle)
{
    return [AIOrder.OF_NON_STOP_INTERMEDIATE];
}

function RoadTransport::ExpandStation(station)
{
    local id = AIStation.GetStationID(station.Location());
    local tiles = AITileList_StationType(id , station.type);
    foreach( tile, dummy in tiles )
    {
        local front = AIRoad.GetRoadStationFrontTile(tile);
        foreach( delta in shuffle(DIRECTIONS) )
        {
	    local newPos = tile + delta;
            local path = BuildRoad([newPos],[front],ROADINESS);
            if (distance(newPos,front)>1 && path)
            {
                local front = path[path.len()-2];
                _logger.info("Expanding station from", location(newPos), "to", location(front) );
                // FIXME: we need to destroy the tile, build the station and rebuild (if needed)
                // the tile. We can do better than this...
                AIRoad.RemoveRoad(newPos, front);
                if (BuildStation(newPos,front, station.type, station.Location())) {
                    AIRoad.BuildRoad(newPos, front);
                    return true;
                }
            }
        }
    }
    return null;
}


function RoadTransport::GetDescription()
{
    return "truck";
}


function RoadTransport::DaysAtStation()
{
    return ROAD_DAYS_AT_STATION;
}


function RoadTransport::Id()
{
    return TransportId.road;
}


function RoadTransport::RoadType()
{
    return AIRoad.ROADTYPE_ROAD;
}


function RoadTransport::BuildStation(orig, dest, vehicleType, stationType)
{
    return AIRoad.BuildRoadStation(orig, dest, vehicleType, stationType) ||
	AIRoad.BuildDriveThroughRoadStation(orig, dest, vehicleType, stationType);
}

/*
 * Return and array [start,second] with the first two tiles of the path
 */
function findStartingTiles(path)
{
    local first = path.GetTile();
    local second = null;
    while (path.GetParent() != null)
    {
	path = path.GetParent();
	second = first;
	first = path.GetTile();
    }
    return [first, second];
    
}

function RoadTransport::_BuildStations(routePlan, path)
{
    local vehicleType = AICargo.HasCargoClass(routePlan.cargo,AICargo.CC_PASSENGERS) ? AIRoad.ROADVEHTYPE_BUS : AIRoad.ROADVEHTYPE_TRUCK;
    local stationType = AICargo.HasCargoClass(routePlan.cargo,AICargo.CC_PASSENGERS) ? AIStation.STATION_BUS_STOP : AIStation.STATION_TRUCK_STOP;
    path = pathToList(path);
    if (path.len() < 2 )
        return null;
    local ok = BuildStation(path[0], path[1], vehicleType, AIStation.STATION_NEW);
    if (!ok) 
    {
	_logger.warning("Cannot build station:", AIError.GetLastErrorString())
	return null;
    }
    ok = BuildStation(path[path.len()-1], path[path.len()-2], vehicleType, AIStation.STATION_NEW);
    if (!ok)
    {
	_logger.warning("Cannot build station:", AIError.GetLastErrorString())
	AIRoad.RemoveRoadStation(path[0]);
	return null;
    }
    return [
	BusStation(path[path.len()-1], path[path.len()-2], stationType, true),
	BusStation(path[0], path[1], stationType, true)];
};


function Transport::DepotIds()
{
    local existing = tableKeys(AIDepotList(TransportType()));
    local t = this;
    _logger.trace("Existing:", dumpArray(map(existing,::location)));
    return filter(existing, function(x):(t) { 
	    return AIRoad.HasRoadType(x, this.RoadType()); 	
    });
}		


function RoadTransport::VehicleType()
{
    return AIVehicle.VT_ROAD;
}


function RoadTransport::TransportType()
{
    return AITile.TRANSPORT_ROAD;
}


function RoadTransport::TransportSubType()
{
    return RoadType();
}


function RoadTransport::_DepotsNeedFinalRoad()
{
    return true;
}
