/* -*- Mode: C++; tab-width: 6 -*- */ 

/*
 *
 * This file is part of Script Communication Protocol (shorten to SCP), which are libraries for OpenTTD NoAI and NoGO
 * Copyright (C) 2012 Krinn <krinn@chez.com> & Zuu
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

class	_SCPLib_Client
{
static	base		= {};
static	CompanyList	= SCPList();

static	Configuration= [1,-1,0,0,0,0,"","",null];	// 0- Number of reserved commands
									// 1- Our companyID, don't touch that
									// 2- AIAIProtocol boolean
									// 3- Error log boolean
									// 4-Info log boolean
									// 5-Events handling boolean
									// 6-Script name
									// 7-Script version
									// 8-AI loopback
		CompanyID		= null;
		ScriptName		= null;
		ScriptVersion	= null;
		LibVersion		= null;
		LibAPI		= null;
		AIAI			= null;			 // 0- disable, 1 -enable

	constructor()
		{
		_SCPLib_Client.UpdateCompanyList();
		}
}

function _SCPLib_Client::GetOurCompanyID()
// return our company ID
{
	return _SCPLib_Client.Configuration[1];
}

function _SCPLib_Client::GetTransportCompanyID(senderCompanyID, receiverCompanyID)
// return the transport CompanyID, it's an ID build with the senderID and receiverID
{
	return senderCompanyID+(receiverCompanyID << 5);
}

function _SCPLib_Client::GetSenderCompanyID(transportCompanyID)
// return the sender CompanyID out of a TransportCompanyID
{
	local receiver=(transportCompanyID >> 5);
	local sender=transportCompanyID - (receiver << 5);
	return sender;
}

function _SCPLib_Client::GetReceiverCompanyID(transportCompanyID)
// return the receiver CompanyID out of a TransportCompanyID
{
	local receiver=(transportCompanyID >> 5);
	local sender=transportCompanyID - (receiver << 5);
	return receiver;
}

function _SCPLib_Client::AddClient(_CompanyID, _LibVersion, _LibAPI, _AIAI, s_name, s_version)
// Add a client to our set
{
	if (_SCPLib_Client.ClientExist(_CompanyID))	return;
	local client=_SCPLib_Client();
	client.CompanyID=_CompanyID;
	client.LibVersion=_LibVersion;
	client.LibAPI=_LibAPI;
	client.AIAI=_AIAI;
	client.ScriptVersion=s_version;
	client.ScriptName=s_name;
	local state="GS Server";
	if (client.CompanyID != 16)	state=SCPCompany.GetName(client.CompanyID);
	local aiai=client.AIAI ? "enable" : "disable";
	_SCPLib_Message.SCPLogInfo("New Client #"+client.CompanyID+"("+state+") - SCPVersion:" +client.LibVersion+" - API:"+client.LibAPI+" - Name: "+client.ScriptName+" - version: "+client.ScriptVersion+"- AIAI: "+aiai);
	_SCPLib_Client.base[client.CompanyID] <- client;
}

function _SCPLib_Client::ClientExist(_CompanyID)
{
	return _CompanyID in _SCPLib_Client.base;
}

function _SCPLib_Client::Load(_CompanyID)
{
	return _CompanyID in _SCPLib_Client.base ? _SCPLib_Client.base[_CompanyID] : null;
}

function _SCPLib_Client::Delete(_CompanyID)
{
	if (!_SCPLib_Client.ClientExist(_CompanyID))	return false;
	delete _SCPLib_Client.base[_CompanyID];
	_SCPLib_Message.SCPLogInfo("Unregistred client #"+_CompanyID+" "+SCPCompany.GetName(_CompanyID));
}

function _SCPLib_Client::RegisterClient(message, self)
{
	_SCPLib_Message.SCPLogInfo("Received "+message.Command+":"+message.CommandSet+" msgID="+message.MessageID+" type="+message.Type+" from="+message.SenderID+" to="+message.ReceiverID);
	if (message.Data.len() < 5)	{ _SCPLib_Message.SCPLogInfo("Refuse to register that client, Data too short "+message.Data.len()); return; }
	if (message.Type)
		{
		_SCPLib_Client.AddClient(message.SenderID, message.Data[0], message.Data[1], message.Data[2], message.Data[3], message.Data[4]);
		local cmdlist=[];
		cmdlist.push(_SCPLib_Share.LibVersion);
		cmdlist.push(_SCPLib_Share.LibAPIVersion);
		cmdlist.push(_SCPLib_Client.Configuration[2]);
		cmdlist.push(_SCPLib_Client.Configuration[6]);
		cmdlist.push(_SCPLib_Client.Configuration[7]);
		local newlist=SCPList();
		newlist.AddList(_SCPLib_Command.SearchID);
		newlist.RemoveBelowValue(_SCPLib_Client.Configuration[0]);	// remove reserved command list ID
		foreach (cmdnumber, dummy in newlist)
			{
			cmdlist.push(_SCPLib_Command.SearchName[cmdnumber]);
			cmdlist.push(cmdnumber);
			}
		SCPLib.Answer(message, cmdlist);
		}
	else	{
		_SCPLib_Message.SCPLogInfo("Register answer from "+message.SenderID);
		local _LibVersion=message.Data[0];
		local _LibAPI = message.Data[1];
		local _AIAI= message.Data[2];
		local _s_name= message.Data[3];
		local _s_version = message.Data[4];
		_SCPLib_Client.AddClient(message.SenderID, _LibVersion, _LibAPI, _AIAI, _s_name, _s_version);
		for (local i=_SCPLib_Client.Configuration[0]; i < _SCPLib_Command.SearchName.len(); i++)
			_SCPLib_Command.SearchID.SetValue(i, -1); // invalidate now all commands so server must valid them, even ones in the buffer
		for (local i=5; i < message.Data.len(); i++)
			{
			local old_id=_SCPLib_Command.GetInternalCommandID(message.Data[i]);
			i++;
			if (old_id == -1)	{ _SCPLib_Message.SCPLogInfo("Unknown command handled by server "+message.Data[i-1]); continue; }
			local new_id=message.Data[i];
			_SCPLib_Command.ServerChangeID(old_id, new_id);
			}
		_SCPLib_MessagePool.DelayedCommandDraw(message.SenderID); // draw the messages kept in buffer
		}
}

function _SCPLib_Client::UpdateCompanyList()
{
	local updateList=SCPList();
	for (local i=0; i < 16; i++)	updateList.AddItem(i,-1);
	updateList.Valuate(_SCPLib_Client.ResolveCompanyID);
	if (_SCPLib_Client.CompanyList.IsEmpty())	_SCPLib_Client.CompanyList.AddList(updateList); // on init
	foreach (companyID, status in _SCPLib_Client.CompanyList)
		{
		if (status == -1 && updateList.GetValue(companyID) != -1)
			{	// new company
			_SCPLib_Message.SCPLogInfo("New company : #"+companyID+" "+SCPCompany.GetName(companyID));
			}
		if ( (status != -1 && updateList.GetValue(companyID) == -1) || (status >=20))
			{	// company disappears
			if (status >=20)	status-=20;
			_SCPLib_Message.SCPLogInfo("Company #"+companyID+" "+SCPCompany.GetName(companyID)+" is bankrupt");
			//_SCPLib_Client.Delete(companyID);
			updateList.SetValue(companyID,-1);
			}
		}
	_SCPLib_Client.CompanyList.Clear();
	_SCPLib_Client.CompanyList.AddList(updateList);
}

function _SCPLib_Client::ResolveCompanyID(_CompanyID)
{
	if (_CompanyID == 16)	return 16;
	return SCPCompany.ResolveCompanyID(_CompanyID);
}

function _SCPLib_Client::IsValidCompany(_CompanyID)
{
	local reso=_SCPLib_Client.ResolveCompanyID(_CompanyID);
	return (_SCPLib_Client.ResolveCompanyID(_CompanyID) != -1);
}

/* AIAI protocol part */

function _SCPLib_Client::CheckQueryCompany()
{
	if (!_SCPLib_Share.LibraryMode && !_SCPLib_Client.ClientAIAIReady(_SCPLib_Client.GetOurCompanyID()))
		{ _SCPLib_Message.SCPLogError("You cannot use QueryCompany when your AIAI protocol is disable, none could answer you."); return false; }
	return true;
}

function _SCPLib_Client::CheckSpeakToCompany(_CompanyID)
// Put errors if we cannot speak with another company
{
	if (_CompanyID < 0 || _CompanyID > 14)	{ _SCPLib_Message.SCPLogError("Invalid company range"); return false; }
	if (_CompanyID == _SCPLib_Client.GetOurCompanyID())	{ _SCPLib_Message.SCPLogError("You cannot speak with yourself"); return false; }
	if (!_SCPLib_Client.ClientAIAIReady(16))	{ _SCPLib_Message.SCPLogError("You cannot use AIAI protocol. GS doesn't allow it"); return false; }
	if (!_SCPLib_Client.ClientAIAIReady(_CompanyID))	{ _SCPLib_Message.SCPLogError("You cannot use AIAI protocol with that client"); return false; }
	return true;
}

function _SCPLib_Client::ClientDiscovery()
// Send to all clients information about a client, a need for AIAI implementation
{
	if (!_SCPLib_Share.LibraryMode)	return; // only the GS is allow to handle that function
}

function _SCPLib_Client::IsClientInvalid(_CompanyID)
// return true if that client is invalidate
{
	local client = _SCPLib_Client.Load(_CompanyID);
	if (client == null)	return false;
	return client.AIAI == null;
}

function _SCPLib_Client::ClientInvalidate(_CompanyID)
// invalidate a client
{
	local client = _SCPLib_Client.Load(_CompanyID);
	client.AIAI = null;
	_SCPLib_Client.Delete(_CompanyID);
}

function _SCPLib_Client::ClientAIAIReady(_CompanyID)
// return true if that client is AIAI protocol ready
{
	if (_CompanyID == _SCPLib_Client.GetOurCompanyID())	return (_SCPLib_Client.Configuration[2] == 1);
	local client = _SCPLib_Client.Load(_CompanyID);
	if (client == null)	return false;
	return (client.AIAI == 1);
}

function _SCPLib_Client::ShowClient()
{
/*	foreach (client, dummy in _SCPLib_Client.CompanyList)
		{
		local cinfo = _SCPLib_Client.Load(client);
		if (cinfo == null)	{ print("unknow company "+client); continue; }
		local aiai=(cinfo.AIAI==1);
		print("#"+client+" Name: "+cinfo.ScriptName+" Versin: "+cinfo.ScriptVersion+" SCPVersion: "+cinfo.LibVersion+" AIAI: "+aiai);
		}
*/
}

function _SCPLib_Client::AISpeak(message, self)
// A dummy function that will get overwrite with user choice
{
	local callback=_SCPLib_Client.Configuration[8];
	callback(message, self);
}
