/* -*- Mode: C++; tab-width: 6 -*- */ 

/*
 *
 * This file is part of Script Communication Protocol (shorten to SCP), which are libraries for OpenTTD NoAI and NoGO
 * Copyright (C) 2012 Krinn <krinn@chez.com> & Zuu
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

class _SCPLib_Command
{
static	base	= {};
static	SearchID = SCPList();	// use to match our internal commandID with the one use by the server
						// item = the internal commandID, value = the serverID
static	SearchName	= [];		// use to match Name+CommandSet -> internalID

		ID		= null;
		serverID	= null;
		Name		= null;
		CommandSet	= null; // match the command set name
		Answer	= null; // true if we should answer to this command (avoid server crash)
		Self		= null; // the instance that call the AddNewCommand
		loopback	= null; // the function to trigger when Command is found
}

function _SCPLib_Command::AddNewCommand(_Name, _CommandSet, self, _loopback=null)
{
	local cmd=_SCPLib_Command();
	cmd.Answer=false;
	if (_loopback != null)
		{	
		if (!typeof(_loopback) == "function")	{ _SCPLib_Message.SCPLogError(_loopback+" is not a function !"); return false; }
		cmd.Answer=true;
		}
	if (_SCPLib_Command.SearchName.len() == 256)	{ _SCPLib_Message.SCPLogError("Not adding this command, commands limit reach ! Max 256 commands allowed"); return false; }
	// command #0 is reserved to registration, but noone could take it except us.
	cmd.Name=_Name;
	cmd.CommandSet=_CommandSet;
	cmd.ID=_SCPLib_Command.SearchName.len();
	cmd.Self=self;
	cmd.loopback=_loopback;
	cmd.serverID=-1;
	_SCPLib_CommandSet.AddCommandToSet(_Name, _CommandSet);
	_SCPLib_Message.SCPLogInfo("Adding new command : "+_Name+" to set "+_CommandSet);
	_SCPLib_Command.base[cmd.ID] <- cmd;
	_SCPLib_Command.SearchID.AddItem(cmd.ID, cmd.ID); // Assume all commands as valid so we could buffer the command. An invalid command value = -1
	_SCPLib_Command.SearchName.push(cmd.CommandSet+":"+cmd.Name);
}

function _SCPLib_Command::ServerChangeID(_oldID, _newID)
{
	local cmd=_SCPLib_Command.LoadCommand(_oldID)
	if (cmd == null)	{ _SCPLib_Message.SCPLogError("The command ID "+_oldID+" doesn't exist"); return false; }
	cmd.serverID = _newID;
	_SCPLib_Command.SearchID.SetValue(_oldID, _newID);
	_SCPLib_Message.SCPLogInfo("Server use command "+_oldID+" as "+_newID);
	return true;
}

function _SCPLib_Command::GetServerID(_ID)
// convert our internal commandID to match the server commandID
{
	if (_SCPLib_Command.SearchID.HasItem(_ID))	return _SCPLib_Command.SearchID.GetValue(_ID);
	_SCPLib_Message.SCPLogInfo("No server is handling that commandID"+_ID);
	return -1;
}

function _SCPLib_Command::GetInternalCommandID(serverID)
// convert a serverID or a CommandName:CommandSetName to match our internal commandID
{
	if (typeof(serverID) == "integer")
		{
		local mlist=SCPList();
		mlist.AddList(_SCPLib_Command.SearchID);
		mlist.KeepValue(serverID);
		if (mlist.IsEmpty())	{
						_SCPLib_Message.SCPLogInfo("No server is handling that commandID"+serverID);
						return -1;
						}
		return mlist.Begin();
		}

	for (local i=0; i < _SCPLib_Command.SearchName.len(); i++)
		{
		if (serverID == _SCPLib_Command.SearchName[i])	{ return i; }
		}
	return -1;
}

function _SCPLib_Command::GetCommnandName(commandID)
// return the name of a command from its commandID
{
	local cmdname="Unknown command";
	if (!_SCPLib_Command.SearchID.HasItem(commandID))	return cmdname;
	return _SCPLib_Command.SearchName[commandID];
}

function _SCPLib_Command::GetServerCommandByte(_cmdName, _setName)
// return the ID of the command as set by the server
{
	local seek=_setName+":"+_cmdName;
	local cmdByte=-1;
	for (local i=0; i < _SCPLib_Command.SearchName.len(); i++)
		{
		if (seek == _SCPLib_Command.SearchName[i])	{ cmdByte=i; break; }
		}
	if (cmdByte == -1)	return -1;
	cmdByte = _SCPLib_Command.GetServerID(cmdByte);
	return cmdByte;
}

function _SCPLib_Command::LoadCommand(_ID)
		{
		return _ID in _SCPLib_Command.base ? _SCPLib_Command.base[_ID] : null;
		}

function _SCPLib_Command::ExecuteCommand(message)
// Execute the command we have get passing data to it
{
	local cmdServerID=_SCPLib_Command.GetInternalCommandID(message.Command);
	if (cmdServerID == -1)	{ _SCPLib_Message.SCPLogError("Cannot find this command : "+message.Command); return false; }
	local cmdobj=_SCPLib_Command.LoadCommand(cmdServerID);
	if (cmdobj == null)	{ _SCPLib_Message.SCPLogError("Unknown command : "+cmdServerID); return false; }
	message.Command = cmdobj.Name; // need to prepare it in case an answer will comes next
	message.CommandSet = cmdobj.CommandSet;
	if (!cmdobj.Answer && message.Type == false)
		{ _SCPLib_Message.SCPLogError("The command "+cmdServerID+"-"+cmdobj.Name+":"+cmdobj.CommandSet+" doesn't expect an answer"); return false; }
	//_SCPLib_Message.SCPLogInfo("executing : "+cmdobj.Name+" source="+message.SenderID+" target="+message.ReceiverID+" type="+message.Type);
	if (cmdobj.loopback == null)	{ _SCPLib_Message.SCPLogError("Attempt to execute a command that don't have any loopback : "+cmdobj.Name+". Fix your script implementation"); return false; }
	cmdobj.loopback(message, cmdobj.Self);
}

function _SCPLib_Command::ServerDiscovery()
// Announce we want communicate
{
	if (_SCPLib_Client.base.len()==0  && !_SCPLib_Share.LibraryMode)
			SCPLib.QueryServer("SCPRegister", "SCPBaseSet", _SCPLib_Share.LibVersion, _SCPLib_Share.LibAPIVersion, _SCPLib_Client.Configuration[2], _SCPLib_Client.Configuration[6], _SCPLib_Client.Configuration[7]);
}

function _SCPLib_Command::HandleHole(message, self)
// Handle a hole query from SCPGetHole command
{
	if (message.Type)
		{
		_SCPLib_Sign.DrawPanel(message.Data[0], message.SenderID, message.Data[1]);
		_SCPLib_Message.SCPLogInfo("Resending part #"+message.Data[1]+" of message "+message.Data[0]);
		SCPLib.Answer(message, message.Data[2]); // it's signPoolID to ease handling we get it back
		}
	else	{ _SCPLib_Message.SCPLogInfo("Got reply to our hole query : "+message.Data[0]); _SCPLib_Sign.SignState.SetValue(message.Data[0], 3); }
}
	
function _SCPLib_Command::Speak(_usingCommand, _usingSet, _ReceiverID, _isQuery, _rawdata, _replyID=null)
// Send a query
{
	local commandID = _SCPLib_Command.GetServerCommandByte(_usingCommand, _usingSet);
	if (commandID == -1)	{ _SCPLib_Message.SCPLogError("Server doesn't handle that command "+_usingSet+":"+_usingCommand); return -1; }
	local compName="";
	if (_ReceiverID == -1)	{ compName="GS server"; _ReceiverID=16; }
				else	compName=SCPCompany.GetName(_ReceiverID);
	_replyID=_SCPLib_MessagePool.AddMessage(_usingCommand, _usingSet, commandID, _ReceiverID, _isQuery, _rawdata, _replyID);
	if (_replyID == -1)	return -1; // Message pool report the error already
	if (!_SCPLib_Client.ClientExist(_ReceiverID) && commandID > 1)
		{
		_SCPLib_Message.SCPLogInfo("Delaying message "+_replyID+", cannot speak yet with "+_ReceiverID+"-"+compName);
		return 1;
		}
	local msg=_SCPLib_MessagePool.LoadMessage(_replyID, _ReceiverID); // load the new create message to update infos
	local tID=_SCPLib_Client.GetTransportCompanyID(msg.SenderID, msg.ReceiverID);
	local codedata=_SCPLib_Sign.DataSplit(_replyID, msg.CommandID, tID, msg.Type, msg.Data);
	if (codedata == -1)	return -1; // the coding will throw out error message on failure
	msg.Data=codedata;
	_SCPLib_Sign.DrawPanel(_replyID, _ReceiverID);
	return 0;
}
