/* -*- Mode: C++; tab-width: 6 -*- */ 

/*
 *
 * This file is part of Script Communication Protocol (shorten to SCP), which are libraries for OpenTTD NoAI and NoGO
 * Copyright (C) 2012 Krinn <krinn@chez.com> & Zuu
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

require("share.nut");
require("events.nut");
require("charcode.nut");
require("messagepool.nut");
require("commandset.nut");
require("command.nut");
require("client.nut");
require("signs.nut");

class SCPLib
{
static	ClientPool		= SCPList();
static	CmdList		= SCPList();
	
	constructor(s_name, s_version, aicall=null)
	// Initalize the Script Communication Protocol library. You must construct this class prior to using any library function.
	// @param s_name a string with short name of your script
	// @param s_version a string with the version of your script
	// @param aicall a callback function that is called when an AIAI-protocol message is received
	// @return instance
	{
	local aiai = (aicall != null);
	if (aiai && !_SCPLib_Share.LibraryMode)
		{
		if (!typeof(aicall) == "function")	{ throw(aicall+" is not a function ! You must gave a valid function to answer to other AIs"); }
		_SCPLib_Client.Configuration[8]=aicall;
		}
	if (_SCPLib_CommandSet.base.len()==0)	SCPLib.LibInit(s_name, s_version, aiai);
	}
}

// public functions: the SCP API

function SCPLib::AddCommand(_Name, _CommandSet, self, _loopback=null)
// Add a command. AI & GS.
// @param _Name a string with the name of the command
// @param _CommandSet a string with the name of the command set
// @param self optional pointer to an instance of the class where the callback function belongs to. In most cases you'll probably pass 'this' as this parameter.
// @param _loopback pointer to callback function
// @return void
{
	SCPLib.CheckInstance(null);
	_SCPLib_Command.AddNewCommand(_Name, _CommandSet, self, _loopback);
}

function SCPLib::QueryServer(_usingCommand, _usingSet, ...)
// Send a query to the server. AI ONLY.
// @param _usingCommand string with the command to send
// @param _usingSet a string with the command set where the command using_command belongs to
// @param ... data to send to the receiver of the command. The data will end up in the order as it is sent in the Data member of the message structure that the receiver callback function get.
// @return -1 on error, 0 on success, 1 delayed message
{
	SCPLib.CheckInstance(false);
	local args = [];
	for (local i=0; i < vargc; i++)	args.push(vargv[i]);
	if (vargc == 1 && typeof(vargv[0]) == "array")	args=vargv[0];
	return _SCPLib_Command.Speak(_usingCommand, _usingSet, -1, true, args);
}

function SCPLib::TellServer(_usingCommand, _usingSet, ...)
// Send a query to the server. AI ONLY.
// By using Tell* instead of Query*, we tell the receiver of this message that we are not expecting to get an answer. 
// @param _usingCommand string with the command to send
// @param _usingSet a string with the command set where the command using_command belongs to
// @param ... data to send to the receiver of the command. The data will end up in the order as it is sent in the Data member of the message structure that the receiver callback function get.
// @return -1 on error, 0 on success, 1 delayed message
{
	SCPLib.CheckInstance(false);
	local args = [];
	for (local i=0; i < vargc; i++)	args.push(vargv[i]);
	if (vargc == 1 && typeof(vargv[0]) == "array")	args=vargv[0];
	return _SCPLib_Command.Speak(_usingCommand, _usingSet, -1, null, args);
}

function SCPLib::QueryAI(_CompanyID, ...)
// Send a query to another AI controlling _CompanyID. AI ONLY.
// @param _CompanyID company ID of the company to send the command to
// @param ... data to send to the receiver of the command. The data will end up in the order as it is sent in the Data member of the message structure that the receiver callback function get.
// @return -1 on error, 0 on success, 1 delayed message
{
	SCPLib.CheckInstance(false);
	if (!_SCPLib_Client.CheckQueryCompany())	return -1;
	if (!_SCPLib_Client.CheckSpeakToCompany(_CompanyID))	return -1;
	local args = [];
	for (local i=0; i < vargc; i++)	args.push(vargv[i]);
	if (vargc == 1 && typeof(vargv[0]) == "array")	args=vargv[0];
	return _SCPLib_Command.Speak("SCPAITalk", "SCPBaseSet", _CompanyID, true, args);
}

function SCPLib::TellAI(_CompanyID, ...)
// Tell something to another AI controlling _CompanyID. AI ONLY.
// By using Tell* instead of Query*, we tell the receiver of this message that we are not expecting to get an answer. 
// @param _usingCommand string with the command to send
// @param _usingSet a string with the command set where the command using_command belongs to
// @param _CompanyID company ID of the company to send the command to
// @param ... data to send to the receiver of the command. The data will end up in the order as it is sent in the Data member of the message structure that the receiver callback function get.
// @return -1 on error, 0 on success, 1 delayed message
{
	SCPLib.CheckInstance(true);
	if (!_SCPLib_Client.CheckSpeakToCompany(_CompanyID))	return -1;
	local args = [];
	for (local i=0; i < vargc; i++)	args.push(vargv[i]);
	if (vargc == 1 && typeof(vargv[0]) == "array")	args=vargv[0];
	return _SCPLib_Command.Speak("SCPAITalk", "SCPBaseSet", _CompanyID, null, args);
}

function SCPLib::QueryCompany(_usingCommand, _usingSet, _CompanyID, ...)
// Send a query to a companyID. GS ONLY.
// @param _usingCommand string with the command to send
// @param _usingSet a string with the command set where the command using_command belongs to
// @param _CompanyID company ID of the company to send the command to
// @param ... data to send to the receiver of the command. The data will end up in the order as it is sent in the Data member of the message structure that the receiver callback function get.
// @return -1 on error, 0 on success, 1 delayed message
{
	SCPLib.CheckInstance(true);
	local args = [];
	for (local i=0; i < vargc; i++)	args.push(vargv[i]);
	if (vargc == 1 && typeof(vargv[0]) == "array")	args=vargv[0];
	return _SCPLib_Command.Speak(_usingCommand, _usingSet, _CompanyID, true, args);
}

function SCPLib::TellCompany(_usingCommand, _usingSet, _CompanyID, ...)
// Tell something a companyID. GS ONLY.
// By using Tell* instead of Query*, we tell the receiver of this message that we are not
// expecting to get an answer. 
// @param _usingCommand string with the command to send
// @param _usingSet a string with the command set where the command using_command belongs to
// @param _CompanyID company ID of the company to send the command to
// @param ... data to send to the receiver of the command. The data will end up in the order as it is sent in the Data member of the message structure that the receiver callback function get.
// @return -1 on error, 0 on success, 1 delayed message
{
	SCPLib.CheckInstance(true);
	local args = [];
	for (local i=0; i < vargc; i++)	args.push(vargv[i]);
	if (vargc == 1 && typeof(vargv[0]) == "array")	args=vargv[0];
	return _SCPLib_Command.Speak(_usingCommand, _usingSet, _CompanyID, null, args);
}

function SCPLib::Answer(message, ...)
// Send an answer to a previous query, it will be sent to who has made the query to you (the server or an AI). AI & GS.
// @param message pass the message instance that you got as parameter to your callback function
// @param ... data to be sent along with the command. The data end up as Data[0], Data[1], .. etc. on the receiver end.
// @return -1 on error, 0 on success, 1 delayed message
{
	SCPLib.CheckInstance(null);
	if (message.Type != true)	{ _SCPLib_Message.SCPLogError("You can only reply to a query"); return -1; }
	local args = [];
	for (local i=0; i < vargc; i++)	args.push(vargv[i]);
	if (vargc == 1 && typeof(vargv[0]) == "array")	args=vargv[0];
	return _SCPLib_Command.Speak(message.Command, message.CommandSet, message.SenderID, false, args, message.MessageID);
}

function SCPLib::CanSpeakWith(companyID = 16)
// @param companyID ID of company to speak with. Defaults to 16 (Game Script). AI & GS.
// @return true if we could speak with that company ID, otherwise false
{
	SCPLib.CheckInstance(null);
	if (!_SCPLib_Client.ClientExist(companyID))	return false; // unknown company, we don't speak with strangers
	if (!_SCPLib_Share.LibraryMode)
		{ // we're an AI
		if (companyID == 16 || companyID == -1)	return true;
		if (!_SCPLib_Client.ClientAIAIReady(16))	return false; // server disallow aiai protocol
		if (!_SCPLib_Client.ClientAIAIReady(companyID))	return false;	// client don't have aiai protocol enable
		}
	return true;
}

function SCPLib::Check()
// You must include a call to this function in the main loop. This function scans for incoming communication signs and if there is any incoming messages the registered callback functions are called. Check() has been designed so that when one task such as handling a incoming message has been done, it returns. If there is more work to do (eg. incoming messages), it will return true to indicate this, otherwise false. So if you have time and prioritize quick responses on SCP communication, you should do something like this: while(my_scp.Check()) { }. AI & GS.
// @return true if there are more tasks for the lib to do, otherwise false
{
	SCPLib.CheckInstance(null);
	_SCPLib_Events.HandleEvents();
	_SCPLib_Sign.BrowseSign();
	return _SCPLib_Sign.ReadCommand();
}

function SCPLib::SetEventHandling(enable)
// Enable or disable the handling of events by the lib (defaults to disabled). AI & GS.
//
// If enabled, the library will read and discard events from OpenTTD by itself when you call Check(). This is recommended if you don't call [AI/GS]EventController.GetNextEvent().
//
// If this setting is disabled, the library will get informed about events automagically when you call [AI/GS]EventController.GetNextEvent().
// @param enable true to enable handling of events by the lib, false if your script handle events itself
{
	SCPLib.CheckInstance(null);
	if (enable)	_SCPLib_Client.Configuration[5]=1;
		else	_SCPLib_Client.Configuration[5]=0;
}

function SCPLib::GetAIAIProtocol(_CompanyID)
// @return true if AIAI protocol is enabled for that company, otherwise false. AI & GS.
{
	SCPLib.CheckInstance(null);
	return _SCPLib_Client.ClientAIAIReady(_CompanyID);
}

function SCPLib::SCPLogging_Error(enable)
// @param enable true to enable SCP loggin error only in console. AI & GS.
{
	if (typeof(enable) != "bool")	return;
	local setting=0;
	if (enable)	setting=1;
	_SCPLib_Client.Configuration[3]=setting;
}

function SCPLib::SCPLogging_Info(enable)
// If you wish help us debugging what's going on, enabling this one should help us, your console will get more noise. This also force error logging if enabled. AI & GS.
// @param enable true to enable SCP loggin info in console
{
	if (typeof(enable) != "bool")	return;
	local setting=0;
	if (enable)	{ setting=1; SCPLib.SCPLogging_Error(enable); }
	_SCPLib_Client.Configuration[4]=setting;
}

function SCPLib::SCPGetCommunicationTile()
// The communication tile is where the signs used to communicate are placed. You can use this function to ignore all signs on this tile if you use signs for some other purposes. AI & GS.
// @return the tile that is use as the Communication Tile
{
	SCPLib.CheckInstance(null);
	return _SCPLib_Sign.CommunicationTile;
}

// private functions
function SCPLib::LibInit(s_name, s_version, aiai)
{
	if (typeof(aiai) != "bool")	throw("AIAI protocol must be a boolean parameter");
	local mode="AI";
	local cID=16;
	if (_SCPLib_Share.LibraryMode)	mode="GS";
						else	cID=_SCPLib_Client.ResolveCompanyID(SCPCompany.COMPANY_SELF);
	_SCPLib_Client.Configuration[1]=cID;
	local proto="enable";
	if (aiai)	{ _SCPLib_Client.Configuration[2]=1; }
		else	{ _SCPLib_Client.Configuration[2]=0; proto="disable"; }
	SCPLog.Info("Script Communcation Protocol: version: "+_SCPLib_Share.LibVersion+" - API: "+_SCPLib_Share.LibAPIVersion+" - CompanyID: "+_SCPLib_Client.GetOurCompanyID()+" - "+mode+" mode"+" - AIAIProtocol: "+proto+" Script name: "+s_name+" Script version: "+s_version);
	SCPLog.Info("You can get the source code here : http://dev.openttdcoop.org/projects/scriptlib-scp");
	_SCPLib_Command.AddNewCommand("SCPRegister", "SCPBaseSet", this, _SCPLib_Client.RegisterClient);
	_SCPLib_Command.AddNewCommand("SCPGetHole", "SCPBaseSet", this, _SCPLib_Command.HandleHole);
	_SCPLib_Command.AddNewCommand("SCPClient", "SCPBaseSet", this, _SCPLib_Client.ClientDiscovery);
	_SCPLib_Command.AddNewCommand("SCPAITalk", "SCPBaseSet", this, _SCPLib_Client.AISpeak);
	_SCPLib_Client.Configuration[0]=_SCPLib_Command.SearchName.len();
	SCPLog.Info("Reserved commands : "+_SCPLib_Client.Configuration[0]);
	local init=_SCPLib_Client();
	_SCPLib_Client.Configuration[6]=s_name;
	_SCPLib_Client.Configuration[7]=s_version;
	_SCPLib_Sign.CleanUp();
	_SCPLib_Command.ServerDiscovery(); // a one time try to register
}

function SCPLib::CheckInstance(value)
// Check SCP has been init before allowing any static call
{
	if (_SCPLib_CommandSet.base.len()==0)	throw("SCP must be init by its constructor.");
	if (value==true && !_SCPLib_Share.LibraryMode)	throw("Attempt to use a GS only function in an AI.");
	if (value==false && _SCPLib_Share.LibraryMode)	throw("Attempt to use an AI only function in an GS.");
}


