/*
	MogulAI - an artificial intelligence for OpenTTD
	Copyright (C) 2009 - 2010 Kazantsev Lev (Dezmond_snz)

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License along
	with this program; if not, write to the Free Software Foundation, Inc.,
	51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

require ("Util.nut");

const DEBUG = 0;

class RoadPathFinder
{
	pf = null;
	
	constructor(start, end, percent = 125)
	{
		pf = PathFinder(start, end, Util.CanPass, Util.GetCost, percent);
	}
	
	function Itterate( steps )
	{
		return pf.Itterate(steps);
	}
	
	function GetCurrentPaths ()
	{
		return pf.GetCurrentPaths ();
	}
}

class PathFinder
{
	MaximumBridgeLength = 0;
	MaximumTunnelLength = 0;
	MaximumPathLength = 0;
	CostBaseStep = 0;
	
	_toVisit = {};
	_toVisitList = [];
	_visited = {};
	
	_start = null;
	_end = null;
	
	_canPassQuery = null;
	_getCostQuery = null;
	
	constructor(startTile, endTile, canPassQuery, getCostQuery, percent)
	{
		if (typeof(canPassQuery) != "function") throw("'canPassQuery' has to be a function-pointer."); 
		if (typeof(getCostQuery) != "function") _getCostQuery = null;
		else _getCostQuery = getCostQuery; 
		
		_canPassQuery = canPassQuery;
		
		_start = startTile;
		_end = endTile;
		
		MaximumBridgeLength = 10;
		MaximumTunnelLength = 10;
		MaximumPathLength = AIMap.DistanceManhattan(_start, _end)*percent/100;
		CostBaseStep = 15;
		
		_toVisit = {};
		_toVisitList = [];
		_visited = {};
		
		local startNode = PathNode(_start, null, ReachableType.Normal, 0, 0);
		addPathNodeToVisitList(startNode);
	}
	
	function Itterate ( steps )
	{
		local foo = AITestMode();
		
		if ( steps == 0 )
			return false;
		
		local result = false;
		for ( local i = 0; i < steps; i++ )
		{
			if ( _toVisitList.len() == 0 )
			{
				LogDebug ( "No more tiles to visit." );
				return true; // finished!
			}
			
			local res = itterate ();
			if ( res )
				result = true;
		}
		return result;
	}
	
	function itterate()
	{
		local pn = _toVisit[ _toVisitList[ 0 ] ];
		_toVisitList.remove ( 0 );
		_toVisit.rawdelete(pn._tile);
		if ( pn._step + AIMap.DistanceManhattan(pn._tile, _end) < MaximumPathLength )
		{
			if (AITunnel.IsTunnelTile(pn._tile) && pn._reachType == ReachableType.Normal) // reuse existing tunnel
			{
				local endTile = AITunnel.GetOtherTunnelEnd(pn._tile);
				if (AIMap.IsValidTile(endTile))
				{
					local cost = pn._currentCost + CostBaseStep;
					local pn2 = PathNode (endTile, pn, ReachableType.Tunnel, pn._step + AIMap.DistanceManhattan(pn._tile, endTile) - 1, cost);
					processNewPathNode(pn2);
				}
			}
			else if (AIBridge.IsBridgeTile(pn._tile) && pn._reachType == ReachableType.Normal) // reuse existing bridge
			{
				local endTile = AIBridge.GetOtherBridgeEnd(pn._tile);
				if (AIMap.IsValidTile(endTile))
				{
					local cost = pn._currentCost + CostBaseStep;
					local pn2 = PathNode (endTile, pn, ReachableType.Bridge, pn._step + AIMap.DistanceManhattan(pn._tile, endTile) - 1, cost);
					processNewPathNode(pn2);
				}
			}
			else
			{
				local par = pn._parentNode == null ? AIMap.TILE_INVALID : pn._parentNode._tile;
				if (lookForTunnel(pn, par))
				{}
				if (lookForBridge(pn, par))
				{}
				
				local list = Util.GetAdjacentTiles ( pn._tile );
				foreach (nextTile,_ in list)
				{
					if ( nextTile == par ) continue;
					if ( _visited.rawin ( nextTile ) ) continue;
					if ( !_canPassQuery ( pn._tile, nextTile, par ) ) continue;
					if (pn._reachType != ReachableType.Normal || (pn._parentNode != null && pn._parentNode._reachType == ReachableType.Tunnel))
						if (Util.GetDirection(par, pn._tile) != Util.GetDirection(pn._tile, nextTile) ) continue; // keep direction on exit from bridge/tunnel
					
					local cost = pn._currentCost + CostBaseStep;
					if ( _getCostQuery != null )
						cost += _getCostQuery ( pn._tile, nextTile, par );
					
					local pn2 = PathNode ( nextTile, pn, ReachableType.Normal, pn._step + 1, cost );
					
					processNewPathNode(pn2);
				} // foreach (adjTile,_ in list)
			} // else
		} // if ( pn._step < MaximumPathLength )
		
		if ( _visited.rawin ( _end ) )
			return true;
		else
			return false;
	}
	
	function lookForTunnel(pn, par)
	{
		if (pn._reachType != ReachableType.Normal || pn._step < 3)
			return false;
		
		if (pn._parentNode == null || pn._parentNode._reachType != ReachableType.Normal) // do not connect tunnels and/or bridges directly!
			return false;
		
		if (AITile.GetSlope(pn._tile) == AITile.SLOPE_FLAT)
			return false;
		
		// check for possible tunnel
		local end = AITunnel.GetOtherTunnelEnd(pn._tile);
		if (!AIMap.IsValidTile(end) || end == _end)
			return false;
		
		if ( Util.GetDirection(par, pn._tile) == Util.GetDirection(pn._tile, end) ) // keep direction
		{
			if (AIMap.DistanceManhattan(pn._tile, end) > MaximumTunnelLength)
				return false;
			if (!AITunnel.BuildTunnel(AIVehicle.VT_ROAD, pn._tile))
				return false;
			
			local cost = pn._currentCost + CostBaseStep;
			if ( _getCostQuery != null )
				cost += _getCostQuery ( pn._tile, end, par );
			
			local pn2 = PathNode ( end, pn, ReachableType.Tunnel, pn._step + AIMap.DistanceManhattan(pn._tile, end) - 1, cost );
			processNewPathNode(pn2);
			return true;
		} // if ( end != _end
		return false;
	}
	
	function lookForBridge(pn, par)
	{
		if (pn._reachType != ReachableType.Normal || pn._step < 3)
			return false;

		if (pn._parentNode == null || pn._parentNode._reachType != ReachableType.Normal) // do not connect tunnels and/or bridges directly!
			return false;
		
		local dir = Util.GetDirection(par, pn._tile);
		if (!Util.IsSuitableForFlatBridge(pn._tile, dir))
			return false;
		local h = AITile.GetMaxHeight(pn._tile);
		
		for (local i = 1; i < MaximumBridgeLength; i++)
		{
			local end = pn._tile + dir * i;
			if (!AIMap.IsValidTile(end) || end == _end)
				return false;
			
			if (!Util.IsSuitableForFlatBridge(end, -dir))
			{
				if (AITile.GetMaxHeight(end) >= h)
					return false;
				continue;
			}
			
			local bList = AIBridgeList_Length (i+1);
			if (bList.Count() == 0 || !AIBridge.BuildBridge(AIVehicle.VT_ROAD, bList.Begin(), pn._tile, end))
				return false;
			// seems good!
			local cost = pn._currentCost + CostBaseStep;
			if ( _getCostQuery != null )
				cost += _getCostQuery ( pn._tile, end, par );
			
			local pn2 = PathNode ( end, pn, ReachableType.Bridge, pn._step + AIMap.DistanceManhattan(pn._tile, end) - 1, cost );
			processNewPathNode(pn2);
			return true;
		} // for (local i = 1
		
		return false;
	}
	
	function addPathNodeToVisitedList ( pn )
	{
		_visited.rawset ( pn._tile, pn );
//		local abc = AIExecMode();
//		AISign.BuildSign(pn._tile, pn._step+" "+pn._currentCost); 
//		AISign.BuildSign(pn._tile, ""+pn._reachType);
	}
	function addPathNodeToVisitList(pn)
	{
		_toVisitList.append(pn._tile);
		_toVisit.rawset(pn._tile, pn);
	}
	function processNewPathNode(newPn)
	{
		if ( _visited.rawin ( newPn._tile ) )
		{
			/*if ( _visited[ newPn._tile ].GetTotalCost() <= newPn.GetTotalCost() )
				return;
			else if (_visited[ newPn._tile ]._reachType == ReachableType.Normal && newPn._reachType == ReachableType.Normal &&
					newPn._parentNode._reachType == ReachableType.Normal && (_visited[ newPn._tile ]._nextNode == null || _visited[ newPn._tile ]._nextNode._reachType == ReachableType.Normal))
			{
				local par = newPn._parentNode._tile;
				if (_visited[ newPn._tile ]._nextNode == null || _canPassQuery ( newPn._tile, _visited[ newPn._tile ]._nextNode._tile, newPn._parentNode._tile ))
				{
					local cost = newPn._parentNode._currentCost + CostBaseStep;
					if ( _getCostQuery != null )
						cost += _getCostQuery ( newPn._parentNode._tile, _visited[ newPn._tile ]._tile, newPn._parentNode._parentNode._tile );
					
					_visited[ newPn._tile ]._parentNode = newPn._parentNode;
					_visited[ newPn._tile ]._currentCost = cost;
					_visited[ newPn._tile ]._step = newPn._step;
				}
				return;
			}
			else//*/
				return;
		}
		
		addPathNodeToVisitedList(newPn);
		
		if (newPn._tile == _end)
		{
			LogDebug ( "Reached destination!" );
			return;
		}
		
		if (_toVisit.rawin(newPn._tile))
		{
			if ( _toVisit[newPn._tile]._currentCost > newPn._currentCost )
				_toVisit.rawset(newPn._tile, newPn);
		}
		else
			addPathNodeToVisitList(newPn);
	}
	
	function GetCurrentPaths ()
	{
		if ( !_visited.rawin ( _end ) )
			return null;
		//return _visited[ _end ];
		local path = [];
		local node = _visited[ _end ];
		local last = null;
		while (node != null)
		{
			if (	last != null &&
					node._parentNode != null &&
					last._reachType == ReachableType.Normal &&
					node._reachType == ReachableType.Normal &&
					node._parentNode._reachType == ReachableType.Normal &&
					Util.GetDirection(last._tile, node._tile) == Util.GetDirection(node._tile, node._parentNode._tile) )
			{}
			else
				path.append(node);
			
			last = node;
			node = node._parentNode;
		}
		path.reverse();
		return path;
	}
	
	function LogInfo(msg)
	{
		AILog.Info("PathFinder: "+msg);
	}
	function LogWarning(msg)
	{
		AILog.Warning("PathFinder: "+msg);
	}
	function LogError(msg)
	{
		AILog.Error("PathFinder: "+msg);
	}
	
	function LogDebug(msg)
	{
		if (DEBUG)
			AILog.Info("(DBG) PathFinder: "+msg);
	}
}
