/*
	MogulAI - an artificial intelligence for OpenTTD
	Copyright (C) 2009 - 2010 Kazantsev Lev (Dezmond_snz)

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License along
	with this program; if not, write to the Free Software Foundation, Inc.,
	51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

require ("Util.nut");
require ("DummyPathFinder.nut");
require ("PathFinder.nut");
require ("PathBuilder.nut");

const DEBUG = 0;

class IndustryPair
{
	Source = null;
	Dest = null;
	Cargo = null;
	constructor(source, dest, cargo)
	{
		Source=source;
		Dest=dest;
		Cargo = cargo;
	}
	function GetRaw()
	{
		local raw = {};
		raw.rawset("src", Source);
		raw.rawset("dst", Dest);
		raw.rawset("crg", Cargo);
		return raw;
	}
	function FromRaw(table)
	{
		local src = table.rawget("src");
		local dst = table.rawget("dst");
		local crg = table.rawget("crg");
		return IndustryPair(src, dst,crg);
	}
}

class RouteBuildManager
{
	_routeManager = null;
	_damnAuthorityManager = null;
	
	_maxDist = 100;
	_minDist = 80;
	_currentPair = null;
	
	_sourceStation = null;
	_destStation = null;
	_depot1 = null;
	_depot2 = null;
	
	_pathFinder = null;
	_path = null;
	_pathBuilder = null;
	_roadCost = -1;
	_vehicleInTheWayWaitTime = -1;
	
	_engine = null;
	_vehicle = null;
	_count = -1;
	
	// task parameters
	_currentState = 0; // initialize
	_nextStepTime = 0;
	
	_startTime = null;
	_srcSign = -1;
	_dstSign = -1;
	
	constructor(routeManager, damnAuthorityManager)
	{
		_routeManager = routeManager;
		_damnAuthorityManager = damnAuthorityManager;
		_nextStepTime = AIController.GetTick();
		_maxDist = 100;
		_minDist = 90;
	}
	
	function MakeStep()
	{
		if (_nextStepTime > AIController.GetTick()) // waiting...
			return;
		
		switch(this._currentState)
		{
			case 0: // Reset
				this.Init();
				break;
			case 1: // Look good service candidate
				this.LookGoodIndustryPairs();
				break;
			case 2:
				this.PrepareService();
				break;
			case 3: // Try to find path
				this.FindPath();
				break;
			case 4:
				this.BuildPath();
				break;
			case 5: // build road
				this.BuildRoad();
				break;
			case 6: // build stations and depots
				this.BuildStationsDepots();
				break;
			case 7: // build vehicles
				this.CreateVehicle();
				break;
			case 8:
				this.BuildVehicles();
				break;
			case 99: // special state: destroy all we already built in this run before reseting...
				this.DestroyCurrentAlreadyBuilt();
				break;
			default:
				LogWarning("Unknown state!!! Reseting...");
				this.Init();
				break;
		}
	}
	
	function Init() // step 0
	{
		LogInfo("Initializing...");
		
		if (AISign.IsValidSign(_srcSign))	AISign.RemoveSign(_srcSign);
		if (AISign.IsValidSign(_dstSign))	AISign.RemoveSign(_dstSign);
		_currentPair = null;
		
		_sourceStation = null;
		_destStation = null;
		_depot1 = null;
		_depot2 = null;
		
		_pathFinder = null;
		_pathBuilder = null;
		_path = null;
		_roadCost = -1;
		
		_vehicleInTheWayWaitTime = -1;
		
		_engine = null;
		_vehicle = null;
		_count = -1;
		
		AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_ROAD);
		_currentState = 1; // now we can start work!
		_nextStepTime = 0;
	}
	
	function LookGoodIndustryPairs() // step 1
	{
		local vehicleList = AIVehicleList();
		local vehicleCount = vehicleList.Count();
		local vehicleMax = AIGameSettings.GetValue("vehicle.max_roadveh");
		if (vehicleCount >= vehicleMax-10)
		{
			_nextStepTime = AIController.GetTick()+100;
			return;
		}
		
		local pairList = [];
		local sourceList = GetSourceIndustryList();
		sourceList.Valuate(__productionValuator);
		sourceList.Sort(AIAbstractList.SORT_BY_VALUE, AIAbstractList.SORT_DESCENDING);
		
		foreach (source,_ in sourceList)
		{
			local cargoList = AICargoList_IndustryProducing(source);
			
			foreach (cargo,_ in cargoList)
			{
				if (_routeManager.CheckIsIndustryAlreadyManaged(source, cargo)) continue;
				if (AIIndustry.GetLastMonthProduction(source, cargo) <= 0) continue;
				
				local percent = AIIndustry.GetLastMonthTransportedPercentage(source, cargo);
				if (percent > 90) continue; // more 90%!! There is really good service already...
				
				local destList = AIIndustryList_CargoAccepting(cargo);
				destList.Valuate(AIIndustry.IsValidIndustry);
				destList.KeepValue(1);
				
				destList.Valuate(AIIndustry.GetDistanceManhattanToTile, AIIndustry.GetLocation(source));
				destList.RemoveBelowValue(this._minDist);
				destList.RemoveAboveValue(this._maxDist);
				destList.Sort(AIAbstractList.SORT_BY_VALUE, AIAbstractList.SORT_DESCENDING);
				
				foreach (dest,_ in destList)
				{
					if (Util.GetAcceptingTileList(dest, cargo).Count() == 0) continue;
					
					pairList.append(IndustryPair(source, dest, cargo));
				} // for (local dest
				
				/*if (AICargo.GetTownEffect(cargo) != AICargo.TE_NONE) // needed for towns?
				{
					destList = AITownList();
					destList.Valuate(Util.TownAcceptCargo, cargo);
					destList.KeepValue(1);
				}*/
				
			} // for (local cargo
			
			if (pairList.len() > 100) // don't see for too many pairs
				break;
		} // for ( local source
		
		if (pairList.len() > 0)
		{
			LogDebug("Total count of suitable pairs is "+pairList.len());
			local rnd = AIBase.RandRange(pairList.len()>10?10:pairList.len());
			local pair = pairList[rnd];
			LogInfo("Found Suitable Industry Pair. Cargo is "+AICargo.GetCargoLabel(pair.Cargo));
			
			this._currentPair = pair;
			this._currentState = 2; // we can begin build service!
		}
		else
		{
			this._currentPair = null;
			if (increaseDist())
				LogInfo("No suitable industry pairs... ("+this._maxDist+")");
		}
	}
	
	function PrepareService()
	{
		if (!LookForStationPlaces(_currentPair.Source, _currentPair.Dest, _currentPair.Cargo))
		{
			_currentState = 0; // initialize
			return;
		}
		
		if (AIController.GetSetting("RouteBuildSigns"))
		{
			if (AISign.IsValidSign(_srcSign))	AISign.RemoveSign(_srcSign);
			if (AISign.IsValidSign(_dstSign))	AISign.RemoveSign(_dstSign);
			_srcSign = AISign.BuildSign(_sourceStation, "!src");
			_dstSign = AISign.BuildSign(_destStation, "!dst");
		}
		
		LogInfo("Trying to find path...");
		_startTime = AIDate.GetCurrentDate();
		
		local dummyPf = DummyRoadPathFinder(_sourceStation, _destStation);
		local path = dummyPf.TryFindPath();
		if (path != null)
		{
			_pathBuilder = PathBuilder(path);
			LogInfo("Dummy path found in "+(AIDate.GetCurrentDate() - _startTime)+" days.");
			_currentState = 4; // build path
			return;
		}
		LogInfo("Dummy path NOT found. Will look for path more wisely... So WAIT!!!");
		preparePathFinder();
		_currentState = 3; // find path
	}
	
	function FindPath()
	{
		local result = _pathFinder.Itterate( 200 );
		if (!result)
			return; // continue pathfinder work...
		
		local path = _pathFinder.GetCurrentPaths ();
		if (path == null)
		{
			LogInfo("Path NOT found.");
			increaseDist();
			_currentState = 0; // initialize
		}
		else
		{
			_pathBuilder = PathBuilder(path);
			LogInfo("Path found in "+(AIDate.GetCurrentDate() - _startTime)+" days.");
			//_pathFinder.Itterate( 500 ); // itterate 500 times more to find better path
			_currentState = 4; // build path
		}
	}
	
	function BuildPath()
	{
		local vehicleList = AIVehicleList();
		local vehicleCount = vehicleList.Count();
		local vehicleMax = AIGameSettings.GetValue("vehicle.max_roadveh");
		if (vehicleCount >= vehicleMax-10)
		{
			LogInfo("I already have too many vehicles. Creating new route is not needed.");
			_currentState = 0; // initialize
			return;
		}
		local result = _pathBuilder.CheckPath();
		if (result == false) // path is invalid or broken
		{
			LogInfo("Check path failed! Trying to find better path...");
			_currentState = 2; // find path
			return;
		}
		
		if (_roadCost != result)
		{
			if (_roadCost == -1)
			{
				LogInfo("Preparing to build road...");
				LogDebug ("Road cost is "+result);
			}
			if (!FundsManager.CheckMoneyConstruction(result, false))
			{
				_nextStepTime = AIController.GetTick() + 5;
				if (_roadCost == -1)
					LogInfo ("Not enough money to build road."); // TODO: Check if we have moneymaker!!!
				_roadCost = result;
				return;
			}
			_roadCost = result;
		}
		else if (!FundsManager.CheckMoneyConstruction(result, false))
		{
			_nextStepTime = AIController.GetTick() + 5;
			return;
		}
		
		_currentState = 5; // building road
		LogInfo ("Building road...");
	}
	
	function BuildRoad()
	{
		local result = _pathBuilder.BuildRoadStep();
		if (result == PathBuilder.RESULT_ERROR)
		{
			_currentState = 0; // initialize
			return;
		}
		if (result == PathBuilder.RESULT_DONE)
			_currentState = 6; // build depots and stations
	}
	
	function BuildStationsDepots()
	{
		_depot2 = BuildDepot(_sourceStation);
		_depot1 = BuildDepot(_destStation);
		
		if (_depot1 == null && _depot2 == null)
		{
			LogWarning("No depots built!");
			_currentState = 99; // clear unfinished route
			return;
		}
		
		local errMsg = buildStation(_sourceStation); 
		if (errMsg==null)
			LogDebug("Built source station.");
		else
		{
			LogError("Source station at "+_sourceStation+" not built: "+errMsg);
			_currentState = 99; // clear unfinished route
			return;
		}
		
		local errMsg = buildStation(_destStation); 
		if (errMsg==null)
			LogDebug("Built dest station.");
		else
		{
			LogError("Dest station at "+_destStation+" not built: "+errMsg);
			_currentState = 99; // clear unfinished route
			return;
		}
		_currentState = 7; // build RV's
	}
	
	function DestroyCurrentAlreadyBuilt() // state 99
	{
		if (_sourceStation != null && AIMap.IsValidTile(_sourceStation) && AIStation.IsValidStation(AIStation.GetStationID(_sourceStation)))
			AITile.DemolishTile(_sourceStation);
		_sourceStation = null;
		
		if (_destStation != null && AIMap.IsValidTile(_destStation) && AIStation.IsValidStation(AIStation.GetStationID(_destStation)))
			AITile.DemolishTile(_destStation);
		_destStation = null;
		
		if (_depot1 != null && AIMap.IsValidTile(_depot1) && AIRoad.IsRoadDepotTile(_depot1))
			AITile.DemolishTile(_depot1);
		_depot1 = null;
		
		if (_depot2 != null && AIMap.IsValidTile(_depot2) && AIRoad.IsRoadDepotTile(_depot2))
			AITile.DemolishTile(_depot2);
		_depot2 = null;
		
		_currentState = 0; // initialize
	}
	
	function CreateVehicle()
	{
		_engine = Util.FindBestEngine(AIRoad.ROADTYPE_ROAD, _currentPair.Cargo);
		if (_engine == null)
		{
			LogWarning("Good engine for "+AICargo.GetCargoLabel(_currentPair.Cargo)+" not found!");
			_currentState = 99; // clear unfinished route
			return false;
		}
		
		_currentState = 8; // build some vehicles
		LogInfo("Building vehicles...");
		BuildVehicles();
	}
	
	function BuildVehicles()
	{
		if (_vehicle == null)
		{
			if (!FundsManager.CheckMoneyForEngine(_engine, 1, false))
			{
				_nextStepTime = AIController.GetTick() + 5;
				return;
			}
			
			if (_depot2 != null)
				_vehicle = AIVehicle.BuildVehicle(_depot2, _engine);
			else
				_vehicle = AIVehicle.BuildVehicle(_depot1, _engine);
			
			if (!AIVehicle.IsValidVehicle(_vehicle))
			{
				LogError("Vehicle with engine "+AIEngine.GetName(_engine)+" not bought! "+AIError.GetLastErrorString());
				_currentState = 99; // clear unfinished route
				return;
			}
			
			if (AIVehicle.GetCapacity(_vehicle, _currentPair.Cargo) == 0)
				AIVehicle.RefitVehicle(_vehicle, _currentPair.Cargo);
			
			AIOrder.AppendOrder(_vehicle, _sourceStation, AIOrder.AIOF_NON_STOP_INTERMEDIATE|AIOrder.AIOF_FULL_LOAD_ANY);
			AIOrder.AppendOrder(_vehicle, _destStation, AIOrder.AIOF_NON_STOP_INTERMEDIATE|AIOrder.AIOF_NO_LOAD);
			
			local group = AIGroup.CreateGroup(AIVehicle.GetVehicleType(_vehicle));
			AIGroup.SetName(group, AICargo.GetCargoLabel(_currentPair.Cargo)+": "+_sourceStation+" -> "+_destStation);
			AIGroup.MoveVehicle(group, _vehicle);
			
			local prod = AIIndustry.GetLastMonthProduction(_currentPair.Source, _currentPair.Cargo) * (100.0-AIIndustry.GetLastMonthTransportedPercentage (_currentPair.Source, _currentPair.Cargo)) / 100;
			if (_count == -1)
				_count = CalculateVehiclesCountForIndustry(_vehicle, _currentPair.Cargo, prod, AITile.GetDistanceManhattanToTile(_sourceStation, _destStation));
			
			if (_count > 10)
				_count = 10;
			
			_count--;
			return;
		}
		
		if (_count > 0)
		{
			if (!FundsManager.CheckMoneyForEngine(_engine, 1, false))
				return;
			local dupl = AIVehicle.CloneVehicle(AIVehicle.GetLocation(_vehicle), _vehicle, true);
			if (!AIVehicle.IsValidVehicle(dupl))
				LogError("Vehicle not bought: "+AIError.GetLastErrorString());
			else
				AIVehicle.StartStopVehicle(dupl);
			_count --;
			return;
		}
		
		AIVehicle.StartStopVehicle(_vehicle);
		
		_routeManager.AddNewRoute(_currentPair.Source, _currentPair.Dest, _sourceStation, _destStation, _currentPair.Cargo, (_depot2 != null? _depot2: _depot1));
		LogInfo("Service route finished.");
		increaseMaxDist();
		_nextStepTime = AIController.GetTick()+10; // don't build next one sometime
		_currentState = 0; // service route finished!
	}
	
	/*********************/
	/*****Util Functions*****/
	/*********************/
	function GetSourceIndustryList()
	{
		local list = AIIndustryList();
		
		list.Valuate(AIIndustry.IsValidIndustry);
		list.KeepValue(1);
		
		list.Valuate(__isOnWater);
		list.KeepValue(0);
		
		list.Valuate(__hasProduction);
		list.KeepValue(1);
		
		return list;
	}
	
	function __isOnWater(industry) { return AIIndustryType.IsBuiltOnWater(AIIndustry.GetIndustryType(industry)); }
	function __hasProduction(industry)
	{
		local cargoList = AICargoList_IndustryProducing(industry);
		if (cargoList.Count() == 0)
			return false;
		
		foreach (cargo,_ in cargoList)
		{
			if (AIIndustry.GetLastMonthProduction(industry, cargo) > 0)
				return true;
		}
		return false;
	}
	function __productionValuator(industry)
	{
		local cargoList = AICargoList_IndustryProducing(industry);
		if (cargoList.Count() == 0)
			return 0;
		
		local max = 0;
		foreach (cargo,_ in cargoList)
		{
			local prod = AIIndustry.GetLastMonthProduction(industry, cargo);
			if (prod > max)
				max = prod;
		}
		return max;
	}
	
	function LookForStationPlaces(source, dest, cargo)
	{
		local town = null;
		local rating = null;
		
		_sourceStation = null;
		_destStation= null;

		_sourceStation = lookForStationPlace(source, dest, cargo, true);
		if (_sourceStation == null)
		{
			LogInfo("Good place for source station not found.");
			return false;
		}
		
		town = AITile.GetClosestTown(_sourceStation);
		rating = AITown.GetRating(town, AICompany.ResolveCompanyID(AICompany.COMPANY_SELF));
		if (rating != AITown.TOWN_RATING_NONE && rating < AITown.TOWN_RATING_POOR)
		{
			_damnAuthorityManager.AddDamnAuthority(town, AITown.TOWN_RATING_POOR);
			LogInfo ("Damn "+AITown.GetName(town)+" authority refuses to allow build source station!");
			return false;
		}
		
		_destStation = lookForStationPlace(dest, source, cargo, false);
		if (_destStation == null)
		{
			LogInfo("Good place for destination station not found.");
			return false;
		}
		
		town = AITile.GetClosestTown(_destStation);
		rating = AITown.GetRating(town, AICompany.ResolveCompanyID(AICompany.COMPANY_SELF));
		if (rating != AITown.TOWN_RATING_NONE && rating < AITown.TOWN_RATING_POOR)
		{
			_damnAuthorityManager.AddDamnAuthority(town, AITown.TOWN_RATING_POOR);
			LogInfo ("Damn "+AITown.GetName(town)+" authority refuses to allow build dest station!");
			return false;
		}
		
		return true;
	}
	
	function lookForStationPlace(indSource, indDest, cargo, isSource)
	{
		local tileS = AIIndustry.GetLocation(indSource);
		local tileD = AIIndustry.GetLocation(indDest);
		local tileList =  null;
		if (isSource)
			tileList = Util.GetProducingTileList(indSource, cargo);
		else
			tileList = Util.GetAcceptingTileList(indSource, cargo);
		
		tileList.Valuate(AITile.IsBuildable);
		tileList.KeepValue(1);
		
		tileList.Valuate(AITile.GetDistanceManhattanToTile, tileD);
		tileList.Sort(AIAbstractList.SORT_BY_VALUE, AIAbstractList.SORT_ASCENDING); 
		
		while (tileList.Count() > 0)
		{
			if (AIBase.RandRange(10) == 0 || tileList.Count() == 1)
				return tileList.Begin();
			else
				tileList.RemoveTop(1);
		}
		return null;
	}
	
	function preparePathFinder()
	{
		_pathFinder = RoadPathFinder(_sourceStation, _destStation);
	}
	
	function buildStation(baseTile)
	{
		if (!AIMap.IsValidTile(baseTile))
			return "base tile is invalid!";
		
		local adjTiles = Util.GetAdjacentTiles(baseTile);
		foreach (nextTile,_ in adjTiles)
		{
			if (!AIMap.IsValidTile(nextTile) || !AIRoad.AreRoadTilesConnected(nextTile, baseTile)) continue;
			
			local errMsg = Util.BuildStationAt(baseTile, nextTile);
			if (errMsg == null)
				return null;
			else
			{
				if (AIError.GetLastError() == AIError.ERR_LOCAL_AUTHORITY_REFUSES)
					_damnAuthorityManager.AddDamnAuthority(AITile.GetClosestTown(baseTile), AITown.TOWN_RATING_POOR);
				
				return errMsg;
			}
		}
		
		return "no suitable place found.";
	}
	
	function BuildDepot(startTile)
	{
		local step = 0;
		local built = false;
		local last = null;
		local current = startTile;
		while(step < 15 && !built)
		{
			local tiles = Util.GetRoadNeighbours(current);
			
			if (tiles.Count() == 0) break;
			if ((tiles.Count() == 1 && tiles.Begin() == last) || step >= 5)
			{
				local aTiles = Util.GetAdjacentTiles(current);
				aTiles.Valuate(AITile.IsBuildable);
				aTiles.KeepValue(1);
				aTiles.Valuate(AIRoad.IsRoadTile);
				aTiles.KeepValue(0);
				
				foreach (dep,_ in aTiles)
				{
					if (Util.CanBuildDepot(current, dep))
					{
						if (Util.BuildDepotStatic(dep, current))
							return dep;
					}
				}
			}
			
			local rnd = AIBase.RandRange(tiles.Count());
			local tile = MogulAI.GetItemAtIndex(tiles, rnd);
			if (tile == last)
			{
				if (tiles.Count() != rnd+1)
					tile = tiles.Next();
				else
					tile = tiles.Begin();
			}
			
			last = current;
			current = tile;
			step++;
		}
		return null;
	}
	
	function CalculateVehiclesCountForIndustry(vehicle, cargo, production, distance)
	{
		local count = 1;
		local capacity = AIVehicle.GetCapacity(vehicle, cargo);
		local speed = AIEngine.GetMaxSpeed(AIVehicle.GetEngineType(vehicle));
		
		local timesPerMonth = 30 / (30.0 * (distance * 2 / speed) + 6); // by 3 days for loading and unloading
		local monthlyCargoAmount = capacity * timesPerMonth;
		
		count = (production / monthlyCargoAmount).tointeger();
		
		if (count < 1)
			count = 1;
		
		return count;
	}
	
	function Save()
	{
		local table = {};
		
		table.rawset("maxdist", _maxDist);
		table.rawset("mindist", _minDist);
		table.rawset("nextsteptime", _nextStepTime - AIController.GetTick());
		
		if (this._currentState < 6)
		{
			table.rawset("state", 0);
		}
		else
		{
			table.rawset("state", _currentState);
			table.rawset("srcst", _sourceStation);
			table.rawset("dstst", _destStation);
			
			table.rawset("pair", _currentPair.GetRaw());
			
			if (_depot1 != null)
				table.rawset("depot1", _depot1);
			if (_depot2 != null)
				table.rawset("depot2", _depot2);
			
			if (_engine != null)
				table.rawset("engine", _engine);
			
			if (_vehicle != null)
				table.rawset("vehicle", _vehicle);
			
			table.rawset("count", _count);
		}
		return table;
	}
	function Load(version, data)
	{
		if (data.rawin("maxdist"))
			_maxDist = data.rawget("maxdist");
		if (data.rawin("mindist"))
			_minDist = data.rawget("mindist");
		if (data.rawin("nextsteptime"))
			_nextStepTime = AIController.GetTick() + data.rawget("nextsteptime");
		
		if (data.rawin("state"))
			_currentState = data.rawget("state");
		else
		{
			_currentState = 0;
			return;
		}
		
		if (_currentState == 0)
			return;
		
		_currentState = data.rawget("state");
		_sourceStation = data.rawget("srcst");
		_destStation = data.rawget("dstst");
		
		_currentPair = IndustryPair.FromRaw(data.rawget("pair"));
		_count = data.rawget("count");
		
		if (data.rawin("depot1"))
			_depot1 = data.rawget("depot1");
		if (data.rawin("depot2"))
			_depot2 = data.rawget("depot2");

		if (data.rawin("engine"))
			_engine = data.rawget("engine");
		if (data.rawin("vehicle"))
			_vehicle = data.rawget("vehicle");
	}
	
	function getRoadCost(start, end)
	{
		local foo = AITestMode();
		local cost = AIAccounting();
		AIRoad.BuildRoad(start, end);
		return cost.GetCosts();		
	}
	
	function increaseMaxDist()
	{
		if (this._maxDist < 400)
			this._maxDist += 10;
		if (this._minDist < 350)
			this._minDist += 10;
	}
	function increaseDist()
	{
		if (this._maxDist < 400)
		{
			this._maxDist += 5;
			return true;
		}
		else if (this._minDist > 30)
		{
			this._minDist -= 5;
			return true;
		}
		return false;
	}
	
	function LogInfo(msg)
	{
		AILog.Info("RouteBuildManager: "+msg);
	}
	function LogWarning(msg)
	{
		AILog.Warning("RouteBuildManager: "+msg);
	}
	function LogError(msg)
	{
		AILog.Error("RouteBuildManager: "+msg);
	}
	
	function LogDebug(msg)
	{
		if (DEBUG)
			AILog.Info("(DBG) RouteBuildManager: "+msg);
	}
}
