/*
	MogulAI - an artificial intelligence for OpenTTD
	Copyright (C) 2009 - 2010 Kazantsev Lev (Dezmond_snz)

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License along
	with this program; if not, write to the Free Software Foundation, Inc.,
	51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

require("Route.nut");

const DEBUG = 0;

class RouteManager
{
	_routeList = [];
	_checkRoutesTime = null;
	
	constructor()
	{
		_checkRoutesTime = AIController.GetTick();
		_routeList = [];
	}
	
	function MakeStep()
	{
		CheckRoutes();
		CheckStatueBuild();
	}
	
	function AddNewRoute(source, dest, sourceStation, destStation, cargo, depot)
	{
		local route = Route(source, dest, sourceStation, destStation, cargo, depot);
		this.AddRoute(route);
	}
	
	function AddRoute(route)
	{
		_routeList.append(route);
	}
	
	function DeleteRoute(route)
	{
		for (local i=_routeList.len()-1; i>= 0; i--)
		{
			if (_routeList[i] == route)
				_routeList.remove(i);
		}
	}
	
	function CheckRoutes()
	{
		if (_checkRoutesTime > AIController.GetTick()) return;
		
//		if (_routeList.len() > 0)
//			LogDebug("Checking routes..."+_routeList.len());
		
		for (local i=_routeList.len()-1; i>= 0; i--)
		{
			if (_routeList[i].IsClosed())
				_routeList.remove(i);
		}
		
		for (local i=0; i< _routeList.len(); i++)
		{
			_routeList[i].SellVehicles();
			if (!_routeList[i].CheckIsRouteValid())
				_routeList[i].CloseRoute();
			else if (_routeList[i].CheckVehiclesReplace())
			{/*_routeList[i].ProcessVehicleReplacment();*/}
			else if (_routeList[i].CheckCargoWaiting())
				_routeList[i].AddVehicles(1);
			else
				_routeList[i].CheckQueuedVehicles();
		}
		_checkRoutesTime = AIController.GetTick()+10;
	}
	
	function CheckIsIndustryAlreadyManaged(industry, cargo)
	{
		for (local i=0; i< _routeList.len(); i++)
		{
			if (_routeList[i].GetSourceIndustry() == industry && _routeList[i].GetCargo() == cargo)
				return true;
		}
		return false;
	}
	
	function CheckIsIndustryAlreadySupplied(industry, cargo)
	{
		for (local i=0; i< _routeList.len(); i++)
		{
			if (_routeList[i].GetDestIndustry() == industry && _routeList[i].GetCargo() == cargo)
				return true;
		}
		return false;
	}
	
	function CheckStatueBuild()
	{
		if (AICompany.GetLoanAmount() == 0 && AICompany.GetBankBalance(AICompany.COMPANY_SELF) > 200000)
		{		
			local townList = AITownList();
			townList.Valuate(AITown.HasStatue);
			townList.RemoveValue(1);
			townList.Valuate(Util.ValuateZero);
			
			local stationList = AIStationList(AIStation.STATION_ANY);
			stationList.Valuate(AIStation.GetNearestTown);
			
			foreach (station, town in stationList)
			{
				if (townList.HasItem(town))
					townList.SetValue(town, townList.GetValue(town) + 1);
			}
			
			townList.KeepAboveValue(0);
			if (townList.Count() > 0)
			{
				townList.Sort(AIAbstractList.SORT_BY_VALUE, AIAbstractList.SORT_DESCENDING);
				local town = townList.Begin();
				if (AITown.PerformTownAction(town, AITown.TOWN_ACTION_BUILD_STATUE))
					LogInfo("Built a statue in " + AITown.GetName(town));
			}
		}
	}
	
	function NotifyVehicleUnprofitable(vehicle)
	{
		for (local i=0; i< _routeList.len(); i++)
		{
			if (_routeList[i].IsVehicleThere(vehicle))
			{
				_routeList[i].VehicleUnprofitable(vehicle);
				return;
			}
		}
	}
	
	function Save()
	{
		local table = {};
		table.rawset("checktime",_checkRoutesTime - AIController.GetTick());
		for (local i=0; i< _routeList.len(); i++)
		{
			table.rawset(i, _routeList[i].GetRaw());
		}
		return table;
	}
	
	function Load(version, data)
	{
		if (_routeList == null)
			_routeList = [];
		
		local i = 0;
		while (data.rawin(i))
		{
			local r = Route.FromRaw(version, data.rawget(i));
			_routeList.append(r);
			i++;
		}
		
		if (data.rawin("checktime"))
			_checkRoutesTime = AIController.GetTick() + data.rawget("checktime");
		
		LogInfo ("Loaded "+_routeList.len()+" routes.");
	}
	
	function LogInfo(msg)
	{
		AILog.Info("RouteManager: "+msg);
	}
	function LogWarning(msg)
	{
		AILog.Warning("RouteManager: "+msg);
	}
	function LogError(msg)
	{
		AILog.Error("RouteManager: "+msg);
	}
	
	function LogDebug(msg)
	{
		if (DEBUG)
			AILog.Info("(DBG) RouteManager: "+msg);
	}
}