/*
	MogulAI - an artificial intelligence for OpenTTD
	Copyright (C) 2009 - 2010 Kazantsev Lev (Dezmond_snz)

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License along
	with this program; if not, write to the Free Software Foundation, Inc.,
	51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

enum ReachableType
{
	None = 0,
	Normal = 1,
	Bridge = 2,
	Tunnel = 3
}
class PathNode
{
	_tile = -1;
	_parentNode = null;
	_nextNode = null;
	_reachType = ReachableType.None;
	_step = 0;
	_currentCost = 0;
	_totalCost = 0;
	
	constructor ( tile, parentNode, reachType, step, cost )
	{
		_tile = tile;
		_parentNode = parentNode;
		_reachType = reachType;
		_step = step;
		_currentCost = cost;
		_totalCost = GetTotalCost();
		if (_parentNode != null)
			_parentNode._nextNode = this;
	}
	function GetTotalCost()
	{
		if (_parentNode != null)
			return _parentNode.GetTotalCost() + _currentCost;
		return _currentCost;
	}
}

class Util
{
	function TileAddOffset(tile, dx, dy)
	{
		local tileX = AIMap.GetTileX(tile);
		local tileY = AIMap.GetTileY(tile);
		
		local newX = tileX + dx;
		local newY = tileY + dy;
		
		if (newX < 0 || newY < 0)
			return AIMap.TILE_INVALID;
		return AIMap.GetTileIndex(newX, newY);
	}
	
	function GetAdjacentTiles(tile)
	{
		local adjTiles = AITileList();
		adjTiles.AddTile(Util.TileAddOffset(tile, 1, 0));
		adjTiles.AddTile(Util.TileAddOffset(tile, 0, 1));
		adjTiles.AddTile(Util.TileAddOffset(tile, -1, 0));
		adjTiles.AddTile(Util.TileAddOffset(tile, 0, -1));
		
		adjTiles.Valuate(AIMap.IsValidTile);
		adjTiles.KeepValue(1);
		
		return adjTiles;
	}
	function AddRectTilesToList(center, dx, dy)
	{
		local tileList = AITileList();
		for (local i = -dx; i <= dx; i++)
		{
			for (local j = -dy; j <= dy; j++)
				tileList.AddTile( Util.TileAddOffset(center, i, j));
		}
		
		tileList.Valuate(AIMap.IsValidTile);
		tileList.KeepValue(1);
		return tileList;
	}
	function GetRoadNeighbours(tile)
	{
		local tileList = AITileList();
		local list = Util.GetAdjacentTiles(tile);
		list.Valuate(AIRoad.AreRoadTilesConnected, tile);
		list.KeepValue(1);
		
		foreach (t,_ in list)
		{
			local end = AIMap.TILE_INVALID;
			if (AIBridge.IsBridgeTile(t))
				end = AIBridge.GetOtherBridgeEnd(t);
			else if (AITunnel.IsTunnelTile(t))
				end = AITunnel.GetOtherTunnelEnd(t);
			
			if (!AIMap.IsValidTile(end)) continue;
			local end2 = end + Util.GetDirection(t,end);
			if (AIRoad.AreRoadTilesConnected(end, end2) && AIRoad.IsRoadTile(end2))
				tileList.AddTile(end2);
		}
		
		list.Valuate(AIRoad.IsRoadTile);
		list.KeepValue(1);

		foreach (t,_ in list)
			tileList.AddTile(t);
		
		return tileList;		
	}
	function GetDirection (srcTile, dstTile)
	{
		return (dstTile - srcTile) / AIMap.DistanceManhattan(srcTile, dstTile);
	}
	function IsSuitableForFlatBridge(tile, direction)
	{
		if (!AIMap.IsValidTile(tile) || AITile.GetSlope(tile) == AITile.SLOPE_FLAT)
			return false;
		local slope = AITile.GetSlope(tile);
		
		switch(direction)
		{
			case -1:
				if ( (slope & AITile.SLOPE_NE ) == 0)		return true;
				return false;
			case 1:
				if ( (slope & AITile.SLOPE_SW ) == 0)	return true;
				return false;
			default:
				if (direction < 0)
				{
					if ( (slope & AITile.SLOPE_NW ) == 0)	return true;
				}
				else
				{
					if ( (slope & AITile.SLOPE_SE ) == 0)		return true;
				}
				return false;
		}
	}
	
	function CanPass(currTile, nextTile, prevTile)
	{
		if (AIRoad.AreRoadTilesConnected(currTile, nextTile) && AIRoad.AreRoadTilesConnected(nextTile, currTile))
			return true;
		
		if ( !(AITile.IsBuildable(nextTile) || AIRoad.IsRoadTile(nextTile)))
			return false;
		
		if (AITile.IsWaterTile(currTile) || AITile.IsWaterTile(nextTile))
			return false;
		
		if (AIMap.IsValidTile(prevTile) && !AIRoad.CanBuildConnectedRoadPartsHere(currTile, prevTile, nextTile))
			return false;
		
		if (!AIRoad.BuildRoad(currTile, nextTile))
		{
			local err = AIError.GetLastError();
			if (
      				err == AIRoad.ERR_ROAD_WORKS_IN_PROGRESS ||
					err == AIRoad.ERR_ROAD_ONE_WAY_ROADS_CANNOT_HAVE_JUNCTIONS ||
    				err == AIRoad.ERR_ROAD_CANNOT_BUILD_ON_TOWN_ROAD ||
					err == AIError.ERR_AREA_NOT_CLEAR ||
		    		err == AIError.ERR_FLAT_LAND_REQUIRED ||
    				err == AIError.ERR_LAND_SLOPED_WRONG ||
    				err == AIError.ERR_SITE_UNSUITABLE
    			)
      			return false;
		}
		
		return true;
	}
	function GetCost(currTile, nextTile, prevTile)
	{
		local cost = 0;
		if (AIMap.IsValidTile(prevTile) && Util.GetDirection(nextTile, currTile) != Util.GetDirection(currTile, prevTile) ) // turn
			cost += 500;
		
		if (AIMap.DistanceManhattan(nextTile, currTile) == 1)
		{
			if (!AIRoad.AreRoadTilesConnected(currTile, nextTile))
			{
				local acc = AIAccounting();
				AIRoad.BuildRoad(currTile, nextTile);
				cost += acc.GetCosts()*2;
			}
		}
		else
			cost += AIMap.DistanceManhattan(nextTile, currTile)*350;
		return cost;
	}
	
	function FindBestEngine(roadType, cargo)
	{
		local list = AIEngineList(AIVehicle.VT_ROAD);
		
		list.Valuate(AIEngine.GetRoadType);
		list.KeepValue(roadType);
		
		list.Valuate(AIEngine.CanRefitCargo, cargo);
		list.KeepValue(1);
		
		list.Valuate(AIEngine.IsArticulated);
		list.KeepValue(0);
		
		list.Valuate(AIEngine.GetDesignDate);
		list.Sort(AIAbstractList.SORT_BY_VALUE, AIAbstractList.SORT_DESCENDING);
		
		if (AIGameSettings.GetValue("vehicle_breakdowns")== 0) // no breakdowns!
			return list.Begin();
		
		local best = null;
		foreach (engine,_ in list)
		{
			if (best == null)
				best = engine;
			else
			{
				local dr = AIEngine.GetReliability(engine) - AIEngine.GetReliability(best);
				local dc = (AIEngine.GetCapacity(best) - AIEngine.GetCapacity(engine))*100/AIEngine.GetCapacity(best);
				if (dr > 0 && dr >  dc)
					best = engine;
				break;
			}
		}
		return best;
	}
	
	function ValuateZero(item)
	{
		return 0;
	}
	
	function TownAcceptCargo(town, cargo)
	{
		if (AICargo.GetTownEffect(cargo) == AICargo.TE_NONE) // needed for towns?
			return false;
		
		
		
		return false;
	}
	
	function CanBuildRoad(from, to)
	{
		local mode = AITestMode();
		if (!AIRoad.BuildRoad(from, to))
		{
			local err = AIError.GetLastError();
			if (	err == AIRoad.ERR_ROAD_WORKS_IN_PROGRESS ||
					err == AIRoad.ERR_ROAD_ONE_WAY_ROADS_CANNOT_HAVE_JUNCTIONS ||
					err == AIError.ERR_FLAT_LAND_REQUIRED ||
					err == AIError.ERR_LAND_SLOPED_WRONG)
				return false;
		}
		return true;
	}
	
	function CanBuildDepot(dep, front)
	{
		if (!Util.CanBuildRoad(dep, front))
			return false;
		
		/*if (AITile.GetSlope(dep) != AITile.SLOPE_FLAT)
			if (AITile.GetMinHeight(dep) == AITile.GetMinHeight(front) && AITile.GetMaxHeight(dep) != AITile.GetMinHeight(dep))
				return false;*/
		
		local mode = AITestMode();
		if (!AIRoad.BuildRoadDepot(dep, front))
		{
			local err = AIError.GetLastError();
			if (	err == AIRoad.ERR_ROAD_WORKS_IN_PROGRESS ||
					err == AIRoad.ERR_ROAD_ONE_WAY_ROADS_CANNOT_HAVE_JUNCTIONS )
				return false;
		}
		return true;
	}
	
	function GetProducingTileList(industry, cargo)
	{
		local tileList = AITileList_IndustryProducing(industry, AIStation.GetCoverageRadius(AIStation.STATION_TRUCK_STOP));
		tileList.Valuate(AITile.GetCargoProduction, cargo, 1, 1, AIStation.GetCoverageRadius(AIStation.STATION_TRUCK_STOP));
		tileList.KeepAboveValue(0);
		tileList.Valuate(AITile.IsBuildable);
		tileList.KeepValue(1);
		return tileList;
	}
	function GetAcceptingTileList(industry, cargo)
	{
		local tileList = AITileList_IndustryAccepting(industry, AIStation.GetCoverageRadius(AIStation.STATION_TRUCK_STOP));
		tileList.Valuate(AITile.GetCargoAcceptance, cargo, 1, 1, AIStation.GetCoverageRadius(AIStation.STATION_TRUCK_STOP));
		tileList.KeepAboveValue(7);
		tileList.Valuate(AITile.IsBuildable);
		tileList.KeepValue(1);
		return tileList;
	}
	
	function MixListRandomly(list)
	{
		list.Valuate(AIBase.RandItem);
		list.Sort(AIAbstractList.SORT_BY_VALUE, AIAbstractList.SORT_ASCENDING);
		return list;
	}
	
	function LevelTileToHeight(tile, height)
	{
		if (AITile.GetMinHeight(tile) > height || AITile.GetMaxHeight(tile) < height)
			return false;
		
		for (local i=0;i<4;i++) // 0, 1, 2, 3
		{
			if (AITile.GetCornerHeight(tile, i) < height)
			{
				if (!AITile.RaiseTile(tile, (1<<i)))
					return false;
			}
			else if (AITile.GetCornerHeight(tile, i) > height)
			{
				if (!AITile.LowerTile(tile, (1<<i)))
					return false;
			}
		}
		return true;
	}
	
	function LevelTileToBase(tile, baseTile)
	{
		if (AITile.GetMaxHeight(tile) - AITile.GetMinHeight(tile) == 2) // double slope!
		{
			if (AITile.GetMaxHeight(tile) == AITile.GetMaxHeight(baseTile))
				return Util.LevelTileToHeight(tile, AITile.GetMaxHeight(tile));
			else
				return Util.LevelTileToHeight(tile, AITile.GetMinHeight(tile)+1);
		}
		else
			return Util.LevelTileToHeight(tile, AITile.GetMinHeight(tile));
	}
	
	function BuildStationAt(stationTile, baseTile, lorry = null)
	{
		if (!AIRoad.BuildRoadStation(stationTile, baseTile, AIRoad.ROADVEHTYPE_TRUCK, (lorry == null || !AIStation.IsValidStation(lorry)?AIStation.STATION_NEW:lorry)))
		{
			if (AIError.GetLastError() == AIError.ERR_LOCAL_AUTHORITY_REFUSES)
			{
				return "damn "+AITown.GetName(AITile.GetClosestTown(stationTile))+" authority refuses to allow build station here!";
			}
			else if (AIError.GetLastError() == AIError.ERR_FLAT_LAND_REQUIRED || AIError.GetLastError() == AIError.ERR_LAND_SLOPED_WRONG)
			{
				AITile.DemolishTile (stationTile);
				
				if (!Util.LevelTileToBase(stationTile, baseTile))
					return "Cannot level tile "+stationTile+": "+AIError.GetLastErrorString();
				
				return Util.BuildStationAt(stationTile, baseTile, lorry);
			}
			else
				return "error building station: "+AIError.GetLastErrorString();
		}
		
		return null;
	}
	
	function BuildRoadStatic(fromTile, toTile)
	{
      	if (!AIRoad.BuildRoad(fromTile, toTile))
      	{
			local err = AIError.GetLastError();
      		if (		err == AIError.ERR_VEHICLE_IN_THE_WAY ||
      					err == AIRoad.ERR_ROAD_WORKS_IN_PROGRESS ||
						err == AIRoad.ERR_ROAD_ONE_WAY_ROADS_CANNOT_HAVE_JUNCTIONS ||
						err == AIError.ERR_AREA_NOT_CLEAR ||
		   				err == AIError.ERR_FLAT_LAND_REQUIRED ||
    	  				err == AIError.ERR_LAND_SLOPED_WRONG ||
    	  				err == AIError.ERR_SITE_UNSUITABLE
    	  			)
      			return false;
      	}
      	return true;
	}
	
	function BuildDepotStatic(depotTile, frontTile)
	{
		if (!Util.BuildRoadStatic(depotTile, frontTile))
			return false;
		if (!AIRoad.BuildRoadDepot(depotTile, frontTile))
		{
			if (AIError.GetLastError() == AIError.ERR_FLAT_LAND_REQUIRED || AIError.GetLastError() == AIError.ERR_LAND_SLOPED_WRONG)
			{
				AITile.DemolishTile (depotTile);
				if (!Util.LevelTileToBase(depotTile, frontTile))
					return false;
				
				return Util.BuildDepotStatic(depotTile, frontTile);
			}
			return false;
		}
		return true;
	}
	
	function Breakpoint(msg, tile=-1)
	{
		AILog.Warning("Breakpoint: "+msg);
		local sign = AISign.BuildSign(tile == -1?AIMap.GetTileIndex(1,1):tile , "breakpoint");
		while(AISign.IsValidSign(sign)) AIController.Sleep(1);
	}
	function TileToString(tile)
	{
		if (AIMap.IsValidTile(tile))
			return AIMap.GetTileX(tile)+", "+AIMap.GetTileY(tile);
		else return "Invalid";
	}
	function Abs(i)
	{
		return (i<0?-i:i);
	}
	function SendVehicleToDepotForever(veh)
	{
		AIOrder.UnshareOrders(veh);
		if (AIVehicle.GetState(veh) == AIVehicle.VS_AT_STATION)
			AIVehicle.SendVehicleToDepot(veh);
		
		local count = AIOrder.GetOrderCount(veh);
		for (local i = 0; i < count; i++)
			AIOrder.RemoveOrder(veh, AIOrder.ORDER_CURRENT);
		
		AIOrder.AppendOrder(veh, AIMap.TILE_INVALID, AIOrder.AIOF_GOTO_NEAREST_DEPOT| AIOrder.AIOF_STOP_IN_DEPOT);
		if(AIVehicle.GetCurrentSpeed(veh) == 0)
			AIVehicle.ReverseVehicle(veh);
	}
	function SendVehicleToDepot(veh)
	{
		AIVehicle.SendVehicleToDepot(veh);
		if (AIBase.RandRange(3) == 0 && AIVehicle.GetCurrentSpeed(veh) == 0)
			AIVehicle.ReverseVehicle(veh);
	}
}