Utils <- class {}

/// getCities() - Inserts cities only to the townList.
/// @return
function Utils::getCities() {
    local townList = AITownList();
    for(local town = townList.Begin(); townList.HasNext(); town = townList.Next()) {
        if(AITown.IsCity(town)) {
            townList.AddItem(town, 0)
        }
    }
}

/// printTownList
///
/// @param list - AIList() of town indices.
/// @return
function Utils::printTownList(list) {
    for(local town = list.Begin(); list.HasNext(); town = list.Next()) {
                AILog.Info(town + " " + AIEngine.GetName(town));
    }
}

/// getNthItem
///
/// @param list - The list to get the item from.
/// @param n - The order of the item.
/// @return - The n-th item of the list, null if not found.
function Utils::getNthItem(list, n) {
    if(list.Count() == 0) {
        AILog.Warning("getNthItem: list is empty!");
        return null;
    }

    if(n >= list.Count()) {
        AILog.Warning("getNthItem: list is too short!");
        return null;
    }

    for (local item = list.Begin(), i = 0; list.HasNext(); item = list.Next(), ++i) {
        if(i == n) {
            //AILog.Info("getNthItem: Found: " + item + " list.GetValue(item)");
            return item;
        }
    }

    return null;
}

/// print2DArray
///
/// @param array - The 2D array to be printed.
/// @return
function Utils::print2DArray(array) {
    for(local i = 0; i < array.len(); ++i) {
        for(local j = 1; j < array[i].len(); ++j) {
            AILog.Info("" + array[i][j]);
        }
    }
}

/// init2DArray
///
/// @param array - The array to initialize.
/// @param n - The number of rows to be initialised.
/// @return

function Utils::init2DArray(array, n) {
    for(local i = 0; i < n; ++i) {
        local localArray = [];
        array.append(localArray);
    }
}

/// getOffsetTile
///
/// @param tile - The starting tile.
/// @param offsetX - The x-axis offset.
/// @param offsetY - The y-axis offset.
/// @return - The offset tile.
function Utils::getOffsetTile(tile, offsetX, offsetY) {
    local oldX = AIMap.GetTileX(tile);
    local oldY = AIMap.GetTileY(tile);

    local newX = oldX + offsetX;
    local newY = oldY + offsetY;

    if(((newX < 1) || (newY < 1)) // 0 if freeform_edges off
        && (((newX > AIMap.GetMapSizeX() - 2) || (newY > AIMap.GetMapSizeY() - 2)))) {
        return AIMap.TILE_INVALID;
    }

    return AIMap.GetTileIndex(newX, newY);
}

/// getAdjacentTiles
///
/// @param tile - The starting tile.
/// @return - The AITileList of adjacent tiles.
function Utils::getAdjacentTiles(tile) {
    local adjTiles = AITileList();

    local offsetTile = Utils.getOffsetTile(tile, 0, -1)
    if(offsetTile != AIMap.TILE_INVALID) {
        adjTiles.AddTile(offsetTile);
    }

    offsetTile = Utils.getOffsetTile(tile, 1, 0)
    if(offsetTile != AIMap.TILE_INVALID) {
        adjTiles.AddTile(offsetTile);
    }

    offsetTile = Utils.getOffsetTile(tile, 0, 1)
    if(offsetTile != AIMap.TILE_INVALID) {
        adjTiles.AddTile(offsetTile);
    }

    offsetTile = Utils.getOffsetTile(tile, -1, 0)
    if(offsetTile != AIMap.TILE_INVALID) {
        adjTiles.AddTile(offsetTile);
    }

    return adjTiles;
};


/// valuateTruckStopTile
///
/// @param tile - The starting tile to valuate.
/// @param cargoClass - should be either AICargo.CC_MAIL, or AICargo.CC_PASSENGERS to get correct cargo acceptance.
/// @return - The tile value.
function Utils::valuateTruckStopTile(tile, cargoClass, stationId) {
    //check if there are other stations squareSize squares nearby

    //if(stationId == AIStation.STATION_NEW) {
        local square = AITileList();
        local squareSize = 0;

        if(!AIController.GetSetting("is_friendly")) {
            //dont care about enemy stations when is_friendly is off
            squareSize = (stationId == null) ? 5 : 2; //6 tiles distance between new stations, 3 tiles distance between adjacent stations

            square.AddRectangle(Utils.getOffsetTile(tile, (-1) * squareSize, (-1) * squareSize),
                Utils.getOffsetTile(tile, squareSize, squareSize));

            //if another station is nearby return 0
            for(local tile = square.Begin(); square.HasNext(); tile = square.Next()) {
                if(AITile.IsStationTile(tile) && (AITile.GetOwner(tile) == AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)) ) { //negate second expression to merge your statitions
                    return 0;
                }
            }
        }
        else {
            squareSize = 3;

            square.AddRectangle(Utils.getOffsetTile(tile, (-1) * squareSize, (-1) * squareSize),
                Utils.getOffsetTile(tile, squareSize, squareSize));

            //if another station is nearby return 0
            for(local tile = square.Begin(); square.HasNext(); tile = square.Next()) {
                if(AITile.IsStationTile(tile)) {
                    return 0;
                }
            }
        }

    //}

    if(! ((AITile.IsBuildable(tile) || AIRoad.IsRoadTile(tile)) && AITile.GetSlope(tile) == AITile.SLOPE_FLAT)) {
        return 0;
    }

    local cargoList = AICargoList();
    cargoList.Valuate(AICargo.HasCargoClass, cargoClass);
    cargoList.KeepValue(1);
    local cargoType = cargoList.Begin();

    local radiusType;
    if(cargoClass == AICargo.CC_MAIL) {
        radiusType = AIStation.STATION_TRUCK_STOP;
    }
    else {
        radiusType = AIStation.STATION_BUS_STOP;
    }

    if(AITile.GetCargoAcceptance(tile,
        cargoType,
        1,
        1,
        AIStation.GetCoverageRadius(radiusType)) < 8) {

            return 0;
    }

    local value = AITile.GetCargoProduction(tile,
                cargoType,
                1,
                1,
                AIStation.GetCoverageRadius(radiusType));

    return value;
}

/// isTileMyStation - check if tile has owners station.
///
/// @param tile - the tile to chcek.
/// @return - 1 if tile has owners station, 0 otherwise.
function Utils::isTileMyStation(tile) {
    if((AITile.IsStationTile(tile)) && (AITile.GetOwner(tile) == AICompany.ResolveCompanyID(AICompany.COMPANY_SELF))) {
        return 1;
    }

    return 0;
}

/// valuateTownRoadEnds - Creates a square around @tile and finds all tiles with one adjacent road tile.
///
/// @param tile - The starting tile to valuate.
/// @return - True if road ends on the tile, false otherwise.
function Utils::valuateTownRoadEnds(tile) {
    if(!AIRoad.IsRoadTile(tile)) {
        return 0;
    }

    local temp = AITileList();
    temp.AddList(getAdjacentTiles(tile));
    temp.Valuate(AIRoad.IsRoadTile);
    temp.KeepValue(1);

    if(temp.Count() == 1) {
        return 1;
    }

    return 0;
}


/// getCargoId - Returns either mail cargo id, or passenger cargo id.
///
/// @param cargoClass - either AICargo.CC_MAIL, or AICargo.CC_PASSENGERS
/// @return - Cargo list.
function Utils::getCargoId(cargoClass) {
    local cargoList = AICargoList();
    cargoList.Valuate(AICargo.HasCargoClass, cargoClass);
    cargoList.KeepValue(1);

    //both AICargo.CC_MAIL and AICargo.CC_PASSENGERS should have only one cargo
    return cargoList.Begin();
}


