/*
 *
 * Copyright (c) 2009, Dustin Andrews
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * Redistributions of source code must retain the above copyright notice, this list
 * of conditions and the following disclaimer.
 * 
 * Redistributions in binary form must reproduce the above copyright notice, this
 * list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 
 * The names of its contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

class AntPathFinder
{  
  path = null;
  start = null;
  goal = null;
  next = null;
  last = null;
  manhattanDistance = null;
  current = null;
  waterPenalty = 15;
  cost = 0;
  iterationLimit = null;
  iterations = 0;
  pointIndex = 0;
  max_len = 100;
    
  //Parameters that control the hueristic of the ant.
  //How much the ant prefers "good nodes".
  goalFocus = null;
  ignoreTiles = null;

  display_thinking = null;
 
  IsGoalMyTrack = null;

   
  constructor(start, goal, ignoreTiles)
  {  
  
    assert(AIMap.IsValidTile(start));
    assert(AIMap.IsValidTile(goal));
    this.start = start;
    current = start;
    this.goal = goal;
    this.display_thinking = AIController.GetSetting("display_thinking");
    
    if(display_thinking)
    {
     Util.DebugReplaceSign(start, "{ANTSTART}");
     Util.DebugReplaceSign(goal, "{ANTGOAL}");
    }

    this.ignoreTiles = ignoreTiles;
    manhattanDistance = AITile.GetDistanceManhattanToTile(start, goal);
    iterationLimit = manhattanDistance * 10;
    
    next = null;
    last = null;
    
    goalFocus = 1.0;
    cost = 0;
    this.pointIndex = 0;
    iterations = 0;
    path = AIList();
    
    this.max_len = AIController.GetSetting("ant_max_len");
    
    if(AIRail.IsRailTile(goal))
    {
      local company = AITile.GetOwner(goal);
      if(AICompany.IsMine(company))
      {
        this.IsGoalMyTrack = true;
      }
    }   

  }

}

function AntPathFinder::FindPath()
{ 
  StationManager.BringTileToLevel(start);
  StationManager.BringTileToLevel(goal);
  
  if(manhattanDistance > this.max_len) {return null;}
  
  iterations = 0;
  while(iterations < iterationLimit && current != goal)
  {
    iterations++;
    this.MoveToNext();
  }
    
  //didn't find a viable path;
  if (iterations == iterationLimit)
  {
    //AILog.Info("Ant finder timed out after " + iterations + " tiles." );
    cost = 99999999999;
    this.goalFocus += 0.5;
  }
  path.Sort(AIAbstractList.SORT_BY_VALUE, true);
  //print ("iterations used:" + iterations);
  return this.path;
}




function AntPathFinder::MoveToNext()
{
  //print("MoveToNext()");
  this.GetNextNode(); 
  this.CalculateCost();

  //print("Ant::MoveToNext() cur:" + current + " next:" + next);
  if(current != next)
  {
    //print ("Adding tile to path:" + current);
    this.path.AddItem(current, this.pointIndex++);
  }
  else
  {
    current = goal;
  }
  
  last = current;
  current = next;
  next = null;
}


function AntPathFinder::CalculateCost()
{
 if(last == null || next == null || current == null){return;}
 local test = AITestMode();
 local costs =  AIAccounting();
 if(AIRail.BuildRail(last, current, next))
 {
  this.cost += costs.GetCosts();
 }
 else
 {
  AILog.Warning("Ant Wouldn't Be Able to Build To Here.");
  AISign.BuildSign(last, "ERR1");
  AISign.BuildSign(current, "ERR2");
  AISign.BuildSign(next, "ERR3");
  this.cost += 10000; //penalty for unbuildable routes.
 }
}

function AntPathFinder::GetNextNode()
{
  
  local nodelist = AntHill.FindCompassPointsNunitsAway(this.current, 1);
  
  nodelist.Valuate(AIMap.IsValidTile);
  nodelist.KeepValue(1);
  
  nodelist.Valuate(CrowPathFinder.IsInPathValuator, path);
  nodelist.KeepValue(0);
  
  nodelist.Valuate(AntPathFinder.isBuildable, ignoreTiles);
  nodelist.KeepValue(1);
  
   if(IsGoalMyTrack && AIMap.DistanceManhattan(current, goal) == 1)
   {
    nodelist.AddTile(goal);
   }

  foreach(i, v in nodelist)
  {
    local verified = this.VerifySegment(i, current, last);
    nodelist.SetValue(i, verified);
  }
  nodelist.KeepValue(1);
  
  if(nodelist.Count() == 0)
  {
    //AILog.Info("Ant Ran into a dead end. " + iterations + " iterations.");
    iterations = iterationLimit;
    return;
  }
  nodelist.Valuate(this.NodeValuator, this.goal, this.waterPenalty, this.path, this.current, nodelist, this.last);
   
  nodelist = AntPathFinder.AddTrackCost(nodelist);
  
  nodelist.Sort(AIAbstractList.SORT_BY_VALUE, true);
  
  // Convert all the scores >= 0;

  local totalWeight = 0;
  
  local scale = 1 - nodelist[nodelist.Begin()];
  nodelist.Valuate(AntPathFinder.AddValuator, nodelist, scale);
 
  
  foreach(i, x in nodelist)
  {
    totalWeight += x;
  }

  //sort high to low.
  nodelist.Sort(AIAbstractList.SORT_BY_VALUE, false);
  
  next = GetWeightedRandomItem(nodelist, totalWeight);
  if(next == null)
  {
    throw("GetWeightedRandomItem returned null;");
  }
}

function AntPathFinder::VerifySegment(tile, current, last)
{
  if(current == null || last == null) { return 1;}
  local test = AITestMode();
  if(AIRail.BuildRail(last,current,tile))
  {
    return 1;
  }
  else
  {
    return 0;
  }
}


function AntPathFinder::AddTrackCost(nodelist)
{
 local test = AITestMode();
 local costs =  AIAccounting(); 
 
 foreach(idx, val in nodelist)
 {
 
   if(! AIRail.BuildRailTrack(idx, AIRail.RAILTRACK_NE_SW))
   {
    AIRail.BuildRailTrack(idx, AIRail.RAILTRACK_NW_SE)
   }
   nodelist[idx] +=  costs.GetCosts();
   costs.ResetCosts();
 }
 
 return nodelist;
}

function AntPathFinder::AddValuator(idx, nodelist, scale)
{
  if( scale  + nodelist[idx] < 0 )
  {  
   //print( " idx " + idx);
   //print( "scale " + scale);
   //print( "nodelist[idx] " + nodelist[idx] );
    throw("Bad Scale.");  
  }
  return nodelist[idx] + scale;
}

function AntPathFinder::GetWeightedRandomItem(nodelist, totalWeight)
{
  //print("getWeightedRandom() totalWeight=" + totalWeight + " totalNodes=" + nodelist.Count());
  local selected = null;
  if(nodelist.Count() == 0) {return null;}
  local keepTrying = true;
  while(keepTrying)
  {
    local random = AIBase.RandRange(totalWeight);
    foreach(i, val in nodelist)
    {
      random -= val * this.goalFocus;
      //print("Random =" + random + " val=" + val + " index:" + i);
      AIController.Sleep(1);
      if (random <= 0)
      {
          //print("Picked One: " + i);  
          keepTrying = false;
          selected = i;
          break;
      }
    }
  }
  
  return selected;

}


function AntPathFinder::DistanceValuator(idx, goal)
{
  return AITile.GetDistanceSquareToTile(idx, goal);
}

//Give relative score to a tile. Positive is more desirable, negative is less.
function AntPathFinder::NodeValuator(tile, goal, waterPenalty, path, current, nodelist, last)
{
 local score = 0;

 //Check to see if it's the goal.
 if(tile == goal)
 {
  return 100;
 }
 
 //local lastDistance = AITile.GetDistanceSquareToTile(current, goal);
 //local currDistance = AITile.GetDistanceSquareToTile(tile, goal);
 
  local lastDistance = AITile.GetDistanceManhattanToTile(current, goal);
  local currDistance = AITile.GetDistanceManhattanToTile(tile, goal);
  score =  (lastDistance - currDistance) / 10;

 //penalize tiles by the distance to the goal.
 //score -= AITile.GetDistanceManhattanToTile(tile, goal) * 2;
 score -= AITile.GetDistanceSquareToTile(tile, goal); //square is better, but much slower.
 
 //penalize water and coastal squares
 //TODO: Make this understand bridges.
 if(AITile.IsWaterTile(tile))
 {
  score -= waterPenalty;
 }
 
 if(AITile.IsCoastTile(tile))
 {
  score -= waterPenalty;
 }
 
 
 local slope = AITile.GetSlope(tile);
 if(AITile.SLOPE_FLAT == slope)
 {
  score += 1;
 }
 
 if(AITile.IsSteepSlope(slope))
 {
  score -= 1;
 }
 
 //penalize elevation changes.
 score -= abs( AITile.GetMinHeight(current) - AITile.GetMinHeight(tile) );
 

 
 //if(AntPathFinder.isStraitLine(path, current, tile))
 //{
 // score += 1;
 //}
 
  
  
 return score;
}


function AntPathFinder::isBuildable(tile, ignoreTiles)
{

    if(ignoreTiles.HasItem(tile)) {return false;}
    
    if(AITile.IsCoastTile(tile))
    {
        return false;
    }
  
    if(AITile.IsBuildable(tile))
    {
      return true;
    }
    
    local stationId = AIStation.GetStationID(tile);
    if(stationId != AIStation.STATION_INVALID)
    {
      return false;
      //AILog.Warning("Own Station tiles are considered buildable!");
    }
    
    return false;
}




function AntPathFinder::isBuildableOrMyRail(tile)
{
    if(AITile.IsCoastTile(tile))
    {
      return false;
    }
    
    if(AITile.IsBuildable(tile))
    {
      return true;
    }
    
    if(AIRail.IsRailTile(tile))
    {
      local company = AITile.GetOwner(tile);
      if(AICompany.IsMine(company))
      {
        return true;
      }
    }
    
    local stationId = AIStation.GetStationID(tile);
    if(stationId != AIStation.STATION_INVALID)
    {
      return false;
      AILog.Warning("Own Station tiles are considered buildable!");
    }
    
    return false;
}


function AntPathFinder::isStraitLine(path, current,  tile)
{
  local pathCopy = path;
  pathCopy.Sort(AIAbstractList.SORT_BY_VALUE, false);
  
  local px = AIMap.GetTileX(pathCopy.Begin());
  local py = AIMap.GetTileY(pathCopy.Begin());
  
  local cx = AIMap.GetTileX(current);
  local cy = AIMap.GetTileY(current);
    
  local tx = AIMap.GetTileX(tile);
  local ty = AIMap.GetTileY(tile);
  
  //check for x axis or y axis lines.
  if(px == cx && cx == tx) {return true;}
  if(py == cy && cy == ty) {return true;}
  
  //prevent divide by 0.
  if((px-cx) == 0) {return false;}
  if((px-tx) == 0) {return false;}
  if((cx-tx) == 0) {return false;}
  
  local slope1=(py-cy)/(px-cx);
  local slope2=(py-ty)/(px-tx);
  local slope3=(cy-ty)/(cx-tx);
  
  if(slope1 == slope2 && slope2 == slope3)
  {
    return true;
  }
  
  
  
  return false;
}
