/*
 *
 * Copyright (c) 2009, Dustin Andrews
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * Redistributions in binary form must reproduce the above copyright notice, this
 * list of conditions and the following disclaimer in the documentation and/or 
 *  other materials provided with the distribution.
 * 
 * The names of its contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

class PassingLane
{  
  trainsize = null;
  constructor(trainsize)
  {
    this.trainsize = trainsize;
  }
}

function PassingLane::BuildPassingLanes(path)
{ 
  local p1 = AIList();
  local p2 = AIList();
  local p3 = AIList();
  local p4 = AIList();
  local transforms = [p1, p2, p3, p4];
  local count = 0;
  local passing = AIList();
  
  foreach(tile, v in path)
  {
    local t1 = AIMap.GetTileIndex( -1, -1) + tile;
    local t2 = AIMap.GetTileIndex( 1, 1) + tile;
    local t3 = AIMap.GetTileIndex( 1, -1) + tile;
    local t4 = AIMap.GetTileIndex( 1, -1) + tile;
    p1.AddItem(t1, v);
    p2.AddItem(t2, v);
    p3.AddItem(t3, v);
    p4.AddItem(t4, v);
  }
  
  //adjust for the offset.
  foreach(index, list in transforms)
  {
    list.RemoveTop(1);
    list.RemoveBottom(1);
  }  
  
  
  local buildable = 0;
  local best = null;
  foreach(index, list in transforms)
  {
    local count = CountBuildable(list);
    if(count > buildable)
    {
        buildable = count;
        best = list; 
    }
  }
 
  local lanes = GetPassingLanes(path, best);//{lane = lane, start = startTile, end = endTile, pathStart = pathStart, pathEnd = pathEnd}
  AILog.Info("Laying down " + lanes.len() + " passing lanes.");
   
  foreach(lane in lanes)
  {
    local cost = AntHill.CostPath(lane.lane);
    Util.GetMaxLoan();
    if(AICompany.GetBankBalance(AICompany.COMPANY_SELF) <  cost * 2) //don't build passing lanes if short on cash.
    {
      AILog.Info("Didn't build lane due to cash shortage.");
      Util.PayDownLoan();
      continue;
    }
    
    if(AITile.GetMaxHeight(lane.pathStart) != AITile.GetMaxHeight(lane.start))
    {
        BridgePathFinder.BringeTileToAltitidue(lane.start, AITile.GetMaxHeight(lane.pathStart));
        //AISign.BuildSign(lane.pathStart, "FROM");
        //AISign.BuildSign(lane.start, "TO");
        //if(AITile.GetMaxHeight(lane.pathStart) != AITile.GetMaxHeight(lane.start)){throw("");}
    }
    
    if(AITile.GetMaxHeight(lane.pathEnd) != AITile.GetMaxHeight(lane.end))
    {
    
        BridgePathFinder.BringeTileToAltitidue(lane.end, AITile.GetMaxHeight(lane.pathEnd));
        //AISign.BuildSign(lane.pathEnd, "FROM");
        //AISign.BuildSign(lane.end, "TO");
        //if(AITile.GetMaxHeight(lane.pathEnd) != AITile.GetMaxHeight(lane.end)) {throw("");}
    }

    BuildSimpleRoute(lane.lane);
    
    StationManager.BuildAllTrack(lane.start);
    StationManager.BuildAllTrack(lane.pathStart);
    
    StationManager.BuildAllTrack(lane.end);
    StationManager.BuildAllTrack(lane.pathEnd);
    OneWaySignalSideBySide(lane.lane, path);
    passing.AddList(lane.lane);
  }
  
  
 return passing;
}

function PassingLane::CountBuildable(path)
{
    local newList = AIList();
    newList.AddList(path);
    newList.Valuate(AITile.IsBuildable);
    local total = 0;
    foreach(tile, v in newList)
    {
        total += v;
    }
    return total;
}


function PassingLane::GetPassingLanes(path, lane)
{
  local lanes = [];
  if(lane == null) {return lanes;}
  local pathStart = path.Begin();
  
  local baseLane = AIList();
  baseLane.AddList(lane);
  baseLane.Sort(AIAbstractList.SORT_BY_VALUE, false);
  local count = 0;
  while(baseLane.Count() > trainsize)
  {
    local lane = AIList();
    lane.AddList(baseLane);
    lane.Sort(AIAbstractList.SORT_BY_VALUE, true);
    lane = GetBuildableTiles(lane);
    
    
    if(lane.Count() == 0)
    {
        baseLane.RemoveTop(1);
        continue;
    }
    
    if(lane.Count() < trainsize + 2)
    {
        baseLane.RemoveList(lane);
        continue;
    }
    
    lane.Sort(AIAbstractList.SORT_BY_VALUE, true);
    TrimToFlatClearTile(lane);
    lane.Sort(AIAbstractList.SORT_BY_VALUE, false);
    TrimToFlatClearTile(lane);
    

    
    lane.Sort(AIAbstractList.SORT_BY_VALUE, true);
    local startTile = lane.Begin();    
    lane.Sort(AIAbstractList.SORT_BY_VALUE, false);
    local endTile = lane.Begin();    
  
      
    local pathStart = GetClosestFlatTileInList(startTile, path);    
    local pathEnd =   GetClosestFlatTileInList(endTile, path);
    
    
    //get good match on pathstart
    while(AIMap.DistanceManhattan(startTile, pathStart) > 1 && lane.Count() > trainsize)
    {
      count++;
      lane.Sort(AIAbstractList.SORT_BY_VALUE, true);      
      lane.RemoveTop(1);
      startTile = lane.Begin();
      pathStart = GetClosestFlatTileInList(startTile, path);  
    }
    
    //get good match on pathend   
    while(AIMap.DistanceManhattan(endTile, pathEnd) > 1 && lane.Count() > trainsize)
    {
      lane.Sort(AIAbstractList.SORT_BY_VALUE, false);
      lane.RemoveTop(1);
      endTile = lane.Begin();
      pathEnd = GetClosestFlatTileInList(endTile, path);
    }
    
    if(lane.Count() > (trainsize + 2) * 2)
    {        
      baseLane.RemoveList(lane);
      lanes.push({lane = lane, start = startTile, end = endTile, pathStart = pathStart, pathEnd = pathEnd});
    }
    else
    {
        baseLane.RemoveTop(1);
    }   
    
    
    if(count++ > path.Count())
    {
      AILog.Error("Broke infinite loop in passing lane routine.");
      break;
    }
  
    
 }
 return lanes;
}

function PassingLane::BuildSimpleRoute(path)
{   
   
    local pp = null;
    local p = null;
    local count = 0;
    foreach(tile, val in path)
    {
    
     //AISign.BuildSign(tile, count++ + ":" + val);
      if(p != null && pp != null)
      {
         
         if (AIMap.DistanceManhattan(p, tile) == 1)
         { 
           AIRail.BuildRail(pp, p, tile);
         }
      }
      pp = p; p = tile;
    }
}

function PassingLane::GetBuildableTiles(path)
{
    //print("get buildable tiles");
    local testPath = AIList();
    testPath.AddList(path);
    local buildable = AIList();
    local pp = null;
    local p = null;
    local count = 0;
    
    
    foreach(tile, val in testPath)
    { 
      if((p == null || pp == null) && AITile.IsBuildable(tile) == false)
      {
        //ran into trouble right at the start.
        return buildable;
      }
      
      if(p != null && pp != null && AIMap.DistanceManhattan(p, tile) == 1)
      {   
        local test = AITestMode();
        local result = (AIRail.BuildRail(pp, p, tile) && AITile.IsBuildable(p));
        if(result == false 
           || AIMap.DistanceManhattan(pp, p) > 1 
           || AIMap.DistanceManhattan(p, tile) > 1)
        {
          return buildable;
        }
    
        buildable.AddItem(tile, val);       
      }              
      count++;
      pp = p; p = tile;
    }
    return buildable;
}


function PassingLane::TrimToFlatClearTile(path)
{
  //print("trim to flat clear.");
  foreach(tile, val in path)
    {
      if(AITile.GetSlope(tile) != AITile.SLOPE_FLAT || !AITile.IsBuildable(tile))
      {
        path.RemoveItem(tile);
      }
      else
      {
        break;
      }
    }
}

function PassingLane::OneWaySignalSideBySide(lane, path)
{

    local count = 1;
    local tile;
    local prev;
    
    lane.Sort(AIAbstractList.SORT_BY_VALUE, true);
    local lStart = lane.Begin();
    local pStart = Util.GetClosestTile(lStart, path);

    lane.Sort(AIAbstractList.SORT_BY_VALUE, false);    
    local lEnd = lane.Begin();
    local pEnd =  Util.GetClosestTile(lEnd, path);
    
    OneWaySignalPath(lane);
    
    local pathSegment = AIList();
    pathSegment.AddList(path);
    
    local bottom = min( path.GetValue(pEnd), path.GetValue(pStart) );
    local top = max( path.GetValue(pEnd), path.GetValue(pStart) );

    pathSegment.RemoveAboveValue(top);
    pathSegment.RemoveBelowValue(bottom);
    pathSegment.Sort(AIAbstractList.SORT_BY_VALUE, true);
    OneWaySignalPath(pathSegment);
}



function PassingLane::OneWaySignalPath(path)
{
  local count = 1;
  local tile;
  local prev;
  foreach(tile, v in path)
  {
    if((count++ % trainsize) == 0)
    {
      AIRail.BuildSignal(tile, prev, AIRail.SIGNALTYPE_PBS_ONEWAY);
    }
    prev = tile;
  }
  
  //AIRail.BuildSignal(tile, prev, AIRail.SIGNALTYPE_ENTRY);
}

function PassingLane::GetClosestFlatTileInList(tile, list)
{
  if(AIMap.IsValidTile(tile) == false)
  {
    AILog.Error("Can't find closest tile to invalid.");
    return list.Begin();
  }
  local newList = AIList();
  newList.AddList(list);
  newList.Valuate(AITile.GetSlope);
  newList.KeepValue(AITile.SLOPE_FLAT);
  newList.Valuate(AIBridge.IsBridgeTile);
  newList.KeepValue(0);
  
  newList.Valuate(AIMap.DistanceManhattan, tile);
  
  return Util.GetClosestTile(tile, newList);
}


function PassingLane::GetClosestTriangulatedTile(tile1, tile2, tile3, list)
{
  local newList = AIList();
  newList.AddList(list);
  newList.RemoveItem(tile1);
  newList.RemoveItem(tile2);
  newList.RemoveItem(tile3);
  newList.Valuate(Util.AddDistancesValuator, tile1, tile2, tile3);
  newList.Sort(AIAbstractList.SORT_BY_VALUE, true);  
  return newList.Begin();
}