/*
 *
 * Copyright (c) 2009, Dustin Andrews
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * Redistributions in binary form must reproduce the above copyright notice, this
 * list of conditions and the following disclaimer in the documentation and/or 
 *  other materials provided with the distribution.
 * 
 * The names of its contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

require("AntPathFinder.nut");
require("AntHill.nut");
require("CrowPathFinder.nut");
require("StationManager.nut");
require("TownManager.nut");
require("PassingLane.nut");
require("Util.nut");
require("Headquarters.nut");


class DenverAndRioGrande extends AIController
{
  industryInfo = null;
  
  trainsize = 8;
  speedConstant = 0.056;
  PathIterations = null;
  routeCount = 0; 
  cabooseId = null;
  deleteCount = 1;
  stationLen = 5;
  hpPerCar = 150;
  ignoreTiles = null;
  
  IndustriesUsedAndDiscarded = null;
  bestScoreIndustryIdx = null;
  activeProducingStations = null;
  lastUpdateDate = 0;
  
  _path = null;
  
  TrainsNeeded = 0;
  display_thinking = null;
  randomize_station_picks = null;    
  bridge_max_len = null;
  ant_max_len = null;
  transport_min_percent = null;
  fiscal_restraint = null;
  min_route_dist = null;
  random_names = null;
  RestrictTerraforming = null;

  constructor()
  {
    this.PathIterations = 10;
    this.routeCount = 0;
    this.deleteCount = 1;
    this.cabooseId = FindCaboose();
    this.IndustriesUsedAndDiscarded = AIList();
    this.bestScoreIndustryIdx = null;
    activeProducingStations = [];
    this.trainsize = 8;
    this.lastUpdateDate = 0;
    this.hpPerCar = 150;
    _path = null;
    this.TrainsNeeded = 0;
    this.RestrictTerraforming = true;
    this.display_thinking = AIController.GetSetting("display_thinking");
    randomize_station_picks = AIController.GetSetting("randomize_station_picks");   
    bridge_max_len = AIController.GetSetting("bridge_max_len");
    ant_max_len = AIController.GetSetting("ant_max_len");;
    transport_min_percent = AIController.GetSetting("transport_min_percent");
    fiscal_restraint = AIController.GetSetting("fiscal_restraint");
    min_route_dist = AIController.GetSetting("min_route_dist");
    random_names = AIController.GetSetting("random_names"); 
    ignoreTiles = AIList();
  }
  
}


function DenverAndRioGrande::Start()
{
  this.Sleep(1);
  
  AILog.Info("Dedicated to the memory of Howard H. Thompson. 1912-1988");
  AILog.Info("Special thanks to Brumi, SirkoZ and Eliandor for help with testing and suggestions.");
  AILog.Info("Denver & Rio Grande by Dustin Andrews started...");
  Util.PayDownLoan();
  SetCompanyName();
  local hq = Headquarters();
  Util.GetMaxLoan();
  while(hq.Place() == false);
  Util.PayDownLoan();
  //find a rail type that can haul frieght
  while(SetRoadAndRailType() == false);
  
  //set a "no fly" zone around towns to make pathfinding smoother.
  foreach(town, v in AITownList())
  {
    local tm = TownManager(town);
    local tiles = tm.GetBoundingRect();
    if(tiles != null)
    {
      ignoreTiles.AddList(tm.GetBoundingRect());
    }
  }
  
  
 //Keep running
 while (true)
 {
  MainLoop();
 }
}


function DenverAndRioGrande::MainLoop()
{  
  Management(this);
  Util.PayDownLoan();
  if(AICompany.GetBankBalance(AICompany.COMPANY_SELF) > 500000) {RestrictTerraforming = false;}
  if(TrainsNeeded < 2)
  {
    BuildRoute();
  }
  else
  {
    AILog.Info("Skipping new route while stations still need trains.");
    AIController.Sleep(1000);
  }

  if(AIDate.GetCurrentDate() > 90 + lastUpdateDate)
  {
    lastUpdateDate =  AIDate.GetCurrentDate();
    AILog.Info("Still Alive. " + GetGameDateString());    
  }
  AIController.Sleep(1);
}

function DenverAndRioGrande::Management(DaRG)
{
    DaRG.ManageStations();
    DaRG.ManageTrains();
}

function DenverAndRioGrande::SetRoadAndRailType()
{

  AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_ROAD);
  
  local types = AIList();
  types.AddList(AIRailTypeList());
  if(types.Count() == 0)
  {
    AILog.Warning("There are no rail types to pick from. Sleeping.");
    AIController.Sleep(5000);
    return false;
  }
  
  local bestIndustryIndex = PickFromBestIndustries();
  local industryType = AIIndustry.GetIndustryType(bestIndustryIndex);
  local produces = AIIndustryType.GetProducedCargo(industryType);
  local cargo = produces.Begin();
  

  AILog.Info("Trying out the different track types. Warnings are normal.");
  foreach(rail, v in types)
  {
    AIRail.SetCurrentRailType(rail);
    local wagons = FindWagons(cargo);
    types.SetValue(rail, wagons.Count());
  }
  
  types.RemoveValue(0);
  
  if(types.Count() == 0)
  {
    AILog.Warning("There are no rail types with wagons that haul freight. Waiting to see if some become available.");
    AIController.Sleep(5000);
    return false;
  }
  
  types.Sort(AIAbstractList.SORT_BY_VALUE, false);
  AIRail.SetCurrentRailType(types.Begin());

  AILog.Info("Rail type selected.");
  return true;
}






function DenverAndRioGrande::Stop()
{
  AILog.Info(" Stopped");
}


function DenverAndRioGrande::Save()
{
 local table = {};	
 //print(" Saved");
 return table;
}

function DenverAndRioGrande::Load(version, data)
{
  AILog.Info(" Loaded");
}


function DenverAndRioGrande::SetCompanyName()
{
    local names =
    [
        "Colorado & Southern",
        "Burlington Northern Railroad",
        "Union Pacific Railroad",
        "Southern Pacific Railroad",
        "Western Pacific Railroad",
        "Chicago, Burlington and Quincy",
        "Durango and Silverton",
        "Sante Fe Railroad",
        "Chicago and North Western",
        "Missouri Pacific Railroad",
        "Missouri-Kansas-Texas", 
        "Chicago and North Western",
        "Atchison Topeka & Santa Fe",
        "Chicago Rock Island & Pacific",
        "Rocky Mountain",
        "Denver Laramie & North Western",
        "Colorado Kansas & Oklahoma"
    ];
    
    
    AICompany.SetPresidentName("H.H. Thompson")
    
    //set a random name if that's choosen.
    local count = 0;
    while(random_names && count < 10)
    {
        local name = names[AIBase.RandRange(names.len())];
        if(AICompany.SetName( name ) )
        {
            AILog.Warning("Doing business as " + name + ".");
            return;
        }
        count++;
    } 


    //Set the default name if it's available.
    if(AICompany.SetName("Denver & Rio Grande"))
    {
        return;
    }    
    
    

    //fall back to random for additional companies.
    while(count < 10)
    {
        local name = names[AIBase.RandRange(names.len())];
        if(AICompany.SetName( name ) )
        {
            AILog.Warning("Doing business as " + name + ".");
            return;
        }
        count++;
    } 

}


function DenverAndRioGrande::ManageStations()
{
  local maint = 0;
  foreach(i, station in activeProducingStations)
  {
    if(station.PeriodicCheckup())
    {
      maint++;
    }
    if(station.Deleted == true)
    {
        activeProducingStations.remove(i);
    }
  }
  
  if(maint != 0)
  {
      AILog.Info("Station maintainence came due on " + maint + " stations and was run.");
  }
}

function DenverAndRioGrande::ManageTrains()
{

    //look for stopped trains and sell them    
    local trains = AIList();
    trains.AddList(AIVehicleList());
    trains.Valuate(AIVehicle.GetState);
    trains.KeepValue(AIVehicle.VS_IN_DEPOT);
    
    foreach(train, v in trains)
    {
        AIVehicle.SellWagonChain(train, 0);
    }

    //check built train ratio
    TrainsNeeded = 0;
    foreach(idx, stationManager in activeProducingStations)
    {
        if(stationManager.Trains.Count() == 0 && stationManager.Deleted == false)
        {
            stationManager.BuildTrains();
            TrainsNeeded++;
        }
    }
}

function DenverAndRioGrande::BuildRoute()
{
  Util.GetMaxLoan();
  local maxloan;
  {
      local testMode = AITestMode();
      maxloan = AICompany.GetMaxLoanAmount();
  }
  if(AICompany.GetBankBalance(AICompany.COMPANY_SELF) < maxloan * (fiscal_restraint / 100.0) )
  {
    Util.PayDownLoan();
    if(AIDate.GetCurrentDate() > 90 + lastUpdateDate)
    {
        AILog.Info("Build Route Waiting for more money... ");
        AIController.Sleep(1000);
    }
    return false;
  }

  Util.PayDownLoan();
  
  PickFromBestIndustries();  

  if(bestScoreIndustryIdx == null || !AIIndustry.IsValidIndustry(bestScoreIndustryIdx) || !industryInfo.rawin(bestScoreIndustryIdx))
  {
    AILog.Info("Looks like we ran out of industries! Lifting restrictions and re-trying places local authority forbid in the past.");
    IndustriesUsedAndDiscarded.Clear();
    RestrictTerraforming = false;
    return false;
  }
  
  local bestIndustry = industryInfo[bestScoreIndustryIdx];
  
  Util.DebugReplaceSign(bestIndustry.location, "Start " + ++routeCount + " index:" +bestIndustry.index); 
  local consumers = DenverAndRioGrande.FindAllConsumersOf(bestIndustry);
  local stationManager = this.ChooseRoute(bestIndustry, consumers); //Chooseroute is the major loop.
  if(stationManager == null) {return false;}
  activeProducingStations.push(stationManager);
  stationManager.BuildTrains();
  
  return true;
}


function DenverAndRioGrande::IsInAIList(idx, ailist)
{
  if(ailist.HasItem(idx))
  {
    return 1;
  }
  else
  {
    return 0;
  }
}




function DenverAndRioGrande::GetAmountOfMySigns(industryIndex)
{
  local signs = AISignList();
  local signLocations = AIList();
  signLocations.AddList(signs);
  signLocations.Valuate(AISign.GetLocation);
  signLocations.KeepValue(AIIndustry.GetLocation(industryIndex));
  return signLocations.Count();
}



function DenverAndRioGrande::ConsumerDistance(idx, producer)
{
    local producerLocation = AIIndustry.GetLocation(producer);
    local distance = AIIndustry.GetDistanceManhattanToTile(idx, producerLocation);
    return distance;
}

function DenverAndRioGrande::getIndustryInfo()
{
  //print("Getting industry info");
  local industryList = AIIndustryList();
  this.industryInfo = {};
  bestScoreIndustryIdx = null;
  foreach(idx, val in industryList)
  {
    local industryType = AIIndustry.GetIndustryType(idx);
    local location = AIIndustry.GetLocation(idx);
    local produces = AIIndustryType.GetProducedCargo(industryType);
    local accepts = AIIndustryType.GetAcceptedCargo(industryType);
    local productionScore = 0;
    
    local alreadyTried = IndustriesUsedAndDiscarded.HasItem(idx);
    if(alreadyTried == true) { continue;}
     
    local score = 0;
    if(produces == null) {continue;} //industry might close between time list is made and cargo is checked.
    foreach(cargoIdx, val in produces)
    {
     local cargoVal = AICargo.GetCargoIncome(cargoIdx, 100, 15);
     local producedLastMonth = AIIndustry.GetLastMonthProduction(idx, cargoIdx);
     local transPortedLastMonth = AIIndustry.GetLastMonthTransported(idx, cargoIdx); 
     productionScore += cargoVal *  (producedLastMonth - transPortedLastMonth);
     score += productionScore; //add up multiple cargos like farms.
    } 
    
    

    //if(bestScoreIndustryIdx == null || ! AIIndustry.IsValidIndustry(bestScoreIndustryIdx)) 
    //{ 
    //  bestScoreIndustryIdx = idx;
    //}
    //else
    //{
    // if(score > industryInfo[bestScoreIndustryIdx].score) { bestScoreIndustryIdx = idx;}
    //}
    
    //insert a table into the array.
    industryInfo[idx] <-
      {
        index = idx,
        type = industryType,
        location = location,
        produces = produces,
        accepts = accepts,
        score = productionScore 
      };
    
  }
 //print("Ending industry info collection.");
 return this.industryInfo;

}


function DenverAndRioGrande::WagonValuator(engineId)
{
  return  AIEngine.GetCapacity(engineId) * AIEngine.GetMaxSpeed(engineId);
}

function DenverAndRioGrande::FindAllConsumersOf(industry)
{
  local industryIdx = industry.index;
  local cargoIdx = industry.produces.Begin()
  local consumers = AIIndustryList_CargoAccepting(cargoIdx);
  //print ("Found " + consumers.Count() + " consumers.");
  consumers.Valuate(DenverAndRioGrande.ConsumerDistance, industryIdx);
  consumers.Sort(AIAbstractList.SORT_BY_VALUE, true);
  return consumers;
}

function DenverAndRioGrande::FindBestWagon(cargoIndex)
{   
    local wagons = DenverAndRioGrande.FindWagons(cargoIndex);
    wagons.Valuate(DenverAndRioGrande.WagonValuator);
    return wagons.Begin();
}


function DenverAndRioGrande::FindWagons(cargoIndex)
{
    AILog.Info("Looking for " + AICargo.GetCargoLabel(cargoIndex) + " wagons.");
    local wagons = AIEngineList(AIVehicle.VT_RAIL);
    wagons.Valuate(AIEngine.IsWagon);
    wagons.RemoveValue(0);
    //print(wagons.Count() + " wagons." );
    wagons.Valuate(AIEngine.IsValidEngine);
    wagons.RemoveValue(0);
    //print(wagons.Count() + " Valid and buildable." );
    wagons.Valuate(AIEngine.CanRefitCargo, cargoIndex);
    wagons.RemoveValue(0);
    //print(wagons.Count() + " Can refit to cargo." );    
    wagons.Valuate(AIEngine.CanRunOnRail, AIRail.GetCurrentRailType());
    wagons.RemoveValue(0);
    //print(wagons.Count() + " Can run on rail." ); 
    //wagons.AddList(nonRefitWagons);
    if(wagons.Count() == 0)
    {
      AILog.Warning("Warning, no wagons can pull or be refitted to this cargo on the current track.");
    }
    return wagons;
}

function DenverAndRioGrande::FindCaboose()
{
  local engines = AIEngineList(AIVehicle.VT_RAIL);
  foreach(i, v in engines)
  {
    if(AIEngine.GetName(i) == "Caboose")
    {
      return i;
    }
  }
  return null;
}

function DenverAndRioGrande::FindBestEngine(wagonId, trainsize, cargoId)
{
    
  local minHP = 175 * trainsize;
  
  local speed = AIEngine.GetMaxSpeed(wagonId);
  if(speed == 0) {speed = 500;}
  local engines = AIEngineList(AIVehicle.VT_RAIL);
  engines.Valuate(AIEngine.IsWagon);
  engines.RemoveValue(1);
  
  engines.Valuate(AIEngine.IsValidEngine);
  engines.RemoveValue(0);
  engines.Valuate(AIEngine.CanPullCargo, cargoId);
  engines.RemoveValue(0);
  engines.Valuate(AIEngine.HasPowerOnRail, AIRail.GetCurrentRailType());
  engines.RemoveValue(0);
  
  engines.Valuate(AIEngine.GetPower);
  
  engines.Sort(AIAbstractList.SORT_BY_VALUE, false);
  if(engines.GetValue(engines.Begin()) < minHP ) //no engine can pull the wagon at it's top speed.
  {
   //print("No engine has enough horsepower to pull all the wagons well.");
  }
  else
  {
    engines.RemoveBelowValue(minHP);
  }
  
  
  engines.Valuate(AIEngine.GetMaxSpeed);
  engines.Sort(AIAbstractList.SORT_BY_VALUE, false);
  
  if(engines.GetValue(engines.Begin()) < speed ) //no engine can pull the wagon at it's top speed.
  {
   AILog.Info("No engine has top speed of wagon. Checking Fastest.");
   AILog.Info("The fastest engine to pull '" + AIEngine.GetName(wagonId) + "'' at full speed ("+ speed +") is '" + AIEngine.GetName(engines.Begin()) +"'" );
   Util.GetMaxLoan();
   local cash = AICompany.GetBankBalance(AICompany.COMPANY_SELF);
   if(cash > AIEngine.GetPrice(engines.Begin()) * 2 || AIVehicleList().Count() > 10)//if there are 10 trains, just return the best one and let it fail.
   {
    return engines.Begin();
   }
   else
   {
    AILog.Info("The company is poor. Picking a slower, cheaper engine.");
    engines.Valuate(AIEngine.GetPrice);
    engines.Sort(AIAbstractList.SORT_BY_VALUE, true);
    AILog.Info("The Cheapest engine to pull '" + AIEngine.GetName(wagonId) + "'  is '" + AIEngine.GetName(engines.Begin()) +"'" );
    return engines.Begin();
   }
  }
  
  engines.RemoveBelowValue(speed);
  engines.Valuate(AIEngine.GetPrice);
  engines.Sort(AIAbstractList.SORT_BY_VALUE, true);
  
  AILog.Info("The cheapest engine to pull '" + AIEngine.GetName(wagonId) + "'' at full speed ("+ speed +") is '" + AIEngine.GetName(engines.Begin()) +"'" );
  return engines.Begin();
  
}



function DenverAndRioGrande::ChooseRoute(bestIndustry, consumers)
{
  local producingTiles = AITileList_IndustryProducing(bestIndustry.index, 3);
  producingTiles = DenverAndRioGrande.GetBuildableFlatTiles(producingTiles);
  local producingStationManager = StationManager(bestIndustry.index, routeCount + " Producer " );
  local success = PlaceProducingStation(producingStationManager);
  
  IndustriesUsedAndDiscarded.AddItem(bestIndustry.index, 0);//add industries to the "done" list to keep things moving.
  
  if(!success)
  {
    return null;
  }
  

  consumers.Sort(AIAbstractList.SORT_BY_VALUE, true)
  
  AILog.Info("Trying " + consumers.Count()+ " consumers.");
  local tooShort = 0;
  local cantFit = 0;
  local count = 1;
  
   //Check existing stations first.
//   foreach(index, producer in activeProducingStations)
//   {
//    local consumer = producter.Consumer;
//    local station = TryLink(producingStationManager, bestInsdustry, producingTiles, consumer);
//    if(station != null)
//    {
//      return station;
//    }
//   }
  
  
  foreach(idx, val in consumers)
  {
    if(display_thinking)
    {
        AISign.BuildSign(AIIndustry.GetLocation(idx), "{" + routeCount + " Consumer " + count++ + "}");
    }
    local distance = AITile.GetDistanceManhattanToTile(AIIndustry.GetLocation(idx), AIIndustry.GetLocation(bestIndustry.index));
    if(distance < min_route_dist) 
    {
        tooShort++;
        continue;
    }
    local station = TryLink(producingStationManager, bestIndustry, producingTiles, idx);
    if(station != null)
    {
        return station;
    }
    else
    {
        cantFit++;
    }

  //Don't forget to manage things while routes are being looked for.
  Management(this);
  Util.PayDownLoan();
    
  }
  
  AILog.Info("Found no route to any consumer.");
  AILog.Info(tooShort + " were too close. ");
  AILog.Info(cantFit + " Didn't have a good spot or authority forbids station.");
  Util.PayDownLoan();
  IndustriesUsedAndDiscarded.AddItem(bestIndustry.index, 0);   
  return null;
}

function DenverAndRioGrande::TryLink(producingStationManager, bestIndustry, producingTiles, consumerIdx)
{

    local route = {};
  
    local consumingTiles = AITileList_IndustryAccepting(consumerIdx, 4);
        
    local consumerStationManager = StationManager(consumerIdx, routeCount + " Consumer");
    consumerStationManager = consumerStationManager.PlaceConsumingStationTest(stationLen, producingStationManager, RestrictTerraforming);
    if(consumerStationManager == null) {return null;}
    
    ignoreTiles.AddList(consumerStationManager.IgnoreTiles);

    local producerLocation = Util.GetClosestTile(consumerStationManager.Location, producingStationManager.JoinTiles )
    local consumerLocation = Util.GetClosestTile(producingStationManager.Location, consumerStationManager.JoinTiles);
    local routeStart = producerLocation;
    local routeGoal = consumerLocation;
        
    //consumingTiles = GetBuildableFlatTiles(consumingTiles);
        
    //local antHill = AntHill(routeStart, routeGoal, producingStationManager.Tiles, postPendTiles);
    local antHill = AntHill(routeStart, routeGoal, AIList(), AIList(), ignoreTiles);
 
    Util.PayDownLoan();
    AILog.Info("Starting Pathfinder. " + GetGameDateString());
    local path = antHill.FindPathMixed(this.Management, this);
    AILog.Info("pathfinder done. " + GetGameDateString());
    
    local minDistSquared = AIMap.DistanceSquare(routeStart, routeGoal);      
    
    if(path != null && path.Count() * path.Count()  <   minDistSquared)
    {
        AILog.Warning("Path was created but it is too short:" + path.Count()); //TODO: this is a bug in CrowPathfinder that needs looking into.
        path = null;
    }   
           
    Util.GetMaxLoan();
    if(path != null && AICompany.GetBankBalance(AICompany.COMPANY_SELF) < CalculateRouteCost(path, bestIndustry.produces.Begin(), antHill) )
    {
        
        local cost = CalculateRouteCost(path, bestIndustry.produces.Begin(), antHill);
        AILog.Warning("Not Enough Money (" + AICompany.GetBankBalance(AICompany.COMPANY_SELF) + ") to build route and trains (" + cost + "). Looking for alternate routes.");
        Util.PayDownLoan();
        path = null;
    }
    Util.PayDownLoan();
       
    if(path == null)
    {
      AILog.Info("found no path.");
      Util.ClearPathSigns();
      return null;
    }

    Util.GetMaxLoan();
    local pResult = producingStationManager._PlaceStation(stationLen);
    local cResult = consumerStationManager._PlaceStation(stationLen);
    if(!pResult || ! cResult)
    {
      AILog.Info("Failed to build one of the stations.");
      producingStationManager.RemoveRoute();
      producingStationManager._DeleteStation();
      consumerStationManager._DeleteStation();
      return null;
    }
    
    AILog.Info("building route.");
    //producingStationManager.SetConsumerStation(consumerStationManager);
    if(!BuildMainRail(path, producingStationManager, consumerStationManager, antHill))
    {
        AILog.Warning("path failed unexpectedly.");
        Util.ClearPathSigns();
        return null;
    }
    
    AILog.Info("building passing lane.");
    Util.ClearPathSigns();
    
    local passing = PassingLane(trainsize);
    local lanes = passing.BuildPassingLanes(path);
        
    route.Start <- producingStationManager.Location;
    route.End <- consumerStationManager.Location;
    route.Depot <- producingStationManager.Depot;
    route.Path <- path;
    route.Passing <- lanes; 
    producingStationManager.Route = route;
    producingStationManager.SetConsumerStation(consumerStationManager);
    
    if(lanes == null)
    {
        producingStationManager.PassingLen = 0;
    }
    else
    {
        producingStationManager.PassingLen = lanes.Count();
    }

    Util.PayDownLoan();
    return producingStationManager;  

}



function DenverAndRioGrande::BuildMainRail(path, producingStationManager, consumerStationManager, antHill)
{
    local count = 0;
    while(!FixPath(path))
    {
        if(count++ > 10)
        {
          AILog.Error("Broke Infinite loop in fixpath.");
          break;
        } 
    }
    
    Util.GetMaxLoan();
    local result = antHill.BuildRoute(path);
    Util.PayDownLoan();
    
    if(result == false)
    {
        producingStationManager.Route = { Path = path, Passing = null };
        producingStationManager.DeleteTrainsTrackAndStations();
    }
    
    return result;
}

function DenverAndRioGrande::PlaceProducingStation(producingStationManager)
{
    Util.GetMaxLoan();
    local success = producingStationManager.PlaceProducingStationTest(stationLen, RestrictTerraforming);
    Util.PayDownLoan();
    
    
    if(!success)
    {
        AILog.Info("No room to build station at prefered industry. Moving on."); 
        Util.DebugReplaceSign(producingStationManager.industryLocation, "station won't fit.");
    }

    return success;
}


function DenverAndRioGrande::CalculateRouteCost(path, cargo, antHill)
{
      local cost = antHill.CostPath(path); 
      cost *= 1.5; //factor in passing lanes.
      cost += 40000; //station cost estimate.     
      local wagon = this.FindBestWagon(cargo);
      local engine = this.FindBestEngine(wagon, trainsize, cargo);
      local power = AIEngine.GetPower(engine);
      cost += AIEngine.GetPrice(wagon) * this.trainsize;
      cost += AIEngine.GetPrice(engine);
      return cost;
}






function DenverAndRioGrande::GetEmptyAdjacentTiles(tile, goal)
{
   local tiles = AIList();
   tiles = AntHill.FindCompassPointsNunitsAway(tile, 1);
   tiles.Valuate(AITile.IsBuildable);
   tiles.KeepValue(1);
   tiles.Valuate(AITile.GetDistanceManhattanToTile, goal);
   tiles.Sort(AIAbstractList.SORT_BY_VALUE, true);
   return tiles;
}


function DenverAndRioGrande::GetNextClosestTile(list, goal)
{
  //print("Locations to choose from:" + list.Count());
  local tileList = AIList();
  tileList.AddList(list);
  tileList.Valuate(AITile.GetDistanceSquareToTile, goal);
  tileList.Sort(AIAbstractList.SORT_BY_VALUE, true);
  tileList.Begin();
  return tileList.Next();
}

function DenverAndRioGrande::GetBuildableFlatTiles(tileList)
{
  tileList.Valuate(AITile.IsBuildable)
  tileList.RemoveValue(0);
  tileList.Valuate(AITile.GetSlope);
  tileList.KeepValue(AITile.SLOPE_FLAT);
  return tileList;
}

function DenverAndRioGrande::CountFlatTiles(tile)
{
  local nodelist = AntHill.FindCompassPointsNunitsAway(tile, 1);
  nodelist.Valuate(AITile.IsBuildable)
  nodelist.RemoveValue(0);
  nodelist.Valuate(AITile.GetSlope);
  nodelist.KeepValue(AITile.SLOPE_FLAT);
  return nodelist.Count();

}




//Remove kinks from path.
function DenverAndRioGrande::FixPath(path)
{
  //print("Fixing Path...");
  foreach(tile, val in path)
  {
   local path2 = AIList();
   path2.AddList(path);
   path2.RemoveItem(tile);
   path2.Valuate(AIMap.DistanceManhattan, tile);
   path2.KeepValue(1);
   if(path2.Count() > 2)
   {
    foreach(i, v in path2)
    {
      path2.SetValue(i, path.GetValue(i));
    }
    
    if(display_thinking == 1)
    {
      AISign.BuildSign(tile, "{K}");
      foreach(index, value in path2)
      {
        AISign.BuildSign(index, "{K}")
      }
      
    }

   
    local val = path.GetValue(tile);
    
    path2.RemoveValue(val+1);
    path2.RemoveValue(val-1);
    
    
    if( TilesAreSameLevelAndFlat( tile, path2.Begin() ) )
    {
      path.RemoveBetweenValue(val, path.GetValue(path2.Begin()));
      return false;
    }
    
   }
  }
 return true;
}


function DenverAndRioGrande::TilesAreSameLevelAndFlat(tile, tile2)
{
  if(AITile.GetSlope(tile) == AITile.SLOPE_FLAT && AITile.GetSlope(tile2) == AITile.SLOPE_FLAT )
  {
    if(AITile.GetMaxHeight(tile) ==  AITile.GetMaxHeight(tile2) )
    {
      return true;
    }
  }
  
  return false;
}






function DenverAndRioGrande::GetGameDateString()
{
  local date = AIDate.GetCurrentDate();
  local year = AIDate.GetYear(date);
  local month = AIDate.GetMonth(date);
  local day = AIDate.GetDayOfMonth(date);
  return year + "-" + month + "-" + day;
}

function DenverAndRioGrande::GetWeightedRandomItem(nodelist, totalWeight)
{
  nodelist.Sort(AIAbstractList.SORT_BY_VALUE, false);
  if(AIController.GetSetting("randomize_station_picks") == 0)
  {
    return nodelist.Begin();
  }
  //print("getWeightedRandom() totalWeight=" + totalWeight + " totalNodes=" + nodelist.Count());
  local selected = null;
  if(nodelist.Count() == 0) {return null;}
  local keepTrying = true;
  while(keepTrying)
  {
    local random = AIBase.RandRange(totalWeight);
    foreach(i, val in nodelist)
    {
      random -= val;
      //print("Random =" + random + " val=" + val + " index:" + i);
      AIController.Sleep(1);
      if (random <= 0)
      {
          //print("Picked One: " + i);  
          keepTrying = false;
          selected = i;
          break;
      }
    }
  }
  //print("Selected: " + selected); 
  return selected;
}

function DenverAndRioGrande::PickFromBestIndustries()
{
  local industryInfo = this.getIndustryInfo();
  
  local pickList = AIList();
  local totalWeight = 0;
  foreach(idx, val in industryInfo)
  {
    if(val.score > 0)
    {
        pickList.AddItem(idx, val.score);
        totalWeight += val.score;
    }
  }
  
  bestScoreIndustryIdx = GetWeightedRandomItem(pickList, totalWeight);
  return bestScoreIndustryIdx;
}
