class SCPClient_CompanyValueGS {

	/* Private members variables */
	static COMMAND_SET = "Company Value GS";

	_scp = null;                   // SCPLib instance
	_goal_mode = null;
	_goal_value = null;
	_company_value_gs_game = null; // Is this a Company Value GS game? (true/null)

	/**
	 * Library constructor
	 * @param scp_ptr Instance of SCPLib. If null is passed, the library
	 * will act as if no Company Value GS has been detected.
	 */
	constructor(scp_ptr)
	{
		this._goal_mode = null;
		this._goal_value = null;
		this._company_value_gs_game = null;

		this._scp = scp_ptr;

		this.RegisterCommands();
		this.AskForGoals();
	}

	/* Public methods */

	/**
	 * Is this a Company Value GS game?
	 * @return true, if the game has a GS that respond to Company Value GS
	 * commands, otherwise false.
	 * @note If you call this method too early, it will return false even if
	 * the game has the Company Value GS with SCP activated because a round-trip
	 * has not yet been completed.
	 */
	function IsCompanyValueGSGame();

	/**
	 * Is Company Value GS running in Goal mode?
	 * In this mode, companies are competing to become the first to reach a
	 * company value target. See GetCurrentTargetValue to get the value.
	 * @return true, if Company Value GS is running in Goal mode.
	 * @note When it returns false, it doesn't necessarily mean that Company Value GS
	 * is running in Ranking mode.
	 * @note When the target value is reached by one of the companies, Company Value GS
	 * pauses the game, and switches from Goal mode to Ranking mode after a user presses
	 * the "Continue" button. Note that AIs are unable to respond when the game is paused.
	 */
	function IsCompanyValueGSInGoalMode();

	/**
	 * Is Company Value GS running in Ranking mode?
	 * In this mode, companies are competing between each other to be the most
	 * valuable company at all times. See GetBestCompanyValue to get the value.
	 * @return true, if Company Value GS is running in Ranking mode.
	 * @note When it returns false, it doesn't necessarily mean that Company Value GS
	 * is running in Goal mode.
	 */
	function IsCompanyValueGSInRankingMode();

	/**
	 * Get the Company ID of the current most valuable company.
	 * @return CompanyID of the company with the highest value at the moment.
	 * @note Returns AICompany.COMPANY_INVALID when Company Value GS SCP isn't active,
	 * or when no companies are found (shouldn't happen, because we're a company too).
	 */
	function GetBestCompanyID();

	/**
	 * Get the value of the current most valuable company.
	 * @return integer value of the company with the best value at the moment.
	 * @note Returns -1 when Company Value GS SCP isn't active, or when no companies
	 * are found (shouldn't happen, because we're a company too).
	 */
	function GetBestCompanyValue();

	/**
	 * Get the current targeted value to be reached.
	 * In Goal mode, the value is defined in the Company Value GS settings.
	 * In Ranking mode, the value is that of the current most valuable company.
	 * @return integer value of the current target to be reached.
	 * @note Returns -1 when Company Value GS SCP isn't active, or when no companies
	 * are found (shouldn't happen, because we're a company too).
	 */
	function GetCurrentTargetValue();

	/**
	 * List of companies, ranked by their respective values in descending order.
	 * @return AIList with Company ID as the item, and Company Value as the value
	 * @note Returns an empty AIList when Company Value GS SCP isn't active, or when
	 * no companies are found (shouldn't happen, because we're a company too).
	 */
	function RankingList();

	/**
	 * Get the rank position of the provided Company ID from a list of running
	 * companies with their respective company values.
	 * @param company_id The Company ID to get the rank of.
	 * @return integer position of provided company in relation to the
	 * others regarding their company value.
	 * @note The rank at the top has a value of 1. The rank at the bottom has a value
	 * dependent on the number of companies running in the game.
	 * @note Returns -1 when the provided Company ID is invalid, or when no companies
	 * are found (shouldn't happen, because we're a company too).
	 */
	function GetCompanyIDRank(company_id);

	/**
	 * Get the company value of the provided Company ID.
	 * @param company_id The Company ID to get the company value of.
	 * @return integer value of the specified company.
	 * @note Returns -1 when Company Value GS SCP isn't active, or when the provided
	 * Company ID is invalid.
	 */
	function GetCompanyIDValue(company_id);

	/**
	 * Get the difference in company value between the current targeted value and the
	 * provided Company ID's value.
	 * @param company_id The Company ID in which the difference is based of.
	 * @return integer value of the difference in company value between the current
	 * targeted value and the specified Company ID's value.
	 * @note The difference is always equal or higher than zero.
	 * @note Returns -1 when Company Value GS SCP isn't active, or when the provided
	 * Company ID is invalid.
	 */
	function GetCompanyIDDiffToTarget(company_id);

	/**
	 * Get the difference in company value between the value of the best company and
	 * the provided Company ID's value.
	 * @param company_id The Company ID in which the difference is based of.
	 * @return integer value of the difference in company value between the best
	 * company and the specified Company ID.
	 * @note The difference is always equal or higher than zero.
	 * @note Returns -1 when Company Value GS SCP isn't active, or when the provided
	 * Company ID is invalid.
	 */
	function GetCompanyIDDiffToBest(company_id);

	/**
	 * Get the difference in company value between the value of the provided company
	 * and the next ahead or behind in the rank list.
	 * @param company_id The Company ID in which the difference is based of.
	 * @param next boolean value (use 'true' to compare with the next
	 * ahead in rank or 'false' to compare with the next behind in rank).
	 * @note The difference is always equal or higher than zero.
	 * @note Returns -1 when Company Value GS SCP isn't active, or when the provided
	 * Company ID is invalid.
	 * @note Returns 0 when there is no company ahead or behind in rank.
	 */
	function GetCompanyIDDiffToNext(company_id, next);
}

/**** Private methods: ****/

function SCPClient_CompanyValueGS::RegisterCommands()
{
	if (this._scp == null) return;

	local self = this;

	// AI -> GS commands:
	this._scp.AddCommand("CurrentGoal", COMMAND_SET, self, SCPClient_CompanyValueGS.ReceivedCurrentGoalCommand);

	// GS -> AI commands:
}

function SCPClient_CompanyValueGS::AskForGoals()
{
	if (this._scp == null) return;

	this._scp.QueryServer("CurrentGoal", COMMAND_SET, [AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)]);
}

/**** Public API methods: ****/

function SCPClient_CompanyValueGS::IsCompanyValueGSGame()
{
	if (this._scp == null) return false;

	return this._company_value_gs_game == true;
}

function SCPClient_CompanyValueGS::IsCompanyValueGSInGoalMode()
{
	if (this._scp == null) return false;

	return this._goal_mode == 1;
}

function SCPClient_CompanyValueGS::IsCompanyValueGSInRankingMode()
{
	if (this._scp == null) return false;

	return this._goal_mode == 0;
}

function SCPClient_CompanyValueGS::GetBestCompanyID()
{
	if (this._scp == null) return AICompany.COMPANY_INVALID;
	if (this._company_value_gs_game != true) return AICompany.COMPANY_INVALID;

	local best_company_id = AICompany.COMPANY_INVALID;
	local best_company_value = -1;
	for (local c_id = AICompany.COMPANY_FIRST; c_id < AICompany.COMPANY_LAST; c_id++) {
		if (AICompany.ResolveCompanyID(c_id) != AICompany.COMPANY_INVALID) {
			local c_value = AICompany.GetQuarterlyCompanyValue(c_id, AICompany.CURRENT_QUARTER);
			if (c_value >= best_company_value) {
				best_company_id = c_id;
				best_company_value = c_value;
			}
		}
	}
	return best_company_id;
}

function SCPClient_CompanyValueGS::GetBestCompanyValue()
{
	if (this._scp == null) return -1;
	if (this._company_value_gs_game != true) return -1;

	local best_company_value = -1;
	for (local c_id = AICompany.COMPANY_FIRST; c_id < AICompany.COMPANY_LAST; c_id++) {
		if (AICompany.ResolveCompanyID(c_id) != AICompany.COMPANY_INVALID) {
			local c_value = AICompany.GetQuarterlyCompanyValue(c_id, AICompany.CURRENT_QUARTER);
			if (c_value >= best_company_value) {
				best_company_value = c_value;
			}
		}
	}
	return best_company_value;
}

function SCPClient_CompanyValueGS::GetCurrentTargetValue()
{
	if (this._scp == null) return -1;
	if (this._company_value_gs_game != true) return -1;

	if (this._goal_mode == 1) return this._goal_value;
	return this.GetBestCompanyValue();
}

function SCPClient_CompanyValueGS::RankingList()
{
	if (this._scp == null) return AIList();
	if (this._company_value_gs_game != true) return AIList();

	local global_list = AIList();
	for (local c_id = AICompany.COMPANY_FIRST; c_id < AICompany.COMPANY_LAST; c_id++) {
		if (AICompany.ResolveCompanyID(c_id) != AICompany.COMPANY_INVALID) {
			local c_value = AICompany.GetQuarterlyCompanyValue(c_id, AICompany.CURRENT_QUARTER);
			global_list.AddItem(c_id, c_value);
		}
	}
	global_list.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);
	return global_list;
}

function SCPClient_CompanyValueGS::GetCompanyIDRank(company_id)
{
	if (this._scp == null) return -1;
	if (this._company_value_gs_game != true) return -1;

	if (AICompany.ResolveCompanyID(company_id) == AICompany.COMPANY_INVALID) return -1;

	local global_list = this.RankingList();
	local rank = 0;
	for (local c_id = global_list.Begin(); !global_list.IsEnd(); c_id = global_list.Next()) {
		rank++;
		if (c_id == company_id) {
			return rank;
		}
	}
	return -1;
}

function SCPClient_CompanyValueGS::GetCompanyIDValue(company_id)
{
	if (this._scp == null) return -1;
	if (this._company_value_gs_game != true) return -1;

	if (AICompany.ResolveCompanyID(company_id) == AICompany.COMPANY_INVALID) return -1;

	return AICompany.GetQuarterlyCompanyValue(company_id, AICompany.CURRENT_QUARTER);
}

function SCPClient_CompanyValueGS::GetCompanyIDDiffToTarget(company_id)
{
	if (this._scp == null) return -1;
	if (this._company_value_gs_game != true) return -1;

	if (AICompany.ResolveCompanyID(company_id) == AICompany.COMPANY_INVALID) return -1;

	local current_target = this.GetCurrentTargetValue();
	local company_id_value = this.GetCompanyIDValue(company_id);

	local difference = -1;
	if (current_target != -1 && company_id_value != -1) {
		difference = current_target - company_id_value;
	}

	return difference;
}

function SCPClient_CompanyValueGS::GetCompanyIDDiffToBest(company_id)
{
	if (this._scp == null) return -1;
	if (this._company_value_gs_game != true) return -1;

	if (AICompany.ResolveCompanyID(company_id) == AICompany.COMPANY_INVALID) return -1;

	local best_company_value = this.GetBestCompanyValue();
	local company_id_value = this.GetCompanyIDValue(company_id);

	local difference = -1;
	if (best_company_value != -1 && company_id_value != -1) {
		difference = best_company_value - company_id_value;
	}

	return difference;
}

function SCPClient_CompanyValueGS::GetCompanyIDDiffToNext(company_id, next)
{
	if (this._scp == null) return -1;
	if (this._company_value_gs_game != true) return -1;

	if (AICompany.ResolveCompanyID(company_id) == AICompany.COMPANY_INVALID) return -1;
	if (typeof(next) != "bool") return -1;

	local global_list = this.RankingList();
	if (global_list.Count() == 0) return -1;

	local difference = 0;
	if (global_list.Count() == 1) return difference;

	local company_id_rank = this.GetCompanyIDRank(company_id);
	if (company_id_rank == -1) return -1;
	if (company_id_rank == 1 && next || company_id_rank == global_list.Count() && !next) return difference;

	local rank = 0;
	local next_company_id = AICompany.COMPANY_INVALID;
	for (local c_id = global_list.Begin(); !global_list.IsEnd(); c_id = global_list.Next()) {
		rank++;
		if (rank == (company_id_rank + (next ? -1 : 1))) {
			next_company_id = c_id;
			break;
		}
	}
	if (AICompany.ResolveCompanyID(next_company_id) == AICompany.COMPANY_INVALID) return -1;

	local company_id_value = this.GetCompanyIDValue(company_id);
	local next_company_id_value = this.GetCompanyIDValue(next_company_id);
	if (company_id_value == -1 || next_company_id_value == -1) return -1;

	if (next) {
		difference = next_company_id_value - company_id_value;
	} else {
		difference = company_id_value - next_company_id_value;
	}
	return difference;
}


/** SCP in/out methods (private to the library) */

/*****************************************************************
 *                                                               *
 *   Outgoing Commands - commands that we can send to GSes       *
 *                                                               *
 *****************************************************************/


/*****************************************************************
 *                                                               *
 *   Incoming Commands - commands that we can get from AIs       *
 *                                                               *
 *****************************************************************/

// These methods are called by the SCP library when we call this._scp.Check() and there is
// a received incoming message.

// Use 'self' instead of 'this'.

function SCPClient_CompanyValueGS::ReceivedCurrentGoalCommand(message, self)
{
	self._company_value_gs_game = true;

	local data0 = message.GetIntData(0);   // Goal Mode
	local data1 = message.GetStringData(1); // Goal Value divided by 1000, received as a string
	local data = [data0, data1];

	self._goal_mode = data0;
	self._goal_value = data1.tointeger() * 1000;

	local s = "[";
	foreach (d in data) {
		if (s != "[") {
			s += ", ";
		}
		if (d == null) {
			s += "null";
		} else if (typeof(d) == "string") {
			s += "\"" + d + "\"";
		} else {
			s += d.tostring();
		}
	}
	s += "]";

	AILog.Info("SCP: Reveived CurrentGoal command with data: " + s);
}