// Perform service on the network


//Each service has a group of vehicles
function RailwAI::CheckServices() {
	local groupList = AIGroupList()
	local numOfGroups = groupList.Count()
	for (local group = groupList.Begin(); groupList.HasNext(); group = groupList.Next()) {
		local vehicleType = AIGroup.GetVehicleType(group)
		local hasEnoughVehicles = true
		local list = AIVehicleList_Group(group)
		local firstVehicle = null
		local secondVehicle = null
		local numVehicles = list.Count()
		local runningVehicles = 0
		local stoppedVehicles = 0
		local profitableVehicles = 0
		local totalProfit = 0
		local maxAge = 0
		local minAge = 99999
		for (local vehicle = list.Begin(); list.HasNext(); vehicle = list.Next()) {
			if (firstVehicle == null)
				firstVehicle = vehicle
			else if (secondVehicle == null) {
				if (AIBase.RandRange(10) < 5) {
					secondVehicle = firstVehicle
					firstVehicle = vehicle
				} else {
					secondVehicle = vehicle
				}
			}
			//Count running, profitable vehicles
			if (AIVehicle.GetCurrentSpeed(vehicle) > 20) {
				runningVehicles += 1		
			} else {
				stoppedVehicles += 1
			}
			local age = AIVehicle.GetAge(vehicle)
			if (age < minAge) minAge = age
			if (age > maxAge) maxAge = age
			local profit = AIVehicle.GetProfitThisYear(vehicle)
			if (profit > 1000) {
				profitableVehicles += 1		
			}
			totalProfit += profit
		}
		if (numVehicles == 0) {
			LogError("I found an empty group. Deleting this group that has name " + AIGroup.GetName(group) + ".")
			AIGroup.DeleteGroup(group)
			return false
		}
		if (numVehicles == 1) {
			hasEnoughVehicles = false
		}
		
		local depot = null
		local station1 = null
		local cargoType = GetVehicleCargoType(firstVehicle)
		local cargoCapacity = AIVehicle.GetCapacity(firstVehicle, cargoType)
		local totalWaitingCargo = 0
		local totalTransportCapacity = 1
		for (local i = 0; i < AIOrder.GetOrderCount(firstVehicle); i++) {
			if (!AIOrder.IsValidVehicleOrder(firstVehicle, i)) {
				AIOrder.RemoveOrder(firstVehicle, i)
				return false
			}
			if (depot == null && AIOrder.IsGotoDepotOrder(firstVehicle, i))
				depot = AIOrder.GetOrderDestination(firstVehicle, i)
			if (AIOrder.IsGotoStationOrder(firstVehicle, i)) {
				local location = AIOrder.GetOrderDestination(firstVehicle, i)
				if (station1 == null)
					station1 = location
				if (AICargo.HasCargoClass(cargoType, AICargo.CC_PASSENGERS)) {
					local flags = AIOrder.GetOrderFlags(firstVehicle, i)
					if (flags & AIOrder.OF_FULL_LOAD_ANY)
						AIOrder.SetOrderFlags(firstVehicle, i, AIOrder.OF_NONE)
					else if (!IsStationProperlyServed(location, cargoType) && AIBase.RandRange(10) < 4)
						AIOrder.SetOrderFlags(firstVehicle, i, AIOrder.OF_FULL_LOAD_ANY)
				}
			}
		}
		list = AIVehicleList_Group(group)
		for (local vehicle = list.Begin(); list.HasNext(); vehicle = list.Next()) {
			totalTransportCapacity += AIVehicle.GetCapacity(vehicle, cargoType)
		}
		list = AIStationList_Vehicle(firstVehicle)
		for (local station = list.Begin(); list.HasNext(); station = list.Next()) {
			totalWaitingCargo += AIStation.GetCargoWaiting(station, cargoType)
		}

		if (station1 != null && !IsStationProperlyServed(station1, cargoType)) {
			hasEnoughVehicles = false
		}
		if (AICargo.HasCargoClass(cargoType, AICargo.CC_PASSENGERS)) {
			if (totalWaitingCargo > totalTransportCapacity)
				hasEnoughVehicles = false
		} else { //Freight, we always want something that loads cargo
			if (totalWaitingCargo > totalTransportCapacity / numVehicles / 2)
				hasEnoughVehicles = false
		}

		
		if (list.Count() > numVehicles)
			hasEnoughVehicles = false
			
		//Now check if we have a condition, in which we are not allowed to buy more vehicles for this connection
		//if (list.Count()*4-2 <= numVehicles)
		//	hasEnoughVehicles = true
		if (runningVehicles < numVehicles/2)
			hasEnoughVehicles = true
		if (stoppedVehicles > 1)
			hasEnoughVehicles = true
		if (AICargo.HasCargoClass(cargoType, AICargo.CC_PASSENGERS)) {
			if (numVehicles >= 2 && profitableVehicles < 2)
				hasEnoughVehicles = true				
			if (numVehicles >= 2 && numVehicles >= numOfGroups*2-2)
				hasEnoughVehicles = true
		} else {
			//if there are multiple vehicles running, the service exists for more than a year and it is unprofitable, don't buy a new vehicle
			if (numVehicles >= 2 && totalProfit / numVehicles < 500 && maxAge > 400)
				hasEnoughVehicles = true
			if (numVehicles >= 2 && numVehicles >= numOfGroups*4) //spread chances
				hasEnoughVehicles = true
		}
		
			
		if (minAge < 30) { //we already bought a new vehicle this month
			hasEnoughVehicles = true
		}
		
		//If the service is not served well, or only has one vehicle, buy a new vehicle
		if (numVehicles == 1 && AIVehicle.GetVehicleType(firstVehicle) == AIVehicle.VT_RAIL) {
			//for trains, we want two different looking trains
			hasEnoughVehicles = BuyTrainLikeThis(firstVehicle)
		}
		if (!hasEnoughVehicles && depot != null) {
			local requestedNewVehicles = 1 + totalWaitingCargo * numVehicles / totalTransportCapacity / 2
			if (AICargo.HasCargoClass(cargoType, AICargo.CC_PASSENGERS))
				requestedNewVehicles = 1
			Log("The group with name " + AIGroup.GetName(group) + " has not enough vehicles. I should add " + requestedNewVehicles + ".")
			if (requestedNewVehicles > 10)
				requestedNewVehicles = 10
			for (local i = 0; i < requestedNewVehicles; i++) {
				local vehicle = AIVehicle.CloneVehicle(depot, firstVehicle, true)
				if (AIVehicle.IsValidVehicle(vehicle)) {
					AIVehicle.StartStopVehicle(vehicle);
					Log("I have cloned a vehicle in group " + AIGroup.GetName(group));
					GiveVehicleAName(vehicle)
				} else if (AIError.GetLastError() == AIVehicle.ERR_VEHICLE_NOT_AVAILABLE) {
					if ( AIVehicle.GetVehicleType(firstVehicle) == AIVehicle.VT_RAIL) {
						//We can not clone this train anymore, just buy a new train
						BuyTrainLikeThis(firstVehicle);
					} else {
						BuyVehicleLike(firstVehicle)
					}
				}
			}
		}
		//For trains older than 1 year, check if the train length should be adjusted
		if (vehicleType == AIVehicle.VT_RAIL) {
			local totalLoad = 0
			local totalCapacity = 0
			local maxCapacity = 0
			list = AIVehicleList_Group(group)
			for (local vehicle = list.Begin(); list.HasNext(); vehicle = list.Next()) {
				local capacity = AIVehicle.GetCapacity(vehicle, cargoType)
				totalCapacity += capacity
				if (capacity > maxCapacity) maxCapacity = capacity
				totalLoad += AIVehicle.GetCargoLoad(vehicle, cargoType)
			}
			local averageCapacity = totalCapacity / list.Count()
			local requestedCapacity = GetRequestedTransportCapacity(firstVehicle, cargoType)
			local capacityOffered = averageCapacity *100 / (requestedCapacity + 1)
			
			if (capacityOffered > 180 && (AICargo.HasCargoClass(cargoType, AICargo.CC_PASSENGERS) || AICargo.HasCargoClass(cargoType, AICargo.CC_MAIL))) {
				//capacity is too high, send some trains to depots
				list = AIVehicleList_Group(group)
				local capacityThreshold = maxCapacity * 3 / 5
				for (local vehicle = list.Begin(); list.HasNext(); vehicle = list.Next()) {
					local capacity = AIVehicle.GetCapacity(vehicle, cargoType)
					if (AIVehicle.GetAge(vehicle) > 180 && AIVehicle.GetNumWagons(vehicle) > 2 && capacity > capacityThreshold &&
						AIVehicle.GetCargoLoad(vehicle, cargoType) * 100 / (capacity+1) <= 60 && AIBase.RandRange(10) < 5) {
						AIVehicle.SendVehicleToDepot(vehicle)
					}
				}
			} else if (capacityOffered < 85) {
				//offered capacity is too low, send some full trains that are not maximum length yet to depots
				local maxTrainLength = 16 * GetPlatformLengthOfTrain(firstVehicle)
				list = AIVehicleList_Group(group)
				local count = 0
				for (local vehicle = list.Begin(); list.HasNext(); vehicle = list.Next()) {
					if (count < 2 && AIVehicle.GetAge(vehicle) > 360 && AIVehicle.GetCargoLoad(vehicle, cargoType) * 100 / (AIVehicle.GetCapacity(vehicle, cargoType)+1) >= 50) {
						local trainLength = AIVehicle.GetLength(vehicle)
						if (trainLength < maxTrainLength / 2 || (trainLength < maxTrainLength*3/4 && AIVehicle.GetNumWagons(vehicle) > 2 && AIBase.RandRange(10) < 5)) {
							AIVehicle.SendVehicleToDepot(vehicle)
							count = count + 1
						}
					}
				}
			}
		}
	}
	return true //please lie that we're doing great
}


/* Search the map for level crossings. We don't like accidents, so we try to avoid them.
 * For adjacent level crossings, we try to build a bridge around
 * For single track level crossings, we build path signals before and after the crossing
 */
function RailwAI::HandleLevelCrossings() {
	local Xmax = AIMap.GetMapSizeX()
	local Ymax = AIMap.GetMapSizeY()
	local myCompany = AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)
	LogWarning("Finding all level crossings on a map with size " + Xmax + ", " + Ymax)
	local crossings = []
	foreach (tile in this.myTiles) {
		if (AIRail.IsLevelCrossingTile(tile))
			crossings.append(tile)
		if (AIRoad.IsRoadTile(tile) && AIBase.RandRange(10) < 1) {
			if (AIRail.IsLevelCrossingTile(tile+dXY(1,0)))
				crossings.append(tile+dXY(1,0))
			if (AIRail.IsLevelCrossingTile(tile+dXY(0,1)))
				crossings.append(tile+dXY(0,1))
		}
	}
	
	//search for crossings that are mine
	foreach (tile in crossings) {
		if (!AIRail.IsLevelCrossingTile(tile))
			continue
		local x = AIMap.GetTileX(tile)
		local y = AIMap.GetTileY(tile)
		//owner contains the owner of the rail
		local railOwner = AITile.GetOwner(tile)
		if (railOwner == myCompany) {
			// Use path signals around the railway crossing
			MigrateToPathSignalsAround(tile, 5)
			//build a bridge over the railway crossing
			this.idea = RailCrossingReplaceIdea(tile)
			if (this.idea.plans.len() > 0) {
				this.idea.Execute()
			}
			this.idea = null			
			//LogTile("My crossing at", tile)
		}
		//check if it still a level crossing, or that it was already replaced by an other construction
		if (!AIRail.IsLevelCrossingTile(tile))
			continue
		foreach (roadType in [AIRoad.ROADTYPE_ROAD, AIRoad.ROADTYPE_TRAM]) {
			if (AIRoad.HasRoadType(tile, roadType)) {
				AIRoad.SetCurrentRoadType(roadType)
				foreach (delta in [dXY(0,1), dXY(1,0)]) {
					if (AIRail.IsLevelCrossingTile(tile - delta)) continue
					local tileB = tile + delta
					while (AIRail.IsLevelCrossingTile(tileB))
						tileB = tileB + delta
					if (AIRoad.AreRoadTilesConnected(tileB, tileB - delta) && AIRoad.AreRoadTilesConnected(tile, tile - delta) &&
					AIRoad.HasRoadType(tileB, roadType) && AIRoad.HasRoadType(tile - delta, roadType)) {
						if (tile +delta != tileB || roadType == AIRoad.ROADTYPE_TRAM || AIBase.RandRange(10) < 3) {
						//I should build a detour route around it
							LogTile("Level crossing should be avoided at", tile)
							LogTile("Level crossing should be avoided at", tileB)
							this.idea = RoadDetourIdea(tile, tileB)
							if (this.idea.plans.len() > 0) {
								this.idea.Execute()
							}
							this.idea = null
						}
					}
				}
			}
		}
	}
}


/* Search the map for locks within canals. We don't like to accidentally block another canal.
 * if we did, we will build a detour canal
 */
function RailwAI::HandleLocks() {
	local Xmax = AIMap.GetMapSizeX()
	local Ymax = AIMap.GetMapSizeY()
	local myCompany = AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)
	LogWarning("Finding all locks on a map with size " + Xmax + ", " + Ymax)
	local locks = []
	for (local x = 1; x < Xmax; x++) {
		for (local y = 1; y < Ymax; y++) {
			local tile = AIMap.GetTileIndex(x,y)
			if (AIMarine.IsLockTile(tile) && AITile.GetSlope(tile) != AITile.SLOPE_FLAT)
				locks.append(tile)
		}
	}
	Log(locks.len() + " locks found")
	foreach (tile in locks) {
		local x = AIMap.GetTileX(tile)
		local y = AIMap.GetTileY(tile)
		//owner contains the owner of the rail
		local lockOwner = AITile.GetOwner(tile)
		if (lockOwner == myCompany || AITile.GetOwner(tile + dXY(1,1)) == myCompany || AITile.GetOwner(tile + dXY(-1,-1)) == myCompany) {
			//We are owner of something around this lock. Let's check if the world looks nice here.
			local directions = [Direction.SW, Direction.NE]
			if (AITile.GetSlope(tile) == AITile.SLOPE_NW || AITile.GetSlope(tile) == AITile.SLOPE_SE) {
				directions = [Direction.SE, Direction.NW]
			}
			foreach (direction in directions) {
				local coords = VC.VectorCoordinates(tile, direction)
				local checkTile = VC.GetTile(coords, [1, 2])
				local refTile = VC.GetTile(coords, [1, 1])
				if (IsAWaterTile(refTile) && !IsAWaterTile(checkTile)) {
					if (AITile.GetMaxHeight(checkTile) > AITile.GetMaxHeight(refTile)) {
						AITile.LowerTile(checkTile, AITile.GetSlope(checkTile))
					} else {
						AITile.RaiseTile(checkTile, AITile.GetComplementSlope(checkTile))
					}
					AIMarine.BuildCanal(checkTile)
				}
				
				checkTile = VC.GetTile(coords, [-1, 2])
				refTile = VC.GetTile(coords, [-1, 1])
				if (IsAWaterTile(refTile) && !IsAWaterTile(checkTile)) {
					if (!AITile.GetSlope(checkTile) != AITile.SLOPE_FLAT) {
						if (AITile.GetMaxHeight(checkTile) > AITile.GetMaxHeight(refTile)) {
							AITile.LowerTile(checkTile, AITile.GetSlope(checkTile))
						} else {
							AITile.RaiseTile(checkTile, AITile.GetComplementSlope(checkTile))
						}
					}
					AIMarine.BuildCanal(checkTile)
				}
			}
		}
	}
}

function RailwAI::BuildHQ() {
	local myCompany = AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)
	if (AICompany.GetCompanyHQ(myCompany) != AIMap.TILE_INVALID) return true
	local deltas = [dXY(AIBase.RandRange(14)+3, AIBase.RandRange(14)+3), dXY(-AIBase.RandRange(14)-3, AIBase.RandRange(14)+3),
		dXY(-AIBase.RandRange(14)-3, -AIBase.RandRange(14)-3), dXY(AIBase.RandRange(14)+3, -AIBase.RandRange(14)-3)] 
	local list = AIStationList(AIStation.STATION_ANY)
	for (local station = list.Begin(); list.HasNext(); station = list.Next()) {
		local tile = AIStation.GetLocation(station)
		foreach (delta in deltas) {
			if (AICompany.BuildCompanyHQ(tile + delta)) {
				LogTile("Company headquarters has been built near", tile, "hq")
				LogTile("Company headquarters has been built at", tile + delta, "hq")
				return true
			}
		}
	}
}

function RailwAI::CheckEvents() {
	while (AIEventController.IsEventWaiting()) {
		local e = AIEventController.GetNextEvent();
		switch (e.GetEventType()) {
			case AIEvent.ET_SUBSIDY_OFFER: 
				break //ignore subsidies
			case AIEvent.ET_SUBSIDY_OFFER_EXPIRED: 
				break //ignore subsidies
			case AIEvent.ET_SUBSIDY_AWARDED: 
				break //ignore subsidies
			case AIEvent.ET_SUBSIDY_EXPIRED: 
				break //ignore subsidies
			case AIEvent.ET_COMPANY_NEW:
				local ec = AIEventCompanyNew.Convert(e)
				local id = ec.GetCompanyID()
				LogWarning("Hello " + AICompany.GetName(id)+"!")
				break
			case AIEvent.ET_COMPANY_IN_TROUBLE: 
				local ec = AIEventCompanyInTrouble.Convert(e)
				local id = ec.GetCompanyID()
				if (id == AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)) {
					LogError("Help! I'm in trouble according to the government! I don't know what to do!")
				}
				break
			case AIEvent.ET_COMPANY_ASK_MERGER:
				local ec = AIEventCompanyAskMerger.Convert(e)
				local id = ec.GetCompanyID()
				if (ec.AcceptMerger()) {
					LogWarning("#YOLO I just bought another company!")
				}
				break
			case AIEvent.ET_COMPANY_MERGER:
				break //do nothing
			case AIEvent.ET_COMPANY_BANKRUPT: 
				local ec = AIEventCompanyBankrupt.Convert(e)
				local id = ec.GetCompanyID()
				if (id == AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)) {
					throw("I went bankrupt. Bye!")
				} else if (id != AICompany.COMPANY_INVALID) {
					LogError("Goodbye " + AICompany.GetName(id) + "! It seems that " + AICompany.GetPresidentName(id) + " didn't do good business.")
				}
				break
			case AIEvent.ET_VEHICLE_CRASHED:
 			case AIEvent.AI_ET_VEHICLE_CRASHED:
				local ec = AIEventVehicleCrashed.Convert(e);
				local v  = ec.GetVehicleID();
				local tile = ec.GetCrashSite();
				this.dangerTiles.AddItem(tile, 0)
				AILog.Info("We have a crashed vehicle (" + v + ")");
				break;
			case AIEvent.ET_VEHICLE_LOST: 
				break //TODO
			case AIEvent.ET_VEHICLE_WAITING_IN_DEPOT: 
				local ec = AIEventVehicleWaitingInDepot.Convert(e)
				local vehicle = ec.GetVehicleID()
				if (AIVehicle.IsValidVehicle(vehicle) && AIVehicle.IsStoppedInDepot(vehicle)) {//it might already be serviced in the depot
					HandleVehicleInDepot(vehicle)
				}
				break
			case AIEvent.ET_VEHICLE_UNPROFITABLE: 
				break
			case AIEvent.ET_INDUSTRY_OPEN: 
				break
			case AIEvent.ET_INDUSTRY_CLOSE: 
				break
			case AIEvent.ET_ENGINE_PREVIEW: 
				local ec = AIEventEnginePreview.Convert(e)
				ec.AcceptPreview() //just accept, we'll maybe use it
				break
			case AIEvent.ET_ENGINE_AVAILABLE:
				break //do nothing
			case AIEvent.ET_STATION_FIRST_VEHICLE:
				local ec = AIEventStationFirstVehicle.Convert(e)
				local station = ec.GetStationID()
				local vehicle = ec.GetVehicleID()
				if (AIStation.IsValidStation(station)) { //this checks whether the station is mine
					//name this vehicle after the city
					AIVehicle.SetName(vehicle, AIVehicle.GetUnitNumber(vehicle) + " " + AITown.GetName(AIStation.GetNearestTown(station)))
				}
				break
			case AIEvent.ET_TOWN_FOUNDED: 
				local ec = AIEventTownFounded.Convert(e)
				LogWarning("Welcome in our world, citizens of " + AITown.GetName(ec.GetTownID()))
				break
			case AIEvent.ET_DISASTER_ZEPPELINER_CRASHED: 
				break //aircraft stuff
			case AIEvent.ET_DISASTER_ZEPPELINER_CLEARED: 
				break //aircraft stuff
			case AIEvent.ET_AIRCRAFT_DEST_TOO_FAR:
				AILog.Error("An aircraft has a destination that is too far. That surprises me, because I don't use aircraft.")
				break
			case AIEvent.ET_EXCLUSIVE_TRANSPORT_RIGHTS: 
				break //TODO
			case AIEvent.ET_ROAD_RECONSTRUCTION:
				break //do not care about road works, just ignore
		}
	}	
}