/* -*- Mode: C++; tab-width: 6 -*- */ 

/*
 *
 * This file is part of Script Communication Protocol (shorten to SCP), which are libraries for OpenTTD NoAI and NoGO
 * Copyright (C) 2012 Krinn <krinn@chez.com> & Zuu
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

class _SCPLib_Charcode
{
static	QuerySpace	= "q"; // this char is add to identify a query.
// Any chars not in the range of our hex special integer will be valid as spacer.
static	AnswerSpace	= "a"; // this char is add to identify an answer
static	LooseSpace = "l"; // this char is add to identify a simple query (don't expect answer)
}

function _SCPLib_Charcode::DataEncode(value)
// Lookout for value and decide how to encode it, then encode it and return it.
{
	local string_delimiter = 35; // the ascii decimal of the character delimiter for string
	local	number_delimiter = 36; // the ascii decimal of the character delimiter for unsign integer
	local negative_delimiter = 37; // the ascii decimal of the character delimiter for sign integer
	local res=-1;
	switch (typeof(value))
	{
		case	"string":
			res = _SCPLib_Charcode.EncodeString(value);
			if (res == -1)	return -1;
			return string_delimiter.tochar()+res;
		case	"integer":
			local signint=(value < 0);
			res = _SCPLib_Charcode.ByteToSCPChar(abs(value));
			if (res == -1)	return -1;
			if (signint)	return negative_delimiter.tochar()+res;
					else	return number_delimiter.tochar()+res;
		case	"null":
			return number_delimiter.tochar();
		case	"bool":
			if (value)	value=1;
				else	value=0;
			res = _SCPLib_Charcode.ByteToSCPChar(value);
			if (res == -1)	return -1;
			return number_delimiter.tochar()+res;
		default:	_SCPLib_Message.SCPLogError("We don't support the type "+typeof(value)+" in "+value);
	}
return -1;
}

function _SCPLib_Charcode::GetCharset()
// return the charset use for chartype
// openttd valid chars in squirrel : 32-126
// chars 35, 36 and 37 are pickup as delimiter for string, unsignint, signint
{
	local charval=[33,34,39,60,63,64,91,92,93,94,95,96,123,124,125,126];
	local charset=[];
	for (local i = 0; i < 16; i++)	charset.push(charval[i].tochar());
	return charset;
}

function _SCPLib_Charcode::IntegerToHex(_number)
{
	local charset=_SCPLib_Charcode.GetCharset();
	local r = _number % 0x10;
	local result;
	if (_number - r == 0)	result = charset[r];
				else	result = _SCPLib_Charcode.IntegerToHex( (_number - r) >> 4 ) + charset[r];
	return result;
}

function _SCPLib_Charcode::ArrayIN(_value, _array)
// return true if _value is in _array else false
{
for (local i=0; i < _array.len(); i++)
	{
	if (_array[i]==_value)	return true;
	}
return false;
}

function _SCPLib_Charcode::ByteToSCPChar(_value)
// on entry, _value is an integer < 0xFFFFFFFE or a char or a string.len()==1
// exit : encoded _value
{
	if (typeof(_value) == "integer" && _value > 0xFFFFFFE)	{ _SCPLib_Message.SCPLogError("integer too high max allow is "+0xFFFFFFE); return -1; }
	if (typeof(_value) != "integer" && typeof(_value) != "string")	{ _SCPLib_Message.SCPLogError("Wrong type <"+typeof(_value)+"> we need an integer or string"); return -1; }
	if (typeof(_value) == "string" && _value.len() != 1)	{ _SCPLib_Message.SCPLogError("we can only handle a char or a string of size 1"); return -1; }
	local charset=_SCPLib_Charcode.GetCharset();
	local codetype=0; // 0- no coding, use the real char, 1- code in our tweak hex format
	if (typeof(_value) == "string")
		{
		if ((_value[0] < 32 || _value[0] > 126) || (_value[0]>34 && _value[0]<38) || _SCPLib_Charcode.ArrayIN(_value, charset))	codetype=1;
		}
	if (typeof(_value) == "integer")	codetype=1;
	if (codetype == 0)	return _value; // we return the same char, unchange
	local number=_value;
	if (typeof(_value) == "string")	number=_value[0]; // get its ascii value
	return _SCPLib_Charcode.IntegerToHex(number);
}

function _SCPLib_Charcode::SCPCharToByte(_value)
// return an integer or a char depending of _value type
{
	if (_value == "")	return null;
	if (typeof(_value) != "string")	{ _SCPLib_Message.SCPLogError("Can only work with an SCPChar format : "+typeof(_value)); return -1; }
	if (_value.len() > 7)	{ _SCPLib_Message.SCPLogError("String too big "+_value); return -1; }
	local charset=_SCPLib_Charcode.GetCharset();
	if (_value.len()==1 && _value[0] > 31 && _value[0] < 127 && _value[0]!=35 && _value[0]!=36 && _value[0]!=37 && _SCPLib_Charcode.ArrayIN(_value, charset)==false)	return _value; // plain char
	local buff=[];
	for (local i=0; i < _value.len(); i++)
		{
		local found=false;
		for (local z=0; z < charset.len(); z++)
			if (_value.slice(i, i+1) == charset[z])	{ buff.push(z); found=true; break; }
		if (!found)	{
				_SCPLib_Message.SCPLogError("Cannot convert this : "+_value.slice(i,i+1)+" from "+_value);
				return -1;
				}
		}
	local shl=0;
	local number=0;
	for (local i = buff.len() -1; i >= 0; i--)
		{
		local mul=(buff[i] << shl);
		number+=mul;
		shl+=4;
		}
	return number;
}

function _SCPLib_Charcode::EncodeString(_str)
// encode a string to SCPChar
{
	if (!typeof(_str)=="string")	{ _SCPLib_Message.SCPLogError("Not a valid string : "+_str); return -1; }
	local encode="";
	local read="";
	for (local i=0; i < _str.len(); i++)
		{
		read=_SCPLib_Charcode.ByteToSCPChar(_str.slice(i, i+1));
		if (read == -1)	return -1;
		encode=encode+read;
		}
	return encode;
}

function _SCPLib_Charcode::DecodeString(_str)
// decode a string that is encoded to SCPChar
{
	local decode="";
	local buff="";
	local read="";
	local buffering=false;
	local charset=_SCPLib_Charcode.GetCharset();

	for (local i=0; i < _str.len(); i++)
		{
		buff=_str.slice(i, i+1);
		if (!buffering)
			{
			if (_SCPLib_Charcode.ArrayIN(buff, charset))
				{
				buffering=true;
				read=buff;
				}
			else	decode=decode+buff;
			}
		else	{
			if (!_SCPLib_Charcode.ArrayIN(buff, charset))
				{
				buffering=false;
				read=_SCPLib_Charcode.SCPCharToByte(read).tochar();
				if (read == -1)	return -1;
				decode=decode+read+buff;
				}
			else	read=read+buff;
			}
		}
		if (buffering) // catch end of datas
			{
				read=_SCPLib_Charcode.SCPCharToByte(read).tochar();
				if (read == -1)	return -1;
				decode=decode+read;
			}
	return decode;
}
