/* -*- Mode: C++; tab-width: 6 -*- */ 

/*
 *
 * This file is part of Script Communication Protocol (shorten to SCP), which are libraries for OpenTTD NoAI and NoGO
 * Copyright (C) 2012 Krinn <krinn@chez.com> & Zuu
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

require("message.nut");

class	_SCPLib_MessagePool
// the MessagePool is a pool of messages, we will handle a max 255 messages per company
{
static	base		= {};			// the pool itself, hold one messagepool object per company
static	PoolCache	= SCPList();	// use to cache pool message to faster find who is who

		MessageBase = null;		// messagepool object : the base holding all messages for a company
		PoolList 	= null;		// messagepool object : the poollist (see constructor part)
		CompanyID	= null;		// messagepool object : the companyID

static	function LoadMessage(_MessageID, _CompanyID)
		{
		local comp = _SCPLib_MessagePool.LoadCompany(_CompanyID);
		if (comp == null)	{ _SCPLib_Message.SCPLogError("Unknown company : "+_CompanyID); return null; }
		return _MessageID in comp.MessageBase ? comp.MessageBase[_MessageID] : null;
		}
	

	constructor(compID)
		{
		this.CompanyID = compID;
		this.MessageBase = {};
		this.PoolList = SCPList();
		for (local i=0; i < 256; i++)	this.PoolList.AddItem(i,0); // 0-free slot 1-undraw 2-draw
		_SCPLib_MessagePool.base[this.CompanyID] <- this;
		}
}

function _SCPLib_MessagePool::MessageStatusDraw(messageID, companyID)
{
	local comp = _SCPLib_MessagePool.LoadCompany(companyID);
	if (comp == null)	return -1;
	comp.PoolList.SetValue(messageID, 2);
}

function _SCPLib_MessagePool::DelayedCommandDraw(_CompanyID)
// Browse message to find command delayed and draw them
{
	local weare= _SCPLib_Client.GetOurCompanyID();
	local comp = _SCPLib_MessagePool.LoadCompany(_CompanyID);
	if (comp == null)	return -1;
	local msgList=SCPList();
	msgList.AddList(comp.PoolList);
	msgList.KeepValue(1); // remove status 0- unused and 2- already draw
	foreach (msgID, dummy in msgList)
		{
		local msg=_SCPLib_MessagePool.LoadMessage(msgID, _CompanyID);
		if (msg == null)	{ _SCPLogError("Cannot find message "+msgID+" in pool"); continue; }
		if (msg.ReceiverID != _CompanyID)	continue;
		local commandID = _SCPLib_Command.GetServerCommandByte(msg.Command, msg.CommandSet); // must convert to new commandID while it was in the buff
		if (commandID == -1)	{ _SCPLib_Message.SCPLogError("Unkown command "+msg.CommandSet+":"+msg.Command); continue; }
		msg.CommandID=commandID;
		local tID=_SCPLib_Client.GetTransportCompanyID(weare, _CompanyID);
		local codedata=_SCPLib_Sign.DataSplit(msg.MessageID, msg.CommandID, tID, msg.Type, msg.Data);
		if (codedata==-1)	continue; // the coding will throw out error message on failure
		msg.Data=codedata;

		_SCPLib_Message.SCPLogInfo("Resending delayed message "+msgID+" to "+msg.ReceiverID);
		_SCPLib_Sign.DrawPanel(msg.MessageID, msg.ReceiverID);
		}
}

function _SCPLib_MessagePool::LoadCompany(_CompanyID)
		{
		local compID = _SCPLib_Client.IsValidCompany(_CompanyID);
		if (_CompanyID in _SCPLib_MessagePool.base)
			{
			if (compID == -1)
				{
				_SCPLib_Message.SCPLogError("Removing bad company : "+_CompanyID);
				delete _SCPLib_MessagePool.base[_CompanyID];
				local clearList=SCPList();
				clearList.AddList(_SCPLib_MessagePool.PoolCache);
				clearlist.KeepValue(_CompanyID);
				foreach (_poolID, dummy in clearlist)	_SCPLib_MessagePool.RemoveItem(_poolID);
				return null;
				}
			return _SCPLib_MessagePool.base[_CompanyID];
			}
		if (compID)
			{
			local nCompany = _SCPLib_MessagePool(_CompanyID);
			return nCompany;
			}
		return null;
		}

function _SCPLib_MessagePool::GetPoolID(_MessageID, _CompanyID)
// return the pool ID
{
	return (_MessageID+(256*_CompanyID));
}

function _SCPLib_MessagePool::AddMessage(_usingCommand, _usingSet, _commandID, _CompanyID, _isquery, _data, _messageReplyID=null)
// Add a new message to our pool
// If _messageReplyID != null we create a message using the same messageID the client gave us, else we create a new messagID
{
	local weare=_SCPLib_Client.GetOurCompanyID();
	local pCompany= _SCPLib_MessagePool.LoadCompany(_CompanyID);
	if (pCompany == null)	return -1;
	local pList=SCPList();
	local msg = _SCPLib_Message();
	pList.AddList(pCompany.PoolList);
	pList.KeepValue(0); // keep only not in use message
	if (pList.IsEmpty())	{ _SCPLib_Message.SCPLogError("Cannot add a new message to "+_CompanyID+" pool. The pool is full ! "); return -1; }
	pList.Valuate(SCPBase.RandItem);
	if (_messageReplyID != null)		msg.MessageID = _messageReplyID;
						else	msg.MessageID = pList.Begin(); // take first avaiable one
	msg.Command=_usingCommand;
	msg.CommandSet=_usingSet;
	msg.CommandID=_commandID;
	msg.Type=_isquery;
	msg.ReceiverID = _CompanyID;
	msg.SenderID = weare;
	msg.PoolID = _SCPLib_MessagePool.GetPoolID(msg.MessageID, msg.ReceiverID);
	msg.Data = _data;
	//_SCPLib_Message.SCPLogInfo("New message : "+msg.MessageID+" / "+msg.PoolID+" for "+_CompanyID+" - "+(pList.Count()-1)+" remain slots");
	pCompany.MessageBase[msg.MessageID] <- msg;
	_SCPLib_MessagePool.PoolCache.AddItem(msg.PoolID, msg.ReceiverID);
	pCompany.PoolList.SetValue(msg.MessageID, 1); // 1 - undraw status
	return msg.MessageID;
}

function _SCPLib_MessagePool::MessageExist(_PoolID)
{
	return _SCPLib_MessagePool.PoolCache.HasItem(_PoolID);
}

function _SCPLib_MessagePool::DeleteMessage(_MessageID, _CompanyID)
{
	local pCompany= _SCPLib_MessagePool.LoadCompany(_CompanyID);
	if (pCompany == null)	return false;
	local msg = _SCPLib_MessagePool.GetPoolID(_MessageID, _CompanyID);
	if (_SCPLib_MessagePool.PoolCache.HasItem(msg))	_SCPLib_MessagePool.PoolCache.RemoveItem(msg);
								else	_SCPLib_Message.SCPLogInfo("Message "+_MessageID+" is not in pool : "+msg);
	if (pCompany.PoolList.HasItem(_MessageID))	pCompany.PoolList.SetValue(_MessageID,0); // reset to 0 to show free slot
	if (_MessageID in pCompany.MessageBase)	delete pCompany.MessageBase[_MessageID];
	//_SCPLib_Message.SCPLogInfo("Closing message "+_MessageID+" / "+msg+" MessagePool size="+_SCPLib_MessagePool.PoolCache.Count());
}
