/* 
 *	Copyright 2013 Varen De Meersman
 *  This file is part of MailAI.
 *
 *  MailAI is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  MailAI is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with MailAI.  If not, see <http://www.gnu.org/licenses/>.
 */
 
class BuildManager {
	constructor() {}
}

function BuildManager::BuildLine(route_array) {
	if(route_array == null){MailAI.Info(0,"Empty route_array in BuildLine");return null;}
	local engine = route_array[1];
	local town_id1 = route_array[0][0];local town_id2 = route_array[0][1];
	if(AIEngine.GetVehicleType(engine) == AIVehicle.VT_ROAD) {	
		if(!TownManager.HasTruckStopPotential(town_id1)) {return null;}
		if(!TownManager.HasTruckStopPotential(town_id2)) {return null;}
		//de articulated wordt niet gebruikt in buildroad, waarom moest ik die dan doorgeven?
		return this.BuildRoadRoute(town_id1,town_id2,AIEngine.IsArticulated(engine));
	} else if(AIEngine.GetVehicleType(engine) == AIVehicle.VT_RAIL) {
		local con1=null,con2=null;
		con1 = TownManager.GetExpandingTrainStopConnection(town_id1, town_id2);
		con2 = TownManager.GetExpandingTrainStopConnection(town_id2, town_id1);	
		if(con1!=null && con2!=null) {//we can only connect one existing station
			if(AITown.GetPopulation(town_id2) > AITown.GetPopulation(town_id1)) {
				if(TownManager.CanBuildNewTrainStation(town_id1,town_id2)) {
					return this.BuildRailRoute(town_id2,con2,town_id1);//reverse towns
				} else {
					con2 = null;	
				}
			} else {
				if(TownManager.CanBuildNewTrainStation(town_id2,town_id1)) {
					return this.BuildRailRoute(town_id1,con1,town_id2);
				} else {
					con1 = null;	
				}
			}		
		}
		if(con1==null && !TownManager.CanBuildNewTrainStation(town_id1,town_id2)) {
			MailAI.Info(0,"No train options in town 1");
			return null;
		}
		if(con2==null && !TownManager.CanBuildNewTrainStation(town_id2,town_id1)) {
			MailAI.Info(0,"No train options in town 2");
			return null;	
		}
		if(con2!=null) {return this.BuildRailRoute(town_id2,con2,town_id1);}//reverse towns
		return this.BuildRailRoute(town_id1,con1,town_id2);
	}
}

function BuildManager::BuildRailRoute(town_id1,con,town_id2) {	
	MailAI.Info(0,"Trying to build a rail route between "+AITown.GetName(town_id1)+" and "+AITown.GetName(town_id2)+".");
	local pre1,cur1,pre2,cur2,switch1,station1_tile,switch2,station2_tile,save_opt1,save_opt2;
	local dir2 = Tile.GetDirectionFromTo(AITown.GetLocation(town_id2),AITown.GetLocation(town_id1));	
	local side2 = Tile.GetSideDirectionFromTo(AITown.GetLocation(town_id2),AITown.GetLocation(town_id1));
	local options2 = TownManager.GetTrainStationOptions(AITown.GetLocation(town_id2),dir2,side2);
	if(options2.len() == 0){MailAI.Info(0,"0.No good place to build a station in "+AITown.GetName(town_id2)+".");return null;}
	foreach(opt in options2) {	
		local tl = AITileList(); tl.AddRectangle(opt[4],opt[5]);
		if(!Tile.ClearArea(tl)){MailAI.Info(0,"Couldn't clear area");continue;}
		foreach(t, _ in tl){if(!Tile.IsBuildableAlmostFlatTile){continue;}}
		station2_tile = opt[0];
		{ local ISINswitchbuildTESTMODE = AITestMode();
			switch2 = RailBuilder.BuildStationSwitch(opt);
		}
		if(switch2 == null){continue;}
		pre2 = switch2[3]; cur2 = switch2[4];
		save_opt2 = opt;
		break;
	}
	if(save_opt2 == null){MailAI.Info(0,"1.No good place to build a station in "+AITown.GetName(town_id2)+".");return null;}	
	if(con != null) {	
		station1_tile = con[2]; local path = null;
		MailAI.Info(1,"Re-using existing train station in "+AITown.GetName(town_id1)+".");
		local expand;
		/*expand = RailBuilder.GetNextJoiningLocation(con[0],con[1],switch2[4],false);
		if(expand != null) {
			{	local ISINjoiningTESTMODE = AITestMode();
				switch1 = RailBuilder.BuildJoiningSwitch(expand);
				if(switch1!=null) {
					pre1 = switch1[3]; cur1 = switch1[4];
					path = RailBuilder.BuildRailPath([[cur2, pre2]],[[cur1, pre1]]);
				}
			}
		}
		if(path == null) {
			MailAI.Info(0,"best location not buildable, trying other routine");*/
			expand = RailBuilder.GetNextJoiningLocation(con[0],con[1],switch2[4],true);
			while(expand != null && expand[0] != null && expand[1] != null) {
				{	local ISINjoiningTESTMODE = AITestMode();
					switch1 = RailBuilder.BuildJoiningSwitch(expand);
					if(switch1!=null) {
						pre1 = switch1[3]; cur1 = switch1[4];
						path = RailBuilder.BuildRailPath([[cur2, pre2]],[[cur1, pre1]]);
					}
				}
				if(path==null){
					MailAI.Info(0,"Trying next expanding location.");
					expand = RailBuilder.GetNextJoiningLocation(expand[0],expand[1],station2_tile,true);
				} else {break;}
			}
		//}
		if(path != null) {
			switch1 = RailBuilder.BuildJoiningSwitch(expand);
			if(switch1 == null){MailAI.Info(0,"Couldnt build joining switch.");return null;}
			pre1 = switch1[3]; cur1 = switch1[4];
		}
		if(path == null){MailAI.Info(0,"Couldnt join existing station.");return null;}	
	} else {	
		local dir1 = Tile.GetDirectionFromTo(AITown.GetLocation(town_id1),AITown.GetLocation(town_id2));	
		local side1 = Tile.GetSideDirectionFromTo(AITown.GetLocation(town_id1),AITown.GetLocation(town_id2));
		local options1 = TownManager.GetTrainStationOptions(AITown.GetLocation(town_id1),dir1,side1);
		if(options1.len() == 0){MailAI.Info(0,"2.No good place to build a station in "+AITown.GetName(town_id1)+".");return null;}
		foreach(opt in options1) {	
			local tl = AITileList(); tl.AddRectangle(opt[4],opt[5]);
			if(!Tile.ClearArea(tl)){MailAI.Info(0,"Couldn't clear area");continue;}
			foreach(t, _ in tl){if(!Tile.IsBuildableAlmostFlatTile){continue;}}
			station1_tile = opt[0];
			{ local ISINswitchbuildTESTMODE = AITestMode();
				switch1 = RailBuilder.BuildStationSwitch(opt);
			}
			if(switch1 == null){continue;}
			pre1 = switch1[3]; cur1 = switch1[4];
			save_opt1 = opt;
			break;
		}
		if(save_opt1 == null){MailAI.Info(0,"3.No good place to build a station in "+AITown.GetName(town_id2)+".");return null;}	
	}
	{ local ISINpathfinderTESTMODE = AITestMode();
		if(cur1 == null || pre1 == null || cur2 == null || pre2 == null){
			MailAI.Info(2,"Null values before building railpath!");return null;}
		local travel_dist = RailBuilder.BuildRailPath([[cur1,pre1]],[[cur2,pre2]]);
		if(travel_dist == null) {MailAI.Info(0,"Couldn't find path between stations.");return null;}
	}
	if(save_opt1!=null) {
		station1_tile = RailBuilder.BuildTrainStation(save_opt1[0],save_opt1[2],save_opt1[4],save_opt1[5]);
		switch1 = RailBuilder.BuildStationSwitch(save_opt1);
	}
	if(switch1 == null){
		MailAI.Info(0,"Problem building switch1.");
		if(con==null && save_opt1[0]!=null){Tile.DemolishTile(save_opt1[0]);}
		return null;
	}
	pre1 = switch1[3]; cur1 = switch1[4];
	if(station1_tile==null){
		MailAI.Info(0,"4.No good place to build a station in "+AITown.GetName(town_id2)+".");
		return null;
	}
	station2_tile = RailBuilder.BuildTrainStation(save_opt2[0],save_opt2[2],save_opt2[4],save_opt2[5]);
	if(station2_tile==null){
		MailAI.Info(0,"5.No good place to build a station in "+AITown.GetName(town_id2)+".");
		if(con==null && save_opt1[0]!=null){Tile.DemolishTile(save_opt1[0]);}
		return null;
	}
	switch2 = RailBuilder.BuildStationSwitch(save_opt2);
	if(switch2 == null){
		MailAI.Info(0,"problem building switch2 in front of station");
		if(con==null && save_opt1!=null){Tile.DemolishTile(save_opt1[0]);}
		if(save_opt2!=null){Tile.DemolishTile(save_opt2[0]);}
		return null;
	}
	pre2 = switch2[3]; cur2 = switch2[4];
	if(cur1 == null || pre1 == null || cur2 == null || pre2 == null){
		MailAI.Info(2,"Null values before building railpath!");
		if(con==null && save_opt1!=null){Tile.DemolishTile(save_opt1[0]);}
		if(save_opt2!=null){Tile.DemolishTile(save_opt2[0]);}
		return null;
	}
	local travel_dist = RailBuilder.BuildRailPath([[cur1,pre1]],[[cur2,pre2]]);
	if(travel_dist == null) {
		MailAI.Info(2,"Couldn't build rails between stations.");
		if(con==null && save_opt1!=null){Tile.DemolishTile(save_opt1[0]);}
		if(save_opt2!=null){Tile.DemolishTile(save_opt2[0]);}
		return null;
	}
	if(switch1[2] != null){switch1 = RailBuilder.UpdateStationSwitch(switch1);}
	if(switch2[2] != null){switch2 = RailBuilder.UpdateStationSwitch(switch2);}
	local passings1 = RailBuilder.BuildPassings(switch1[0],switch1[1],switch1[3],switch1[4],travel_dist);
	local passings2 = RailBuilder.BuildPassings(switch2[0],switch2[1],switch2[3],switch2[4],travel_dist);
	if(passings1 == null){passings1 = passings2;}
	if(passings1 == null){MailAI.Info(1,"Could not build passing lanes on route!");
	} else {travel_dist = passings1[1] + 20;}//guestimate of switches and station length
	local depot1 = null, depot2 = null;
	if(con == null) {depot1 = RailBuilder.BuildDepot(switch1);}
	depot2 = RailBuilder.BuildDepot(switch2);
	if(depot2 == null && depot1 == null) {depot2 = RailBuilder.BuildExtraRailDepot(passings1[0]);}
	if(depot2 == null && depot1 != null) {depot2 = depot1;}
	if(depot2 == null) {MailAI.Info(2,"(no train depot)?");return null;}
	local engine = EngineMaster.GetTrainEngine(travel_dist,0,true);
	if(engine == null){MailAI.Info(3,"Exiting buildroute with null as engine!");return null;}
	local replaced_towns = [];//don't store town_id, station might think it's closer to another town!
	if(AITile.GetClosestTown(station1_tile) != town_id1){
		replaced_towns.append([AITile.GetClosestTown(station1_tile),town_id1]);}
	if(AITile.GetClosestTown(station2_tile) != town_id2){
		replaced_towns.append([AITile.GetClosestTown(station2_tile),town_id2]);}
	local sorted_towns = [AITile.GetClosestTown(station1_tile)];
	sorted_towns = MailAI.Append(sorted_towns,AITile.GetClosestTown(station2_tile));
	local station_list = AITileList();
	station_list.AddTile(station1_tile);station_list.AddTile(station2_tile);
	//foreach(station_tile, _ in station_list) {MailAI.Info(0,"stat"+station_tile);}//the _ is important!
	return Route(sorted_towns,station_list,depot2,travel_dist,engine,replaced_towns);
}

function BuildManager::BuildFourTruckStops(town,route) {
		local engine = EngineMaster.GetTruckEngine(20,0,true); if(engine==null){return null;}
		local articulated = AIEngine.IsArticulated(engine);
		local town_tile = AITown.GetLocation(town);
		local tl = AITileList(), station_saved = null;
		foreach(station_tile, _ in route.GetStationList()) {
			if(AITile.GetClosestTown(station_tile) == town) {
				tl = TownManager.GetAdjacentRoadStationTiles(station_tile);
				station_saved = station_tile;
				break;
			}
		}
		local st_tile = this.BuildTruckStop(tl, true, AIStation.STATION_JOIN_ADJACENT);	
		if(st_tile == null){
			local tile_list = StationManager.GetSurroundingTiles(station_saved);
			foreach(building_tile, _ in tile_list) {
				local connect_tiles = Tile.GetAdjacentTiles(building_tile);
				connect_tiles.Valuate(AIRoad.IsRoadTile);
				connect_tiles.KeepValue(1);
				if(!AITile.IsBuildable(building_tile)
				&& connect_tiles.Count() > 0) {
					if(Tile.DemolishTile(building_tile)){
						tl = AITileList();tl.AddTile(building_tile);
						st_tile = this.BuildTruckStop(tl, true, AIStation.STATION_JOIN_ADJACENT);
						break;
					}
				}
			}
		}
		if(st_tile == null) {MailAI.Info(0,"No truck connection to train");return null;}
		local stats = AITileList(); stats.AddTile(st_tile);		
		tl = TownManager.GetNewFourTruckStopList(town_tile,"NW",articulated);
		local tile = this.BuildTruckStop(tl,articulated,AIStation.STATION_NEW);
		if(tile != null && tile != st_tile){stats.AddTile(tile);}
		tl = TownManager.GetNewFourTruckStopList(town_tile,"NE",articulated);
		tile = this.BuildTruckStop(tl,articulated,AIStation.STATION_NEW);
		if(tile != null && tile != st_tile){stats.AddTile(tile);}
		tl = TownManager.GetNewFourTruckStopList(town_tile,"SW",articulated);
		tile = this.BuildTruckStop(tl,articulated,AIStation.STATION_NEW);
		if(tile != null && tile != st_tile){stats.AddTile(tile);}
		tl = TownManager.GetNewFourTruckStopList(town_tile,"SE",articulated);
		tile = this.BuildTruckStop(tl,articulated,AIStation.STATION_NEW);
		if(tile != null && tile != st_tile){stats.AddTile(tile);}
		local replaced_towns = route.GetReplacedTowns();
		if(replaced_towns != null) {
			foreach(rep in replaced_towns) {
				if(rep[0] == town){MailAI.Info(0,"also looking around: "+AITown.GetName(rep[1]));
					local rep_town = AITown.GetLocation(rep[1]);
					tl = TownManager.GetNewFourTruckStopList(rep_town,"NW",articulated);
					tile = this.BuildTruckStop(tl,articulated,AIStation.STATION_NEW);
					if(tile!=null){stats.AddTile(tile);}
					tl = TownManager.GetNewFourTruckStopList(rep_town,"NE",articulated);
					tile = this.BuildTruckStop(tl,articulated,AIStation.STATION_NEW);
					if(tile!=null){stats.AddTile(tile);}
					tl = TownManager.GetNewFourTruckStopList(rep_town,"SW",articulated);
					tile = this.BuildTruckStop(tl,articulated,AIStation.STATION_NEW);
					if(tile!=null){stats.AddTile(tile);}
					tl = TownManager.GetNewFourTruckStopList(rep_town,"SE",articulated);
					tile = this.BuildTruckStop(tl,articulated,AIStation.STATION_NEW);
					if(tile!=null){stats.AddTile(tile);}
					this.BuildRoadPath(rep_town,town_tile);		
				}
			}
		}	
		local depot_tile = null;
		foreach(new_stat, _ in stats) {
			if(AITile.GetClosestTown(new_stat) != town) {
				//we've build in a neighbouring town
				if(!this.BuildRoadPath(town_tile,new_stat)) {
					if(!this.BuildRoadPath(town_tile,AIRoad.GetRoadStationFrontTile(new_stat))) {
						if(!this.BuildRoadPath(town_tile,AITown.GetLocation(AITile.GetClosestTown(new_stat)))) {
							MailAI.Info(2,"Road station probably not connected.");
						}
					}
				}	
			}
			if(depot_tile==null && stats.Count() > 1){
				depot_tile = TownManager.GetRoadDepot(town,null);
				if(depot_tile==null) {
					MailAI.Info(0,"Building road depot in "+AITown.GetName(AITile.GetClosestTown(new_stat)));
					depot_tile=this.BuildTownRoadDepot(new_stat,null);
				}
			}
		}
		if(depot_tile == null && stats.Count() > 1) {MailAI.Info(2,"couldn't build road depot!");return null;}	
		return [stats,depot_tile,engine];
}

/**
*	We're forcing a PtP Truck line from an existing station to somewhere else.
*/
function BuildManager::BuildForcedTruckLine(station1_tile,town_id2,depot_tile) {
	local town_id1 = AITile.GetClosestTown(station1_tile);	
	local road1 = AIRoad.GetRoadStationFrontTile(station1_tile);	
	local reuse2 = false, station2_tile = null, road2 = AITown.GetLocation(town_id2);
	local expand2 = TownManager.GetFirstTruckStopToExpand(town_id2);
	if(expand2 != null) {reuse2 = true;station2_tile = expand2[0];road2 = expand2[1];}
	local travel_dist = null; {
		local ISINroadroutefindingTESTMODE = AITestMode();
		travel_dist = this.BuildRoadPath(AITown.GetLocation(town_id1),AITown.GetLocation(town_id2));
		if(travel_dist == null || travel_dist == 0) {return null;}
	}
	local height_difference = abs(AITile.GetMaxHeight(station1_tile)-AITile.GetMaxHeight(AITown.GetLocation(town_id2)));
	local engine = EngineMaster.GetTruckEngine(travel_dist,height_difference,true);
	if(engine==null){return null;}
	if(station2_tile == null) {
		local tl = TownManager.GetNewTruckStopList(town_id2,AIEngine.IsArticulated(engine));
		if(tl.IsEmpty()) {return null;}
		station2_tile = this.BuildTruckStop(tl, AIEngine.IsArticulated(engine),AIStation.STATION_NEW);
		if(station2_tile == null) {
			MailAI.Info(1,"Couldn't build station in "+AITown.GetName(town_id2)+".");
			return null;
		}
	}
	travel_dist = this.BuildRoadPath(station1_tile,station2_tile);
	if(travel_dist == null || travel_dist == 0) {
		travel_dist = this.BuildRoadPath(AITown.GetLocation(town_id1),AITown.GetLocation(town_id2));
		if(travel_dist == null || travel_dist == 0) {
			MailAI.Info(2,"Couldn't build roads, removing infrastructure.");
			if(!reuse2){AITile.DemolishTile(station2_tile);}
			return null;
		}	
	}
	//store closest town instead of passed town_id
	local sorted_towns = [AITile.GetClosestTown(station1_tile)];
	sorted_towns = MailAI.Append(sorted_towns,AITile.GetClosestTown(station2_tile));
	local station_list = AITileList();station_list.AddTile(station1_tile);station_list.AddTile(station2_tile);
	local depot1_tile = null, depot2_tile = null;
	if(depot_tile == null) {
		//MailAI.Info(0,"Forced truck route needs Road Depot.");
		depot1_tile = TownManager.GetRoadDepot(town_id1,null);
		depot2_tile = TownManager.GetRoadDepot(town_id2,null);
		if(depot1_tile == null){depot1_tile = this.BuildRoadDepot(station2_tile,station1_tile);}
		if(depot2_tile == null){depot2_tile = this.BuildRoadDepot(station1_tile,station2_tile);}
		if(depot1_tile != null){return Route(sorted_towns,station_list,depot1_tile,travel_dist,engine,null);}
		if(depot2_tile != null){return Route(sorted_towns,station_list,depot2_tile,travel_dist,engine,null);}
		MailAI.Info(2,"No Road Depot!");
	}
	return Route(sorted_towns,station_list,depot_tile,travel_dist,engine,null);
}
 
/**
*	Build two stations, a depot and a road between them.
*	@param town_id1 First town to build a station in
*	@param town_id2 Second town
*	@return a newly created Route object if succesfull, otherwise null
*	@todo Uses a random station if existing ones, now there will always be
*	one per city but this might change later so should check for best station
*/
function BuildManager::BuildRoadRoute(town_id1,town_id2,articulated) {
	local expand1 = TownManager.GetFirstTruckStopToExpand(town_id1);
	local road1 = AITown.GetLocation(town_id1), reuse1 = false, station1_tile = null;
	if(expand1 != null) {	
		reuse1 = true;
		MailAI.Info(1,"Re-using existing truck station in "+AITown.GetName(town_id1)+".");
		station1_tile = expand1[0];
		road1 = expand1[1];
	}		
	local expand2 = TownManager.GetFirstTruckStopToExpand(town_id2);	
	local road2 = AITown.GetLocation(town_id2), reuse2 = false, station2_tile = null;
	if(expand2 != null) { 	
		reuse2 = true;
		MailAI.Info(1,"Re-using existing truck station in "+AITown.GetName(town_id2)+".");
		station2_tile = expand2[0];
		road2 = expand2[1];
	}	
	local travel_dist = null; {
		local ISINroadroutefindingTESTMODE = AITestMode();
		travel_dist = this.BuildRoadPath(road1,road2);
		if(travel_dist == null || travel_dist == 0) {return null;}
	}
	local engine = EngineMaster.GetTruckEngine(travel_dist,0,true);
	if(engine == null) {
		MailAI.Info(2,"No engine while building road between "+AITown.GetName(town_id1)+" and "+AITown.GetName(town_id2)+".");
		return null;
	}
	//if not reusing a station then build new station
	if(station1_tile == null) {
		local tl = TownManager.GetNewTruckStopList(town_id1,AIEngine.IsArticulated(engine));
		if(tl.IsEmpty()) {return null;}
		station1_tile = this.BuildTruckStop(tl,AIEngine.IsArticulated(engine),AIStation.STATION_NEW);
		if(station1_tile == null) {
			MailAI.Info(1,"Couldn't build station in "+AITown.GetName(town_id1)+".");
			return null;
		}
	}
	if(station2_tile == null) {
		local tl = TownManager.GetNewTruckStopList(town_id2,AIEngine.IsArticulated(engine));
		if(tl.IsEmpty()) {return null;}
		station2_tile = this.BuildTruckStop(tl,AIEngine.IsArticulated(engine),AIStation.STATION_NEW);
		if(station2_tile == null) {
			MailAI.Info(1,"Couldn't build station in "+AITown.GetName(town_id2)+".");
			if(!reuse1){AITile.DemolishTile(station1_tile);}
			return null;
		}
	}
	//build the road
	travel_dist = this.BuildRoadPath(station1_tile,station2_tile);
	//travel_dist = this.BuildRoadPath(road1,road2);
	if(travel_dist == null || travel_dist == 0) {
		MailAI.Info(2,"Something went wrong building the RoadPath, trying something else.");
		travel_dist = this.BuildRoadPath(AITown.GetLocation(town_id1),AITown.GetLocation(town_id2));
		if(travel_dist == null || travel_dist == 0) {
			MailAI.Info(2,"Still couldn't build, removing infrastructure.");
			if(!reuse2){AITile.DemolishTile(station2_tile);}
			if(!reuse1){AITile.DemolishTile(station1_tile);}
			return null;
		}	
	}	
	//build depot
	local depot_tile = null, depot1_tile = null, depot2_tile = null;
	if(reuse1){depot1_tile = TownManager.GetRoadDepot(town_id1,null);}
	if(reuse2){depot2_tile = TownManager.GetRoadDepot(town_id2,null);}
	if(depot1_tile == null) {
		if(travel_dist > 70 || depot2_tile == null) {
			depot1_tile = this.BuildRoadDepot(station2_tile,station1_tile);
		}
	}
	if(depot1_tile!=null){depot_tile = depot1_tile;}
	if(depot2_tile == null) {
		if(travel_dist > 70 || depot1_tile == null) {
			depot2_tile = this.BuildRoadDepot(station1_tile,station2_tile);
		}
	}
	if(depot2_tile!=null){depot_tile = depot2_tile;}
	local engine = EngineMaster.GetTruckEngine(travel_dist,0,true);
	if (depot_tile == null || engine == null) {	
		MailAI.Info(2,"Failed completing the new road route,no depot or null engine...");
		if(!reuse2){AITile.DemolishTile(station2_tile);}
		if(!reuse1){AITile.DemolishTile(station1_tile);}
		return null;
	}
	local sorted_towns = [AITile.GetClosestTown(station1_tile)];
	sorted_towns = MailAI.Append(sorted_towns,AITile.GetClosestTown(station2_tile));
	local station_list = AITileList();station_list.AddTile(station1_tile);station_list.AddTile(station2_tile);
	return Route(sorted_towns,station_list,depot_tile,travel_dist,engine,null);
}

/**
*	Builds a road between the tiles provided.
*	@param t1 starttile
*	@param t2 endtile
*	@return the length in tiles of the path build, null if no path found
*/
function BuildManager::BuildRoadPath(t1,t2) {
	local pathfinder = MyRoadPF();
	pathfinder.cost.max_bridge_length = MailAI.max_truck_bridge_length;
	pathfinder.cost.bridge_per_tile = MailAI.bridge_cost;
	//set max_cost to prevent pathfinder looping for a very long time when no routes are found
	pathfinder.cost.max_cost = (AIMap.DistanceManhattan(t1, t2)+5) * 3 * (pathfinder.cost.tile);		
	local path = false, bool = false;
	pathfinder.InitializePath([t1],[t2]);
	local counter = 0;
	while (counter < 500 && path == false) {
		path = pathfinder.FindPath(500);//returns false if not finished, otherwise path or null
		MailAI.Sleep(1);
		counter++;
	}
	if (path == null || path == false) {return null;}
	local saved_path = path;
	counter = 0;
	/* If a path was found, build a road over it. */
	while (path != null) {
		counter++;
		if(path == false){MailAI.Info(2,"path is false in buildroadpath, returning null");return null;}
		local par = path.GetParent(), last_node;	
		if (par != null) {
			local prev_node = last_node;
			last_node = path.GetTile();
			if (AIMap.DistanceManhattan(path.GetTile(), par.GetTile()) == 1 ) {
				//if(AIRoad.HasRoadType(par.GetTile(),AIRoad.ROADTYPE_TRAM))
					//{MailAI.Info(2,"Check to see if road has been build over tram tile.");}
				local it_count = 0;
				while(!AIRoad.BuildRoad(path.GetTile(), par.GetTile()) && it_count < 33
					&& AIError.GetLastErrorString() != "ERR_ALREADY_BUILT") {
					EventManager.HandleCommonErrors(path.GetTile(), par.GetTile());	
					it_count++;
				}
				if(it_count >= 33) {
					MailAI.Info(2,"Couldn't build road: "+AIError.GetLastErrorString());
					return null;
				}
			} else {	/* Build a bridge or tunnel. */
				if (!AIBridge.IsBridgeTile(path.GetTile()) && !AITunnel.IsTunnelTile(path.GetTile())) {
					/* If it was a road tile, demolish it first. Do this to work around expended roadbits. */
					if (AIRoad.IsRoadTile(path.GetTile())) {AITile.DemolishTile(path.GetTile());}
					if (AITunnel.GetOtherTunnelEnd(path.GetTile()) == par.GetTile()) {	
						local it_count = 0; 
						while(!AITunnel.BuildTunnel(AIVehicle.VT_ROAD, path.GetTile()) && it_count < 33
							&& AIError.GetLastErrorString() != "ERR_ALREADY_BUILT") {
							if(!EventManager.HandleCommonErrors(path.GetTile(),null)){return null;}		
							it_count++;
						}
						if(it_count >= 33) {
							MailAI.Info(1,"Couldn't build tunnel: "+AIError.GetLastErrorString());
							return null;
						}
						/*
						if (!AITunnel.BuildTunnel(AIVehicle.VT_ROAD, path.GetTile()))
						{
							local build = false, loan = true;
							while(AIError.GetLastErrorString() == "ERR_NOT_ENOUGH_CASH" && !build && loan)
							{
								loan = BankManager.LoanOne();
								build = AITunnel.BuildTunnel(AIVehicle.VT_ROAD, path.GetTile());
							}
							if(!build)AILog.Error("error building tunnel: "+AIError.GetLastErrorString());
						}*/
					} else {
            			local bridge_list = AIBridgeList_Length(AIMap.DistanceManhattan(path.GetTile(), par.GetTile()) + 1);
            			bridge_list.Valuate(AIBridge.GetMaxSpeed);
            			bridge_list.Sort(AIList.SORT_BY_VALUE, false);
						local it_count = 0;
						while(!AIBridge.BuildBridge(AIVehicle.VT_ROAD, bridge_list.Begin(), path.GetTile(), par.GetTile()) && it_count < 33
							&& AIError.GetLastErrorString() != "ERR_ALREADY_BUILT") {
							if(!EventManager.HandleCommonErrors(path.GetTile(), par.GetTile())){return null;}		
							it_count++;
						}
						if(it_count >= 33) {
							MailAI.Info(1,"Couldn't build bridge: "+AIError.GetLastErrorString());
							return null;
						}
						/*
	            			if (!AIBridge.BuildBridge(AIVehicle.VT_ROAD, bridge_list.Begin(), path.GetTile(), par.GetTile()))
	            			{
								local build = false, loan = true;
								while(AIError.GetLastErrorString() == "ERR_NOT_ENOUGH_CASH" && !build && loan)
								{
									loan = BankManager.LoanOne();
									build = AIBridge.BuildBridge(AIVehicle.VT_ROAD, bridge_list.Begin(), path.GetTile(), par.GetTile());
								}
								if(!build)AILog.Error("error building bridge: "+AIError.GetLastErrorString());
	            			}
						*/
          			}
        		}
      		}
    	}
		path = par;
	}
	return counter;
}

function BuildManager::BuildTruckStop(try_list, articulated, new) {
	if(try_list==null || try_list.IsEmpty()){return null;}
	local stat = null;
	foreach(tile, _ in try_list) {
		if(AIRoad.IsRoadStationTile(tile) && AICompany.IsMine(AITile.GetOwner(tile))) {
			return tile;//return if it's an existing road station of our own
		}
		stat = this.TryBuildRoadStation(tile, articulated, new);
		if(stat!=null){break;}
	}
	if(stat==null){return null;}
	local station_id = AIStation.GetStationID(stat);
	local town = AIStation.GetNearestTown(station_id);
	local named = false;
	if(new == AIStation.STATION_JOIN_ADJACENT){named = true;}
	local town_name = AITown.GetName(town);
	if(town_name.len() > 8) {town_name = town_name.slice(0,5);}
	if(articulated) {
		if(!named){named = AIStation.SetName(station_id,town_name+" Mailbox");}
		if(!named){named = AIStation.SetName(station_id,town_name+" Postbox");}
		if(!named){named = AIStation.SetName(station_id,town_name+" Mail Drop");}
		if(!named){named = AIStation.SetName(station_id,town_name+" Box");}
	} else {
		if(!named){named = AIStation.SetName(station_id,town_name+" Post Office");}
		if(!named){named = AIStation.SetName(station_id,town_name+" Postal Office");}
		if(!named){named = AIStation.SetName(station_id,town_name+" Mail Office");}
	}
	if(!named){named = AIStation.SetName(station_id,town_name+" Mail");}
	if(!named){named = AIStation.SetName(station_id,town_name+" Post");}
	if(!named){named = AIStation.SetName(station_id,town_name+" PO");}
	return stat;
}

/**
*	Builds a station on the given tile
*	@param station Tile to build station on
*	@param articulated Wheter the station needs to be drive through or not
*	@param new AIStation.STATION_NEW or AIStation.STATION_JOIN_ADJACENT
*/
function BuildManager::TryBuildRoadStation(tile, articulated, new) {
	if(tile == null){throw("tile null in trybuildroadstation");}
	if(AIRoad.GetNeighbourRoadCount(tile) == 0) {
		local it_count = 0; 
		while(BuildManager.BuildRoadPath(tile,AITown.GetLocation(AITile.GetClosestTown(tile))) == null
			 	&& it_count < 3) {// && AIError.GetLastErrorString() != "ERR_NONE") {//when err_none it still continues, is this a problem?
			EventManager.HandleCommonErrors(tile,null);		
			it_count++;
		} if(it_count >= 3) {return null;}
	}
	local adjacent_tiles = Tile.GetAdjacentTiles(tile);
	foreach(road, _ in adjacent_tiles) {//check road access	
		if(AIRoad.IsRoadTile(road)) {
			if(!AIRoad.IsRoadTile(tile)) {//if tile is allready road no need to build road
				local it_count = 0; 
				while(!AIRoad.BuildRoad(tile, road) && it_count < 33
						&& AIError.GetLastErrorString() != "ERR_ALREADY_BUILT") {
					EventManager.HandleCommonErrors(tile, road);		
					it_count++;
				}
				if(it_count >= 33) {
					MailAI.Info(0,"Couldn't build road for station: "+AIError.GetLastErrorString());
					return null;
				}
			}
			if(!AIRoad.AreRoadTilesConnected(tile,road)){continue;}	 
			if(articulated) {
				local it_count = 0; 
				while(!AIRoad.BuildDriveThroughRoadStation(tile,road,AIRoad.ROADVEHTYPE_TRUCK,new) && it_count < 33
						&& AIError.GetLastErrorString() != "ERR_ALREADY_BUILT") {
					EventManager.HandleCommonErrors(tile, road);		
					it_count++;
				}
				if(it_count >= 33) {
					MailAI.Info(0,"Couldn't build drive through station: "+AIError.GetLastErrorString());
					return null;
				}
			} else {
				local it_count = 0;
				while(!AIRoad.BuildRoadStation(tile,road,AIRoad.ROADVEHTYPE_TRUCK,new)
					 && it_count < 33 && AIError.GetLastErrorString() != "ERR_ALREADY_BUILT") {
						MailAI.Info(0,"Couldn't build truck station: "+AIError.GetLastErrorString());
					EventManager.HandleCommonErrors(tile, road);		
					it_count++;
				}
				if(it_count >= 33) {
					MailAI.Info(0,"Couldn't build road station: "+AIError.GetLastErrorString());
					return null;
				}	
			}
			return tile;
		}//end if road tile
	}//end for each road
	return null;
}

function BuildManager::BuildRoadDepot(stat1,stat2) {
	local ret_tile = this.BuildTownRoadDepot(stat1,stat2);
	if(ret_tile == null){return this.BuildTownRoadDepot(stat2,stat1);}
	return ret_tile;
}

function BuildManager::BuildTownRoadDepot(stat1,stat2) {
	if(AIRoad.IsDriveThroughRoadStationTile(stat1)) {
		if(AIRoad.IsRoadTile(AIRoad.GetRoadStationFrontTile(stat1))
			&& AITile.IsBuildable(AIRoad.GetDriveThroughBackTile(stat1))) {
			local it_count = 0; 
			while(!AIRoad.BuildRoadDepot(AIRoad.GetDriveThroughBackTile(stat1),stat1) && it_count < 33
				&& AIError.GetLastErrorString() != "ERR_ALREADY_BUILT") {
				EventManager.HandleCommonErrors(AIRoad.GetDriveThroughBackTile(stat1),null);		
				it_count++;
			}
			if(it_count < 33) {return AIRoad.GetDriveThroughBackTile(stat1);}
		}	
	}
	local tl = AITileList();
	local town_tile = AITown.GetLocation(AITile.GetClosestTown(stat1));
	local x = AIMap.GetTileX(town_tile), y = AIMap.GetTileY(town_tile);
	tl.AddRectangle(AIMap.GetTileIndex(x-12,y-12), AIMap.GetTileIndex(x+12,y+12));
   	tl.Valuate(AIRoad.IsRoadDepotTile);
   	tl.KeepValue(1);
   	foreach(tile, _ in tl){
   		if(AITile.GetOwner(tile)==AICompany.COMPANY_SELF){return tile;}
   	}
	tl.AddRectangle(AIMap.GetTileIndex(x-12,y-12), AIMap.GetTileIndex(x+12,y+12));
	tl.Valuate(Valuator.IsGoodRoadDepotTile); tl.KeepValue(1);		
   	if (tl.IsEmpty()) {return null;}
   	if(stat2 == null) {tl.Valuate(AITile.GetDistanceManhattanToTile, town_tile);}
	else {tl.Valuate(AITile.GetDistanceManhattanToTile, stat2);}
	tl.Sort(AITileList.SORT_BY_VALUE, true);	
	foreach(aitile, _ in tl) {   
		local adjacent_tiles = Tile.GetAdjacentTiles(aitile);
		foreach(tile2, _ in adjacent_tiles) {	
	   		if(AIRoad.IsRoadTile(tile2) && Tile.IsAlmostFlat(tile2) ) {
				if (AIRoad.BuildRoad(tile2, aitile)) {
					local it_count = 0; 
					while(!AIRoad.BuildRoadDepot(aitile, tile2) && it_count < 33
						&& AIError.GetLastErrorString() != "ERR_ALREADY_BUILT") {
						EventManager.HandleCommonErrors(aitile, null);		
						it_count++;
					}
					if(it_count >= 33) {
						MailAI.Info(1,"Couldn't build roaddepot: "+AIError.GetLastErrorString());
						return null;
					}
					return aitile;
				}
			}
   		}//end for loop adjacent tiles
	}//end for loop tilelist
	return null;
}