/* 
 *	Copyright 2013 Varen De Meersman
 *  This file is part of MailAI.
 *
 *  MailAI is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  MailAI is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with MailAI.  If not, see <http://www.gnu.org/licenses/>.
 */
class EngineMaster {
	constructor() {}
}

/**
*	Function that will be called when deciding if a possible route has a good distance when
*	compared to the available engines (both truck and train)
*/
function EngineMaster::GetEngine(distance, height_difference, use_trucks, use_trains, force) {	
	local engine1 = null, engine2 = null;
	if(use_trains) {
		engine1 = EngineMaster.GetTrainEngine(distance, height_difference, force);
	}
	if(use_trucks) {
		engine2 = EngineMaster.GetTruckEngine(distance, height_difference, force);
	}
	if(engine1 != null) {
		//prefer trucks
		if(engine2 != null && distance < MailAI.max_medium_route_distance){return engine2;}
		return engine1;
	}	
	if(engine2 != null) {return engine2;}
	return null;
}

function EngineMaster::GetTruckEngine(distance, height_difference, force) {
	local engine_list = EngineMaster.GetTruckEngineList();
	if(engine_list == null){return null;}
	local compare_list = AIList(); compare_list.AddList(engine_list);	
	local speed = abs((distance * 17.86 / MailAI.early_mail));//17,86 is speed for 1 tile/day
	compare_list.Valuate(AIEngine.GetMaxSpeed); compare_list.KeepAboveValue(abs(speed*1.6));//1.6 to convert mph
	if(compare_list.IsEmpty()) {
		compare_list.AddList(engine_list);	
		speed = abs((distance * 17.86 / MailAI.normal_mail));//a bit slower
		compare_list.Valuate(AIEngine.GetMaxSpeed); compare_list.KeepAboveValue(abs(speed*1.6));
		if(compare_list.IsEmpty()) {
			if(force) {	//if no is not an option
				compare_list.AddList(engine_list);
				compare_list.Valuate(AIEngine.GetMaxSpeed);
				compare_list.Sort(AIList.SORT_BY_VALUE, false);
				if(compare_list.IsEmpty()){MailAI.Info(2,"No trucks available!");}
				return compare_list.Begin(); //no good engines for this distance, take fastest (or still null!)
			} else {return null;}
		}
	}	
	compare_list.Valuate(Valuator.GetTruckCostCapacity);
	compare_list.Sort(AIList.SORT_BY_VALUE, true);
	return compare_list.Begin(); //take best cost per capacity
}

function EngineMaster::GetTrainEngine(distance, height_difference, force) {
	local engine_list = EngineMaster.GetTrainEngineList();
	if(engine_list.IsEmpty() || EngineMaster.GetWagonList().IsEmpty()) {
		MailAI.Info(2,"Enginelist or wagonlist null!");return null;
	}
	local height_bonus = 500;
	if(height_difference != null) {
		height_bonus = 500 - (height_difference*100);
	}//arbitrary value to encourage more powerful engines when height difference is big
	local fast_speed = abs((distance * 17.86 / MailAI.early_mail));//17,86 is speed for 1 tile/day
	local slow_speed = abs((distance * 17.86 / MailAI.normal_mail));	
	local lesser_list = AIList(), good_list = AIList();	
	lesser_list.AddList(engine_list);
	lesser_list.Valuate(Valuator.HasEnoughPower, slow_speed, height_bonus); lesser_list.RemoveValue(0);	
	lesser_list.Valuate(AIEngine.GetMaxSpeed); lesser_list.KeepAboveValue(abs(slow_speed*1.6));
	if(lesser_list.IsEmpty()) {
			local neededHP = abs(150 * slow_speed * 0.08);//estimate
			//MailAI.Info(0"no engine fast and powerful enough, needed: "+neededHP+"hp and "+abs(slow_speed*1.6));
		if(force) {
			engine_list.Valuate(AIEngine.GetPower);
			engine_list.Sort(AIList.SORT_BY_VALUE, false);
			if(engine_list.IsEmpty()){MailAI.Info(2,"No train engines available!");}
			//MailAI.Info(0,"no engines powerfull enough, taking most powerful (forced)");
			return engine_list.Begin();
		}
	}
	good_list.AddList(lesser_list);
	good_list.Valuate(Valuator.HasEnoughPower, fast_speed, height_bonus); good_list.RemoveValue(0);
	good_list.Valuate(AIEngine.GetMaxSpeed); good_list.KeepAboveValue(abs(fast_speed*1.6));//1.6 to convert mph
	if(!good_list.IsEmpty()) {
		good_list.Valuate(AIEngine.GetRunningCost);
		good_list.Sort(AIList.SORT_BY_VALUE, true);
		//MailAI.Info(0,"power enough and fast enough, taking cheapest running cost");
		return good_list.Begin();
	}
	good_list.AddList(lesser_list);
	//is lesser list allready --> good_list.Valuate(Valuator.HasEnoughPower, slow_speed, height_bonus); good_list.RemoveValue(0);
	good_list.Valuate(AIEngine.GetMaxSpeed); good_list.KeepAboveValue(abs(slow_speed*1.6));//1.6 to convert mph
	if(!good_list.IsEmpty()) {	
		good_list.Valuate(AIEngine.GetMaxSpeed);
		good_list.Sort(AIList.SORT_BY_VALUE, false);
		//MailAI.Info(0,"power enough but less fast, taking fastest");
		return good_list.Begin();
	}
	good_list.AddList(lesser_list);
	good_list.Valuate(AIEngine.GetMaxSpeed); good_list.KeepAboveValue(abs(fast_speed*1.6));//1.6 to convert mph
	if(!good_list.IsEmpty()) {	
		good_list.Valuate(Valuator.HasEnoughPower, slow_speed, height_bonus);
		good_list.Sort(AIList.SORT_BY_VALUE, false);
		//MailAI.Info(0,"fast enough but lesser power, taking most powerful");
		return good_list.Begin();
	}
	good_list.AddList(lesser_list);
	good_list.Valuate(AIEngine.GetMaxSpeed); good_list.KeepAboveValue(abs(slow_speed*1.6));//1.6 to convert mph
	if(!good_list.IsEmpty()) {	
		good_list.Valuate(AIEngine.GetMaxSpeed);
		good_list.Sort(AIList.SORT_BY_VALUE, false);
		//MailAI.Info(0,"less fast, less power, taking fastest");
		return good_list.Begin();
	}
	if(!force){return null;}
	good_list.AddList(lesser_list);
	good_list.Valuate(AIEngine.GetMaxSpeed); good_list.Sort(AIList.SORT_BY_VALUE, false);
	//MailAI.Info(0,"not fast, no power, returning fastest (forced)");
	return good_list.Begin();
}

function EngineMaster::GetCheapTrainEngine(distance) {
	local engine_list = EngineMaster.GetTrainEngineList();
	if(engine_list.IsEmpty() || EngineMaster.GetWagonList().IsEmpty()) {
		MailAI.Info(1,"Enginelist or wagonlist null!");return null;
	}
	local slow_speed = abs((distance * 17.86 / MailAI.normal_mail));	
	local min_list = AIList(), good_list = AIList();	
	min_list.AddList(engine_list);	good_list.AddList(engine_list);
	min_list.Valuate(AIEngine.GetMaxSpeed); min_list.KeepAboveValue(abs(slow_speed*1.6));
	if(min_list.IsEmpty()) {
		good_list.Valuate(AIEngine.GetMaxSpeed); good_list.Sort(AIList.SORT_BY_VALUE, false);
		return good_list.Begin();
	}
	min_list.Valuate(AIEngine.GetRunningCost);
	min_list.Sort(AIList.SORT_BY_VALUE, true);
	return min_list.Begin();
}

/**
*	Looks for mail carrying wagons fit for the given engine.
*	Returns the lightest one if multiple are fast enough, otherwise the fastest.
*	@param engine The engine that is going to pull the wagons.
*	@param capacity The lowest capacity each wagon needs.
*	@return Lightest wagon fast enough or fastest wagon.
*/
function EngineMaster::GetWagon(engine,capacity) {
	if(engine == null) {throw("engine should not be null in EngineMaster.GetWagon");}
	local speed = AIEngine.GetMaxSpeed(engine);
	local wagon_list = EngineMaster.GetWagonList();
	if(wagon_list.IsEmpty()){throw("wagon list should not be empty in EngineMaster.GetWagon");}
	wagon_list.Valuate(AIEngine.GetMaxSpeed);
	wagon_list.RemoveBetweenValue(0,speed);//0 is unlimited speed
	if(wagon_list.IsEmpty()) {	//no wagon fast enough, return fastest
		wagon_list = EngineMaster.GetWagonList();
		wagon_list.Valuate(AIEngine.GetMaxSpeed);	
		wagon_list.Sort(AIList.SORT_BY_VALUE, false);
		return wagon_list.Begin();
	}
	local hold_list = AIList();
	hold_list.AddList(wagon_list);
	if(capacity != null) {
		wagon_list.Valuate(Valuator.GetMailCapacity);
		wagon_list.RemoveBelowValue(capacity);
		if(!wagon_list.IsEmpty()) {	//cap reached, use cheapest
			wagon_list.Valuate(AIEngine.GetPrice);
			wagon_list.Sort(AIList.SORT_BY_VALUE, true);
			return wagon_list.Begin();
		}
	}
	//cap not reached or not provided, use highest cap
	hold_list.Valuate(Valuator.GetMailCapacity);
	hold_list.Sort(AIList.SORT_BY_VALUE, false);
	return hold_list.Begin();
}

/**
*	This function will weed out unreliable, low capacity or not allowed engines.
*/
function EngineMaster::GetTruckEngineList() {
	local truck_list = AIEngineList(AIVehicle.VT_ROAD);	
	truck_list.Valuate(AIEngine.GetRoadType); truck_list.KeepValue(AIRoad.ROADTYPE_ROAD);		
	if(!AIController.GetSetting("use_art_trucks")){
		truck_list.Valuate(AIEngine.IsArticulated); truck_list.KeepValue(0);
	}
	//truck_list.Valuate(AIEngine.GetCargoType);
	//foreach(e, v in truck_list) {MailAI.Info(0,"engine: "+AIEngine.GetName(e)+" has "+v);}
	truck_list.Valuate(Valuator.CanPullMail); truck_list.KeepValue(1);
	local hold_list	= AIList();
	hold_list.AddList(truck_list);	
	truck_list.Valuate(AIEngine.GetReliability); truck_list.KeepAboveValue(MailAI.try_reliability);
	if(truck_list.IsEmpty()) {
		hold_list.Valuate(AIEngine.GetReliability); hold_list.KeepAboveValue(MailAI.min_reliability);
		if(hold_list.IsEmpty()) {return null;}
		return hold_list;
	}
	return truck_list;
}

/**
*	This function will weed out unreliable, low capacity or not allowed engines.
*/
function EngineMaster::GetTrainEngineList() {
	local engine_list = AIEngineList(AIVehicle.VT_RAIL);
	local hold_list = AIEngineList(AIVehicle.VT_RAIL);
	hold_list.Valuate(AIEngine.CanRunOnRail,0); hold_list.KeepValue(1);	
	engine_list.Valuate(AIEngine.CanRunOnRail,1); engine_list.KeepValue(1);
	engine_list.AddList(hold_list);
	engine_list.Valuate(Valuator.CanPullMail); engine_list.KeepValue(1);			
	hold_list.Clear();
	engine_list.Valuate(AIEngine.GetPrice); engine_list.Sort(AIList.SORT_BY_VALUE, true);
	local max_price = AIEngine.GetPrice(engine_list.Begin())*50;
	//MailAI.Info(0,"max engine price: "+max_price+" engine: "+AIEngine.GetName(engine_list.Begin()));
	engine_list.RemoveAboveValue(max_price);
	hold_list.AddList(engine_list);	
	engine_list.Valuate(AIEngine.GetReliability); engine_list.KeepAboveValue(MailAI.try_reliability);
	if(engine_list.IsEmpty()) {
		hold_list.Valuate(AIEngine.GetReliability); hold_list.KeepAboveValue(MailAI.min_reliability);
		return hold_list;
	}
	return engine_list;
}

/**
*	This function will weed out low capacity or not allowed wagons.
*/
function EngineMaster::GetWagonList() {
	local wagon_list = AIEngineList(AIVehicle.VT_RAIL);
	wagon_list.Valuate(AIEngine.IsWagon); wagon_list.KeepValue(1);
	wagon_list.Valuate(AIEngine.CanRefitCargo,MailAI.GetMailID()); wagon_list.KeepValue(1);
	wagon_list.Valuate(AIEngine.CanRunOnRail,0); wagon_list.KeepValue(1);
	//wagon_list.Valuate(Valuator.GetMailCapacity); wagon_list.KeepAboveValue(MailAI.min_capacity);
	return wagon_list;
}

/**
*	Simply sells the vehicle and all wagons left if any.
*	@param veh The vehicle to be sold.
*	@return The boolean from the sell command.
*/
function EngineMaster::SellVehicle(veh) {
	if(veh == null){return false;}
	if(AIVehicle.GetVehicleType(veh)==AIVehicle.VT_RAIL) {
		AIVehicle.SellWagonChain(veh,1);//should leave engine
	}
	if(!AIVehicle.SellVehicle(veh)) {
		MailAI.Info(3,"Couldn't sell vehicle: "+AIError.GetLastErrorString());
		return false;
	}
	return true;	
}

function EngineMaster::SellWagon(veh) {
	if(veh == null){return false;}
	if(AIVehicle.GetVehicleType(veh)!=AIVehicle.VT_RAIL) {
		throw("Couldn't sell Wagon because no Rail vehicle in SellWagon.");
		return false;
	}
	if(!AIVehicle.SellWagon(veh,AIVehicle.GetNumWagons(veh)-1)) {
		MailAI.Info(3,"Couldn't sell Wagon: "+AIError.GetLastErrorString());
		return false;
	}
	return true;
}


/**
*	Replaces the 'old' veh with a new one (possibly another design).
*	Keeps the same orders and calls AddWagons to add those to the engine.
*	@param veh The vehicle to be replaced.
*	@return null if failed, true if succeeded, false if order copy failed
*/
function EngineMaster::RenewVehicle(veh) {
	MailAI.Info(1,"Renewing: "+AIVehicle.GetName(veh));
	local st1_tile = AIOrder.GetOrderDestination(veh,0);
	local st2_tile = AIOrder.GetOrderDestination(veh,1);
	local dist = AITile.GetDistanceManhattanToTile(st1_tile,st2_tile);
	local model;
	if(AIVehicle.GetVehicleType(veh)==AIVehicle.VT_RAIL) {
		model = EngineMaster.GetTrainEngine(dist,null,true);
	} else if(AIVehicle.GetVehicleType(veh)==AIVehicle.VT_ROAD) {
		model = EngineMaster.GetTruckEngine(dist,null,true);
	} else {throw("Wrong vehicletype in EngineMaster.RenewVehicle");}
	if(model==null) {MailAI.Info(2,"No Engine in RenewVehicle!");return null;}
	local new_veh = BankManager.BuildVehicle(AIVehicle.GetLocation(veh), model);
	MailAI.Info(1,"Renewed: "+AIVehicle.GetName(veh)+" as "+AIVehicle.GetName(new_veh));
	AIVehicle.RefitVehicle(new_veh,AICargo.CC_MAIL);
	if(!AIOrder.CopyOrders(new_veh,veh)) {MailAI.Info(2,"Couldn't copy orders: "+AIError.GetLastErrorString());}
	if(AIVehicle.GetVehicleType(new_veh)==AIVehicle.VT_RAIL) {
		EngineMaster.AddWagons(new_veh,dist);
	}
	EngineMaster.SellVehicle(veh);
	AIVehicle.StartStopVehicle(new_veh);
	return true;
}

function EngineMaster::RenewVehicleCheaper(veh) {
	MailAI.Info(1,"Renewing cheaper: "+AIVehicle.GetName(veh));
	local st1_tile = AIOrder.GetOrderDestination(veh,0);
	local st2_tile = AIOrder.GetOrderDestination(veh,1);
	local dist = AITile.GetDistanceManhattanToTile(st1_tile,st2_tile);
	local model = EngineMaster.GetCheapTrainEngine(dist);
	if(model==null) {MailAI.Info(2,"No Engine in RenewVehicleCheaper");return null;}
	local new_veh = BankManager.BuildVehicle(AIVehicle.GetLocation(veh), model);
	MailAI.Info(1,"Renewed Cheaper: "+AIVehicle.GetName(veh)+" as "+AIVehicle.GetName(new_veh));
	AIVehicle.RefitVehicle(new_veh,AICargo.CC_MAIL);
	if(!AIOrder.CopyOrders(new_veh,veh)) {MailAI.Info(2,"Couldn't copy orders: "+AIError.GetLastErrorString());}
	if(!AIVehicle.MoveWagonChain(veh, 0, new_veh, 0)) {
		MailAI.Info(2,"Couldn't move wagons: "+AIError.GetLastErrorString());
		EngineMaster.AddWagons(new_veh,dist);
	}
	EngineMaster.SellVehicle(veh);
	AIVehicle.StartStopVehicle(new_veh);
	return true;
}

/**
*	Adds wagons to the provided new_veh.
*	Capacity will be calculated based on distance and population of the towns.
*	@param new_veh The new AIEngine to add to, must be at depot.
*	@param town1 First town the vehicle will service.
*	@param town2 Second town the vehicle will service.
*	@return Null if failed, true if succeeded and refitted, false if not refitted.
*/
function EngineMaster::AddWagons(new_veh,dist) {
	local depot = AIVehicle.GetLocation(new_veh);
	local new_wagon = EngineMaster.GetWagon(AIVehicle.GetEngineType(new_veh),null);
	local total_length = AIVehicle.GetLength(new_veh);//in 1/16th of a tile, will grow with MoveWagon
	local moved = false, force_wagon = null;
	local source_wagon = BankManager.BuildVehicle(depot, new_wagon);
	if(source_wagon != null){moved = AIVehicle.MoveWagon(source_wagon,0,new_veh,0);}
	if(!moved) {	
		//fail safe in case newGRF's don't allow certain wagons on certain engines
		MailAI.Info(1,"New GRF not allowing wagon, trying others...");
		if(source_wagon != null){AIVehicle.SellVehicle(source_wagon);}
		new_wagon = null;
		local wagon_list = EngineMaster.GetWagonList();
		wagon_list.Valuate(AIEngine.GetMaxSpeed);	
		wagon_list.Sort(AIList.SORT_BY_VALUE, false);
		foreach(wagon, _ in wagon_list) {
			source_wagon = BankManager.BuildVehicle(depot, wagon);
			if(source_wagon != null) {moved = AIVehicle.MoveWagon(source_wagon,0,new_veh,0);}
			if(moved) {
				MailAI.Info(1,"Found another wagon that suits the job.");
				new_wagon = wagon;
				force_wagon = wagon;
				break;
			} else {
				if(source_wagon != null){AIVehicle.SellVehicle(source_wagon);}
			}
		}	
	}
	if(new_wagon == null) {
		MailAI.Info(2,"Couldn't attach wagons to "+AIVehicle.GetName(new_veh)+"!");
		return null;
	}	
	local min_cap = dist;//calculate min. capacity for the route
	if(min_cap < MailAI.min_train_route_capacity) {min_cap = MailAI.min_train_route_capacity;}
	local rest_cap = min_cap - AIVehicle.GetRefitCapacity(new_veh,MailAI.GetMailID());
	local wagon_length = AIVehicle.GetLength(new_veh) - total_length;//there is one wagon attached
	total_length = AIVehicle.GetLength(new_veh);
	local old_wagon = new_wagon;
	if(force_wagon == null) {new_wagon = EngineMaster.GetWagon(AIVehicle.GetEngineType(new_veh),rest_cap);}
	while(total_length + wagon_length <= 48 && rest_cap > 0) {
		source_wagon = BankManager.BuildVehicle(depot, new_wagon);
		if(source_wagon != null) {moved = AIVehicle.MoveWagon(source_wagon,0,new_veh,AIVehicle.GetNumWagons(new_veh)-1);}
		if(!moved) {	
			MailAI.Info(2,"Something wrong moving wagons: "+AIError.GetLastErrorString());
			if(AIVehicle.GetRefitCapacity(new_veh,MailAI.GetMailID()) > min_cap)
				{break;} else {return null;}
		}	
		wagon_length = AIVehicle.GetLength(new_veh) - total_length;
		total_length = AIVehicle.GetLength(new_veh);
		rest_cap = min_cap - AIVehicle.GetRefitCapacity(new_veh,MailAI.GetMailID());
		if(total_length > 48) {	
			//if the last wagon would exceed the platform take the previous wagon
			AIVehicle.SellWagon(new_veh,AIVehicle.GetNumWagons(new_veh)-1);
			source_wagon = BankManager.BuildVehicle(depot, old_wagon);
			if(source_wagon != null) {AIVehicle.MoveWagon(source_wagon,0,new_veh,0);}
			break;
		}
		old_wagon = new_wagon;
		if(force_wagon == null) {new_wagon = EngineMaster.GetWagon(AIVehicle.GetEngineType(new_veh),rest_cap);}
	}
	return AIVehicle.RefitVehicle(new_veh,MailAI.GetMailID());
}
