/* 
 *	Copyright 2013 Varen De Meersman
 *  This file is part of MailAI.
 *
 *  MailAI is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  MailAI is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with MailAI.  If not, see <http://www.gnu.org/licenses/>.
 */
 
class RailBuilder {
	constructor() {}
}

/**
*	Finds a 2*4 (kind off) to build a switch
*	@param left_switch The left_switch tile from stationbuilding
*	@direction The direction of the station
*	@side The direction it will offset if not finding spots to build
*	@return The left entry railtile and the direction of the switch
*/
function RailBuilder::GetSwitchSquare(left_switch,stat_dir,side_dir) {			
	if(left_switch==null || stat_dir==null || side_dir==null){throw("null in getswitchsquare");}
	local x = Tile.SideToX(stat_dir,1); local y = Tile.SideToY(stat_dir,1);
	local sx = Tile.SideToX(side_dir,1); local sy = Tile.SideToY(side_dir,1);	
	local left, left2, found, found2, counter = 0;
	local start = left_switch + 3 * AIMap.GetTileIndex(x,y);
	while(counter < 8) {		
		counter++; found = true; found2 = true;
		left = start + counter * AIMap.GetTileIndex(x,y);
		left2 = left + counter * AIMap.GetTileIndex(sx,sy);
		if(!Tile.IsBuildableRailTileSlope(left,stat_dir)) {found = false;}
		if(!Tile.IsBuildableRailTileSlope(left + AIMap.GetTileIndex(y,-x),stat_dir)) {found = false;}	
		if(!Tile.IsBuildableRailTileSlope(left2,stat_dir)) {found2 = false;}
		if(!Tile.IsBuildableRailTileSlope(left2 + AIMap.GetTileIndex(y,-x),stat_dir)) {found2 = false;}	
		left += AIMap.GetTileIndex(x,y);// left = l0 (el-nul)
		left2 += AIMap.GetTileIndex(x,y);
		if(!Tile.IsBuildableSwitchTile(left,left + AIMap.GetTileIndex(y,-x))) {found = false;}	
		if(!Tile.IsBuildableSwitchTile(left2,left2 + AIMap.GetTileIndex(y,-x))) {found2 = false;}	
		left += AIMap.GetTileIndex(x,y);// left = l1
		left2 += AIMap.GetTileIndex(x,y);
		if(!AITile.IsBuildable(left)) {found = false;}
		if(!AITile.IsBuildable(left + AIMap.GetTileIndex(y,-x))) {found = false;}
		if(!Tile.WillSlopeTheSame(left,left + AIMap.GetTileIndex(y,-x),stat_dir)) {found = false;}	
		if(!AITile.IsBuildable(left2)) {found2 = false;}
		if(!AITile.IsBuildable(left2 + AIMap.GetTileIndex(y,-x))) {found2 = false;}
		if(!Tile.WillSlopeTheSame(left2,left2 + AIMap.GetTileIndex(y,-x),stat_dir)) {found2 = false;}		
		left += AIMap.GetTileIndex(x,y);// left = l2
		left2 += AIMap.GetTileIndex(x,y);
		if(!Tile.IsBuildableRailTileSlope(left + AIMap.GetTileIndex(y,-x),stat_dir)) {found = false;}
		if(!Tile.IsBuildableRailTileSlope(left2 + AIMap.GetTileIndex(y,-x),stat_dir)) {found2 = false;}		
		left = start + counter*AIMap.GetTileIndex(x,y);// left = l3???
		left2 = left + counter*AIMap.GetTileIndex(sx,sy);
		if(found) {return left;}//if we reach this it's a good location	
		if(found2) {return left2;}
	}
	return null;
}

function RailBuilder::GetMailToThePeopleLocation(pre,start) {
	if(pre==null || start==null){throw("null in GetMailToThePeopleLocation");}
	local pre_tile, work_tile = pre, next_tile = start;
	local first_connect, skip = true;
	local next_tiles = [], side_option;
	if(AIRail.IsRailStationTile(pre)) {//skip the first 10 tracks out of the station (mostly switch anyway)
		local c = 10;
		while(c>0) {
			c--; pre_tile = work_tile; work_tile = next_tile;
			next_tiles = Tile.GetNextRailTile(pre_tile, work_tile);	
			pre_tile = next_tiles[0]; work_tile = next_tiles[1]; next_tile = next_tiles[2];
		}
	}
	while(true) {
		pre_tile = work_tile; work_tile = next_tile;
		next_tiles = Tile.GetNextRailTile(pre_tile, work_tile);
		if(next_tiles == null){return null;}
		if(next_tiles[0] != null) {//skipping bridge or tunnel	
			pre_tile = next_tiles[0]; work_tile = next_tiles[1];
		} 	
		next_tile = next_tiles[2];
		local exit_dir = Tile.GetDirectionFromTo(work_tile,next_tile);	
		if(Tile.GetNeighbourRailCount(next_tile) > 2) {skip=true; continue;}		
		if(skip) {skip=false;continue;}
		if(!Tile.IsAlmostFlat(work_tile) || !Tile.IsAlmostFlat(next_tile)) {continue;}
		local around = AITileList();
		around.AddRectangle(work_tile+AIMap.GetTileIndex(-20, -20),tile + AIMap.GetTileIndex(20, 20));
		around.Valuate(Valuator.ContainsMailToThePeopleTown);
		around.KeepValue(1);
		if(around.Count()==0){skip = true; continue;}
		foreach(t, _ in around) {MailAI.BuildSign(t,"haha");}
		
//straight track
		if(AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NW_SE || AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NE_SW) {
			if(AIRail.GetRailTracks(next_tile) != AIRail.GetRailTracks(work_tile)) {skip = true; continue;}
			
			local first_left = Tile.GetLeftTile(work_tile,exit_dir);
			local first_right = Tile.GetRightTile(work_tile,exit_dir);
			if(Tile.IsCloser(first_left,first_right,a_few_tiles_earlier)) {
				if(Tile.IsCloser(first_left,keep[0],a_few_tiles_earlier)
				&& exit_dir != Tile.GetReverseDirection(Tile.GetDirectionFromTo(work_tile,a_few_tiles_earlier))) {
					local left = Tile.GetLeftTile(next_tile,exit_dir);
					local leftleft = Tile.GetLeftTile(left,exit_dir);
					local leftfwd = Tile.GetForwardTile(left,exit_dir);
					local corner = Tile.GetForwardLeftTile(left,exit_dir);
					if(Tile.IsBuildableSwitchTile(first_left,work_tile) && Tile.IsBuildableSwitchTile(left,work_tile)
					&& Tile.IsBuildableSwitchTile(leftleft,work_tile) && Tile.IsBuildableSwitchTile(leftfwd,work_tile)
					&& Tile.IsBuildableSwitchTile(corner,work_tile)) {	
						if(Tile.GetDirectionFromTo(left,a_few_tiles_earlier) != exit_dir) {			
							keep = [work_tile,next_tile,left,first_left,leftleft,Tile.GetLeftDirection(exit_dir),exit_dir];
						} else {
							keep = [work_tile,next_tile,leftleft,first_left,leftleft,exit_dir,Tile.GetLeftDirection(exit_dir)];
						}
						if(force){return RailBuilder.ReturnClosest(side_option,keep,a_few_tiles_earlier);}
					}
					continue;
				} else {return RailBuilder.ReturnClosest(side_option,keep,a_few_tiles_earlier);}	
			} else {
				if(Tile.IsCloser(first_right,keep[0],a_few_tiles_earlier)
				&& exit_dir != Tile.GetReverseDirection(Tile.GetDirectionFromTo(work_tile,a_few_tiles_earlier))) {
					local right = Tile.GetRightTile(next_tile,exit_dir);
					local rightright = Tile.GetRightTile(right,exit_dir);
					local rightfwd = Tile.GetForwardTile(right,exit_dir);
					local corner = Tile.GetForwardRightTile(right,exit_dir);
					if(Tile.IsBuildableSwitchTile(first_right,work_tile) && Tile.IsBuildableSwitchTile(right,work_tile)
					&& Tile.IsBuildableSwitchTile(rightright,work_tile)	&& Tile.IsBuildableSwitchTile(rightfwd,work_tile)
					&& Tile.IsBuildableSwitchTile(corner,work_tile)) {
						if(Tile.GetDirectionFromTo(right,a_few_tiles_earlier) != exit_dir) {
							keep = [work_tile,next_tile,rightfwd,first_right,rightright,Tile.GetRightDirection(exit_dir),exit_dir];
						} else {
							keep = [work_tile,next_tile,right,first_right,rightright,exit_dir,Tile.GetRightDirection(exit_dir)];
						}
						if(force){return RailBuilder.ReturnClosest(side_option,keep,a_few_tiles_earlier);}
					}
					continue;
				} else {return RailBuilder.ReturnClosest(side_option,keep,a_few_tiles_earlier);}
			}	
		}		
//north left, east up, south right, west down
		if((AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NW_NE && exit_dir == "NW")		
		|| (AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NE_SE && exit_dir == "NE")
		|| (AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_SW_SE && exit_dir == "SE")	
		|| (AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NW_SW && exit_dir == "SW")) {
			if(AIRail.GetRailTracks(work_tile) != AIRail.GetRailTracks(Tile.GetForwardLeftTile(work_tile,exit_dir))) {
				//MailAI.BuildSign(work_tile,"w");MailAI.BuildSign(Tile.GetForwardLeftTile(work_tile,exit_dir),"n");
				skip = true; continue;
			}
			local left = Tile.GetLeftTile(work_tile,exit_dir);
			local leftfwd = Tile.GetForwardLeftTile(work_tile,exit_dir);
			if(Tile.IsCloser(Tile.GetLeftTile(left,exit_dir),leftfwd,a_few_tiles_earlier)) {	
				if(Tile.IsCloser(Tile.GetLeftTile(left,exit_dir),keep[0],a_few_tiles_earlier)
				&& exit_dir != Tile.GetReverseDirection(Tile.GetDirectionFromTo(work_tile,a_few_tiles_earlier))) {
					if(Tile.IsBuildableSwitchTile(left,work_tile) && Tile.IsAlmostFlat(leftfwd)
					&& Tile.IsBuildableAlmostFlatTile(Tile.GetLeftTile(left,exit_dir))) {
						keep = [work_tile,leftfwd,left,left,null,Tile.GetLeftDirection(exit_dir),exit_dir];
						if(force){return RailBuilder.ReturnClosest(side_option,keep,a_few_tiles_earlier);}
					}
				} else {return RailBuilder.ReturnClosest(side_option,keep,a_few_tiles_earlier);}
			}			
		}	
//north right, east down, south left, west up
		if((AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NW_NE && exit_dir == "NE") 
		|| (AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NE_SE && exit_dir == "SE") 
		|| (AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_SW_SE && exit_dir == "SW") 	
		|| (AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NW_SW && exit_dir == "NW")) {
			if(AIRail.GetRailTracks(work_tile) != AIRail.GetRailTracks(Tile.GetForwardRightTile(work_tile,exit_dir))) {
				//MailAI.BuildSign(work_tile,"w");MailAI.BuildSign(Tile.GetForwardRightTile(work_tile,exit_dir),"n");
				skip = true; continue;
			}
			local right = Tile.GetRightTile(work_tile,exit_dir);
			local rightfwd = Tile.GetForwardRightTile(work_tile,exit_dir);
			if(Tile.IsCloser(Tile.GetRightTile(work_tile,exit_dir),rightfwd,a_few_tiles_earlier)) {
				if(Tile.IsCloser(Tile.GetRightTile(work_tile,exit_dir),keep[0],a_few_tiles_earlier)
				&& exit_dir != Tile.GetReverseDirection(Tile.GetDirectionFromTo(work_tile,a_few_tiles_earlier))) {	
					if(Tile.IsBuildableSwitchTile(right,work_tile) && Tile.IsAlmostFlat(rightfwd)
					&& Tile.IsBuildableAlmostFlatTile(Tile.GetRightTile(work_tile,exit_dir))) {
						keep = [work_tile,rightfwd,rightfwd,right,null,Tile.GetRightDirection(exit_dir),exit_dir];
						if(force){return RailBuilder.ReturnClosest(side_option,keep,a_few_tiles_earlier);}
					}	
				} else {return RailBuilder.ReturnClosest(side_option,keep,a_few_tiles_earlier);}
			}			
		}			
	}//end while true
}

/**
*	Runs over a rail line to find a spot from where it can split into two (three) tracks.
*/
function RailBuilder::GetNextJoiningLocation(pre,start,goal_tile,force) {
	if(pre==null || start==null){throw("null in getnextjoininglocation");}
	local pre_tile, work_tile = pre, next_tile = start;
	local first_connect, second_connect, skip = true;
	local joining1, joining2, keep = [null,null,null,null,null,null,null];
	local next_tiles = [], side_option;
	local sw_dir = Tile.GetDirectionFromTo(pre,start);
	local a_few_tiles_earlier = goal_tile + AIMap.GetTileIndex(Tile.SideToX(sw_dir,3),Tile.SideToY(sw_dir,3))
	if(AIRail.IsRailStationTile(pre)) {//skip the first 10 tracks out of the station (mostly switch anyway)
		local c = 10;
		while(c>0) {
			c--; pre_tile = work_tile; work_tile = next_tile;
			next_tiles = Tile.GetNextRailTile(pre_tile, work_tile);	
			if(next_tiles == null || next_tiles[2] == null){return null;}
			pre_tile = work_tile; work_tile = next_tile; next_tile = next_tiles[2];
		}
	}
	//MailAI.BuildSign(pre_tile,"pre");MailAI.BuildSign(work_tile,"start");
	while(true) {
//fill tile values
		pre_tile = work_tile; work_tile = next_tile;
		next_tiles = Tile.GetNextRailTile(pre_tile, work_tile);
		if(next_tiles == null){if(keep[0]!=null){return keep;}else{return null;}}
		if(next_tiles[0] != null) {//skipping bridge or tunnel	
			pre_tile = next_tiles[0]; work_tile = next_tiles[1];
		} 
		next_tile = next_tiles[2];
		local exit_dir = Tile.GetDirectionFromTo(work_tile,next_tile);
		local to_go_dir = Tile.GetDirectionFromTo(work_tile,a_few_tiles_earlier);	
		if(Tile.GetNeighbourRailCount(next_tile) > 2) {
			local first_tile = AIRail.GetRailTracks(work_tile);
			if(first_tile == 1 || first_tile == 2){skip=true; continue;}
			local second_tile = AIRail.GetRailTracks(next_tile);
			if(first_tile==second_tile) {
				joining1 = work_tile;
				joining2 = Tile.GetLeftTile(work_tile,exit_dir);
				if(!AIRail.IsRailTile(joining2)) {
					joining2 = Tile.GetRightTile(work_tile,exit_dir);
				}
				//MailAI.BuildSign(joining1,""+first_tile);MailAI.BuildSign(joining2,""+second_tile);
				side_option = RailBuilder.GetNextJoiningLocation(joining1,joining2,a_few_tiles_earlier,force);
			} else {skip=true; continue;}			
		}
		if(skip){skip=false;continue;}
		if(!Tile.IsAlmostFlat(work_tile) || !Tile.IsAlmostFlat(next_tile)) {continue;}
//straight track
		if(keep[0]!=null && AIMap.DistanceManhattan(keep[0],a_few_tiles_earlier) < 40){return keep;}
		if(AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NW_SE || AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NE_SW) {
			if(AIRail.GetRailTracks(next_tile) != AIRail.GetRailTracks(work_tile)) {
				skip = true; continue;
			}
			local first_left = Tile.GetLeftTile(work_tile,exit_dir);
			local first_right = Tile.GetRightTile(work_tile,exit_dir);
			if(Tile.IsCloser(first_left,first_right,a_few_tiles_earlier)) {
				if(Tile.IsCloser(first_left,keep[0],a_few_tiles_earlier)
				&& exit_dir != Tile.GetReverseDirection(Tile.GetDirectionFromTo(work_tile,a_few_tiles_earlier))) {
					local left = Tile.GetLeftTile(next_tile,exit_dir);
					local leftleft = Tile.GetLeftTile(left,exit_dir);
					local leftfwd = Tile.GetForwardTile(left,exit_dir);
					local corner = Tile.GetForwardLeftTile(left,exit_dir);
					if(Tile.IsBuildableSwitchTile(first_left,work_tile) && Tile.IsBuildableSwitchTile(left,work_tile)
					&& Tile.IsBuildableSwitchTile(leftleft,work_tile) && Tile.IsBuildableSwitchTile(leftfwd,work_tile)
					&& Tile.IsBuildableSwitchTile(corner,work_tile)) {	
						if(Tile.GetDirectionFromTo(left,a_few_tiles_earlier) != exit_dir) {			
							keep = [work_tile,next_tile,left,first_left,leftleft,Tile.GetLeftDirection(exit_dir),exit_dir];
						} else {
							keep = [work_tile,next_tile,leftleft,first_left,leftleft,exit_dir,Tile.GetLeftDirection(exit_dir)];
						}
						if(force){return RailBuilder.ReturnClosest(side_option,keep,a_few_tiles_earlier);}
					}
					continue;
				} else {return RailBuilder.ReturnClosest(side_option,keep,a_few_tiles_earlier);}	
			} else {
				if(Tile.IsCloser(first_right,keep[0],a_few_tiles_earlier)
				&& exit_dir != Tile.GetReverseDirection(Tile.GetDirectionFromTo(work_tile,a_few_tiles_earlier))) {
					local right = Tile.GetRightTile(next_tile,exit_dir);
					local rightright = Tile.GetRightTile(right,exit_dir);
					local rightfwd = Tile.GetForwardTile(right,exit_dir);
					local corner = Tile.GetForwardRightTile(right,exit_dir);
					if(Tile.IsBuildableSwitchTile(first_right,work_tile) && Tile.IsBuildableSwitchTile(right,work_tile)
					&& Tile.IsBuildableSwitchTile(rightright,work_tile)	&& Tile.IsBuildableSwitchTile(rightfwd,work_tile)
					&& Tile.IsBuildableSwitchTile(corner,work_tile)) {
						if(Tile.GetDirectionFromTo(right,a_few_tiles_earlier) != exit_dir) {
							keep = [work_tile,next_tile,rightfwd,first_right,rightright,Tile.GetRightDirection(exit_dir),exit_dir];
						} else {
							keep = [work_tile,next_tile,right,first_right,rightright,exit_dir,Tile.GetRightDirection(exit_dir)];
						}
						if(force){return RailBuilder.ReturnClosest(side_option,keep,a_few_tiles_earlier);}
					}
					continue;
				} else {return RailBuilder.ReturnClosest(side_option,keep,a_few_tiles_earlier);}
			}	
		}		
//north left, east up, south right, west down
		if((AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NW_NE && exit_dir == "NW")		
		|| (AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NE_SE && exit_dir == "NE")
		|| (AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_SW_SE && exit_dir == "SE")	
		|| (AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NW_SW && exit_dir == "SW")) {
			if(AIRail.GetRailTracks(work_tile) != AIRail.GetRailTracks(Tile.GetForwardLeftTile(work_tile,exit_dir))) {
				//MailAI.BuildSign(work_tile,"w");MailAI.BuildSign(Tile.GetForwardLeftTile(work_tile,exit_dir),"n");
				skip = true; continue;
			}
			local left = Tile.GetLeftTile(work_tile,exit_dir);
			local leftfwd = Tile.GetForwardLeftTile(work_tile,exit_dir);
			if(Tile.IsCloser(Tile.GetLeftTile(left,exit_dir),leftfwd,a_few_tiles_earlier)) {	
				if(Tile.IsCloser(Tile.GetLeftTile(left,exit_dir),keep[0],a_few_tiles_earlier)
				&& exit_dir != Tile.GetReverseDirection(Tile.GetDirectionFromTo(work_tile,a_few_tiles_earlier))) {
					if(Tile.IsBuildableSwitchTile(left,work_tile) && Tile.IsAlmostFlat(leftfwd)
					&& Tile.IsBuildableAlmostFlatTile(Tile.GetLeftTile(left,exit_dir))) {
						keep = [work_tile,leftfwd,left,left,null,Tile.GetLeftDirection(exit_dir),exit_dir];
						if(force){return RailBuilder.ReturnClosest(side_option,keep,a_few_tiles_earlier);}
					}
				} else {return RailBuilder.ReturnClosest(side_option,keep,a_few_tiles_earlier);}
			}			
		}	
//north right, east down, south left, west up
		if((AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NW_NE && exit_dir == "NE") 
		|| (AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NE_SE && exit_dir == "SE") 
		|| (AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_SW_SE && exit_dir == "SW") 	
		|| (AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NW_SW && exit_dir == "NW")) {
			if(AIRail.GetRailTracks(work_tile) != AIRail.GetRailTracks(Tile.GetForwardRightTile(work_tile,exit_dir))) {
				//MailAI.BuildSign(work_tile,"w");MailAI.BuildSign(Tile.GetForwardRightTile(work_tile,exit_dir),"n");
				skip = true; continue;
			}
			local right = Tile.GetRightTile(work_tile,exit_dir);
			local rightfwd = Tile.GetForwardRightTile(work_tile,exit_dir);
			if(Tile.IsCloser(Tile.GetRightTile(work_tile,exit_dir),rightfwd,a_few_tiles_earlier)) {
				if(Tile.IsCloser(Tile.GetRightTile(work_tile,exit_dir),keep[0],a_few_tiles_earlier)
				&& exit_dir != Tile.GetReverseDirection(Tile.GetDirectionFromTo(work_tile,a_few_tiles_earlier))) {	
					if(Tile.IsBuildableSwitchTile(right,work_tile) && Tile.IsAlmostFlat(rightfwd)
					&& Tile.IsBuildableAlmostFlatTile(Tile.GetRightTile(work_tile,exit_dir))) {
						keep = [work_tile,rightfwd,rightfwd,right,null,Tile.GetRightDirection(exit_dir),exit_dir];
						if(force){return RailBuilder.ReturnClosest(side_option,keep,a_few_tiles_earlier);}
					}	
				} else {return RailBuilder.ReturnClosest(side_option,keep,a_few_tiles_earlier);}
			}			
		}			
	}//end while true
}

/**
*	Extracted to compact code in GetNextJoiningLocation
*/
function RailBuilder::ReturnClosest(side_option,keep,a_few_tiles_earlier) {
	if(side_option == null || side_option[0] == null) {return keep;}
	if(Tile.IsCloser(side_option[0],keep[0],a_few_tiles_earlier)) {
		return side_option;	
	}
	return keep;
}

function RailBuilder::BuildTrainStation(start_tile,stat_dir,rect_start,rect_end) {
	MailAI.Info(0,"Building train station");
	local tl = AITileList(); tl.AddRectangle(rect_start,rect_end);
	Tile.ClearArea(tl);
	local track_dir = Tile.GetTrackDir(stat_dir);
	if(!AIController.GetSetting("use_ISR")
		|| !AIRail.BuildNewGRFRailStation(start_tile,Tile.GetTrackDir(stat_dir),2,3,AIStation.STATION_NEW,
			10, 0xFF, 0xFF, 0, false)) {
		if(!AIRail.BuildRailStation(start_tile,Tile.GetTrackDir(stat_dir),2,3,AIStation.STATION_NEW)) {
			MailAI.Info(2,"Failed building station: "+AIError.GetLastErrorString());
			return null;
		}
	}
	local town = AITile.GetClosestTown(start_tile);
	local set = AIStation.SetName(AIStation.GetStationID(start_tile),AITown.GetName(town)+" Sorting Station");
	if(!set)set = AIStation.SetName(AIStation.GetStationID(start_tile),AITown.GetName(town)+" Secondary");
	if(!set)AIStation.SetName(AIStation.GetStationID(start_tile),AITown.GetName(town)+" Sort");	
	return start_tile;
}

function RailBuilder::BuildDepot(arr) {	
	if(arr==null || arr[0]==null){throw("Empty array in builddepot");}
	local left_switch, right_switch, dir = Tile.GetDirectionFromTo(arr[0],arr[1]);
	local x = Tile.SideToX(dir,1); local y = Tile.SideToY(dir,1);
	local town_loc = AITown.GetLocation(AITile.GetClosestTown(arr[0]));
	if(arr[2]==null) {//nice switch has been build, arr[0] is left_switch
		left_switch = arr[0]; right_switch = Tile.GetRightTile(left_switch,dir);	
		if(Tile.IsCloser(left_switch-AIMap.GetTileIndex(y,-x),right_switch+AIMap.GetTileIndex(y,-x),town_loc)){
			//first look on side away from town
			if(Tile.IsBuildableAlmostFlatTile(right_switch + AIMap.GetTileIndex(y,-x))
			&& EventManager.BuildRail(right_switch + AIMap.GetTileIndex(y,-x),right_switch,left_switch)
			&& EventManager.BuildRail(right_switch + AIMap.GetTileIndex(y,-x),right_switch,right_switch - AIMap.GetTileIndex(x,y))
			&& EventManager.BuildRail(right_switch + AIMap.GetTileIndex(y,-x),right_switch,right_switch + AIMap.GetTileIndex(x,y))){
				if(AIRail.IsRailTile(right_switch + AIMap.GetTileIndex(x,y))) {
					EventManager.BuildRail(right_switch + AIMap.GetTileIndex(y,-x),right_switch,right_switch + AIMap.GetTileIndex(x,y));
				}
				if(AIRail.BuildRailDepot(right_switch + AIMap.GetTileIndex(y,-x),right_switch)) {
					return right_switch + AIMap.GetTileIndex(y,-x);
				} else {Tile.DemolishTile(right_switch + AIMap.GetTileIndex(y,-x));}
			}
		}
		if(Tile.IsBuildableAlmostFlatTile(left_switch - AIMap.GetTileIndex(y,-x))
		&& EventManager.BuildRail(left_switch - AIMap.GetTileIndex(y,-x),left_switch,right_switch)
		&& EventManager.BuildRail(left_switch - AIMap.GetTileIndex(y,-x),left_switch,left_switch - AIMap.GetTileIndex(x,y))
		&& EventManager.BuildRail(left_switch,right_switch,right_switch - AIMap.GetTileIndex(x,y))) {
			if(AIRail.IsRailTile(left_switch + AIMap.GetTileIndex(x,y))) {
				EventManager.BuildRail(left_switch - AIMap.GetTileIndex(y,-x),left_switch,left_switch + AIMap.GetTileIndex(x,y));
			}
			if(AIRail.BuildRailDepot(left_switch - AIMap.GetTileIndex(y,-x),left_switch)) {
				return left_switch - AIMap.GetTileIndex(y,-x);
			} else {Tile.DemolishTile(left_switch - AIMap.GetTileIndex(y,-x));}
		}
		if(Tile.IsBuildableAlmostFlatTile(right_switch + AIMap.GetTileIndex(y,-x))
		&& EventManager.BuildRail(right_switch + AIMap.GetTileIndex(y,-x),right_switch,left_switch)
		&& EventManager.BuildRail(right_switch + AIMap.GetTileIndex(y,-x),right_switch,right_switch - AIMap.GetTileIndex(x,y))
		&& EventManager.BuildRail(right_switch + AIMap.GetTileIndex(y,-x),right_switch,right_switch + AIMap.GetTileIndex(x,y))){
			if(AIRail.IsRailTile(right_switch + AIMap.GetTileIndex(x,y))) {
				EventManager.BuildRail(right_switch + AIMap.GetTileIndex(y,-x),right_switch,right_switch + AIMap.GetTileIndex(x,y));
			}
			if(AIRail.BuildRailDepot(right_switch + AIMap.GetTileIndex(y,-x),right_switch)) {
				return right_switch + AIMap.GetTileIndex(y,-x);
			} else {Tile.DemolishTile(right_switch + AIMap.GetTileIndex(y,-x));}
		}
		MailAI.Info(0,"Couldn't build train depot!");
		return null;
	}
	left_switch = arr[0]; right_switch = arr[2];
	if(Tile.GetLeftTile(left_switch,dir) == right_switch) {
		left_switch = arr[2]; right_switch = arr[0];
	}
	local depot_left = left_switch - AIMap.GetTileIndex(y,-x);
	local depot_right = right_switch + AIMap.GetTileIndex(y,-x);
	if(Tile.IsCloser(depot_right,depot_left,town_loc)){	
		if(Tile.IsBuildableAlmostFlatTile(depot_left) && Tile.IsAlmostFlat(left_switch)) {
			if(AIRail.BuildRailDepot(depot_left,left_switch)
			&& EventManager.BuildRail(depot_left,left_switch,right_switch)
			&& EventManager.BuildRail(depot_left,left_switch,left_switch - AIMap.GetTileIndex(x,y))) {
				if(AIRail.IsRailTile(left_switch + AIMap.GetTileIndex(x,y))) {
					EventManager.BuildRail(depot_left,left_switch,left_switch + AIMap.GetTileIndex(x,y));
				} else { 
					EventManager.BuildRail(left_switch,right_switch, right_switch - AIMap.GetTileIndex(x,y));
				}
			return depot_left;
			}
		}
	}
	if(Tile.IsBuildableAlmostFlatTile(depot_right) && Tile.IsAlmostFlat(right_switch)) {
		if(AIRail.BuildRailDepot(depot_right,right_switch)
		&& EventManager.BuildRail(depot_right,right_switch,left_switch)
		&& EventManager.BuildRail(depot_right,right_switch,right_switch - AIMap.GetTileIndex(x,y))) {
			if(AIRail.IsRailTile(right_switch + AIMap.GetTileIndex(x,y))) {
				EventManager.BuildRail(depot_right,right_switch,right_switch + AIMap.GetTileIndex(x,y));
			} else {
				EventManager.BuildRail(right_switch, left_switch, left_switch - AIMap.GetTileIndex(x,y));
			}
		return depot_right;
		}
	}
	//failsafe repeater
	if(Tile.IsBuildableAlmostFlatTile(depot_left) && Tile.IsAlmostFlat(left_switch)) {
		if(AIRail.BuildRailDepot(depot_left,left_switch)
		&& EventManager.BuildRail(depot_left,left_switch,right_switch)
		&& EventManager.BuildRail(depot_left,left_switch,left_switch - AIMap.GetTileIndex(x,y))) {
			if(AIRail.IsRailTile(left_switch + AIMap.GetTileIndex(x,y))) {
				EventManager.BuildRail(depot_left,left_switch,left_switch + AIMap.GetTileIndex(x,y));
			} else { 
				EventManager.BuildRail(left_switch,right_switch, right_switch - AIMap.GetTileIndex(x,y));
			}
		return depot_left;
		}
	}
	//the other way around if we're building at the switch location
	if(Tile.IsCloser(left_switch + AIMap.GetTileIndex(x,y),right_switch + AIMap.GetTileIndex(x,y),town_loc)){	
		if(Tile.IsBuildableAlmostFlatTile(left_switch + AIMap.GetTileIndex(x,y))
			&& AIRail.BuildRailDepot(left_switch + AIMap.GetTileIndex(x,y),left_switch)
			&& EventManager.BuildRail(left_switch + AIMap.GetTileIndex(x,y),left_switch,right_switch)
			&& EventManager.BuildRail(left_switch + AIMap.GetTileIndex(x,y),left_switch,left_switch - AIMap.GetTileIndex(x,y))
			&& EventManager.BuildRail(left_switch,right_switch,right_switch - AIMap.GetTileIndex(x,y))) {
			return left_switch + AIMap.GetTileIndex(x,y);
		}
	}
	if(Tile.IsBuildableAlmostFlatTile(right_switch + AIMap.GetTileIndex(x,y))
		&& AIRail.BuildRailDepot(right_switch + AIMap.GetTileIndex(x,y),right_switch)
		&& EventManager.BuildRail(right_switch + AIMap.GetTileIndex(x,y),right_switch,left_switch)
		&& EventManager.BuildRail(right_switch + AIMap.GetTileIndex(x,y),right_switch,right_switch - AIMap.GetTileIndex(x,y))
		&& EventManager.BuildRail(right_switch,left_switch,left_switch - AIMap.GetTileIndex(x,y))) {
		return right_switch + AIMap.GetTileIndex(x,y);
	}
	if(Tile.IsBuildableAlmostFlatTile(left_switch + AIMap.GetTileIndex(x,y))
		&& AIRail.BuildRailDepot(left_switch + AIMap.GetTileIndex(x,y),left_switch)
		&& EventManager.BuildRail(left_switch + AIMap.GetTileIndex(x,y),left_switch,right_switch)
		&& EventManager.BuildRail(left_switch + AIMap.GetTileIndex(x,y),left_switch,left_switch - AIMap.GetTileIndex(x,y))
		&& EventManager.BuildRail(left_switch,right_switch,right_switch - AIMap.GetTileIndex(x,y))) {
		return left_switch + AIMap.GetTileIndex(x,y);
	}
	MailAI.Info(1,"Couldn't build train depot!");
	return null;
}

function RailBuilder::BuildExtraRailDepot(tile_list) {
	foreach(right_switch, _ in tile_list) {
		local adj = Tile.GetAdjacentTiles(right_switch);
		foreach(out_tile, _ in adj) {
			if(Tile.IsBuildableAlmostFlatTile(out_tile)) {
				local dir = Tile.GetDirectionFromTo(out_tile,right_switch);
				if(AIRail.BuildRailDepot(out_tile,right_switch)
				&& EventManager.BuildRail(out_tile,right_switch,Tile.GetForwardTile(right_switch,dir))
				&& EventManager.BuildRail(out_tile,right_switch,Tile.GetLeftTile(right_switch,dir))
				&& EventManager.BuildRail(out_tile,right_switch,Tile.GetRightTile(right_switch,dir))) {
					return out_tile;
				}
			}
		}
	}
	return null;
}

function RailBuilder::BuildStationSwitch(arr) {
	local left_switch = arr[1], stat_dir = arr[2];
	local side_dir = arr[3], track_dir = Tile.GetTrackDir(stat_dir);
	local x = Tile.SideToX(stat_dir,1), y = Tile.SideToY(stat_dir,1);
	local right_switch = left_switch + AIMap.GetTileIndex(y,-x);
	local left_exit = left_switch + AIMap.GetTileIndex(x,y);
	local right_entry = left_exit + AIMap.GetTileIndex(y,-x);			
//while switch tiles not good yet
	while(!Tile.IsBuildableAlmostFlatTile(left_switch) || !Tile.IsBuildableAlmostFlatTile(right_switch)) {
		if(!Tile.DemolishTile(left_switch) || !Tile.DemolishTile(right_switch)) {
			MailAI.Info(0,"Couldn't destroy switch out of station! "+AIError.GetLastErrorString());return null;
		}
		//if both buildable go out of while and see if we can build a nice switch
		if(Tile.IsBuildableAlmostFlatTile(left_switch) && Tile.IsBuildableAlmostFlatTile(right_switch)){break;}
		//if we can avoid a slope build without switch
		if(Tile.IsBuildableAlmostFlatTile(left_switch) && Tile.IsBuildableAvoidSlopeTile(right_switch,left_switch)) {
			if(!AITile.IsBuildable(left_exit)){Tile.DemolishTile(left_exit);}
			if(Tile.IsBuildableRailTileSlope(left_exit,stat_dir)) {
				return RailBuilder.BuildWithoutSwitch(left_switch,left_exit,right_switch,right_entry,stat_dir,null,null);
			}
		}
		if(Tile.IsBuildableAlmostFlatTile(right_switch) && Tile.IsBuildableAvoidSlopeTile(left_switch,right_switch)) {
			if(!AITile.IsBuildable(right_entry)){Tile.DemolishTile(right_entry);}
			if(Tile.IsBuildableRailTileSlope(right_entry,stat_dir)) {
				return RailBuilder.BuildWithoutSwitch(left_switch,left_exit,right_switch,right_entry,stat_dir,null,null);
			}
		}
		//try to make switch spot a bit better
		Tile.FixSlopeUpForSwitch(left_switch,stat_dir);Tile.FixSlopeUpForSwitch(right_switch,stat_dir);
		Tile.FixSlopeDownForSwitch(left_switch,stat_dir);Tile.FixSlopeDownForSwitch(right_switch,stat_dir);	
		Tile.FixSlopeSideWaysForSwitch(left_switch,stat_dir);Tile.FixSlopeSideWaysForSwitch(right_switch,stat_dir);
		if(Tile.IsBuildableAlmostFlatTile(left_switch) && Tile.IsBuildableAlmostFlatTile(right_switch)){break;}
		//see if we need to extend the switch location out a bit
		if((Tile.WillSlopeUp(left_switch,stat_dir) || Tile.WillSlopeDown(left_switch,stat_dir) || Tile.IsAlmostFlat(left_switch))
		&& (Tile.WillSlopeUp(left_switch,stat_dir) || Tile.WillSlopeDown(left_switch,stat_dir) || Tile.IsAlmostFlat(left_switch))) {
			if(!EventManager.BuildRailTrack(left_switch,Tile.GetTrackDir(stat_dir)) 
			|| !EventManager.BuildRailTrack(right_switch,Tile.GetTrackDir(stat_dir))) {
				MailAI.Info(0,"Couldn't extend switch out of station! "+AIError.GetLastErrorString());return null;
			}
			left_switch = left_switch + AIMap.GetTileIndex(x,y);
			right_switch = left_switch + AIMap.GetTileIndex(y,-x);
			left_exit = left_exit + AIMap.GetTileIndex(x,y);
			right_entry = left_exit + AIMap.GetTileIndex(y,-x);
		} else {
			//MailAI.BuildSign(left_switch,"ls");MailAI.BuildSign(right_switch,"rs");
			//MailAI.Info(0,"Couldn't slope switch tiles out of station!"+left_switch+right_switch);
			return null;
		}
	}
//both switch tiles are good, now let's check the tiles outward
	if(!AITile.IsBuildable(right_entry)){Tile.DemolishTile(right_entry);}
		if(!Tile.IsBuildableRailTileSlope(right_entry,stat_dir)) {
			//MailAI.Info(0,"right_entry not good, build without switch");
			return RailBuilder.BuildWithoutSwitch(left_switch,left_exit,right_switch,right_entry,stat_dir,null,null);
	}
	if(!AITile.IsBuildable(left_exit)){Tile.DemolishTile(left_exit);}
		if(!Tile.IsBuildableRailTileSlope(left_exit,stat_dir)) {
			//MailAI.Info(0,"left_exit not good, build without switch");
			return RailBuilder.BuildWithoutSwitch(left_switch,left_exit,right_switch,right_entry,stat_dir,null,null);
	}
	if(!AITile.IsBuildable(left_exit) && !AITile.IsBuildable(right_entry)) {
		MailAI.Info(0,"Both tiles after station switch not buildable!");return null;
	}
//both tiles out are buildable, try the switch routine		
	local l0 = RailBuilder.GetSwitchSquare(left_switch,stat_dir,side_dir);
	if(l0 == null) {
		//MailAI.Info(0,"couldn't find switch location , build without switch");	
		return RailBuilder.BuildWithoutSwitch(left_switch,left_exit,right_switch,right_entry,stat_dir,null,null);
	}
	local l9 = l0 - AIMap.GetTileIndex(x,y); 
	local l1 = l0+AIMap.GetTileIndex(x,y); local l2 = l1+AIMap.GetTileIndex(x,y);
	local r1 = l1+AIMap.GetTileIndex(y,-x);	local r2 = l2+AIMap.GetTileIndex(y,-x);	
	if(!Tile.IsBuildableSwitchTile(r1,l1)) {	
		MailAI.Info(0,"switch location was bad, build without switch");
		return RailBuilder.BuildWithoutSwitch(left_switch,left_exit,right_switch,right_entry,stat_dir,null,null);
	}
	local in_path, ex_path, extra_switch_side;
	//if(Tile.IsSideDirLeft(stat_dir,side_dir)) {
		{	local ISINepathbuildingTESTMODE = AITestMode();	
			ex_path = RailBuilder.BuildRailPath([[l9, l0]],[[left_exit, left_switch]]);	
		}
		if(ex_path==null || ex_path > AIMap.DistanceManhattan(l0,left_exit)+5) {
			//MailAI.Info(0,"no predicted path to switch, build without switch");
			return RailBuilder.BuildWithoutSwitch(left_switch,left_exit,right_switch,right_entry,stat_dir,null,null);
		}
		ex_path = RailBuilder.BuildRailPath([[l9, l0]],[[left_exit, left_switch]]);	
		if(ex_path==null) {
			//MailAI.Info(0,"no actual path to switch, build without switch");
			return RailBuilder.BuildWithoutSwitch(left_switch,left_exit,right_switch,right_entry,stat_dir,null,null);
		}
		//if(AIRail.IsRailTile(right_entry) && AICompany.IsMine(right_entry)) {
		//	EventManager.BuildRailTrack(right_entry,AIRail.GetRailTrack(left_exit));
		//	right_switch = right_entry;	
		//	right_entry = Tile.GetForwardTile(right_entry,Tile.GetDirectionFromTo(left_exit,right_entry));		
		//}	
		{	local ISINipathbuildingTESTMODE = AITestMode();
			in_path = RailBuilder.BuildRailPath([[right_entry,right_switch]],[[l0+AIMap.GetTileIndex(y,-x),r1]]);
		}
		if (in_path == null || in_path > ex_path + 2) {//would otherwise sometimes bridge the other track and turn back etc..
			//remember that exit_track has been build so we need to pass extra parameters to the return function
			//MailAI.Info(0,"only expath to switch, build without switch");
			return RailBuilder.BuildWithoutSwitch(left_switch,left_exit,right_switch,right_entry,stat_dir,l9,l0);
		}
		in_path = RailBuilder.BuildRailPath([[right_entry,right_switch]],[[l0+AIMap.GetTileIndex(y,-x),r1]]);
		if(in_path==null) {
			//MailAI.Info(0,"only expath to switch after actual trying, build without switch");
			return RailBuilder.BuildWithoutSwitch(left_switch,left_exit,right_switch,right_entry,stat_dir,l9,l0);
		}
	//either we've build two tracks or returned allready
	if(!AIRail.BuildSignal(right_entry,right_entry+AIMap.GetTileIndex(x,y),AIRail.SIGNALTYPE_PBS_ONEWAY)) {
		//track is bending outward instead of going straight
		AIRail.BuildSignal(right_entry,right_entry+AIMap.GetTileIndex(y,-x),AIRail.SIGNALTYPE_PBS_ONEWAY);
	}
	EventManager.BuildRail(l9, l0, l0+AIMap.GetTileIndex(x,y));//straight l0
	EventManager.BuildSignal(l0,l9,AIRail.SIGNALTYPE_PBS_ONEWAY);
	if(Tile.IsSideDirLeft(stat_dir,side_dir)) {
		EventManager.BuildRail(r1-AIMap.GetTileIndex(x,y), r1, l1);//cross r1
		EventManager.BuildRail(r1, l1, l2);//cross l1
		EventManager.BuildRail(l0, l1, l2);//straigth l1
		EventManager.BuildRail(l1, l2, l2+AIMap.GetTileIndex(x,y));//straight l2
		RailBuilder.BuildStationSwitchTrack(left_switch,right_switch,stat_dir,true,true);
		return [left_switch,left_exit,null,l2,l2+AIMap.GetTileIndex(x,y)];
	} else {
		EventManager.BuildRail(l0, l1, r1);//cross l1
		EventManager.BuildRail(l1, r1, r2);//cross r1
		EventManager.BuildRail(r1-AIMap.GetTileIndex(x,y), r1, r2);//straigth r1
		EventManager.BuildRail(r1, r2, r2+AIMap.GetTileIndex(x,y));//straight r2
		RailBuilder.BuildStationSwitchTrack(left_switch,right_switch,stat_dir,true,true);
		return [left_switch,left_exit,null,r2,r2+AIMap.GetTileIndex(x,y)];
	}
}

/**
*	Looks for the best side to build the only entry/exit track and builds both switch squares accordingly.
*	@param left_switch
*	@param left_exit
*	@param right_switch
*	@param right_entry
*	@param stat_dir
*	@param extra1
*	@param extra2
*	@return An array with switch square of chosen side, exit/entry tile, other switch, extra1 and extra2
*/
function RailBuilder::BuildWithoutSwitch(left_switch,left_exit,right_switch,right_entry,stat_dir,extra1,extra2) {
	if(left_switch == null || left_exit == null || right_switch == null || right_entry == null) {
			throw("null tiles provided to railbuilder.buildwithoutswitch");}
	//first if one path has allready been build
	if(extra1 != null) {
		//the path out crosses in front of the switch
		if(Tile.IsRailMine(left_exit) && Tile.IsRailMine(right_entry)) {
			if(Tile.IsRailMine(Tile.GetForwardTile(right_entry,stat_dir))) {//right is good side
				//MailAI.BuildSign(left_exit,"clear1");
				Tile.DemolishTile(left_exit); Tile.DemolishTile(right_entry);
				if(EventManager.BuildRailTrack(right_entry,stat_dir)
				&& RailBuilder.BuildStationSwitchTrack(left_switch,right_switch,stat_dir,false,true)) {
					return [right_switch,right_entry,left_switch,extra1,extra2];
				} else { MailAI.Info(0,"Failed stationswitch recover cross right"); return null;}
			}
			local side = Tile.GetRightTile(right_entry,stat_dir);
			if(Tile.IsRailMine(side)) {
				//MailAI.BuildSign(left_exit,"clear2");
				Tile.DemolishTile(left_exit); Tile.DemolishTile(right_entry);
				if(EventManager.BuildRail(right_switch,right_entry,side)
				&& RailBuilder.BuildStationSwitchTrack(left_switch,right_switch,stat_dir,false,true)) {
					return [right_switch,right_entry,left_switch,extra1,extra2];
				} else { MailAI.Info(0,"Failed stationswitch recover cross right to side"); return null;}
			}  
			side = Tile.GetLeftTile(left_exit,stat_dir);
			if(Tile.IsRailMine(Tile.GetForwardTile(right_entry,stat_dir))) {//left is good side
				//MailAI.BuildSign(right_entry,"clear");
				Tile.DemolishTile(left_exit); Tile.DemolishTile(right_entry);
				if(EventManager.BuildRailTrack(left_exit,stat_dir)
				&& RailBuilder.BuildStationSwitchTrack(left_switch,right_switch,stat_dir,true,false)) {
					return [left_switch,left_exit,right_switch,extra1,extra2];
				} else { MailAI.Info(0,"Failed stationswitch recover cross left"); return null;}
			} 
			if(Tile.IsRailMine(side)) {
				//MailAI.BuildSign(right_entry,"clear2");
				Tile.DemolishTile(left_exit); Tile.DemolishTile(right_entry);
				if(EventManager.BuildRail(left_switch,left_exit,side)
				&& RailBuilder.BuildStationSwitchTrack(left_switch,right_switch,stat_dir,true,false)) {
					return [left_switch,left_exit,right_switch,extra1,extra2];
				} else { MailAI.Info(0,"Failed stationswitch recover cross left"); return null;}
			}		
		}
		//path doesn't cross
		if(Tile.IsRailMine(right_entry)) {
			if(RailBuilder.BuildStationSwitchTrack(left_switch,right_switch,stat_dir,false,true)) {
				return [right_switch,right_entry,left_switch,extra1,extra2];
			} else { MailAI.Info(0,"Failed BSST right"); return null;}
		}
		if(Tile.IsRailMine(left_exit)) {
			if(RailBuilder.BuildStationSwitchTrack(left_switch,right_switch,stat_dir,true,false)) {
				return [left_switch,left_exit,right_switch,extra1,extra2];
			} else { MailAI.Info(0,"Failed BSST left"); return null;}
		}
		return null;
	}
	//then if no ipath or epath has been build
	if(Tile.IsBuildableAvoidSlopeTile(right_switch,left_switch)
	|| !Tile.IsBuildableRailTileSlope(right_entry,stat_dir)) {
		//MailAI.Info(0,"Right entry not buildable, trying left exit.");
		if(RailBuilder.BuildStationSwitchTrack(left_switch,right_switch,stat_dir,true,false)) {
			return [left_switch,left_exit,right_switch,left_switch,left_exit];
		} else { MailAI.Info(0,"Left also not buildable! "+AIError.GetLastErrorString()); return null;}
	}
	if(Tile.IsBuildableAvoidSlopeTile(left_switch,right_switch)
	|| !Tile.IsBuildableRailTileSlope(left_exit,stat_dir)) {
		//MailAI.Info(0,"Left exit not buildable, trying right entry.");
		if(RailBuilder.BuildStationSwitchTrack(left_switch,right_switch,stat_dir,false,true)) {
			return [right_switch,right_entry,left_switch,right_switch,right_entry];}
		else { MailAI.Info(0,"Right also not buildable! "+AIError.GetLastErrorString()); return null;}
	}
	//both sides are buildable
	//MailAI.Info(0,"both exits buildable, taking left one");
	if(RailBuilder.BuildStationSwitchTrack(left_switch,right_switch,stat_dir,true,false)) {
		return [left_switch,left_exit,right_switch,left_switch,left_exit];
	}
	MailAI.Info(0,"BuildWithoutSwitch failed! "+AIError.GetLastErrorString()); return null;
}

/**
*	Builds the tracks right in front of the station.
*	@param left_switch The left switch tile in front of the station.
*	@param right_switch The right switch tile in front of the station.
*	@param direction The exit direction of the station.
*	@param full_left Whether to connect to left_exit tile.
*	@param full_right Whether to connect to right_entry tile.
*	@return True if all requested tracks have been build.
*/
function RailBuilder::BuildStationSwitchTrack(left_switch,right_switch,direction,full_left,full_right) {
	local stat_dir, sw1, sw2, sw3, sw4;	
	switch (direction) {
		case "SW":stat_dir=AIRail.RAILTRACK_NE_SW;
		sw1=AIRail.RAILTRACK_NW_NE;sw2=AIRail.RAILTRACK_NW_SW;
		sw3=AIRail.RAILTRACK_SW_SE;sw4=AIRail.RAILTRACK_NE_SE;break;
		case "NE":stat_dir=AIRail.RAILTRACK_NE_SW;
		sw1=AIRail.RAILTRACK_SW_SE;sw2=AIRail.RAILTRACK_NE_SE;
		sw3=AIRail.RAILTRACK_NW_NE;sw4=AIRail.RAILTRACK_NW_SW;break;
		case "SE":stat_dir=AIRail.RAILTRACK_NW_SE;
		sw1=AIRail.RAILTRACK_NW_SW;sw2=AIRail.RAILTRACK_SW_SE;
		sw3=AIRail.RAILTRACK_NE_SE;sw4=AIRail.RAILTRACK_NW_NE;break;
		case "NW":stat_dir=AIRail.RAILTRACK_NW_SE;
		sw1=AIRail.RAILTRACK_NE_SE;sw2=AIRail.RAILTRACK_NW_NE;
		sw3=AIRail.RAILTRACK_NW_SW;sw4=AIRail.RAILTRACK_SW_SE;break;
		default: throw("Wrong direction in BuildStationSwitchTrack: "+direction);
	}	
	if(full_left) {//right platform to left exit
		if(!EventManager.BuildRailTrack(left_switch,stat_dir)){MailAI.Info(0,"RBT1"+AIError.GetLastErrorString());return false;}		
		if(!EventManager.BuildRailTrack(left_switch,sw2)){MailAI.Info(0,"RBT2"+AIError.GetLastErrorString());return false;}
		if(!EventManager.BuildRailTrack(right_switch,sw4)){MailAI.Info(0,"RBT3"+AIError.GetLastErrorString());return false;}
	}
	if(full_right) {//right_entry to left platform
		if(!EventManager.BuildRailTrack(right_switch,stat_dir)){MailAI.Info(0,"RBT4"+AIError.GetLastErrorString());return false;}		
		if(!EventManager.BuildRailTrack(left_switch,sw1)){MailAI.Info(0,"RBT5"+AIError.GetLastErrorString());return false;}
		if(!EventManager.BuildRailTrack(right_switch,sw3)){MailAI.Info(0,"RBT6"+AIError.GetLastErrorString());return false;}	
	}	
	return true;
}

/**
*	lot of duplicate code from buildstationswitch!
*	check if improvements should also go there
*/
function RailBuilder::BuildJoiningSwitch(arr) {
	if(arr==null){throw("should I throw a null error buildjoiningswitch");}
	local left_switch = arr[2], stat_dir = arr[5], side_dir = arr[6];
	if(left_switch == null || stat_dir == null || side_dir == null){return null;}//don't throw error?
	MailAI.Info(0,"BuildJoiningSwitch");
	local track_dir = Tile.GetTrackDir(stat_dir);
	local x = Tile.SideToX(stat_dir,1), y = Tile.SideToY(stat_dir,1);
	local left_exit = left_switch + AIMap.GetTileIndex(x,y);
	local right_switch = left_switch + AIMap.GetTileIndex(y,-x);
	local right_entry = left_exit + AIMap.GetTileIndex(y,-x);		
	local l0 = RailBuilder.GetSwitchSquare(left_switch,stat_dir,side_dir);
	if(l0 == null || !Tile.IsBuildableRailTileSlope(right_entry,stat_dir) || !Tile.IsBuildableRailTileSlope(left_exit,stat_dir)) {	
		MailAI.Info(0,"no good joining switch square available");
		return [left_switch,left_exit,null,left_switch,left_exit];
	}
	local l9 = l0-AIMap.GetTileIndex(x,y);
	local l1 = l0+AIMap.GetTileIndex(x,y); local l2 = l1+AIMap.GetTileIndex(x,y);
	local r1 = l1+AIMap.GetTileIndex(y,-x); local r2 = l2+AIMap.GetTileIndex(y,-x);
	if(!Tile.IsBuildableSwitchTile(r1,l1)) {	
		MailAI.Info(0,"Do we need to check this? BuildJoiningSwitch... +- line 748");
		RailBuilder.BuildHalfJoiningSwitch(arr,true,false);
		return [left_switch,left_exit,null,left_switch,left_exit];
	}
	local in_path, ex_path, extra_switch_side;
	{	local ISINepathbuildingTESTMODE = AITestMode();	
		ex_path = RailBuilder.BuildRailPath([[l9, l0]],[[left_exit, left_switch]]);	
	}
	if(ex_path==null || ex_path > AIMap.DistanceManhattan(l0,left_exit)+5) {
		RailBuilder.BuildHalfJoiningSwitch(arr,true,false);
		return [left_switch,left_exit,null,left_switch,left_exit];
	}
	ex_path = RailBuilder.BuildRailPath([[l9, l0]],[[left_exit, left_switch]]);	
	if(ex_path==null) {
		RailBuilder.BuildHalfJoiningSwitch(arr,true,false);
		return [left_switch,left_exit,null,left_switch,left_exit];
	}	
	/*if(AIRail.IsRailTile(right_entry) && AICompany.IsMine(right_entry)) {	
		throw("this never reaches...");	
		MailAI.BuildSign(right_entry,"re");
		EventManager.BuildRailTrack(right_entry,AIRail.GetRailTrack(left_exit));
		right_switch = right_entry;	
		right_entry = Tile.GetForwardTile(right_entry,Tile.GetDirectionFromTo(left_exit,right_entry));		
		MailAI.BuildSign(right_entry,"re2");
	}*/	
	{	local ISINipathbuildingTESTMODE = AITestMode();
		in_path = RailBuilder.BuildRailPath([[right_entry,right_switch]],[[l0+AIMap.GetTileIndex(y,-x),r1]]);
	}
	if (in_path == null || in_path > ex_path + 2) {
		RailBuilder.BuildHalfJoiningSwitch(arr,true,false);
		return [left_switch,left_exit,null,l9,l0];
	}
	in_path = RailBuilder.BuildRailPath([[right_entry,right_switch]],[[l0+AIMap.GetTileIndex(y,-x),r1]]);
	if(in_path==null) {
		RailBuilder.BuildHalfJoiningSwitch(arr,true,false);
		return [left_switch,left_exit,null,l9,l0];
	}
	if(!AIRail.BuildSignal(right_entry,right_entry+AIMap.GetTileIndex(x,y),AIRail.SIGNALTYPE_PBS_ONEWAY)) {
		AIRail.BuildSignal(right_entry,right_entry+AIMap.GetTileIndex(y,-x),AIRail.SIGNALTYPE_PBS_ONEWAY);
	}
	EventManager.BuildRail(l9, l0, l0+AIMap.GetTileIndex(x,y));//straight l0
	EventManager.BuildSignal(l0,l9,AIRail.SIGNALTYPE_PBS_ONEWAY);
	if(Tile.IsSideDirLeft(stat_dir,side_dir)) {
		EventManager.BuildRail(r1-AIMap.GetTileIndex(x,y), r1, l1);//cross r1
		EventManager.BuildRail(r1, l1, l2);//cross l1
		EventManager.BuildRail(l0, l1, l2);//straigth l1
		EventManager.BuildRail(l1, l2, l2+AIMap.GetTileIndex(x,y));//straight l2
		RailBuilder.BuildHalfJoiningSwitch(arr,true,true);
		return [left_switch,left_exit,null,l2,l2+AIMap.GetTileIndex(x,y)];
	} else {
		EventManager.BuildRail(l0, l1, r1);//cross l1
		EventManager.BuildRail(l1, r1, r2);//cross r1
		EventManager.BuildRail(r1-AIMap.GetTileIndex(x,y), r1, r2);//straigth r1
		EventManager.BuildRail(r1, r2, r2+AIMap.GetTileIndex(x,y));//straight r2
		RailBuilder.BuildHalfJoiningSwitch(arr,true,true);
		return [left_switch,left_exit,null,r2,r2+AIMap.GetTileIndex(x,y)];
	}
}

function RailBuilder::BuildHalfJoiningSwitch(arr,left,right) {
	local stat_dir = arr[5], side_dir = Tile.GetDirectionFromTo(arr[0],arr[3]), temp;
	if(!Tile.IsSideDirLeft(stat_dir,side_dir)){
		temp = left; left = right; right = temp;
	}
	if(Tile.GetForwardLeftTile(arr[0],stat_dir)==arr[1] || Tile.GetForwardRightTile(arr[0],stat_dir)==arr[1]) {
		//diagonal track
		if(left) {EventManager.BuildRailTrack(arr[0],Tile.GetTrackDir(stat_dir));}
		if(right) {EventManager.BuildRailTrack(arr[1],Tile.GetTrackDir(stat_dir));}
		if(left) {EventManager.BuildRailTrack(arr[3],Tile.GetTrackDir(stat_dir));}		
	} else {//3 is firstleft 4 is leftleft
		local tr_dir = Tile.GetDirectionFromTo(arr[0],arr[1]);
		local sidefwd = Tile.GetForwardTile(arr[3],tr_dir);
		if(left) {EventManager.BuildRail(Tile.GetBackwardTile(arr[0],tr_dir),arr[0],arr[3]);}
		if(right) {EventManager.BuildRail(arr[0],arr[1],sidefwd);}
		if(left) {EventManager.BuildRail(arr[0],arr[3],sidefwd);}
		if(right) {EventManager.BuildRail(arr[1],sidefwd,Tile.GetForwardTile(sidefwd,tr_dir));}
		if(left) {EventManager.BuildRail(arr[3],sidefwd,arr[4]);}
		if(tr_dir==stat_dir){
			if(left) {EventManager.BuildRail(sidefwd,arr[4],Tile.GetForwardTile(arr[4],tr_dir));}
		} else {
			if(right) {EventManager.BuildRail(sidefwd,Tile.GetForwardTile(sidefwd,tr_dir),Tile.GetForwardTile(arr[4],tr_dir));}
		}
	}	
}

function RailBuilder::BuildCrossing(left_tile,right_tile,dir) {
	local x_start = 0, x_end = 0, y_start = 0, y_end = 0;
	switch (dir) {
	    case "NE": {x_start = -4; y_end = 1; break;}
	    case "SW": {x_end = 4; y_start = -1; break;}
	    case "SE": {x_end = 1; y_end = 4; break;}
	    case "NW": {x_start = -1; y_start = -4; break;}
	    default: throw("Wrong direction in BuildCrossing!"+dir);
	} 	
	local tl = AITileList();					
	tl.AddRectangle(left_tile+AIMap.GetTileIndex(x_start,y_start),left_tile+AIMap.GetTileIndex(x_end,y_end));	
	tl.RemoveTile(left_tile);tl.RemoveTile(right_tile);
	tl.Valuate(AIRail.GetRailTracks);
	tl.KeepBetweenValue(0,3);
	if(tl.Count()==10){
		tl.Valuate(Valuator.HasSignal);
		tl.KeepValue(0);
		if(tl.Count()!=10){return null;}//don't mess with allready built passings
		local end_cross_left = left_tile + AIMap.GetTileIndex(Tile.SideToX(dir,5),Tile.SideToY(dir,5));
		local end_signal = Tile.GetBackwardTile(end_cross_left,dir);
		local start_signal = Tile.GetForwardTile(right_tile,dir);
		local bool = RailBuilder.BuildStationSwitchTrack(left_tile,right_tile,dir,true,true);
		if(bool){bool = RailBuilder.BuildStationSwitchTrack(end_cross_left,Tile.GetRightTile(end_cross_left,dir),dir,true,true);}
		if(bool){bool = EventManager.BuildSignal(end_signal,Tile.GetBackwardTile(end_signal,dir),AIRail.SIGNALTYPE_PBS_ONEWAY);}
		if(bool){bool = EventManager.BuildSignal(start_signal,Tile.GetForwardTile(start_signal,dir),AIRail.SIGNALTYPE_PBS_ONEWAY);}
		local next_one = RailBuilder.BuildCrossing(end_cross_left,Tile.GetRightTile(end_cross_left,dir),dir);
		if(next_one != null) {return next_one;}
		return [end_signal,end_cross_left,Tile.GetForwardTile(end_cross_left,dir),6];
	}
	return null;
}

//now only uses pre,start and length 
function RailBuilder::BuildPassings(sw1,sw2,pre,start,length) {
	local pass_length_min = 5, pass_length_max = 9, tiles_between_passings = 6;
	local total_track_length = 0, tiles_after_last = 0, tiles_before_end = length;
	local left_straight = 0, left_NE = 0, left_SW = 0;
	local pre_tile, work_tile = pre, next_tile = start, next_tiles, return_list = AITileList();
	local left_start, left_saved, left_list = AITileList();
	while(tiles_before_end > 0) {
		pre_tile = work_tile; work_tile = next_tile;
		next_tiles = Tile.GetNextRailTile(pre_tile, work_tile);//[pre,work,next,bridge/tunnel_length]
		if(next_tiles == null){return [return_list,total_track_length];}
		if(next_tiles[0] != null) {
			pre_tile = next_tiles[0]; work_tile = next_tiles[1];
		} 
		next_tile = next_tiles[2];
		total_track_length += next_tiles[3];
		tiles_after_last += next_tiles[3];
		tiles_before_end = tiles_before_end - next_tiles[3];
		if(next_tiles[0] != null) {//MailAI.Info(0,"Avoiding bridge/tunnel");
			left_list.Clear(); left_saved = null; left_start = null;
			left_straight = 0; left_NE = 0; left_SW = 0; tiles_after_last = 0;
			continue;
		} 
		local exit_dir = Tile.GetDirectionFromTo(work_tile, next_tile);
		local left_tile = Tile.GetLeftTile(work_tile, exit_dir);
		local right_tile = Tile.GetRightTile(work_tile, exit_dir);	
		if(Tile.GetNeighbourRailCount(work_tile) > 2) {	
			//avoid building passings when there's other passings or joinings
			local first_track = AIRail.GetRailTracks(work_tile);
			if(first_track != 1 && first_track != 2){//a passing in the other direction
				if(left_saved != null && left_saved != work_tile) {
					if(left_straight > pass_length_min + 1) {
						//use previous dir!
						return_list.AddTile(RailBuilder.BuildStraightPassing(left_start,left_saved,left_list,
							Tile.GetDirectionFromTo(pre_tile,work_tile)));
					}
					if(left_NE > pass_length_min + 1 || left_SW > pass_length_min + 1) {
						return_list.AddTile(RailBuilder.BuildDiagPassing(left_start,left_saved,left_list,exit_dir));
					}
				}
				left_list.Clear(); left_saved = null; left_start = null;
				left_straight = 0; left_NE = 0; left_SW = 0;
				tiles_after_last=0; continue;
			}
			if(first_track == AIRail.GetRailTracks(left_tile)) {
				if(Valuator.HasSignal(work_tile) || Valuator.HasSignal(left_tile)){
					left_list.Clear(); left_saved = null; left_start = null;
					left_straight = 0; left_NE = 0; left_SW = 0;
					tiles_after_last=0;
					continue;
				}
				next_tiles = RailBuilder.BuildCrossing(left_tile,work_tile,exit_dir);
				if(next_tiles != null) {//other track is on the left so shift the returned tiles
					pre_tile = Tile.GetRightTile(next_tiles[0],exit_dir);
					work_tile = Tile.GetRightTile(next_tiles[1],exit_dir);
					next_tile = Tile.GetRightTile(next_tiles[2],exit_dir);
					total_track_length += next_tiles[3];
					tiles_before_end = tiles_before_end - next_tiles[3];
				}
				left_list.Clear(); left_saved = null; left_start = null;
				left_straight = 0; left_NE = 0; left_SW = 0;
				tiles_after_last=0;
				continue;
			}
			if(first_track == AIRail.GetRailTracks(right_tile)) {
				if(Valuator.HasSignal(work_tile) || Valuator.HasSignal(right_tile)){
					left_list.Clear(); left_saved = null; left_start = null;
					left_straight = 0; left_NE = 0; left_SW = 0;
					tiles_after_last=0;
					continue;
				}
				next_tiles = RailBuilder.BuildCrossing(work_tile,right_tile,exit_dir);	
				if(next_tiles != null) {
					pre_tile = next_tiles[0]; work_tile = next_tiles[1];next_tile = next_tiles[2];
					total_track_length += next_tiles[3];
					tiles_before_end = tiles_before_end - next_tiles[3];
					left_list.Clear(); left_saved = null; left_start = null;
					left_straight = 0; left_NE = 0; left_SW = 0;
					tiles_after_last=0;
					continue;
				}			
			}
		}
//pass_length_max exceeded (using saved but work is pass_length_max-pass_length_min tiles further)
		if(left_straight > pass_length_max && left_saved != null) {
			return_list.AddTile(RailBuilder.BuildStraightPassing(left_start,left_saved,left_list,exit_dir));
			left_list.Clear(); left_saved = null; left_start = null;
			left_straight = 0; left_NE = 0; left_SW = 0;
			tiles_after_last = pass_length_max-pass_length_min; 	
			continue;
		}
		if(left_NE > pass_length_max &&  left_saved != null) {//use exit_dir of previous tile!
			exit_dir = Tile.GetDirectionFromTo(Tile.GetLeftTile(work_tile,exit_dir),work_tile);
			return_list.AddTile(RailBuilder.BuildDiagPassing(left_start,left_saved,left_list,exit_dir));
			left_list.Clear(); left_saved = null; left_start = null;
			left_straight = 0; left_NE = 0; left_SW = 0;
			tiles_after_last = pass_length_max-pass_length_min; 
			continue;
		}
		if(left_SW > pass_length_max &&  left_saved != null) {//use exit_dir of previous tile!
			exit_dir = Tile.GetDirectionFromTo(Tile.GetLeftTile(work_tile,exit_dir),work_tile);
			return_list.AddTile(RailBuilder.BuildDiagPassing(left_start,left_saved,left_list,exit_dir));
			left_list.Clear(); left_saved = null; left_start = null;
			left_straight = 0; left_NE = 0; left_SW = 0;
			tiles_after_last = pass_length_max-pass_length_min;  
			continue;		
		}		
//if straight track		
		if(AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NW_SE
		|| AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NE_SW) {
			//if not buildable see if we have a saved one to build
			if(!AITile.IsBuildable(left_tile)) {
				if(left_saved != null) {
					return_list.AddTile(RailBuilder.BuildStraightPassing(left_start,left_saved,left_list,exit_dir));
					tiles_after_last = 0;
				}
				left_list.Clear(); left_saved = null; left_start = null;
				left_straight = 0; left_NE = 0; left_SW = 0; 
				continue;
			}
//track keeps going straight			
			if(AIRail.GetRailTracks(next_tile) == AIRail.GetRailTracks(work_tile)) {
				if(tiles_after_last > tiles_between_passings) {
					if(left_start != null) {left_straight++;}
					if(Tile.IsBuildableStartSwitchTile(left_tile,work_tile)) {
						if(left_straight == 0 && left_start == null) {
							left_start = work_tile; left_list.Clear();
							left_straight = 1;
						}
					}
					if(Tile.IsBuildableEndSwitchTile(left_tile,work_tile)) {
						if(left_straight > pass_length_min && left_saved == null) {	
							left_saved = work_tile;
						}
					}
					if(left_straight > 1) {left_list.AddTile(left_tile);}
				}
				continue;
			}
		}
//straight track bends away for left, build if saved	
		if((AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NW_SE
		&& (AIRail.GetRailTracks(next_tile) == AIRail.RAILTRACK_NW_SW || AIRail.GetRailTracks(next_tile) == AIRail.RAILTRACK_NE_SE))
		|| (AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NE_SW
		&& (AIRail.GetRailTracks(next_tile) == AIRail.RAILTRACK_NW_NE || AIRail.GetRailTracks(next_tile) == AIRail.RAILTRACK_SW_SE))) {
			if(left_straight > pass_length_min && Tile.IsBuildableEndSwitchTile(left_tile,work_tile)) {
				return_list.AddTile(RailBuilder.BuildStraightPassing(left_start,work_tile,left_list,exit_dir));
			} else if(left_saved != null) {
				return_list.AddTile(RailBuilder.BuildStraightPassing(left_start,left_saved,left_list,exit_dir));
			}
			left_list.Clear(); left_saved = null; left_start = null;
			left_straight = 0; left_NE = 0; left_SW = 0;
			continue;
		}	
//straight incoming bend for left, build if saved (connect to bend!)
		if((AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NE_SW
		&& (AIRail.GetRailTracks(next_tile) == AIRail.RAILTRACK_NE_SE || AIRail.GetRailTracks(next_tile) == AIRail.RAILTRACK_NW_SW))
		|| (AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NW_SE
		&& (AIRail.GetRailTracks(next_tile) == AIRail.RAILTRACK_NW_NE || AIRail.GetRailTracks(next_tile) == AIRail.RAILTRACK_SW_SE))) {		
			local left_end = Tile.GetLeftTile(next_tile,exit_dir);
			if(left_straight > pass_length_min && Tile.IsAlmostFlat(left_end)) {
				left_list.AddTile(left_tile);
				left_list.AddTile(left_end);
				return_list.AddTile(RailBuilder.BuildStraightPassing(left_start,left_end,left_list,exit_dir)); 
			} else if(left_saved != null) {
				return_list.AddTile(RailBuilder.BuildStraightPassing(left_start,left_saved,left_list,exit_dir)); 
			}
			left_list.Clear(); left_saved = null; left_start = null;
			left_straight = 0; left_NE = 0; left_SW = 0;
			continue;
		}				
//diag turning right, build if saved
		if(((AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NW_NE || AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_SW_SE)
		&& AIRail.GetRailTracks(next_tile) == AIRail.RAILTRACK_NW_SE)
		||((AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NW_SW  || AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NE_SE)
		&& AIRail.GetRailTracks(next_tile) == AIRail.RAILTRACK_NE_SW)) {		
			if(left_NE >= pass_length_min || left_SW >= pass_length_min) {
				return_list.AddTile(RailBuilder.BuildDiagPassing(left_start,work_tile,left_list,exit_dir));
			} else if(left_saved != null) {
				return_list.AddTile(RailBuilder.BuildDiagPassing(left_start,left_saved,left_list,exit_dir));
			}
			left_list.Clear(); left_saved = null; left_start = null;
			left_straight = 0; left_NE = 0; left_SW = 0;
			continue;
		}		
//diag turning left, build if saved
		if(((AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NW_SW || AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NE_SE)
		&& AIRail.GetRailTracks(next_tile) == AIRail.RAILTRACK_NW_SE)
		||((AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NW_NE || AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_SW_SE)
		&& AIRail.GetRailTracks(next_tile) == AIRail.RAILTRACK_NE_SW)) {		
			if((left_NE >= pass_length_min || left_SW >= pass_length_min) 
			&& Tile.IsAlmostFlat(next_tile)) {
				left_list.AddTile(left_tile); left_list.AddTile(next_tile); 
				exit_dir = Tile.GetDirectionFromTo(left_tile,work_tile);//use exit_dir of previous tile!
				return_list.AddTile(RailBuilder.BuildDiagPassing(left_start,next_tile,left_list,exit_dir)); 		 			
				tiles_after_last = 0;
			} else if(left_saved != null) {	
				exit_dir = Tile.GetDirectionFromTo(left_tile,work_tile);//use exit_dir of previous tile!
				return_list.AddTile(RailBuilder.BuildDiagPassing(left_start,left_saved,left_list,exit_dir)); 	
				tiles_after_last = 3;//guesstimate 
			}
			left_list.Clear(); left_saved = null; left_start = null;
			left_straight = 0; left_NE = 0; left_SW = 0;
			continue;	
		}	
//North going left or East going up		
		if(AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NW_NE && exit_dir == "NW" 
		|| AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NE_SE && exit_dir == "NE") {
			if(!Tile.FixBuildableDiagTile(left_tile,exit_dir)) {
				left_list.Clear(); left_saved = null; left_start = null;
				left_straight = 0; left_NE = 0; left_SW = 0;
				continue;
			}	
			if(tiles_after_last > tiles_between_passings) {
				left_NE++;		 
				if(left_NE == 1 || left_start == null) {
					left_start = work_tile; left_list.Clear();
					left_NE = 1; //MailAI.BuildSign(work_tile,"s");
				}
				if(left_NE >= pass_length_min && left_saved == null) {			
					left_saved = work_tile; //MailAI.BuildSign(work_tile,"e");
				}
				if(left_NE > 1) {left_list.AddTile(work_tile);}
			}
			continue;
		}
//South Right or West Down, not bending
		if(AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_SW_SE && exit_dir == "SE" 
		|| AIRail.GetRailTracks(work_tile) == AIRail.RAILTRACK_NW_SW && exit_dir == "SW") {
			if(!Tile.FixBuildableDiagTile(left_tile,exit_dir)) {
				left_list.Clear(); left_saved = null; left_start = null;
				left_straight = 0; left_NE = 0; left_SW = 0;
				continue;
			}
			if(tiles_after_last > tiles_between_passings) {
				left_SW++;
				if(left_SW == 1 || left_start == null) {
					left_start = work_tile; left_list.Clear();
					left_SW = 1; //MailAI.BuildSign(work_tile,"s");
				}
				if(left_SW >= pass_length_min && left_saved == null) {
					left_saved = work_tile;	//MailAI.BuildSign(work_tile,"e");		 
				}
				if(left_SW > 1) {left_list.AddTile(work_tile);}
			}
			continue;						
		}	
	}//end of while(tiles_before_end > 0)
	return [return_list,total_track_length];	
}

/**
*	Builds the actual passing, this one for diagonal passing lanes.
*/
function RailBuilder::BuildDiagPassing(start_pass,end_pass,pass_list,exit_dir) {
	if(start_pass == null){throw("start_pass is null in RailBuilder::BuildDiagPassing");}
	local sp_track,spl_track,epl_track,ep_track;
	if(exit_dir == "NE"){//East track going up
		pass_list.Valuate(AIMap.GetTileX); pass_list.Sort(AIList.SORT_BY_VALUE,false);
		sp_track=AIRail.RAILTRACK_NW_SE;spl_track=AIRail.RAILTRACK_NE_SE;
		epl_track=AIRail.RAILTRACK_NW_SW;ep_track=AIRail.RAILTRACK_NE_SW;
	} else if(exit_dir == "NW"){//North track going left
		pass_list.Valuate(AIMap.GetTileY); pass_list.Sort(AIList.SORT_BY_VALUE,false);
		sp_track=AIRail.RAILTRACK_NE_SW;spl_track=AIRail.RAILTRACK_NW_NE;
		epl_track=AIRail.RAILTRACK_SW_SE;ep_track=AIRail.RAILTRACK_NW_SE;
	} else if(exit_dir == "SE"){//South track going right
		pass_list.Valuate(AIMap.GetTileY); pass_list.Sort(AIList.SORT_BY_VALUE,true);
		sp_track=AIRail.RAILTRACK_NE_SW;spl_track=AIRail.RAILTRACK_SW_SE;
		epl_track=AIRail.RAILTRACK_NW_NE;ep_track=AIRail.RAILTRACK_NW_SE;
	} else if(exit_dir == "SW"){//West track going down
		pass_list.Valuate(AIMap.GetTileX); pass_list.Sort(AIList.SORT_BY_VALUE,true);
		sp_track=AIRail.RAILTRACK_NW_SE;spl_track=AIRail.RAILTRACK_NW_SW;
		epl_track=AIRail.RAILTRACK_NE_SE;ep_track=AIRail.RAILTRACK_NE_SW;
	} else {throw("invalid direction in RailBuilder::BuildDiagPassing");}							
	local end_pass_left = Tile.GetLeftTile(end_pass,exit_dir);
	while(true) {
		local bool = false, sav = null, signal = null;
		local back = Tile.GetBackwardTile(start_pass,exit_dir);
		local quick_left = Tile.GetRightTile(back,exit_dir);
		if(Tile.IsRailMine(quick_left) && Tile.FixBuildableDiagTile(back,exit_dir)) {
			if(Tile.IsRailMine(Tile.GetBackwardTile(quick_left,exit_dir))) {
				bool = EventManager.BuildRailTrack(quick_left,epl_track);
			} else {//track to the right of quick_left
				bool = EventManager.BuildRailTrack(quick_left,sp_track);
			}
			if(bool){ bool = EventManager.BuildRailTrack(back,spl_track);}
			if(bool){ bool = EventManager.BuildRailTrack(start_pass,epl_track);
			signal = Tile.GetRightTile(start_pass,exit_dir);}
		}
		if(!bool) {
			signal = Tile.GetForwardTile(start_pass,exit_dir);
			if(!EventManager.BuildRailTrack(start_pass,sp_track)) {break;}
		}
		if(!EventManager.BuildRailTrack(Tile.GetLeftTile(start_pass,exit_dir),spl_track)) {break;}
		bool = true;
		foreach(t, _ in pass_list) {//build two tracks per pass
			if(bool && t != end_pass){bool = EventManager.BuildRailTrack(t,epl_track);}	
			if(bool && Tile.GetLeftTile(t,exit_dir) != end_pass_left) {
				bool = EventManager.BuildRailTrack(Tile.GetLeftTile(t,exit_dir),spl_track);
				sav = Tile.GetLeftTile(t,exit_dir);
			}
			if(t == end_pass){break;}	
		}
		if(!bool) {break;}
		if(AIRail.GetRailTracks(end_pass) == sp_track) {//joining bend
			bool = EventManager.BuildRailTrack(end_pass,epl_track);//last joining track
		} else {
			bool = EventManager.BuildRailTrack(end_pass,ep_track);//normal last track	
		}
		if(!bool){//bend has not build so build normal last track on previous tile		
			bool = EventManager.BuildRailTrack(Tile.GetRightTile(sav,exit_dir),ep_track);
			sav = Tile.GetRightTile(Tile.GetBackwardTile(sav,exit_dir),exit_dir);
			MailAI.BuildSign(sav,"failsafe");
		}
		if(!bool){break;}
		AIRail.BuildSignal(signal,Tile.GetLeftTile(signal,exit_dir),AIRail.SIGNALTYPE_PBS_ONEWAY);
		AIRail.BuildSignal(sav,Tile.GetRightTile(sav,exit_dir),AIRail.SIGNALTYPE_PBS_ONEWAY);
		return end_pass;	
	}
	MailAI.Info(0,"didnt build pass : "+AIError.GetLastErrorString());
	MailAI.BuildSign(start_pass,"!");
	return start_pass;
}

function RailBuilder::BuildStraightPassing(start_pass,end_pass,pass_list,exit_dir) {
	if(start_pass == null || end_pass == null){MailAI.Info(0,"null in buildstraightpassing, should not be");return null;}
	local track_dir = AIRail.RAILTRACK_NE_SW;
	if(exit_dir == "NW" || exit_dir == "SE") {track_dir = AIRail.RAILTRACK_NW_SE;}
	local start1_track,start2_track,end1_track,end2_track;
	//determine the diagonal track directions
	if(exit_dir == "NE") {//left view
		pass_list.Valuate(AIMap.GetTileX);
		pass_list.Sort(AIList.SORT_BY_VALUE,false);
		start1_track=AIRail.RAILTRACK_NW_SW;start2_track=AIRail.RAILTRACK_NE_SE;
		end1_track=AIRail.RAILTRACK_SW_SE;end2_track=AIRail.RAILTRACK_NW_NE;
	}
	if(exit_dir == "SW") {//right view
		pass_list.Valuate(AIMap.GetTileX);
		pass_list.Sort(AIList.SORT_BY_VALUE,true);
		start1_track=AIRail.RAILTRACK_NE_SE;start2_track=AIRail.RAILTRACK_NW_SW;
		end1_track=AIRail.RAILTRACK_NW_NE;end2_track=AIRail.RAILTRACK_SW_SE;
	}					
	if(exit_dir == "NW") {//left view
		pass_list.Valuate(AIMap.GetTileY);
		pass_list.Sort(AIList.SORT_BY_VALUE,false);
		start1_track=AIRail.RAILTRACK_SW_SE;start2_track=AIRail.RAILTRACK_NW_NE;
		end1_track=AIRail.RAILTRACK_NE_SE;end2_track=AIRail.RAILTRACK_NW_SW;
	}					
	if(exit_dir == "SE") {//right view
		pass_list.Valuate(AIMap.GetTileY);
		pass_list.Sort(AIList.SORT_BY_VALUE,true);
		start1_track=AIRail.RAILTRACK_NW_NE;start2_track=AIRail.RAILTRACK_SW_SE;
		end1_track=AIRail.RAILTRACK_NW_SW;end2_track=AIRail.RAILTRACK_NE_SE;
	}					
	while(true) {
		local bool = false, last = null;	
		local start_pass_left = Tile.GetLeftTile(start_pass,exit_dir);
		local end_pass_left = Tile.GetLeftTile(end_pass,exit_dir);
		local signal_start = Tile.GetForwardTile(start_pass,exit_dir);
		local signal_start_before = Tile.GetForwardTile(signal_start,exit_dir);
		local quick = Tile.GetBackwardTile(start_pass_left,exit_dir);
		if(Tile.IsRailMine(quick)) {
			if(Tile.IsRailMine(Tile.GetBackwardTile(quick,exit_dir))) {
				bool = EventManager.BuildRailTrack(quick,track_dir);
			} else if(Tile.IsRailMine(Tile.GetLeftTile(quick,exit_dir))) {
				bool = EventManager.BuildRailTrack(quick,end2_track);
			}
			if(bool){
				bool=EventManager.BuildRailTrack(start_pass_left,track_dir);
				signal_start_before = signal_start;
				signal_start = start_pass;
			}	
		}
		if(!bool) {
			if(!EventManager.BuildRailTrack(start_pass,start1_track)){
				Tile.DemolishTile(start_pass);//road might be crossing there
				EventManager.BuildRailTrack(start_pass,track_dir);
				if(!EventManager.BuildRailTrack(start_pass,start1_track)) {
					MailAI.BuildSign(start_pass,"X1");break;
				}
			}	
			if(!EventManager.BuildRailTrack(start_pass_left,start2_track)){
				MailAI.BuildSign(start_pass_left,"X2");break;
			}	
		} 
		bool=true;	
		foreach(t, _ in pass_list) {
			if(!bool){break;}
			if(t == end_pass) {
				if(AIRail.GetRailTracks(end_pass) == start2_track) {
					bool = EventManager.BuildRailTrack(t,track_dir);	
				} else {
					bool = EventManager.BuildRailTrack(t,start1_track);
				}	
				last = t;
				break;
			}
			if(t == end_pass_left) {//avoid an excess tile in the list
				break;
			}
			if(bool){bool = EventManager.BuildRailTrack(t,track_dir);}		
		}
		if(!bool){break;}
		local signal_end = Tile.GetBackwardTile(end_pass,exit_dir);
		if(last == null) {	
			if(!EventManager.BuildRailTrack(end_pass_left,end1_track)){break;}
			if(!EventManager.BuildRailTrack(end_pass,end2_track)){break;}
			signal_end = Tile.GetBackwardTile(end_pass_left,exit_dir);
		}
		local signal_end_before = Tile.GetBackwardTile(signal_end,exit_dir);					
		//build the signals last so we know that all tracks have been build	
		if(!EventManager.BuildSignal(signal_start,signal_start_before,AIRail.SIGNALTYPE_PBS_ONEWAY)) {
			MailAI.BuildSign(signal_start,"HERE2");break;
		}
		if(!EventManager.BuildSignal(signal_end,signal_end_before,AIRail.SIGNALTYPE_PBS_ONEWAY)) {
			MailAI.BuildSign(signal_end,"HERE3");break;
		}
		return end_pass;	
	}
	MailAI.Info(0,"didnt build pass : "+AIError.GetLastErrorString());MailAI.BuildSign(start_pass,"!");
	return start_pass;
}

function RailBuilder::BuildRailPath(a1,a2) {//actually builds from a2 towards a1
	//MailAI.Info(0,"pathlooking");
	local railfinder = MyRailPF();
	railfinder.cost.max_bridge_length = MailAI.max_train_bridge_length;
	railfinder.cost.bridge_per_tile = MailAI.bridge_cost;
	//set max_cost to prevent pathfinder looping for a very long time when no routes are found
	railfinder.cost.max_cost = AIMap.DistanceManhattan(a1[0][0], a2[0][0]) * 4 * (railfinder.cost.tile);
	local path = false; local next_path = null;
  	railfinder.InitializePath(a1,a2);
   	local counter = 0; 	
   	local timer = AIMap.DistanceManhattan(a1[0][0], a2[0][0]) * 5;
	while (counter < 50 && path == false) {
		path = railfinder.FindPath(timer);//returns false if not finished, otherwise path or null
		MailAI.Sleep(1); counter++;
	}
   	if (path == null) {
		MailAI.Info(0,"Railfinder returns null.");
		return null;
	}
	if (!path) {MailAI.Info(0,"Railfinder is taking too long, aborting.");return null;}
  	local save1 = a1[0][0], save2 = a1[0][1], counter = 0; 	
  	/* If a path was found, build a rail over it. */
	local double_build = false, triple_build = false, next2, next, curr, prev, prev2, prev3;
	//MailAI.Info(0,"pathbuilding");
	while (path != null) {
  		prev3 = prev2; prev2 = prev;
    	prev = curr; curr = next;
    	next = next2;
    	next2 = path.GetTile();
    	path = path.GetParent();
    	counter++;
    	if(triple_build) {triple_build=false;double_build=true;continue;}
    	if(double_build) {double_build=false;continue;}
  		if (prev != null) {//wait till prev, curr, next and next2 are filled
    		if (AIMap.DistanceManhattan(curr,next) > 1) {	
      			if (AITunnel.GetOtherTunnelEnd(curr) == next) {
      				local it_count = 0; 
					while(!AITunnel.BuildTunnel(AIVehicle.VT_RAIL, curr) && it_count < 33
					&& AIError.GetLastErrorString() != "ERR_ALREADY_BUILT") {
						EventManager.HandleCommonErrors(curr,next);		
						it_count++;
					}
					if(it_count >= 33) {
						MailAI.Info(2,"Couldn't build tunnel! "+AIError.GetLastErrorString());
					} else {
						double_build=true;
					}
      			} else {
        			local bridge_list = AIBridgeList_Length(AIMap.DistanceManhattan(curr,next) + 1);
        			bridge_list.Valuate(AIBridge.GetMaxSpeed); bridge_list.Sort(AIList.SORT_BY_VALUE, false);
        			local it_count = 0; 
					while(!AIBridge.BuildBridge(AIVehicle.VT_RAIL, bridge_list.Begin(), curr, next) && it_count < 33
					&& AIError.GetLastErrorString() != "ERR_ALREADY_BUILT") {
						EventManager.HandleCommonErrors(curr,next);		
						it_count++;
					}
					if(it_count >= 33) {
						MailAI.Info(2,"Couldn't build bridge! "+AIError.GetLastErrorString());
					} else {
						double_build=true;
					}
        		}
        		counter += AIMap.DistanceManhattan(curr,next);
		    } else {
		    	local bool = false;
				local dir_to_next = Tile.GetDirectionFromTo(curr,next);
				if(dir_to_next == Tile.GetDirectionFromTo(next,next2)
					&& dir_to_next == Tile.GetDirectionFromTo(prev,curr)) {
					if(Tile.WillSlopeDown(curr,dir_to_next) && Tile.WillSlopeUp(next,dir_to_next)) {
						local bridge_list = AIBridgeList_Length(2);
        				bridge_list.Valuate(AIBridge.GetMaxSpeed); bridge_list.Sort(AIList.SORT_BY_VALUE, false);
        				bool = AIBridge.BuildBridge(AIVehicle.VT_RAIL,bridge_list.Begin(),curr,next);	
						if(bool) {double_build=true;}
					} else if(Tile.WillSlopeUp(curr,dir_to_next) && Tile.WillSlopeDown(next,dir_to_next)) {
						if (AITunnel.GetOtherTunnelEnd(curr) == next) {//checking it's only mini tunnel
        					bool = AITunnel.BuildTunnel(AIVehicle.VT_RAIL, curr);
        					if(bool) {double_build=true;}
						}
					}
				}		
				if(!bool) {
					if(!EventManager.BuildRail(prev, curr, next)) {	
			    		if((AIRail.IsRailTile(curr) || AIRoad.IsRoadTile(curr)) && Tile.IsBuildableAlmostFlatTile(next)) {
			    			//if just one rail or road, make a quick bridge over it
			    			AITile.DemolishTile(prev);
			    			local bridge_list = AIBridgeList_Length(3);
	        				bridge_list.Valuate(AIBridge.GetMaxSpeed); bridge_list.Sort(AIList.SORT_BY_VALUE, false);
			    			bool = AIBridge.BuildBridge(AIVehicle.VT_RAIL, bridge_list.Begin(), prev, next);
			    			if(bool) {double_build=true;}
			    		}
			    		local dir_to_curr = Tile.GetDirectionFromTo(prev,curr);
			    		local dir_to_next = Tile.GetDirectionFromTo(curr,next);
			    		if(dir_to_curr == dir_to_next
			    		&& (AIRail.IsRailTile(curr) || AIRoad.IsRoadTile(curr))
			    		&& (AIRail.IsRailTile(next) || AIRoad.IsRoadTile(next))
			    		&& Tile.IsBuildableAlmostFlatTile(Tile.GetForwardTile(next,dir_to_next))) {
			    			//if just two rail or road, make a quick bridge over it
			    			AITile.DemolishTile(prev);
			    			local bridge_list = AIBridgeList_Length(4);
	        				bridge_list.Valuate(AIBridge.GetMaxSpeed); bridge_list.Sort(AIList.SORT_BY_VALUE, false);
			    			bool = AIBridge.BuildBridge(AIVehicle.VT_RAIL, bridge_list.Begin(), prev, Tile.GetForwardTile(next,dir_to_next));
			    			if(bool) {triple_build=true;}
			    		}
			    		local it_count = 0; 
						if(!EventManager.BuildRail(prev, curr, next)) {
							MailAI.Info(2,"Failed railbuilding because: "+AIError.GetLastErrorString());
			    			AITile.DemolishTile(prev);AITile.DemolishTile(prev2);
			    			MailAI.BuildSign(curr,"--");MailAI.BuildSign(next,"->");
			    			local c = RailBuilder.BuildRailPath([[save1,save2]],[[prev2,prev3]]);
			    			if(c==null) {return null;}
			    			MailAI.Info(2,"Saved the day!");
			    			return counter+c;
						}
					}
		    	}//end if not bool
		    }//end else (distancee=1)
  		}//end if prev!=null
	}//end while
//got one tile left to build!
	//MailAI.Info(0,"pathbuild");
 	prev3 = prev2;prev2 = prev;
   	prev = curr;curr = next;
   	next = next2;
    if(!double_build){
    	counter++;
		local it_count = 0;
		while(!EventManager.BuildRail(prev, curr, next) && it_count < 33
		&& AIError.GetLastErrorString() != "ERR_ALREADY_BUILT") {
			EventManager.HandleCommonErrors(curr,next);		
			it_count++;
		}
		if(it_count >= 33) {
			MailAI.Info(2,"failed last tile of railbuilding because: "+AIError.GetLastErrorString());
		}
    }
	return counter;
}

function RailBuilder::UpdateStationSwitch(arr) {//[switch,out,other_switch,not needed,not needed]
	local left_switch = arr[0], left_out = arr[1]; 
	local dir = Tile.GetDirectionFromTo(left_switch,left_out);
	local right_switch = arr[2], right_out = Tile.GetForwardTile(right_switch,dir) ;
	if(!AIRail.AreTilesConnected(left_switch,left_out,right_out)){return arr;}
//start is right but need to be left
	if(Tile.GetLeftTile(left_switch,dir) == right_switch) {
		left_switch = arr[2]; left_out = Tile.GetForwardTile(left_switch,dir);
		right_switch = arr [0]; right_out = arr[1];
		if(Tile.IsSlopedSideWays(left_out,dir)){return arr;}
		local connect_tile = Tile.GetForwardTile(left_out,dir);
		if(!AIRail.AreTilesConnected(right_out,left_out,connect_tile))
			{connect_tile = Tile.GetLeftTile(left_out,dir);}
		if(!AIRail.AreTilesConnected(right_out,left_out,connect_tile))
			{MailAI.Info(0,"Shouldn't be this direction in updatestationswitch1");return arr;}
		Tile.DemolishTile(left_switch);Tile.DemolishTile(left_out);
		Tile.DemolishTile(right_switch);Tile.DemolishTile(right_out);
		if(EventManager.BuildRail(left_switch,left_out,connect_tile) 
		&& RailBuilder.BuildStationSwitchTrack(left_switch,right_switch,dir,true,false)){
			return [arr[2],left_out,arr[0],arr[3],arr[4]];
		} else {
			MailAI.Info(3,"opened up a perfectly good track1 :(");return arr;
		}	
//start is left but need to be right
	} else {
		if(Tile.IsSlopedSideWays(right_out,dir)){return arr;}
		local connect_tile = Tile.GetForwardTile(right_out,dir);
		if(!AIRail.AreTilesConnected(left_out,right_out,connect_tile))
			{connect_tile = Tile.GetRightTile(right_out,dir);}
		if(!AIRail.AreTilesConnected(left_out,right_out,connect_tile))
			{MailAI.Info(0,"Shouldn't be this direction in updatestationswitch2");return arr;}
		Tile.DemolishTile(left_switch);Tile.DemolishTile(left_out);
		Tile.DemolishTile(right_switch);Tile.DemolishTile(right_out);
		if(EventManager.BuildRail(right_switch,right_out,connect_tile) 
		&& RailBuilder.BuildStationSwitchTrack(left_switch,right_switch,dir,false,true)){
			return [arr[2],right_out,arr[0],arr[3],arr[4]];
		} else {
			MailAI.Info(3,"opened up a perfectly good track1 :(");return arr;
		}	
	}
}
