/* 
 *	Copyright 2013 Varen De Meersman
 *  This file is part of MailAI.
 *
 *  MailAI is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  MailAI is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with MailAI.  If not, see <http://www.gnu.org/licenses/>.
 */
 
class Tile {
	constructor() {}
}

/**
*	Get the major and side direction between towns.
*	@param t1 TownID1
*	@param t2 TownID2
*	@return Main and side directions for both directions and boolean if the stations are
*		more then twentyfive tiles apart on the side-axis.
*/
function Tile::GetDirections(t1_tile,t2_tile) {
	local main_direction1, side_direction1, main_direction2, side_direction2;
	local x1=AIMap.GetTileX(t1_tile),x2=AIMap.GetTileX(t2_tile);
	local y1=AIMap.GetTileY(t1_tile),y2=AIMap.GetTileY(t2_tile);
	local xdif = x1-x2, ydif = y1-y2;
	local turn = false;
	if(abs(xdif)>abs(ydif)) {
		if(xdif < 0){main_direction1="SW";main_direction2="NE";}
		else{main_direction1="NE";main_direction2="SW";}	
		if(ydif<0){side_direction1="SE";side_direction2="NW";}
		else{side_direction1="NW";side_direction2="SE";}
		if(ydif > 25 || ydif < -25) {turn = true;}
	} else {
		if(ydif<0){main_direction2="NW";main_direction1="SE";}
		else{main_direction2="SE";main_direction1="NW";}	
		if(xdif<0){side_direction1="SW";side_direction2="NE";}
		else{side_direction1="NE";side_direction2="SW";}
		if(xdif > 25 || xdif < -25) {turn = true;}
	}	
	return [main_direction1,side_direction1,main_direction2,side_direction2,turn];	
}

function Tile::GetReverseDirection(dir) {
	switch (dir) {
		case "SW":return "NE";
		case "NE":return "SW";
		case "SE":return "NW";
		case "NW":return "SE";
		default: throw("Wrong direction in getleftdirection");
	}	
}

function Tile::GetLeftDirection(dir) {
	switch (dir) {
		case "SW":return "SE";
		case "NE":return "NW";
		case "SE":return "NE";
		case "NW":return "SW";
		default: throw("Wrong direction in getleftdirection");
	}	
}

function Tile::GetRightDirection(dir) {
	switch (dir) {
		case "SW":return "NW";
		case "NE":return "SE";
		case "SE":return "SW";
		case "NW":return "NE";
		default: throw("Wrong direction in getleftdirection");
	}	
}

function Tile::IsCloser(tile_left,tile_right,goal) {
	if(tile_left == null){throw("null tile in iscloser");}
	if(tile_right == null){return true;}
	if(AIMap.DistanceManhattan(tile_left,goal) 
	<= AIMap.DistanceManhattan(tile_right,goal)){return true;}
	return false;
}

function Tile::GetDirectionFromTo(tile1, tile2) {
	if(tile1==null||tile2==null)throw("null tiles in getdirectionfromto");
	local x = abs(AIMap.GetTileX(tile1)-AIMap.GetTileX(tile2));
	local y = abs(AIMap.GetTileY(tile1)-AIMap.GetTileY(tile2));
	if(x>=y) {
		if(AIMap.GetTileX(tile1) < AIMap.GetTileX(tile2)) {return "SW";}
		if(AIMap.GetTileX(tile1) >= AIMap.GetTileX(tile2)) {return "NE";}
	} else {
		if(AIMap.GetTileY(tile1) < AIMap.GetTileY(tile2)) {return "SE";}
		if(AIMap.GetTileY(tile1) >= AIMap.GetTileY(tile2)) {return "NW";}
	}	
}

function Tile::GetSideDirectionFromTo(tile1, tile2) {
	if(tile1==null||tile2==null)throw("null tiles in getdirectionfromto");
	local x = abs(AIMap.GetTileX(tile1)-AIMap.GetTileX(tile2));
	local y = abs(AIMap.GetTileY(tile1)-AIMap.GetTileY(tile2));
	if(x<y) {
		if(AIMap.GetTileX(tile1) <= AIMap.GetTileX(tile2)) {return "SW";}
		if(AIMap.GetTileX(tile1) > AIMap.GetTileX(tile2)) {return "NE";}
	} else {
		if(AIMap.GetTileY(tile1) <= AIMap.GetTileY(tile2)) {return "SE";}
		if(AIMap.GetTileY(tile1) > AIMap.GetTileY(tile2)) {return "NW";}
	}
}

function Tile::IsSideDirLeft(dir,side) {
	if(dir == "SW" && side == "SE"){return true;}
	if(dir == "NE" && side == "NW"){return true;}
	if(dir == "SE" && side == "NE"){return true;}
	if(dir == "NW" && side == "SW"){return true;}
	return false;
}

/**
*
*/
function Tile::SideToX(side,value) {
	switch (side) {
		case "SW":return value;
		case "NE":return -value;
		case "SE":return 0;
		case "NW":return 0;
		default: throw("Wrong side in sidetox");
	}	
}
/**
*
*/
function Tile::SideToY(side,value) {
	switch (side) {
		case "SW":return 0;
		case "NE":return 0;
		case "SE":return value;
		case "NW":return -value;
		default: throw("Wrong side in sidetoy");
	}	
}	

function Tile::GetTrackDir(dir) {
	switch (dir) {
	    case "NE": //return AIRail.RAILTRACK_NE_SW;
	    case "SW": return AIRail.RAILTRACK_NE_SW;
	    case "SE": //return AIRail.RAILTRACK_NW_SE;
	    case "NW": return AIRail.RAILTRACK_NW_SE;
		default: throw("Wrong direction in getrackdir");
	}
}

/**
*	returns the four tiles around given tile
*	@param t center tile
*	@return AITileList with four adjacent tiles
*/
function Tile::GetAdjacentTiles(t) {
	local return_list = AITileList();
	return_list.AddTile(t - AIMap.GetTileIndex(1,0));
	return_list.AddTile(t - AIMap.GetTileIndex(0,1));
	return_list.AddTile(t - AIMap.GetTileIndex(-1,0));
	return_list.AddTile(t - AIMap.GetTileIndex(0,-1));
	return return_list;
}

/**
*	returns the eight tiles around given tile
*	@param t center tile
*	@return AITileList with eight adjacent tiles
*/
function Tile::GetEightAdjacentTiles(t) {	
	local return_list = AITileList();
	return_list.AddList(Tile.GetAdjacentTiles(t));
	return_list.AddTile(t - AIMap.GetTileIndex(1,1));
	return_list.AddTile(t - AIMap.GetTileIndex(-1,-1));
	return_list.AddTile(t - AIMap.GetTileIndex(-1,1));
	return_list.AddTile(t - AIMap.GetTileIndex(1,-1));
	return return_list;
}

function Tile::GetSurroundingTiles(tl1,tl2) {	
	local tile_list = AITileList();
	foreach (t, _ in tl1) {
		tile_list.AddList(Tile.GetEightAdjacentTiles(t));
	}
	if(tl2 != null) {
		foreach (t, _ in tl2) {
			tile_list.AddList(Tile.GetEightAdjacentTiles(t));
		}
		tile_list.RemoveList(tl2);
	}
	tile_list.RemoveList(tl1);	
	return tile_list;
}

/**
*	Only takes almostflat, non coast? tiles as buildable!
*	@param tile The tile to check
*	@return true if almostflat and buildable
*/
function Tile::IsBuildableAlmostFlatTile(tile) {
	if(tile == null){return false;}
	if(Tile.IsAlmostFlat(tile) && AITile.IsBuildable(tile)){return true;}	
	return false;
}

/**
*	Checks to see if tile is flat or only has one lowered corner.
*	@param tile The tile to check.
*	@return true if tile is almost flat.
*/
function Tile::IsAlmostFlat(tile) {
	if(tile == null){return false;}
	switch(AITile.GetSlope(tile)) {
		case AITile.SLOPE_FLAT:
		case AITile.SLOPE_NWS:
		case AITile.SLOPE_WSE:
		case AITile.SLOPE_SEN:
		case AITile.SLOPE_ENW:return true;
		default: return false;
	}
}

/**
*	Check for buildalbe or road tile
*	@param tile tile to check
*	@param road if true, the tile is allowed to be a RoadTile
*	@return true if buildable
*	@todo handle non-flat tiles as well
*/
function Tile::IsBuildableRoadTile(tile) {
	if (tile!=null && Tile.IsAlmostFlat(tile)) {
		if(AITile.IsBuildable(tile) || AIRoad.IsRoadTile(tile)) {
		   	return true;
		}	
	}
	return false;
}

/**
*	Checks to see if tile is almostflat and raises a corner to make it flat.
*	@param tile The tile to flatten.
*	@return True if tile has been made flat.
*/
function Tile::MakeTileFlat(tile) { 
	switch(AITile.GetSlope(tile)) {
		case AITile.SLOPE_FLAT: return true;
		case AITile.SLOPE_NWS: return AITile.RaiseTile(tile,AITile.SLOPE_E);
		case AITile.SLOPE_WSE: return AITile.RaiseTile(tile,AITile.SLOPE_N);
		case AITile.SLOPE_SEN: return AITile.RaiseTile(tile,AITile.SLOPE_W);
		case AITile.SLOPE_ENW: return AITile.RaiseTile(tile,AITile.SLOPE_S);
		case AITile.SLOPE_NW: return AITile.RaiseTile(tile,AITile.SLOPE_SE);
		case AITile.SLOPE_SW: return AITile.RaiseTile(tile,AITile.SLOPE_NE);
		case AITile.SLOPE_NE: return AITile.RaiseTile(tile,AITile.SLOPE_SW);
		case AITile.SLOPE_SE: return AITile.RaiseTile(tile,AITile.SLOPE_NW);
		case AITile.SLOPE_EW: return AITile.RaiseTile(tile,AITile.SLOPE_NS);
		case AITile.SLOPE_NS: return AITile.RaiseTile(tile,AITile.SLOPE_EW);
		case AITile.SLOPE_N: return AITile.LowerTile(tile,AITile.SLOPE_N);
		case AITile.SLOPE_S: return AITile.LowerTile(tile,AITile.SLOPE_S);
		case AITile.SLOPE_E: return AITile.LowerTile(tile,AITile.SLOPE_E);
		case AITile.SLOPE_W: return AITile.LowerTile(tile,AITile.SLOPE_W);
		default: throw("Provided tile in MakeTileFlat has unhandled slope: "+AITile.GetSlope(tile));
	}
	return false;
}

/**
*	Checks to see if tile is flat or only has one lowered corner.
*	@param tile The tile to check.
*	@return true if tile is almost flat.
*/
function Tile::DemolishTile(tile) {
	if(tile == null){return false;}
	//we don't use handlecommonerrors here because this function is called in that routine
	local i = 0;
	for(i = 0; !AITile.DemolishTile(tile) && i < 20 ; i++) {
		if(AIError.GetLastErrorString() == "ERR_LOCAL_AUTHORITY_REFUSES") {
			//MailAI.BuildSign(tile,"demolished");
			EventManager.HandleRating(AITile.GetClosestTown(tile));
		}
		if(AIError.GetLastErrorString() == "ERR_NOT_ENOUGH_CASH") {
			//MailAI.BuildSign(tile,"demolished");
			BankManager.LoanOne();
		} 	
		if(AIError.GetLastErrorString() == "ERR_VEHICLE_IN_THE_WAY") {
			//MailAI.BuildSign(tile,"demolished");
			MailAI.Sleep(50);
		} 
	}
	if(i >= 19) {
		MailAI.Info(0,"Didn't demolish because "+AIError.GetLastErrorString());
		return false;
	}
	return true;
}

function Tile::ClearArea(tl) {
	foreach(t, _ in tl) {
		//road tiles are counted as buildable if they're only half the tile
		if(!AITile.IsBuildable(t) || AIRoad.IsRoadTile(t)) {
			if(!Tile.DemolishTile(t)){return false;}
		}
	}	
	return true;
}

/**
*	returns tile in front considering direction
*	@param tile The start tile
*	@param dir Direction to look in, NE,SE,SW,NW
*	@return The tile in front considering direction provided.
*/
function Tile::GetForwardTile(tile,dir) {
	if(tile==null){throw("null tile in getforwardtile");}
	switch(dir) {
		case "NE": return tile+AIMap.GetTileIndex(-1,0);
		case "SE": return tile+AIMap.GetTileIndex(0,1);
		case "SW": return tile+AIMap.GetTileIndex(1,0);
		case "NW": return tile+AIMap.GetTileIndex(0,-1);
		default: throw("Wrong direction in getforwardtile: "+dir);
	}
}

function Tile::GetForwardLeftTile(tile,dir) {
	return Tile.GetForwardTile(Tile.GetLeftTile(tile,dir),dir);
}
function Tile::GetForwardRightTile(tile,dir) {
	return Tile.GetForwardTile(Tile.GetRightTile(tile,dir),dir);
}

/**
*	returns tile in back considering direction, carefull, this is only for straight track!
*	@param tile The start tile
*	@param dir Direction to look in, NE,SE,SW,NW
*	@return The tile in back considering direction provided.
*/
function Tile::GetBackwardTile(tile,dir) {
	if(tile==null){throw("null tile in getbacktile");}
	switch(dir) {
		case "NE": return tile+AIMap.GetTileIndex(1,0);
		case "SE": return tile+AIMap.GetTileIndex(0,-1);
		case "SW": return tile+AIMap.GetTileIndex(-1,0);
		case "NW": return tile+AIMap.GetTileIndex(0,1);
		default: throw("Wrong direction in getbackwardtile: "+dir);
	}
}

/**
*	returns tile to the left considering direction
*	@param tile The start tile
*	@param dir Direction to look in, NE,SE,SW,NW
*	@return The tile to the left considering direction provided.
*/
function Tile::GetLeftTile(tile,dir) {
	if(tile==null){throw("null tile in getlefttile");}
	switch(dir) {
		case "NE": return tile+AIMap.GetTileIndex(0,-1);
		case "SE": return tile+AIMap.GetTileIndex(-1,0);
		case "SW": return tile+AIMap.GetTileIndex(0,1);
		case "NW": return tile+AIMap.GetTileIndex(1,0);
		default: throw("Wrong direction in getlefttile: "+dir);
	}
}

/**
*	returns tile to the right considering direction
*	@param tile The start tile
*	@param dir Direction to look in, NE,SE,SW,NW
*	@return The tile to the right considering direction provided.
*/
function Tile::GetRightTile(tile,dir) {
	if(tile==null){throw("null tile in getrighttile");}
	switch(dir) {
		case "NE": return tile+AIMap.GetTileIndex(0,1);
		case "SE": return tile+AIMap.GetTileIndex(1,0);
		case "SW": return tile+AIMap.GetTileIndex(0,-1);
		case "NW": return tile+AIMap.GetTileIndex(-1,0);
		default: throw("Wrong direction in getrighttile: "+dir);
	}
}

/**
*	to see if a tile is raised to 'flat' because of building track
*/
function Tile::IsRaisedTrack(tile,dir) {
	local slope = AITile.GetSlope(tile);
	if(dir=="SW" || dir=="NE") {
		if(slope == AITile.SLOPE_SE || slope == AITile.SLOPE_NW)
			{return true;} else {return false;}
	}
	if(dir=="SE" || dir=="NW") {
		if (slope == AITile.SLOPE_NE || slope == AITile.SLOPE_SW)
			{return true;} else {return false;}
	}
	throw("Wrong direction in israisedtrack "+dir);
}

/**
*	Put left tile as first param
*/
function Tile::WillSlopeTheSame(t1,t2,dir) {
	if(Tile.IsAlmostFlat(t1) && Tile.IsAlmostFlat(t2)){return true;}
	if(Tile.WillSlopeUp(t1,dir) && Tile.WillSlopeUp(t2,dir)){return true;}
	if(Tile.WillSlopeDown(t1,dir) && Tile.WillSlopeDown(t2,dir)){return true;}
	if(Tile.IsAlmostFlat(t1) && Tile.IsRaisedTrack(t2,dir)){return true;}
	return false;	
}

function Tile::IsBuildableAvoidSlopeTile(tile,to) {
	if(tile==null || to==null) {throw("null tile in canavoidslope");}
	if(!AITile.IsBuildable(tile)) {return false;}
	local slope = AITile.GetSlope(tile);
	local dir = Tile.GetDirectionFromTo(to,tile);	
	switch(dir) {
		case "NE":
			if(slope == AITile.SLOPE_N || slope == AITile.SLOPE_E) {return true;} 
			break;
		case "SE": 
			if(slope == AITile.SLOPE_S || slope == AITile.SLOPE_E) {return true;}
			break;
		case "SW":
			if(slope == AITile.SLOPE_S || slope == AITile.SLOPE_W) {return true;}
			break;
		case "NW":
			if(slope == AITile.SLOPE_N || slope == AITile.SLOPE_W) {return true;}
			break;
		default: throw("Wrong direction in canavoidslope: "+dir);
	}
	return false;
}

function Tile::FixBuildableDiagTile(tile,dir) {
	if(tile==null) {throw("null tile in IsBuildableDiagTile");}
	if(!AITile.IsBuildable(tile)) {return false;}
	local slope = AITile.GetSlope(tile);
	if(Tile.IsAlmostFlat(tile)) {return true;}	
	switch(dir) {
		case "NE":
			if(slope == AITile.SLOPE_W) {return true;} 
			if(slope == AITile.SLOPE_NW) {return AITile.LowerTile(tile,AITile.SLOPE_N);}
			break;
		case "SE": 
			if(slope == AITile.SLOPE_N) {return true;}
			if(slope == AITile.SLOPE_NE) {return AITile.LowerTile(tile,AITile.SLOPE_E);}
			break;
		case "SW":
			if(slope == AITile.SLOPE_E) {return true;}
			if(slope == AITile.SLOPE_SE) {return AITile.LowerTile(tile,AITile.SLOPE_S);}
			break;
		case "NW":
			if(slope == AITile.SLOPE_S) {return true;}
			if(slope == AITile.SLOPE_SW) {return AITile.LowerTile(tile,AITile.SLOPE_W);}
			break;
		default: throw("Wrong direction in IsBuildableDiagTile: "+dir);
	}
	return false;
}

function Tile::IsSlopedSideWays(tile,dir) {
	local slope = AITile.GetSlope(tile);
	if(dir=="NE") {
		if(slope == AITile.SLOPE_NW || slope == AITile.SLOPE_SE) {return true;}	
	}		
	if(dir=="SE") {
		if(slope == AITile.SLOPE_NE || slope == AITile.SLOPE_SW) {return true;}
	}
	if(dir=="SW") {
		if(slope == AITile.SLOPE_NW || slope == AITile.SLOPE_SE) {return true;}
	}
	if(dir=="NW") {
		if(slope == AITile.SLOPE_NE || slope == AITile.SLOPE_SW) {return true;}
	}
	return false;
}

function Tile::FixSlopeSideWaysForSwitch(tile,dir) {
	local slope = AITile.GetSlope(tile);
	if(dir=="NE" && slope == AITile.SLOPE_NW) {
		return AITile.RaiseTile(tile,AITile.SLOPE_S);
	}
	if(dir=="NE" && slope == AITile.SLOPE_SE) {
		return AITile.RaiseTile(tile,AITile.SLOPE_W);
	}			
	if(dir=="SE" && slope == AITile.SLOPE_NE) {
		return AITile.RaiseTile(tile,AITile.SLOPE_W);
	}
	if(dir=="SE" && slope == AITile.SLOPE_SW) {
		return AITile.RaiseTile(tile,AITile.SLOPE_N);
	}
	if(dir=="SW" && slope == AITile.SLOPE_NW) {
		return AITile.RaiseTile(tile,AITile.SLOPE_E);
	}
	if(dir=="SW" && slope == AITile.SLOPE_SE) {
		return AITile.RaiseTile(tile,AITile.SLOPE_N);
	}
	if(dir=="NW" && slope == AITile.SLOPE_NE) {
		return AITile.RaiseTile(tile,AITile.SLOPE_S);
	}
	if(dir=="NW" && slope == AITile.SLOPE_SW) {
		return AITile.RaiseTile(tile,AITile.SLOPE_E);
	}
	return false;
}

function Tile::FixSlopeUpForSwitch(tile,dir) {
	local slope = AITile.GetSlope(tile);
	if((dir=="SW" && slope == AITile.SLOPE_S)
	|| (dir=="NW" && slope == AITile.SLOPE_N)) {
		return AITile.RaiseTile(tile,AITile.SLOPE_W);
	}			
	if((dir=="SW" && slope == AITile.SLOPE_W)
	|| (dir=="SE" && slope == AITile.SLOPE_E)) { 
		return AITile.RaiseTile(tile,AITile.SLOPE_S);
	}
	if((dir=="NE" && slope == AITile.SLOPE_N)
	|| (dir=="SE" && slope == AITile.SLOPE_S)) {
		return AITile.RaiseTile(tile,AITile.SLOPE_E);
	}
	if((dir=="NE" && slope == AITile.SLOPE_E)
	|| (dir=="NW" && slope == AITile.SLOPE_W)) {
		return AITile.RaiseTile(tile,AITile.SLOPE_N);
	}
	return false;
}

function Tile::FixSlopeDownForSwitch(tile,dir) {
	local slope = AITile.GetSlope(tile);
	if((dir=="SW" && slope == AITile.SLOPE_ENW)
	|| (dir=="NW" && slope == AITile.SLOPE_WSE)) {
		return AITile.LowerTile(tile,AITile.SLOPE_W);
	}			
	if((dir=="SW" && slope == AITile.SLOPE_SEN)
	|| (dir=="SE" && slope == AITile.SLOPE_NWS)) { 
		return AITile.LowerTile(tile,AITile.SLOPE_S);
	}
	if((dir=="NE" && slope == AITile.SLOPE_WSE)
	|| (dir=="SE" && slope == AITile.SLOPE_ENW)) {
		return AITile.LowerTile(tile,AITile.SLOPE_E);
	}
	if((dir=="NE" && slope == AITile.SLOPE_NWS)
	|| (dir=="NW" && slope == AITile.SLOPE_SEN)) {
		return AITile.LowerTile(tile,AITile.SLOPE_N);
	}
	return false;
}

function Tile::WillSlopeUp(tile,dir) {
	local slope = AITile.GetSlope(tile);
	if(dir=="SW") {
		if(slope == AITile.SLOPE_SW || slope == AITile.SLOPE_S || slope == AITile.SLOPE_W)
			{return true;} else {return false;}
	}
	if(dir=="NE") {
		if (slope == AITile.SLOPE_NE || slope == AITile.SLOPE_N || slope == AITile.SLOPE_E)
			{return true;} else {return false;}
	}
	if(dir=="SE") {
		if (slope == AITile.SLOPE_SE || slope == AITile.SLOPE_S || slope == AITile.SLOPE_E)
			{return true;} else {return false;}
	}
	if(dir=="NW") {
		if (slope == AITile.SLOPE_NW || slope == AITile.SLOPE_N || slope == AITile.SLOPE_W)
			{return true;} else {return false;}
	}
	throw("Wrong direction in willslopeup "+dir);
}

function Tile::WillSlopeDown(tile,dir) {
	local slope = AITile.GetSlope(tile);
	if(dir=="NE") {
		if(slope == AITile.SLOPE_SW || slope == AITile.SLOPE_S || slope == AITile.SLOPE_W)
			{return true;} else {return false;}
	}
	if(dir=="SW") {
		if (slope == AITile.SLOPE_NE || slope == AITile.SLOPE_N || slope == AITile.SLOPE_E)
			{return true;} else {return false;}
	}
	if(dir=="NW") {
		if (slope == AITile.SLOPE_SE || slope == AITile.SLOPE_S || slope == AITile.SLOPE_E)
			{return true;} else {return false;}
	}
	if(dir=="SE") {
		if (slope == AITile.SLOPE_NW || slope == AITile.SLOPE_N || slope == AITile.SLOPE_W)
			{return true;} else {return false;}
	}
	throw("Wrong direction in willslopedown "+dir);
}

function Tile::ConnectAtSameHeight(tile1,tile2,dir) {
	if(tile1 == null || tile2 == null || dir == null) {return null;}
	if(Tile.GetExitHeight(tile1,dir) == Tile.GetEntryHeight(tile2,dir)) {return true;}
	return false;	
}

function Tile::GetExitHeight(tile,dir) {
	if(dir=="NE") {	
		if(AITile.GetCornerHeight(tile,AITile.CORNER_N) > AITile.GetCornerHeight(tile,AITile.CORNER_E))
			{return AITile.GetCornerHeight(tile,AITile.CORNER_N);}
		else {return AITile.GetCornerHeight(tile,AITile.CORNER_E);}
	}
	if(dir=="SW") {	
		if(AITile.GetCornerHeight(tile,AITile.CORNER_S) > AITile.GetCornerHeight(tile,AITile.CORNER_W))
			{return AITile.GetCornerHeight(tile,AITile.CORNER_S);}
		else {return AITile.GetCornerHeight(tile,AITile.CORNER_W);}
	}
	if(dir=="NW") {	
		if(AITile.GetCornerHeight(tile,AITile.CORNER_N) > AITile.GetCornerHeight(tile,AITile.CORNER_W))
			{return AITile.GetCornerHeight(tile,AITile.CORNER_N);}
		else {return AITile.GetCornerHeight(tile,AITile.CORNER_W);}
	}
	if(dir=="SE") {	
		if(AITile.GetCornerHeight(tile,AITile.CORNER_S) > AITile.GetCornerHeight(tile,AITile.CORNER_E))
			{return AITile.GetCornerHeight(tile,AITile.CORNER_S);}
		else {return AITile.GetCornerHeight(tile,AITile.CORNER_E);}
	}
	return null;
}

function Tile::GetEntryHeight(tile,dir) {
	if(dir=="NE") {return Tile.GetExitHeight(tile,"SW");}
	if(dir=="SW") {return Tile.GetExitHeight(tile,"NE");}
	if(dir=="NW") {return Tile.GetExitHeight(tile,"SE");}
	if(dir=="SE") {return Tile.GetExitHeight(tile,"NW");}
	return null;	
}

/**
*	Used to run over existing one track rail, returns next build railtrack in array[2]
*	@param before_tile Used to see where the track comes from.
*	@param current_tile The tile previous to the returned 'next' tile.
*	@return [pre,work,next_tile,length_if_bridge/tunnel] first two null if no bridge/tunnel
*/
function Tile::GetNextRailTile(before_tile,current_tile) {
	local dir = Tile.GetDirectionFromTo(before_tile,current_tile);
	local t = Tile.GetRightTile(current_tile,dir);
	if(AIRail.IsRailDepotTile(t) || !AIRail.AreTilesConnected(before_tile,current_tile,t)) {
		t = Tile.GetForwardTile(current_tile,dir);
		if(AIRail.IsRailDepotTile(t) || !AIRail.AreTilesConnected(before_tile,current_tile,t)) {
			t = Tile.GetLeftTile(current_tile,dir);
			if(AIRail.IsRailDepotTile(t) || !AIRail.AreTilesConnected(before_tile,current_tile,t)) {
				MailAI.Info(0,"end of rail line");
				return null;
			}
		}
	}
	if(AIBridge.IsBridgeTile(t)) {
		local bridge_end = AIBridge.GetOtherBridgeEnd(t);
		local x = AIMap.GetTileX(t) - AIMap.GetTileX(current_tile);
		local y = AIMap.GetTileY(t) - AIMap.GetTileY(current_tile);
		local next = bridge_end + AIMap.GetTileIndex(x,y);
		while(AIBridge.IsBridgeTile(next) || AITunnel.IsTunnelTile(next)) {
			if(AIBridge.IsBridgeTile(next)) {
				bridge_end = AIBridge.GetOtherBridgeEnd(next);
			}
			if(AITunnel.IsTunnelTile(next)) {
				bridge_end = AITunnel.GetOtherTunnelEnd(next);
			}
			next = bridge_end + AIMap.GetTileIndex(x,y);
		}
		local bridge_length = 0;
		if(abs(AIMap.GetTileX(t) - AIMap.GetTileX(bridge_end))
			> abs(AIMap.GetTileY(t) - AIMap.GetTileY(bridge_end))) {
		 	bridge_length = abs(AIMap.GetTileX(t) - AIMap.GetTileX(bridge_end));
		} else {
			bridge_length = abs(AIMap.GetTileY(t) - AIMap.GetTileY(bridge_end));
		}
		local before_end = bridge_end - AIMap.GetTileIndex(x,y);
		return [before_end,bridge_end,next,bridge_length];
	}
	if(AITunnel.IsTunnelTile(t)) {
		local tunnel_end = AITunnel.GetOtherTunnelEnd(t);
		local x = AIMap.GetTileX(t) - AIMap.GetTileX(current_tile);
		local y = AIMap.GetTileY(t) - AIMap.GetTileY(current_tile);
		local next = tunnel_end + AIMap.GetTileIndex(x,y);
		while(AIBridge.IsBridgeTile(next) || AITunnel.IsTunnelTile(next)) {
			if(AIBridge.IsBridgeTile(next)) {
				tunnel_end = AIBridge.GetOtherBridgeEnd(next);
			}
			if(AITunnel.IsTunnelTile(next)) {
				tunnel_end = AITunnel.GetOtherTunnelEnd(next);
			}
			next = tunnel_end + AIMap.GetTileIndex(x,y);
		}
		local tunnel_length = 0;
		if(abs(AIMap.GetTileX(t) - AIMap.GetTileX(tunnel_end))
			> abs(AIMap.GetTileY(t) - AIMap.GetTileY(tunnel_end))) {
		 	tunnel_length = abs(AIMap.GetTileX(t) - AIMap.GetTileX(tunnel_end));
		} else {
			tunnel_length = abs(AIMap.GetTileY(t) - AIMap.GetTileY(tunnel_end));
		}
		local before_end = tunnel_end - AIMap.GetTileIndex(x,y);
		return [before_end,tunnel_end,next,tunnel_length];
	}
	return [null,null,t,1];
}

/**
*	Make sure left tile is the first param
*/
function Tile::IsBuildableSwitchTile(t1,t2) {
	if(t1 != null && t2 != null) {
		if(Tile.IsBuildableAlmostFlatTile(t1) && Tile.IsAlmostFlat(t2)
			&& AITile.GetMaxHeight(t1) == AITile.GetMaxHeight(t2)) {
				return true;	
			}
	}
	return false;
}

function Tile::IsBuildableStartSwitchTile(left_tile,right_tile) {
	if(left_tile==null || right_tile==null) {throw("null tile in IsBuildableStartSwitchTile");}
	if(!Tile.IsAlmostFlat(right_tile)){return false;}
	local left_right_dir = Tile.GetDirectionFromTo(left_tile,right_tile)
	if(!Tile.ConnectAtSameHeight(left_tile,right_tile,left_right_dir)) {return false;}
	if(Tile.IsAlmostFlat(left_tile)) {return true;}	
	local track_dir = Tile.GetLeftDirection(left_right_dir);
	local slope = AITile.GetSlope(left_tile);	
	switch(track_dir) {
		case "NE":
			if(slope == AITile.SLOPE_W) {return true;} 
			break;
		case "SE": 
			if(slope == AITile.SLOPE_N) {return true;}
			break;
		case "SW":
			if(slope == AITile.SLOPE_E) {return true;}
			break;
		case "NW":
			if(slope == AITile.SLOPE_S) {return true;}
			break;
		default: throw("Wrong direction in IsBuildableStartSwitchTile: "+dir);
	}
	return false;
}

function Tile::IsBuildableEndSwitchTile(left_tile,right_tile) {
	if(left_tile==null || right_tile==null) {throw("null tile in IsBuildableEndSwitchTile");}
	if(!Tile.IsAlmostFlat(right_tile)){return false;}
	local left_right_dir = Tile.GetDirectionFromTo(left_tile,right_tile)
	if(!Tile.ConnectAtSameHeight(left_tile,right_tile,left_right_dir)) {return false;}
	if(Tile.IsAlmostFlat(left_tile)) {return true;}	
	local track_dir = Tile.GetLeftDirection(left_right_dir);
	local slope = AITile.GetSlope(left_tile);
	switch(track_dir) {
		case "NE":
			if(slope == AITile.SLOPE_N) {return true;} 
			break;
		case "SE": 
			if(slope == AITile.SLOPE_E) {return true;}
			break;
		case "SW":
			if(slope == AITile.SLOPE_S) {return true;}
			break;
		case "NW":
			if(slope == AITile.SLOPE_W) {return true;}
			break;
		default: throw("Wrong direction in IsBuildableEndSwitchTile: "+dir);
	}
	return false;	
}

function Tile::IsBuildableRailTileSlope(tile,dir) {
	if(tile == null || dir == null) {return false;}
	if(Tile.IsBuildableAlmostFlatTile(tile)) {return true;}
	if(!AITile.IsBuildable(tile)) {return false;}
	if(AIRoad.IsRoadTile(tile)) {
		if(AIRoad.IsRoadTile(Tile.GetForwardTile(tile,dir))) {
			return false;//more then one is problems
		} else {return true;}//one road tile can easily be built over or demolished
	}
	if(AITile.GetSlope(tile) == AITile.SLOPE_STEEP) {return false;}
	if(Tile.WillSlopeUp(tile,dir) || Tile.WillSlopeDown(tile,dir)){return true;}
	return false;
}

function Tile::IsRailMine(tile) {
	if(AIRail.GetRailTracks(tile) != null && AICompany.IsMine(AITile.GetOwner(tile))) {return true;}
	return false;
}

function Tile::GetNeighbourRailCount(tile) {
	local tl = Tile.GetAdjacentTiles(tile);
	tl.Valuate(Tile.IsRailMine);
	tl.KeepValue(1);
	return tl.Count();
}