/* 
 *	Copyright 2013 Varen De Meersman
 *  This file is part of MailAI.
 *
 *  MailAI is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  MailAI is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with MailAI.  If not, see <http://www.gnu.org/licenses/>.
 */
 
 class Valuator {
	constructor() {}
}

function Valuator::HasHownManyStations(town_id) {
	local station_list = AIStationList(AIStation.STATION_ANY);//should only be own stations
	station_list.Valuate(AIStation.GetNearestTown);
	station_list.KeepValue(town_id);
	return station_list.Count();
}

/*
function Valuator::NeedsExtraService(stat) {
	local cargo_waiting = AIStation.GetCargoWaiting(stat,AICargo.CC_MAIL);
	local cargo_rating = AIStation.GetCargoRating(stat,AICargo.CC_MAIL);
	if(AIStation.HasStationType(stat,AIStation.STATION_TRAIN)
	&& cargo_waiting > (MailAI.min_train_cargo_waiting - AIController.GetSetting("level_of_easiness"))
	&& cargo_rating < MailAI.max_train_cargo_rating) {
		MailAI.Info(0,"Train station at cargo: "+cargo_waiting+ "("+cargo_rating+")");
		return 1;	
	}
	if(!AIStation.HasStationType(stat,AIStation.STATION_TRAIN)
	&& cargo_waiting > (MailAI.min_train_cargo_waiting - AIController.GetSetting("level_of_easiness"))
	&& cargo_rating < MailAI.max_truck_cargo_rating) {	
		MailAI.Info(0,"Truck station at cargo: "+cargo_waiting+ "("+cargo_rating+")");
		return 1;
	}
	return 0;
}*/

/**
*	Has neighbouring station within 6 tiles.
*	@param tile The tile to check.
*	@return 1 if there exists a neighbouring station otherwise 0
*/
function Valuator::HasNeighbouringStation(tile) {
	local around_list = AITileList();
	around_list.AddRectangle(tile + AIMap.GetTileIndex(-7, -7),tile + AIMap.GetTileIndex(7, 7));
	foreach (t, _ in around_list) { 
		if(AITile.IsStationTile(t) && AICompany.IsMine(AITile.GetOwner(t))) {
			return true;
		}	
	}
	around_list.Clear();
	around_list.AddRectangle(tile + AIMap.GetTileIndex(-2, -2),tile + AIMap.GetTileIndex(2, 2));
	foreach (t, _ in around_list) { 
		if(AITile.IsStationTile(t)) {
			return true;
		}	
	}
	return false;
}

function Valuator::IsGoodForTruckStop(tile,articulated) {
	if(Valuator.HasNeighbouringStation(tile)){return 0;}
	local cover = AITile.GetCargoProduction(tile,MailAI.GetMailID(),1,1,
				AIStation.GetCoverageRadius(AIStation.STATION_TRUCK_STOP));
	if(cover < (MailAI.min_truck_coverage + AIController.GetSetting("level_of_easiness"))) {return 0;}
	if(!Tile.IsBuildableAlmostFlatTile(tile)){
		if(articulated && AIRoad.IsRoadTile(tile)
			&& AITile.GetSlope(tile) == AITile.SLOPE_FLAT
			&& AIRoad.GetNeighbourRoadCount(tile) <= 2){return cover;}
		return 0;}
	return cover; 
}

/**
*	Only pass station tiles. Will check if coverage is still good and enough cargo waiting.
*/
function Valuator::IsReusableTruckStop(tile) {
	local cover = AITile.GetCargoProduction(tile,MailAI.GetMailID(),1,1,
				AIStation.GetCoverageRadius(AIStation.STATION_TRUCK_STOP));
	if(cover < (MailAI.min_truck_coverage + AIController.GetSetting("level_of_easiness"))) {return 0;}
	local cargo_waiting = AIStation.GetCargoWaiting(AIStation.GetStationID(tile),MailAI.GetMailID());
	if(cargo_waiting > MailAI.min_truck_cargo_waiting) {return 1;}
	return 0; 
}

function Valuator::IsGoodRoadDepotTile(tile) {
	if(!Tile.IsBuildableAlmostFlatTile(tile)){return 0;}
	if(AIRoad.GetNeighbourRoadCount(tile) == 0){return 0;}
	return 1;
}

function Valuator::IsGoodForTrainStation(tile) {
	if(Valuator.HasNeighbouringStation(tile)){return 0;}
	if(AITile.IsWaterTile(tile)){return 0;}
	//if(!Tile.IsBuildableAlmostFlatTile(tile)){return 0;}
	local cover = AITile.GetCargoProduction(tile,MailAI.GetMailID(),1,1,
				AIStation.GetCoverageRadius(AIStation.STATION_TRAIN));
	if(cover < (MailAI.min_train_coverage + AIController.GetSetting("level_of_easiness"))){return 0;}
	return cover; 
}

function Valuator::CanPullMail(engine) {		
	if(AIEngine.GetVehicleType(engine) == AIVehicle.VT_ROAD
	&& AIEngine.GetCapacity(engine) > 9) {
		if(AIEngine.CanRefitCargo(engine,MailAI.GetMailID())) {return 1;}
		if(AIEngine.GetCargoType(engine) == MailAI.GetMailID()) {return 1;}
	}
	if(AIEngine.GetVehicleType(engine) == AIVehicle.VT_RAIL
		&& AIEngine.CanPullCargo(engine,MailAI.GetMailID())) {
		if(AIEngine.IsWagon(engine)) {return 0;}
		if(AIEngine.GetWeight(engine) == 1) {return 0;}
		if(AIEngine.GetCapacity(engine) > 0) {
			if(AIEngine.CanRefitCargo(engine,MailAI.GetMailID())) {
				return 1;
			} else {return 0;}
		}
		return 1;	
	}
	return 0;
}

/**
*	Returns mail capacity. Works good for engines as they either have mail or 1/2passengers.
*	Is also used for wagons but need to check if this is then still correct.
*/
function Valuator::GetMailCapacity(engine) {
	//can't use the handy AIVehicle because it's not build yet,
	//most non mail wagons start out as passenger and need the cap halved
	if(AIEngine.GetCargoType(engine) == AICargo.CC_MAIL) {return AIEngine.GetCapacity(engine);}
	else {return abs(AIEngine.GetCapacity(engine)/2);}
}

/**
*	Valuator that returns a value determined by speed provided.
*	Higher value is better HorsePower.
*	@param engine The engine that is being valuated.
*	@param speed If speed needed is lower then the engine's max speed. May be null.
*	@return higher value is better HP
*/
function Valuator::HasEnoughPower(engine, opt_speed, bonus) {
	local eng_speed = AIEngine.GetMaxSpeed(engine);
	local wagon = EngineMaster.GetWagon(engine,null);//wagons may be faster then opt_speed
	if(wagon == null){return 0};
	if(opt_speed != null && opt_speed < eng_speed){eng_speed = opt_speed;}
	if(AIEngine.GetMaxSpeed(wagon) > 0 && AIEngine.GetMaxSpeed(wagon) < eng_speed) {
		eng_speed = AIEngine.GetMaxSpeed(wagon);
	}
	//kN should always be enough
	//To move 1ton up one slope -> 0,135kn, To move 1ton -> 0,035kN
	//if(AIEngine.GetMaxTractiveEffort(engine) < min_kN)
	//determine the horsepower	
	//weight * speed in mph * 0,08 = needed horsepower to handle 1 slope every x tiles for a train of length x tiles
	//amount of wagons is estimated on wagons and engine each being 1/2 of a tile long and fixed 3 long station
	local total_weight = (AIEngine.GetWeight(wagon)*5)+AIEngine.GetWeight(engine);
	local neededHP = abs(total_weight * eng_speed * 0.08);
	if(AIEngine.GetPower(engine) + bonus > neededHP) {return AIEngine.GetPower(engine);}
	return 0;
} 

/**
*	Gets running cost of provided truck and devides by capacity of that truck.
*	Punishes Articulated vehicles by MailAI.punish_articulated.
*/
function Valuator::GetTruckCostCapacity(eng) {
	local cost = AIEngine.GetRunningCost(eng);
	local cap = AIEngine.GetCapacity(eng);
	return abs(cost/cap);
}

function Valuator::CanRoadConnectToAlmostFlatTile(road,tile) {
	if(!Tile.IsBuildableAlmostFlatTile(tile)){return 0;}
	if(!(AITile.IsBuildable(road)||AIRoad.IsRoadTile(road))){return 0;}
	local c1,c2;
	if(AIMap.GetTileX(road) < AIMap.GetTileX(tile)){c1 = AITile.CORNER_S; c2 = AITile.CORNER_W;}
	if(AIMap.GetTileX(road) > AIMap.GetTileX(tile)){c1 = AITile.CORNER_N; c2 = AITile.CORNER_E;}
	if(AIMap.GetTileY(road) < AIMap.GetTileY(tile)){c1 = AITile.CORNER_S; c2 = AITile.CORNER_E;}
	if(AIMap.GetTileY(road) > AIMap.GetTileY(tile)){c1 = AITile.CORNER_N; c2 = AITile.CORNER_W;} 
	if(AITile.GetCornerHeight(road,c1) == AITile.GetMaxHeight(tile)
		|| AITile.GetCornerHeight(road,c2) == AITile.GetMaxHeight(tile)) {return 1;}
	return 0;	
}

function Valuator::HasSignal(tile) {
	local tl = Tile.GetAdjacentTiles(tile);
	tl.Valuate(AIRail.IsRailTile);
	tl.KeepValue(1);
	foreach(t, _ in tl) {
		if(AIRail.GetSignalType(tile,t) != AIRail.SIGNALTYPE_NONE){return true;}	
	}
	return false;
}