/**
 *    This file is part of OtviAI.
 *
 *    OtviAI is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    OtviAI is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with OtviAI.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2008-2013 Otto Visser.
 * Suggestions/comments and bugs can go to: otviAI@otvi.nl
 */

/**
 * list of tested NEWGrf's:
 * ECS (all vectors; performance can be improved)
 * FIRS
 * HEQS (combination with ECS is borked though)
 * Dutch Tram Set
 * New Tram tracks
 * Generic Tram Set
 * eGRVTS
 * 2cc
 * Industrial Stations Renewal
 * UK Renewal train set
 */

require("util.nut");
require("map.nut");

require("train.nut");
require("road.nut");
require("tram.nut");
require("marine.nut");
require("air.nut");

enum ServicedBy {
	TRAIN = 1,
	ROAD = 2,
	TRAM = 4,
	AIR = 8,
	BOAT = 16,
};

enum Direction {
	NE = 1,
	SE = 3,
	SW = 5,
	NW = 7,
};

class OtviAI extends AIController {
	MIN_DISTANCE = 58;	// ends at 8
	MAX_DISTANCE = 112;	// ends at 512
	MAXTICKS = 1024;	// TODO this is an initial guess value
	MAX_TRANSPORT_ATTEMPTS = 2;	// gets increased to transport count later; 2 allows for water and then road for instance
	TICKS_PER_DAY = 1;

	Map = null;

	transports = null;
	roadTransport = null;
	tramTransport = null;
	trainTransport = null;
	boatTransport = null;
	airTransport = null;

	reserveMoney = 0;
	biggestTown = -1;	
	firstIndustry = true;
	lastEventTick = 0;

	atLarge = false;
	tradeCenters = [null, null, null, null];	// NE, SE, SW, NW

	function Save() {
		// TODO
		return {};
	}

	function Load(version, data) {
		Debug("TODO: implement load and save");
	}

	function constructor() {
		::INSTANCE <- this;
		::breakdownLevel <- AIGameSettings.GetValue("difficulty.vehicle_breakdowns"); 

		local cargoList = AICargoList();
		foreach (cargo, val in cargoList) {
			if (AICargo.HasCargoClass(cargo, AICargo.CC_PASSENGERS) && AICargo.GetTownEffect(cargo) == AICargo.TE_PASSENGERS) {
				Debug("Found passenger cargo: " + AICargo.GetCargoLabel(cargo));
				::PASSENGER_ID <- cargo;
			}
			if (AICargo.HasCargoClass(cargo, AICargo.CC_MAIL) && AICargo.GetTownEffect(cargo) == AICargo.TE_MAIL) {
				Debug("Found mail cargo: " + AICargo.GetCargoLabel(cargo));
				::MAIL_ID <- cargo;
			}
		}

		Map = OtviMap();
		transports  = [];

		if (AIGameSettings.IsDisabledVehicleType(AIVehicle.VT_RAIL) || AIGameSettings.GetValue("vehicle.max_trains") == 0) {
			Warning("Trains are disabled!!!");
		} else {
			trainTransport = OtviTrAIn();
			transports.append(trainTransport);
		}

		if (AIGameSettings.IsDisabledVehicleType(AIVehicle.VT_ROAD) || AIGameSettings.GetValue("vehicle.max_roadveh") == 0) {
			Warning("Road vehicles are disabled!!!");
		} else { 
			roadTransport = OtviRoad();
			transports.append(roadTransport);
			if (AIRoad.IsRoadTypeAvailable(AIRoad.ROADTYPE_TRAM)) {
				tramTransport = OtviTram();
				transports.append(tramTransport);
			} else {
				Warning("To enable the use of trams load a tram newgrf!");
			}
		}

		if (AIGameSettings.IsDisabledVehicleType(AIVehicle.VT_WATER) || AIGameSettings.GetValue("vehicle.max_ships") == 0) {
			Warning("Water transport is disabled!!!");
		} else { 
			boatTransport = OtviMarine();
			transports.append(boatTransport);
		}

		if (AIGameSettings.IsDisabledVehicleType(AIVehicle.VT_AIR) || AIGameSettings.GetValue("vehicle.max_planes") == 0) {
			Warning("Air transport is disabled!!!");
		} else { 
			airTransport = OtviAIr();
			transports.append(airTransport);
		}

		local towns = AITownList();
		towns.Valuate(AITown.GetPopulation);
		towns.Sort(AIList.SORT_BY_VALUE, false);
		biggestTown = towns.Begin();

		// @Large part
		// FIXME TODO better way of determining
		if (GetVersion() == 305659905) {
			Warning("Running in an @Large version!")
			atLarge = true;
		} else {
			// Debug("game version: " + GetVersion());
		}
	}

	function Start() {
		if (!AICompany.SetName("OTVI")) {
			local i = 0;
			Map.setTownServed(biggestTown);	// prevent clusterfuck with other instance of OtviAI
			local towns = AITownList();
			towns.Valuate(AITown.GetPopulation);
			towns.Sort(AIList.SORT_BY_VALUE, false);
			biggestTown = towns.Begin();
			biggestTown = towns.Next();
			while (!AICompany.SetName("OTVI franchise #" + i)) {
				Map.setTownServed(biggestTown);	// prevent clusterfuck with other instance of OtviAI
				biggestTown = towns.Next();
				i++;
			}
		}
		AICompany.SetPresidentName("I.R. Boss");
		AICompany.SetPresidentGender(AICompany.GENDER_MALE);

		if (AIGameSettings.IsValid("difficulty.vehicle_breakdowns") && breakdownLevel != 0) {
			Warning("Breakdowns are on; enabling replacement of vehicles...");
			if (!AICompany.SetAutoRenewStatus(true))
				Warning("Couldn't enable auto renew; renewing the manual way");
			AICompany.SetAutoRenewMonths(breakdownLevel == 1 ? 6 : 0);
			AICompany.SetAutoRenewMoney(100000); //	TODO calculate!!!! this way it shouldn't interfere with normal building
		} else {
			AICompany.SetAutoRenewStatus(false);
		}

		TICKS_PER_DAY = GetTicksPerDay();

		foreach (transport in transports) {
			transport.updateEngines();
		}

		if (transports.len() == 0) {
			Error("Can't build anything, going into a coma...");
			while(true) {
				manageLoan();
				Sleep(1024);
			}
		}
		
		local canOperate = false;
		while(!canOperate) {
			foreach (transport in transports) {
				if (transport.isOperational()) {
					Debug("Transport operational: " + transport.getName());
					canOperate = true;
				} else {
					Warning("Tranport non-operational: " + transport.getName());
				}
			}

			checkEvents();	// wait for suitable equipment
			
			Sleep(TICKS_PER_DAY);
		}

		
		// the main control loop
		local curdate = AIDate.GetCurrentDate();
		local currentYear = AIDate.GetYear(curdate);
		local currentMonth = AIDate.GetMonth(curdate);
		local lastYear = currentYear;
		local endYear = currentYear + 50;	// last year to chance search radius for new town connections
		local lastExpansionMonth = currentMonth;	// TODO convert to daycount?
		local hqBuilt = false;

		// delay for two reasons: don't pick what other ai is picking as a first start
		// starting up is already tough enough :P
		// reason 2: When using ECS grf's; cargo acceptance starts 256 ticks after start
		while (currentMonth < 2) {
			curdate = AIDate.GetCurrentDate();
			currentMonth = AIDate.GetMonth(curdate);
			Sleep(TICKS_PER_DAY);
		}
		
		rondjeCheck();
		reserveMoney = 250;

		while(true){
			curdate = AIDate.GetCurrentDate();
			currentYear = AIDate.GetYear(curdate);
			currentMonth = AIDate.GetMonth(curdate);

			if(currentYear != lastYear) {
//				Warning("Small airport maintenance cost factor: " + AIAirport.GetMaintenanceCostFactor(AIAirport.AT_SMALL));
//				Warning("road maintenance cost factor: " + AIRoad.GetMaintenanceCostFactor(AIRoad.ROADTYPE_ROAD));
//				Warning("rail maintenance cost factor: " + AIRail.GetMaintenanceCostFactor(0));
				arrangeBalance(5000);	// in case we need to get rid of stations
				cleanUp();
				lastYear = currentYear;
				if (currentYear < endYear) {
					MAX_DISTANCE += 10;
					MAX_TRANSPORT_ATTEMPTS += 1;
					MIN_DISTANCE -= 1;
				}
			}
			
			// TEST
			/*
			local towns = AITownList();
			towns.Valuate(AITown.GetPopulation);
			towns.Sort(AIList.SORT_BY_VALUE, false);
			biggestTown = towns.Begin();
			while (!towns.IsEnd()) {
				Train.BuildTrainStation(biggestTown);
				biggestTown = towns.Next();
			}*/
			// END TEST

			// deal with stuffs before expanding
			checkEvents();	// building a sub takes long, so check again

			transports.sort(OtviTransport.enhanceSort);
			local available = getAvailableMoney();
			foreach (transport in transports) {
				local cost = transport.getEnhanceCost();
				if (transport.canEnhance() && available > cost && lastExpansionMonth != currentMonth) {
					arrangeBalance(cost);
					if (transport.enhance())
						lastExpansionMonth = currentMonth;
				}
			}

			checkEvents();	// expanding takes some time, check again

			transports.sort(OtviTransport.expansionSort);
			local cost = transports[0].getExpansionCost(::PASSENGER_ID, MAX_DISTANCE);
			if(getAvailableMoney() > cost) {
				Debug("We've got enough money to connect stuffs!");
				// TODO lazy evaluation?
				if (!connectIndustry()) {
					checkEvents();
					if (!connect2Towns()) {
						//if (!connectIndustry()) {
						checkEvents();
						Sleep(1);
					}
				}
			}
						
			checkEvents();

			if (!hqBuilt) {
				buildHQ(biggestTown);
				hqBuilt = true;
			}

			Sleep(1);
		}
	}

	function rondjeCheck() {
		::RONDJE_ALERT <- false;
		for (local i = AICompany.COMPANY_FIRST; i < AICompany.COMPANY_LAST; i++) {
			local j = AICompany.ResolveCompanyID(i);
			if (!AICompany.IsMine(j) && j != AICompany.COMPANY_INVALID) {
				local comName = AICompany.GetName(j);
				Warning("Found competitor: " + comName);
				if (comName == "Rondje Om De Kerk") {
					Error("Enabling Rondje avoidance");
					::RONDJE_ALERT <- true;
				}
			}
		}
	}
	
	function checkEvents() {
		local diffTick = AIController.GetTick() - lastEventTick;
		local skipLong = diffTick > MAXTICKS;
		local recheckEngines = false;
		if (skipLong) {
			Warning("Last action took too long, skipping events that take long time");
		}
		manageLoan();
		while (AIEventController.IsEventWaiting()) {
			local e = AIEventController.GetNextEvent();
			//Debug("Got an eventA: " + e);
			switch (e.GetEventType()) {
				case AIEvent.ET_INVALID:
					Error("Received an invalid event?!");
					break;
				case AIEvent.ET_TEST:
					Error("Received a test event?!");
					break;
				case AIEvent.ET_SUBSIDY_OFFER:
					local ec = AIEventSubsidyOffer.Convert(e);
					local s = ec.GetSubsidyID();
					Debug("New subsidy detected: " + s);
					if (!AISubsidy.IsValidSubsidy(s)) {
						Warning("Invalid (already expired?) subsidy detected; ignoring");
					} else if (AISubsidy.IsAwarded(s)) {
						Warning("Subsidy already awarded to " + AICompany.GetName(AISubsidy.GetAwardedTo(s)));
					} else if (skipLong) {
						Warning("Ignoring; we are too busy already...");
					} else {
						local cost = transports[0].getExpansionCost(AISubsidy.GetCargoType(s), 70);
						if (getAvailableMoney() > cost) {	// current max manhattan distance for subs
							Debug("We've got enough money to get the subsidized route!");
							arrangeBalance(cost);
							connectSubsidy(s);
						} else {
							Warning("Not enough money to get this one now.");
						}
					}
					break;
				case AIEvent.ET_SUBSIDY_OFFER_EXPIRED:
					Warning("A subsidy offer expired");
					break;
				case AIEvent.ET_SUBSIDY_AWARDED:
					local ec = AIEventSubsidyAwarded.Convert(e);
					local s = ec.GetSubsidyID();
					if (AICompany.IsMine(AISubsidy.GetAwardedTo(s)))
						Debug("We got the subsidy!!!!");
					else
						Warning("We failed to win a subsidy!!!! Company that won: " + AICompany.GetName(AISubsidy.GetAwardedTo(s)));
					break;
				case AIEvent.ET_SUBSIDY_EXPIRED:
					Warning("A subsidy ended");
					break;
				case AIEvent.ET_ENGINE_PREVIEW:
					local ec = AIEventEnginePreview.Convert(e);
					Warning("We can try a new engine: " + ec.GetName());
					local accepted = ec.AcceptPreview();	// the new engine functions will determine whether it's a usefull one or not
					Warning("Accepted the new engine offer: " + accepted);
					if (accepted)
						recheckEngines = true;
					break;
				case AIEvent.ET_COMPANY_NEW:
					local ec = AIEventCompanyNew.Convert(e);
					local name = AICompany.GetName(ec.GetCompanyID());
					Warning("New competitor: " + name);
					rondjeCheck();
					break;
				case AIEvent.ET_COMPANY_IN_TROUBLE:
					local ec = AIEventCompanyInTrouble.Convert(e);
					if (AICompany.IsMine(ec.GetCompanyID())) {
						reserveMoney += 2500;
						Error("We seem to be in trouble!!! Increasing reserves to " + reserveMoney);
					} else {
						Warning("Company in trouble: " + AICompany.GetName(ec.GetCompanyID()));
					}
					break;
				case AIEvent.ET_COMPANY_ASK_MERGER:
					local ec = AIEventCompanyAskMerger.Convert(e);
					local comID = ec.GetCompanyID();
					if (AICompany.ResolveCompanyID(comID) == AICompany.COMPANY_INVALID) {
						Warning("There was a company for sale, but it's already bankrupt it seems...");
					} else {
						Warning("For sale: " + AICompany.GetName(comID) + ", for: " + ec.GetValue());
						if (ec.GetValue() < getAvailableMoney()) {
							arrangeBalance(ec.GetValue());
							Warning("Buying " + AICompany.GetName(comID));
							if (ec.AcceptMerger()) {
								Warning("Succesfully bought it!");
								Warning("Cleaning up in old company... firing people, selling stuffs....");
								cleanUp();
							} else {
								Error("Failed to buy");
							}
						} else {
							Warning("Too expensive, not buying it");
						}
					}
					break;
				case AIEvent.ET_COMPANY_MERGER:
					local ec = AIEventCompanyMerger.Convert(e);
					Warning("Company " + AICompany.GetName(ec.GetNewCompanyID()) + " bought " + ec.GetOldCompanyID());
					break;
				case AIEvent.ET_COMPANY_BANKRUPT:
					local ec = AIEventCompanyBankrupt.Convert(e);
					local name = AICompany.GetName(ec.GetCompanyID());
					Warning("A company has gone bankrupt");
					rondjeCheck();
					break;
				case AIEvent.ET_VEHICLE_CRASHED:
					local ec = AIEventVehicleCrashed.Convert(e);
					local v  = ec.GetVehicleID();
					local reason = ec.GetCrashReason();
					Warning("We have a crashed vehicle (" + v + "), due to: " + reason);
					// TODO: Handle the crashed vehicle; aka: clone?
					break;
				case AIEvent.ET_VEHICLE_LOST:
					local ec = AIEventVehicleLost.Convert(e);
					local v  = ec.GetVehicleID();
					Warning("A vehicle got lost! " + AIVehicle.GetName(v));
					// TODO: nuke vehicle?? recheck routes?
					break;
				case AIEvent.ET_VEHICLE_WAITING_IN_DEPOT:
					local ec = AIEventVehicleWaitingInDepot.Convert(e);
					local v  = ec.GetVehicleID();
					if (!AIVehicle.IsValidVehicle(v))
						Warning("Vehicle in depot event; but vehicle already gone/autoreplaced?");
					else {
						if (AIVehicle.GetAgeLeft(v) < 0) {
							Debug("A vehicle arrived to be replaced: (" + v + ") "+ AIVehicle.GetName(v));
							arrangeBalance(AIEngine.GetPrice(AIVehicle.GetEngineType(v)));
							// TODO don't clone, but get newer type?
							local vehicle = AIVehicle.CloneVehicle(AIVehicle.GetLocation(v), v, false);
							if (AIVehicle.IsValidVehicle(vehicle)) {
								AIVehicle.StartStopVehicle(vehicle);
								Debug("Replaced: " + vehicle);
							} else {
								Error("Couldn't replace vehicle: " + AIError.GetLastErrorString());
								// TODO can't clone? make new!
							}
						} else {
							Debug("A vehicle arrived in the depot for selling: " + AIVehicle.GetName(v));
						}
						if (!AIVehicle.SellVehicle(v))
							Warning("Couldn't sell vehicle: " + AIError.GetLastErrorString());
					}
					break;
				case AIEvent.ET_VEHICLE_UNPROFITABLE:
					// TODO: evaluate on one more year?
					local ec = AIEventVehicleUnprofitable.Convert(e);
					local v  = ec.GetVehicleID();
					if (AIVehicle.IsValidVehicle(v)) {
						if (AIVehicle.GetProfitThisYear(v) > 0) {
							Warning("Unprofitable vehicle detected; but seems to be making a profit now; giving it benefit of doubt: " + AIVehicle.GetName(v));
						} else {
							Warning("Unprofitable vehicle detected; sending to depot: " + AIVehicle.GetName(v));
							if (AIVehicle.IsStoppedInDepot(v)) {
								Warning("It is already waiting in depot?!");
							} else {
								AIVehicle.SendVehicleToDepot(v);
								local c = AIOrder.GetOrderCount(v);
								local i;
								for (i = 0; i < c; i++) {
									if (!AIOrder.RemoveOrder(v, 0))
										Warning("Couldn't remove order " + i + " from " + AIVehicle.GetName(v));
								}
							}
						}
					} else {
						Warning("Unprofitable vehicle detected, but already autoreplaced?");
					}
					break;
				case AIEvent.ET_INDUSTRY_OPEN:
					// TODO add to map lists?
					local ec = AIEventIndustryOpen.Convert(e);
					Warning("A new industry came to life: " + AIIndustry.GetName(ec.GetIndustryID()));
					break;	// it will automatically be considered, so event handled.
				case AIEvent.ET_INDUSTRY_CLOSE: 
					local ec = AIEventIndustryClose.Convert(e);
					local indID = ec.GetIndustryID();
					if (AIIndustry.IsValidIndustry(indID)) {
						Warning("An industry is going to close: " + AIIndustry.GetName(indID));
						if (Map.isIndustryServed(indID))
							Error("TODO clean up our stations and trucks, we are/were servicing it!");
					} else {
						Error("An industry closed, but can't check which one anymore...");
					}
					break;
				case AIEvent.ET_ENGINE_AVAILABLE:
					local ec = AIEventEngineAvailable.Convert(e);
					Debug("Scheduling a recheck of engines; new engine available: " + AIEngine.GetName(ec.GetEngineID()));
					recheckEngines = true;
					break;
				case AIEvent.ET_STATION_FIRST_VEHICLE: 
					local ec = AIEventStationFirstVehicle.Convert(e);
					Debug(AIStation.GetName(ec.GetStationID()) + " got it's first visitor: " + AIVehicle.GetName(ec.GetVehicleID()));
					break;
				case AIEvent.ET_DISASTER_ZEPPELINER_CRASHED: 
					local ec = AIEventDisasterZeppelinerCrashed.Convert(e);
					Warning("A zeppeliner crashed on " + AIStation.GetName(ec.GetStationID()));
					break;
				case AIEvent.ET_DISASTER_ZEPPELINER_CLEARED: 
					local ec = AIEventDisasterZeppelinerCleared.Convert(e);
					Warning("The crashed zeppeliner and the blood have been removed from the runway on  " + AIStation.GetName(ec.GetStationID()));
					break;
				case AIEvent.ET_TOWN_FOUNDED: 
					local ec = AIEventTownFounded.Convert(e);
					Warning("A new town was founded; gratz mayor of " + AITown.GetName(ec.GetTownID()));
					break;	// TODO add to map list??
				case AIEvent.ET_AIRCRAFT_DEST_TOO_FAR: 
					// TODO
					Warning("An aircraft has a destination that is out of reach!");
					break;
				case AIEvent.ET_ADMIN_PORT:
					Error("An interesting event: admin port something?!");
					break;
				case AIEvent.ET_WINDOW_WIDGET_CLICK: 
					Error("An interesting event: window widget clicked?!");
					break;
				case AIEvent.ET_GOAL_QUESTION_ANSWER:
					Error("An interesting event: goal question answered?!");
					break;
				default:
					if (atLarge) {
						switch (e.GetEventType()) {
							case AIEvent.ET_TRADE_PROPOSED:
								local ec = AIEventTradeProposed.Convert(e);
								Warning("Another player suggested a trade; accepting: " + ec.GetPlayerName());
								// TODO check for delivery point
								// TODO smart stuff :P
								ec.AcceptTrade();
								break;
							case AIEvent.ET_NEW_TRADE:
								local ec = AIEventNewTrade.Convert(e);
								Info("New trade active");
								if (skipLong) {
									Warning("Ignoring; too busy...");
								} else {
									// TODO smart stuff :P
									buildReceivingTrade(ec.GetDirection(), ec.GetCargoID());
								}
								break;
							default:
								Error("Unhandled event:" + e);
						}
					} else {
						// Cant' happen?
						Error("Unhandled event:" + e);
					}
			}
			manageLoan();
		}
		if (recheckEngines) {
			Warning("Engine recheck triggered...");
			foreach (transport in transports)
				transport.updateEngines();
		}

		lastEventTick = AIController.GetTick();
	}

	function buildReceivingTrade(direction, cargoID) {
		Debug("Building for trading in direction: " + direction);
		local place1 = buildTC(direction);
		Debug("Looking for cargo acceptance in the neigbourhood... " + AICargo.GetCargoLabel(cargoID));

		// TODO FIXME duplicate code with connectIndustry
		local deliveryPoint;
		local destIsTown;
		local distance;
		if (AICargo.GetTownEffect(cargoID) > 0) {
			destIsTown = true;
			Debug("Cargo wants to go to town...");
			local townTiles = AITileList();
			SafeAddRectangle(townTiles, place1, MAX_DISTANCE, MIN_DISTANCE);
			townTiles.Valuate(AITile.GetCargoAcceptance, cargoID, 1, 1, 4);
			townTiles.RemoveBelowValue(8);
			townTiles.Valuate(townIncomeSort, place1, cargoID);
			townTiles.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);
			local townTile = townTiles.Begin();
			if (townTiles.GetValue(townTile) == 0) {
				Warning("This industry has no viable delivery options now");
				return false;
			}
			deliveryPoint = AITile.GetClosestTown(townTile);
			distance = AITile.GetDistanceManhattanToTile(place1, AITown.GetLocation(deliveryPoint));
			Debug("Chosen delivery town: " + AITown.GetName(deliveryPoint) + " at distance: " + distance);
		} else {
			destIsTown = false;
			Debug("Looking for suitable dropoff industry...");
			local industries = AIIndustryList();
			industries.Valuate(AIIndustry.IsCargoAccepted, cargoID);
			industries.KeepValue(AIIndustry.CAS_ACCEPTED);
			industries.Valuate(cargoIncomeSort, place1, cargoID);
			industries.Sort(AIList.SORT_BY_VALUE, false);
			deliveryPoint = industries.Begin();			
			if(industries.GetValue(deliveryPoint) == 0) {
				Warning("This industry has no viable delivery options now");
				return false;
			}

			local place2 = AIIndustry.GetLocation(deliveryPoint);
			distance = AIMap.DistanceManhattan(place1, place2);
			while (distance > MAX_DISTANCE || distance < MIN_DISTANCE) {
				if (industries.IsEnd()) {
					Warning("Could not find a suitable dropoff industry nearby");
					return false;
				}
				deliveryPoint = industries.Next();
				place2 = AIIndustry.GetLocation(deliveryPoint);
				distance = AIMap.DistanceManhattan(place1, place2);
			}
			Debug("Chosen delivery industry: " + AIIndustry.GetName(deliveryPoint) + " at distance: " + distance);
		}

		::startLoc <- place1;
		::endLoc <- deliveryPoint;
		::cargo_ID <- cargoID;
		transports.sort(OtviTransport.cargoTransportSort);
		local connected = false;
		local i = 0;
		while (!connected && i < transports.len() && i < MAX_TRANSPORT_ATTEMPTS) {
			local cost = transports[i].getExpansionCost(cargoID, distance);
			if (cost < getAvailableMoney()) {
				//arrangeBalance(cost)
				Warning("Trying to connect by " + transports[i].getName());
				connected = transports[i].connectTrade(place1, deliveryPoint, cargoID, true, destIsTown);
				if (!connected) {
					Warning("Couldn't connect with " + transports[i].getName());
					checkEvents();
				}
			} else {
				Warning("Skipping transport due to high cost: " + transports[i].getName());
			}
			i++;
		}
		if (connected) {
			Debug("Connected trade to neighbour.");
			return true;
		} else {
			Warning("Could not connect this trade");
			return false;
		}
	}

	function cargoValuate(industry, cargoID) {
		return AIIndustry.GetLastMonthProduction(industry, cargoID) - AIIndustry.GetLastMonthTransported(industry, cargoID);
	}

	// TODO: put in same file as townincomesort and subtract maintenance!

	function townIncomeSort(target, startTile, cargoID) {
		local distance = AIMap.DistanceManhattan(startTile, AITown.GetLocation(target));
		local days = distance/7 + 1;	//underestimate
		return AICargo.GetCargoIncome(cargoID, distance, days) * (365/days);
	}

	function cargoIncomeSort(target, startTile, cargoID) {
		local distance = AIMap.DistanceManhattan(startTile, AIIndustry.GetLocation(target));
		local days = distance/7 + 1;	//underestimate
		return AICargo.GetCargoIncome(cargoID, distance, days) * (365/days);
	}
	
	function getAvailableMoney() {
		return AICompany.GetMaxLoanAmount() - AICompany.GetLoanAmount() + AICompany.GetBankBalance(AICompany.COMPANY_SELF) - reserveMoney;
	}

	function manageLoan() {
		//repay loan
		local balance = AICompany.GetBankBalance(AICompany.COMPANY_SELF);
		balance -= reserveMoney;	// have reserveMoney in bank to prevent bankruptcy etc
		local balanceMod = (balance / AICompany.GetLoanInterval()) * AICompany.GetLoanInterval();
		local loan = AICompany.GetLoanAmount();
		local max = AICompany.GetMaxLoanAmount();
		
		if (balance < 0) {
			// not good... we lost money?
			arrangeBalance(reserveMoney);
			balance = AICompany.GetBankBalance(AICompany.COMPANY_SELF);
			balanceMod = (balance / AICompany.GetLoanInterval()) * AICompany.GetLoanInterval();
			loan = AICompany.GetLoanAmount();
			return;
		}
		
		if (loan > 0 && balanceMod > 0) {
			if (balanceMod > loan)
				balanceMod = loan;
			AICompany.SetLoanAmount(loan - balanceMod);
		}
	}

	function buildHQ(town) {
		Debug("building HQ...");
		// Find empty 2x2 square as close to town centre as possible
		local curRange = 2;
		local maxRange = sqrt(AITown.GetPopulation(town)/100) + 5; //TODO check value correctness
		local area = AITileList();

		arrangeBalance(500);
		
		while (curRange < maxRange) {
			SafeAddRectangle(area, AITown.GetLocation(town), curRange, curRange - 2); 
			area.Valuate(AITile.IsBuildableRectangle, 2, 2);
			area.KeepValue(1);
			if (area.Count()) {
				foreach (tile, val in area) {
					if (AICompany.BuildCompanyHQ(tile)) {
						// TODO different name if franchise
						AISign.BuildSign(tile, ("HQ OTVI"));
						return tile;
					}
				}
				curRange++;
			} else {
				curRange++;
			}
		}
		Warning("No possible HQ location found");
		return null;
	}

	/**
	 * TODO FIXME have border as parameter
	 */
	function buildTC(direction) {
		Debug("building a Trade Center..." + direction);
		// Find empty 7x7 square along the specified border
		local area = AITileList();
		local text;
		
		arrangeBalance(20000);	// TODO

		switch (direction) {
			case Direction.NE:
				area.AddRectangle(AIMap.GetTileIndex(1, 2), AIMap.GetTileIndex(1, AIMap.GetMapSizeY() - 9));
				text = "TC OTVI NE";
				break;
			case Direction.SE:
				area.AddRectangle(AIMap.GetTileIndex(2, AIMap.GetMapSizeY() - 8), AIMap.GetTileIndex(AIMap.GetMapSizeX() - 9, AIMap.GetMapSizeY() - 8));
				text = "TC OTVI SE";
				break;
			case Direction.SW:
				area.AddRectangle(AIMap.GetTileIndex(AIMap.GetMapSizeX() - 8, AIMap.GetMapSizeY() - 9), AIMap.GetTileIndex(AIMap.GetMapSizeX() - 8, 2));
				text = "TC OTVI SW";
				break;
			case Direction.NW:
				area.AddRectangle(AIMap.GetTileIndex(AIMap.GetMapSizeX() - 9, 1), AIMap.GetTileIndex(2, 1));
				text = "TC OTVI NW";
				break;
			default:
				Error("TC location not yet supported: " + direction);
				return;
		}

		//AISign.BuildSign(AIMap.GetTileIndex(1, 1), ("TC start"));
		//AISign.BuildSign(AIMap.GetTileIndex(1, AIMap.GetMapSizeY() - 8), ("TC end"));
		area.Valuate(AITile.IsBuildableRectangle, 7, 7);
		area.KeepValue(1);
		if (area.Count()) {
			Debug("Possible TC locations: " + area.Count());
			foreach (tile, val in area) {
				if (AICompany.BuildTradeCenter(tile)) {
					// TODO different name if franchise
					AISign.BuildSign(tile, text);
					return tile;
				}
			}
			Error("Couldn't place TC: " + AIError.GetLastErrorString());
		}

		Warning("No possible Trade Center location found for this direction");
		return null;
	}

	// TODO: open up possibilities somehow?
	// TODO: cleanup left over infrastructure (elect. rail while there's no elect.trains anymore)
	function cleanUp() {
		Debug("Scanning for vehicles that somehow missed replacement or selling...");
		local vehicles = AIVehicleList();
		vehicles.Valuate(AIVehicle.IsStoppedInDepot);
		vehicles.KeepValue(1);
		foreach(vehicle, i in vehicles) {
			Warning("Selling ignored vehicle in depot: " + AIVehicle.GetName(vehicle));
			if (!AIVehicle.SellVehicle(vehicle))
				Error("Couldn't sell vehicle " + vehicle);
		}
		if (breakdownLevel > 0) {
			Debug("Breakdowns are on, scanning for vehicles that need renewal...");
			vehicles = AIVehicleList();
			vehicles.Valuate(AIVehicle.GetAgeLeft);
			vehicles.KeepBelowValue(0);
			// limit to the ones making a profit, the others are already taken care of by being unprofitable
			// and double gotodepot is like double negation
			vehicles.Valuate(AIVehicle.GetProfitLastYear);
			vehicles.KeepAboveValue(0);
			vehicles.Valuate(AIVehicle.GetProfitThisYear);
			vehicles.KeepAboveValue(0);

			foreach(vehicle, i in vehicles) {
				Warning("Sending old vehicle to depot for replacement: " + AIVehicle.GetName(vehicle));
				AIVehicle.SendVehicleToDepot(vehicle);
			}
		}

		Map.cleanUp();
	}

	function connect2Towns() {
		// look for biggest towns
		local townlist = AITownList();
		townlist.Valuate(Map.isTownServed);
		townlist.KeepValue(0);
		townlist.Valuate(Map.getTownPotential);
		townlist.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);
		if (townlist.IsEmpty()) {
			Debug("No more towns left");
			return false;
		}

		local previousBigTown = biggestTown;
		biggestTown = townlist.Begin();
		
		Debug("biggest town: " + AITown.GetName(biggestTown));
		local bigLoc = AITown.GetLocation(biggestTown);
		// find connecting town between MIN and MAX manhattan distance
		townlist = AITownList();
		townlist.Valuate(Map.incomeSort, bigLoc, ::MAIL_ID);
		townlist.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);
		local connectTown = townlist.Begin();
		local distance = AITile.GetDistanceManhattanToTile(bigLoc, AITown.GetLocation(connectTown));
		while (distance > MAX_DISTANCE || distance < MIN_DISTANCE || connectTown == previousBigTown) {
			//Debug("skipping town: " + AITown.GetName(connectTown) + " at distance " + AITile.GetDistanceManhattanToTile(bigLoc, AITown.GetLocation(connectTown)) + " value: " + townlist.GetValue(connectTown));
			if (townlist.IsEnd()) {	// TODO: add check like: OR skipped 20 or value < maxvalue/4 or something
				Debug("Out of towns");
				return false; // too bad, out of towns
			}
			connectTown = townlist.Next();
			distance = AITile.GetDistanceManhattanToTile(bigLoc, AITown.GetLocation(connectTown));
		}
		
		Debug("Connecting town: " + AITown.GetName(connectTown) + " at a manhattan distance of " + AITile.GetDistanceManhattanToTile(bigLoc, AITown.GetLocation(connectTown)) + " value: " + townlist.GetValue(connectTown));

		::startLoc <- biggestTown;
		::endLoc <- connectTown;
		transports.sort(OtviTransport.townTransportSort);// : (biggestTown, connectTown);
		local connected = false;
		local i = 0;
		while (!connected && i < transports.len()) {
			local cost = transports[i].getExpansionCost(::PASSENGER_ID, distance);
			if (cost < getAvailableMoney()) {
				//arrangeBalance(cost)
				Warning("Trying to connect by " + transports[i].getName());
				connected = transports[i].connect2Towns(biggestTown, connectTown);
				if (!connected) {
					Warning("Couldn't connect with " + transports[i].getName());
					checkEvents();
				}
			} else {
				Warning("Skipping transport due to high cost: " + transports[i].getName());
			}
			i++;
		}
		if (connected) {
			Debug("Connected.");
			Map.setTownServed(biggestTown);
		} else {
			Warning("Could not connect town with any available transport means; marking town...");
			Map.setTownServed(biggestTown, -1);
		}

		/*if (AITown.GetPopulation(biggestTown) < 1000 || AITown.GetPopulation(connectTown) < 1000)
			return connect2TownsByRoad(biggestTown, connectTown);
		else {
			if(connect2TownsByRail(biggestTown, connectTown) == false) {
				Warning("Train failed for this route, trying busses instead...");
				return connect2TownsByRoad(biggestTown, connectTown);	// try road if rail fails
			}
		}*/
	}

	function connectIndustry() {
		local cargoTypeList = AIIndustryTypeList();
		//Debug("cargotypelist count: " + cargoTypeList.Count());
		// remove consumer-only like power plant:
		foreach (typ, val in cargoTypeList) {
			if (AIIndustryType.GetProducedCargo(typ).Count() == 0) {
				cargoTypeList.RemoveItem(typ);
				// Debug("Industry type has no produced cargo " + AIIndustryType.GetName(typ));
			}
		}

		// for the first industry, the only choice is raw
		if (firstIndustry) {
			cargoTypeList.Valuate(AIIndustryType.IsRawIndustry);
			cargoTypeList.KeepValue(1);
		}

		local numberOfIndustries = cargoTypeList.Count();
		//local rand = abs(rand()) % cargoTypeList.Count();
		local rand = AIBase.RandRange(numberOfIndustries); //new command to pick a random industry:
		cargoTypeList.KeepBottom(rand + 1);
		local cargoType = cargoTypeList.Begin();
		Debug("Randomly picked industry type: " + AIIndustryType.GetName(cargoType) + ": " + rand + " out of " + (numberOfIndustries));
		
		local industries = AIIndustryList();
		industries.Valuate(AIIndustry.GetIndustryType);
		industries.KeepValue(cargoType);
		local cargoList = AIIndustryType.GetProducedCargo(cargoType);
		local cargoID = cargoList.Begin();	// TODO: simplified
		while (!cargoList.IsEnd() && AIBase.Chance(50, 100)) {
			Debug("Chosing other cargo due to random chance");
			cargoID = cargoList.Next();
		}
		Debug("Chosen cargo type: " + AICargo.GetCargoLabel(cargoID));

		industries.Valuate(Map.isIndustryServed);
		industries.KeepValue(0);
		industries.Valuate(cargoValuate, cargoID);
		industries.RemoveValue(0);
		if (industries.IsEmpty()) {
			Warning("This industry has no or barely active producers now");
			return false;
		}
		industries.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);
		local theChosenOne = industries.Begin();
		//local engine = getEngine(cargoID);

		// This way, we don't try to always make the same route that might fail for a given cargo
		while (!industries.IsEnd() && AIBase.Chance(50, 100)) {
			Debug("Picking next producer due to random chance...");
			theChosenOne = industries.Next();
		}

		local chosenProd = industries.GetValue(theChosenOne);
		Debug("Chosen industry: " + AIIndustry.GetName(theChosenOne) + " with a production of: " + chosenProd);
		local place1 = AIIndustry.GetLocation(theChosenOne);

		local deliveryPoint;
		local destIsTown;
		local distance;
		local place2;
		if (AICargo.GetTownEffect(cargoID) > 0) {
			destIsTown = true;
			Debug("Cargo wants to go to town...");
			local townTiles = AITileList();
			SafeAddRectangle(townTiles, place1, MAX_DISTANCE, MIN_DISTANCE);
			townTiles.Valuate(AITile.GetCargoAcceptance, cargoID, 1, 1, 4);
			townTiles.RemoveBelowValue(8);
			townTiles.Valuate(townIncomeSort, place1, cargoID);
			townTiles.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);
			local townTile = townTiles.Begin();
			if (townTiles.GetValue(townTile) == 0) {
				Warning("This industry has no viable delivery options now");
				return false;
			}
			deliveryPoint = AITile.GetClosestTown(townTile);
			place2 = AITown.GetLocation(deliveryPoint);
			distance = AITile.GetDistanceManhattanToTile(place1, place2);
			Debug("Chosen delivery town: " + AITown.GetName(deliveryPoint) + " at distance: " + distance);
		} else {
			destIsTown = false;
			Debug("Looking for suitable dropoff industry...");
			industries = AIIndustryList();
			industries.Valuate(AIIndustry.IsCargoAccepted, cargoID);
			industries.KeepValue(AIIndustry.CAS_ACCEPTED);
			industries.Valuate(cargoIncomeSort, place1, cargoID);
			industries.Sort(AIList.SORT_BY_VALUE, false);
			deliveryPoint = industries.Begin();			
			if(industries.GetValue(deliveryPoint) == 0) {
				Warning("This industry has no viable delivery options now");
				return false;
			}

			place2 = AIIndustry.GetLocation(deliveryPoint);
			distance = AIMap.DistanceManhattan(place1, place2);
			while (distance > MAX_DISTANCE || distance < MIN_DISTANCE) {
				if (industries.IsEnd()) {
					Warning("Could not find a suitable dropoff industry nearby");
					return false;
				}
				deliveryPoint = industries.Next();
				place2 = AIIndustry.GetLocation(deliveryPoint);
				distance = AIMap.DistanceManhattan(place1, place2);
			}
			Debug("Chosen delivery industry: " + AIIndustry.GetName(deliveryPoint) + " at distance: " + distance);
		}

		::startLoc <- place1;
		::endLoc <- place2;
		::cargo_ID <- cargoID;
		transports.sort(OtviTransport.cargoTransportSort);
		local connected = false;
		local i = 0;
		while (!connected && i < transports.len()) {
			local cost = transports[i].getExpansionCost(cargoID, distance);
			if (cost < getAvailableMoney()) {
				//arrangeBalance(cost)
				Warning("Trying to connect by " + transports[i].getName());
				connected = transports[i].connectIndustry(theChosenOne, deliveryPoint, cargoID, destIsTown);
				if (!connected) {
					Warning("Couldn't connect with " + transports[i].getName());
					checkEvents();
				}
			} else {
				Warning("Skipping transport due to high cost: " + transports[i].getName());
			}
			i++;
		}
		if (connected) {
			if (firstIndustry)
				firstIndustry = false;
			Debug("Connected.");
			Map.setIndustryServed(theChosenOne);
			return true;
		} else {
			Warning("Could not connect this industry");
			return false;
		}
	}

	function connectSubsidy(subID) {		
		if (!AISubsidy.IsValidSubsidy(subID)) {
			Error("Invalid subsidy detected");
			return false;
		}
		if (AISubsidy.GetSourceType(subID) == AISubsidy.SPT_TOWN) {
			local cargo = AISubsidy.GetCargoType(subID);
			local town1 = AISubsidy.GetSourceIndex(subID);
			local town2 = AISubsidy.GetDestinationIndex(subID);
			Debug("Connecting subsidized town: " + AITown.GetName(town1) + " to: " + AITown.GetName(town2));
			::startLoc <- town1;
			::endLoc <- town2;
			transports.sort(OtviTransport.townTransportSort);
			local connected = false;
			local i = 0;
			while (!connected && i < transports.len()) {
				connected = transports[i].connect2Towns(town1, town2, cargo);
				i++;
			}
			if (connected) {
				Debug("Connected.");
			} else {
				Warning("Could not connect town with any available transport means; marking town...");
			}
		} else if (AISubsidy.GetSourceType(subID) == AISubsidy.SPT_INDUSTRY) {
			// provided:
			local source = AISubsidy.GetSourceIndex(subID);
			local dest = AISubsidy.GetDestinationIndex(subID);
			local cargo = AISubsidy.GetCargoType(subID);

			local loc1 = AIIndustry.GetLocation(source);
			local loc2;
			local destIsTown;
			if (AISubsidy.GetDestinationType(subID) == AISubsidy.SPT_TOWN) {
				destIsTown = true;
				loc2 = AITown.GetLocation(dest);
			} else {
				destIsTown = false;
				loc2 = AIIndustry.GetLocation(dest);
			}
			local distance = AITile.GetDistanceManhattanToTile(loc1, loc2);
			Debug("Route from industry to destination is " + distance);

			::startLoc <- loc1;
			::endLoc <- loc2;
			::cargo_ID <- cargo;
			transports.sort(OtviTransport.cargoTransportSort);
			local connected = false;
			local i = 0;
			while (!connected && i < transports.len()) {
				connected = transports[i].connectIndustry(source, dest, cargo, destIsTown);
				i++;
			}
			if (connected) {
				Debug("Connected.");
			} else {
				Warning("Could not connect subsidy");
			}
		} else {
			Error("Invalid subsidy detected: " + subID);
		}
	}
}

