/**
 *    This file is part of OtviAI.
 *
 *    OtviAI is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    OtviAI is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with OtviAI.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2008-2013 Otto Visser.
 * Suggestions/comments and bugs can go to: otviAI@otvi.nl
 */

require("WaterChecker/ConnectedChecker.nut");

class OtviMarine extends OtviTransport {
	passengerBoatEngine = -1;
	dockRadius = -1;
	maxShips = -1;

	myDockTownList = null;
	myDepotTownList = null;

	function constructor() {
		Debug("Initializing water transportation...");
		dockRadius = AIStation.GetCoverageRadius(AIStation.STATION_DOCK);
		maxShips = AIGameSettings.GetValue("vehicle.max_ships");
		Debug("Max boats: " + maxShips);

		myDockTownList = AITownList();
		myDepotTownList = AITownList();
	}

	/**
	 * obvious function is obvious
	 */
	function getName() {
		return "water";
	}

	/**
	 * Returns whether anything can be done with this transport type
	 * Returns boolean
	 */
	function isOperational() {
		if (passengerBoatEngine == -1)
			return false;

		if (maxShips == 0)
			return false;

		return true
	}

	/**
	 * Returns minimum amount of money needed to establish a new route
	 * default distance used for calculation is 100
	 * Returns max int if expansion not possible
	 */
	function getExpansionCost(cargo, distance = 100) {
		local engine = cargo == ::PASSENGER_ID ? passengerBoatEngine : getEngine(cargo);
		return AIEngine.GetPrice(engine) + 2 * (AIMarine.GetBuildCost(AIMarine.BT_DOCK) + AIMarine.GetBuildCost(AIMarine.BT_DEPOT));
	}

	/**
	 * Returns minimum amount of money needed to enhance the last build route
	 * i.e.: add more vehicles to the route
	 * returns max int if enhancing is not possible
	 */
	function getEnhanceCost() {
		return AIEngine.GetPrice(passengerBoatEngine);
	}

	/**
	 * returns whether there is a route that could use enhancement at the moment
	 */
	function canEnhance() {
		local vehList = AIVehicleList();
		vehList.Valuate(AIVehicle.GetVehicleType);
		vehList.KeepValue(AIVehicle.VT_WATER);
		if (vehList.Count() >= maxShips) {
			//Warning("Reached maximum amount of vehicles!");
			return false;
		}
		// TODO FIXME
		return false;
	}

	/**
	 * does the actual enhancing
	 */
	function enhance() {
		return;
	}

	/**
	 * Returns transport preference
	 * Lower is more preferable
	 */
	function getTownTransportCost(town1, town2) {
		local vehList = AIVehicleList();
		vehList.Valuate(AIVehicle.GetVehicleType);
		vehList.KeepValue(AIVehicle.VT_WATER);
		if (vehList.Count() > maxShips) {
			Warning("Reached maximum amount of ships!");
			return 254;
		}

		if (passengerBoatEngine == -1) {
			Warning("Can't connect towns as there's no suitable boat available");
			return 254;
		}

		local sourceTiles = AITileList();
		SafeAddRectangle(sourceTiles, AITown.GetLocation(town1), dockRadius * 2);
		sourceTiles.Valuate(AITile.GetCargoAcceptance, ::PASSENGER_ID, 1, 1, dockRadius);
		sourceTiles.KeepAboveValue(7);
		sourceTiles.Valuate(AITile.GetCargoProduction, ::PASSENGER_ID, 1, 1, dockRadius);
		sourceTiles.RemoveValue(0);

		sourceTiles.Valuate(AITile.IsCoastTile);
		sourceTiles.KeepValue(1);

		if (sourceTiles.Count() == 0) {
			//Debug(AITown.GetName(town1) + " is not near the coast");
			return 254;
		}

		sourceTiles.Valuate(AITile.IsBuildable);
		sourceTiles.KeepValue(1);

		if (sourceTiles.Count() == 0) {
			Debug(AITown.GetName + " has no room for a dock");
			return 254;
		}

		local destTiles = AITileList();
		SafeAddRectangle(destTiles, AITown.GetLocation(town2), dockRadius * 2);
		destTiles.Valuate(AITile.GetCargoAcceptance, ::PASSENGER_ID, 1, 1, dockRadius);
		destTiles.KeepAboveValue(7);
		destTiles.Valuate(AITile.GetCargoProduction, ::PASSENGER_ID, 1, 1, dockRadius);
		destTiles.RemoveValue(0);

		destTiles.Valuate(AITile.IsCoastTile);
		destTiles.KeepValue(1);

		if (destTiles.Count() == 0) {
			//Debug(AITown.GetName(town2) + " is not near the coast");
			return 254;
		}

		destTiles.Valuate(AITile.IsBuildable);
		destTiles.KeepValue(1);

		if (destTiles.Count() == 0) {
			Debug(AITown.GetName(town2) + " has no room for a dock");
			return 254;
		}

		Debug("Both towns are close to water; connection might be available...");
		//if (!connectedCheck(sourceTiles, destTiles)) {
		//	Debug("There's no direct connection between source and destination by water");
		//	return 254;
		//}
		//Debug("There is a connection between source and destination by water!");

		return 0;
	}

	function getIndustryTransportCost(supplier, endLocation, cargoID) {
		if (getEngine(cargoID) == -1) {
			Warning("Can't connect industries as there's no suitable boat available");
			return 255;
		}
		local vehList = AIVehicleList();
		vehList.Valuate(AIVehicle.GetVehicleType);
		vehList.KeepValue(AIVehicle.VT_WATER);
		if (vehList.Count() > maxShips) {
			Warning("Reached maximum amount of ships!");
			return 254;
		}

		local ret = 255;

		local sourceTiles;

		if (AIIndustry.IsBuiltOnWater(supplier) && AIIndustry.HasDock(supplier)) {
			Debug("Industry on water and already has dock!");
			sourceTiles = AITileList_IndustryProducing(supplier, dockRadius);
			ret = 1;
		} else {
			sourceTiles = AITileList_IndustryProducing(supplier, dockRadius);
			sourceTiles.Valuate(AITile.IsCoastTile);
			sourceTiles.KeepValue(1);

			if (sourceTiles.Count() == 0) {
				Debug("Industry not near the coast");
				return 255;
			}
			sourceTiles.Valuate(AITile.IsBuildable);
			sourceTiles.KeepValue(1);
			if (sourceTiles.Count() == 0) {
				Debug("No room for a dock near supplier");
				return 255;
			}

			Debug("Industry is close to water: " + sourceTiles.Count());
			ret = 2;
		}

		local destTiles = AITileList_IndustryAccepting(AIIndustry.GetIndustryID(endLocation), dockRadius);
		destTiles.Valuate(AITile.IsCoastTile);
		destTiles.KeepValue(1);

		if (destTiles.Count() == 0) {
			Debug("Destination is not near the coast");
			return 254;
		}

		destTiles.Valuate(AITile.IsBuildable);
		destTiles.KeepValue(1);

		if (destTiles.Count() == 0) {
			Debug("Destination has no room for a dock");
			return 254;
		}

		Debug("Destination is close to water; a connection might be available..." + destTiles.Count());
		//if (!connectedCheck(sourceTiles, destTiles)) {
		//	Debug("There's no direct connection between source and destination by water");
		//	return 254;
		//}
		//Debug("There is a connection between source and destination by water!");

		return ret;
	}

	function getEngine(cargoID, distance = 100, production = 0) {
		// TODO cache result and update all engines on update, not just pass + mail
		local engine_list = AIEngineList(AIVehicle.VT_WATER);
		engine_list.Valuate(AIEngine.CanRefitCargo, cargoID);
		engine_list.KeepValue(1);

		// TODO: sort with valuator based on combination of breakdownLevel, speed and capacity
		/*if (::breakdownLevel == 0)
			engine_list.Valuate(AIEngine.GetCapacity);
		else
			engine_list.Valuate(AIEngine.GetReliability);
*/
		engine_list.Valuate(OtviTransport.profitValuator, cargoID, distance, production);
		engine_list.KeepAboveValue(0);

		engine_list.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);
		if (engine_list.IsEmpty()) {
			Warning("No suitable/profitable " + AICargo.GetCargoLabel(cargoID) + " boats available (yet?)!!");
			return -1;
		}
		// TODO if value < 0: return -2 
		
		local engine = engine_list.Begin();
		while (AIEngine.GetMaximumOrderDistance(engine) != 0 && AIEngine.GetMaximumOrderDistance(engine) < (distance + 20) && !(engine_list.IsEnd())) {
			Warning("Picked next engine due to order distance limitations on " + AIEngine.GetName(engine));
			engine = engine_list.Next();
		}

		if (AIEngine.IsValidEngine(engine) && (AIEngine.GetMaximumOrderDistance(engine) == 0 || AIEngine.GetMaximumOrderDistance(engine) >= (distance + 20))) {
			Debug("Picked boat for cargo " + AICargo.GetCargoLabel(cargoID) + ": " + AIEngine.GetName(engine));
			return engine;
		}
		return -1;
	}
	/**
	 * Updates engines
	 */
	function updateEngines() {
		passengerBoatEngine = getEngine(::PASSENGER_ID);
	}

	function drawPath(path) {
		while (path != null) {
			local par = path.GetParent();
			if (par != null) {
				if (AIMap.DistanceManhattan(path.GetTile(), par.GetTile()) == 1 ) {
					if (!AISign.IsValidSign(AISign.BuildSign(path.GetTile(), ".")))
						Error("Could not build sign!");
				} else {
					Warning("Trying to build a water bridge?");
				}
			}
			path = par;
		}
	}

	function appendWaterTiles(tiles, coastTile) {
		local offsets = [AIMap.GetTileIndex(0, 1), AIMap.GetTileIndex(0, -1), AIMap.GetTileIndex(1, 0), AIMap.GetTileIndex(-1, 0)];
		/* Check all tiles adjacent to the current tile. */
		foreach (offset in offsets) {
			local next_tile = coastTile + offset;
			if (AITile.IsWaterTile(next_tile)) {
				tiles.append(next_tile);
			}
		}
	}
				
	function connectedCheck(sourceTiles, destTiles) {
		local max_cost = 3 * AIMap.DistanceManhattan(sourceTiles.Begin(), destTiles.Begin());
		Debug("Checking for a water connection of max length " + max_cost);
		local pathfinder = ConnectedChecker(max_cost);

		local sourceArray = [];
		local destArray = [];
		// the sources and destinations are the possible land tiles for the docks
		// we need the corresponding water tile for the connected check
		foreach (tile, val in sourceTiles)
			appendWaterTiles(sourceArray, tile);

		foreach (tile, val in destTiles)
			appendWaterTiles(destArray, tile);

		if (sourceArray.len() == 0 || destArray.len() == 0) {
			Warning("It's actually not near the coast??");
			return false;
		}

		pathfinder.InitializePath(sourceArray, destArray);

		local path = false;
		path = pathfinder.FindPath(25 * max_cost);
		if (path) {
			Debug("W00tNess!!! there is a watery connection available; cost: " + path.GetCost());
			//drawPath(path);
			return path;
		}
		Warning("There is no connection by water available");

		return false;
	}

	/**
	 * tries to build a dock on one of the given tiles
	 * @param tiles possible locations
	 * @returns a tile location it was built on, or -1 on failure
	 */
	function buildDock(tiles) {
		arrangeBalance(AIMarine.GetBuildCost(AIMarine.BT_DOCK));

		foreach (tile, val in tiles) {
			if (AIMarine.BuildDock(tile, AIStation.STATION_JOIN_ADJACENT))
				return tile;
		}
		Warning("Couldn't build dock; last error: " + AIError.GetLastErrorString());
		return -1;
	}

	/**
	 * tries to build a depot in the neighbourhood of a dock
	 * @param dockTile the target
	 * @return the tile the depot is on or -1 on failure
	 */
	function buildDepot(dockTile) {
		arrangeBalance(AIMarine.GetBuildCost(AIMarine.BT_DEPOT));

		local sites = AITileList();
		// TODO: think about range and stuff
		SafeAddRectangle(sites, dockTile, 15, 4);	// don't block the dock
		sites.Valuate(AITile.IsWaterTile);
		sites.KeepValue(1);
		sites.Valuate(AIMap.DistanceManhattan, dockTile);
		sites.Sort(AIList.SORT_BY_VALUE, AIList.SORT_ASCENDING);

		foreach (tile, val in sites) {
			// TODO other orientations
			if (AIMarine.BuildWaterDepot(tile, tile + AIMap.GetTileIndex(0, 1)))
				return tile;
		}

		Warning("Couldn't build a depot");
		return -1;
	}

	function connectDocks(sourceDock, destDock, sourceDepot, cargoID, production) {
		local engine = getEngine(cargoID, AIMap.DistanceManhattan(sourceDock, destDock), production);
		arrangeBalance(AIEngine.GetPrice(engine));
		local boat = AIVehicle.BuildVehicle(sourceDepot, engine);
		if (!AIVehicle.IsValidVehicle(boat)) {
			Error("Failure to build boat: " + AIError.GetLastErrorString());
			return false;
		}

		if (AIEngine.CanRefitCargo(engine, cargoID))
			AIVehicle.RefitVehicle(boat, cargoID);

		if (!AIOrder.AppendOrder(boat, sourceDock, AIOrder.OF_FULL_LOAD_ANY))
			Error("Couldn't append order: " + AIError.GetLastErrorString());
		if (::breakdownLevel != 0)
			AIOrder.AppendOrder(boat, sourceDock, AIOrder.OF_GOTO_NEAREST_DEPOT | AIOrder.OF_SERVICE_IF_NEEDED);
		if(!AIOrder.AppendOrder(boat, destDock, AIOrder.OF_NONE))
			Error("Couldn't append order: " + AIError.GetLastErrorString());
		if (::breakdownLevel != 0)
			AIOrder.AppendOrder(boat, destDock, AIOrder.OF_GOTO_NEAREST_DEPOT | AIOrder.OF_SERVICE_IF_NEEDED);
		AIVehicle.StartStopVehicle(boat);

		return true;
	}

	function getDock(town) {
		local stationTile = myDockTownList.GetValue(town);
		local statID = AIStation.GetStationID(stationTile);
		if (AIStation.IsValidStation(statID) && AIStation.HasStationType(statID, AIStation.STATION_DOCK)) {
			return stationTile;
		}
	}

	function addDock(town, station) {
		myDockTownList.SetValue(town, station);
	}

	/**
	 * Connects two given towns and starts transporting passengers (and mail if possible)
	 * Returns: true iff succesful, false otherwise
	 */
	function connect2Towns(town1, town2, cargo = 0) {
		if (cargo == 0)
			cargo = ::PASSENGER_ID;

		local vehList = AIVehicleList();
		vehList.Valuate(AIVehicle.GetVehicleType);
		vehList.KeepValue(AIVehicle.VT_WATER);
		if (vehList.Count() > maxShips) {
			Warning("Reached maximum amount of ships!");
			return false;
		}

		if (getEngine(cargo, AIMap.DistanceManhattan(AITown.GetLocation(town1), AITown.GetLocation(town2))) == -1) {
			Warning("Can't connect towns as there's no suitable boat available");
			return false;
		}

		local sourceTiles = AITileList();
		SafeAddRectangle(sourceTiles, AITown.GetLocation(town1), dockRadius * 2);
		sourceTiles.Valuate(AITile.IsCoastTile);
		sourceTiles.KeepValue(1);
		sourceTiles.Valuate(AITile.IsBuildable);
		sourceTiles.KeepValue(1);
		sourceTiles.Valuate(AITile.GetCargoAcceptance, cargo, 1, 1, dockRadius);
		sourceTiles.KeepAboveValue(7);
		sourceTiles.Valuate(AITile.GetCargoProduction, cargo, 1, 1, dockRadius);
		sourceTiles.RemoveValue(0);
		sourceTiles.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);

		if (sourceTiles.Count() == 0) {
			Debug(AITown.GetName(town1) + " is not near the coast or there's no room");
			return false;
		}

		local destTiles = AITileList();
		SafeAddRectangle(destTiles, AITown.GetLocation(town2), dockRadius * 2);
		destTiles.Valuate(AITile.IsCoastTile);
		destTiles.KeepValue(1);
		destTiles.Valuate(AITile.IsBuildable);
		destTiles.KeepValue(1);
		destTiles.Valuate(AITile.GetCargoAcceptance, cargo, 1, 1, dockRadius);
		destTiles.KeepAboveValue(7);
		destTiles.Valuate(AITile.GetCargoProduction, cargo, 1, 1, dockRadius);
		destTiles.RemoveValue(0);
		destTiles.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);

		if (destTiles.Count() == 0) {
			Debug(AITown.GetName(town2) + " is not near the coast or there's no room for a dock");
			return false;
		}

		Debug("Both towns are close to water; checking connection...");
		local path = connectedCheck(sourceTiles, destTiles);
		if (!path) {
			Debug("There's no direct connection between source and destination by water");
			return false;
		}
		Debug("There is a connection between source and destination by water!");

		local buildDest = false;
		local destDock = getDock(town2);
		if (!destDock) {
			destDock = buildDock(destTiles);
			buildDest = true;
		}
		if (destDock == -1) {
			Debug("Building a destination dock failed...");
			return false;
		}

		local sourceDock = getDock(town1);
		local buildSource = false;
		if (!sourceDock) {
			sourceDock = buildDock(sourceTiles);
			buildSource = true;
		}
		if (sourceDock == -1) {
			Debug("Building a source dock failed...");
			if (buildDest)
				AITile.DemolishTile(destDock);
			return false;
		}

		local sourceDepot = myDepotTownList.GetValue(town1);
		if (!sourceDepot) {
			sourceDepot = buildDepot(sourceDock);
			if (sourceDepot != -1)
				myDepotTownList.SetValue(town1, sourceDepot);
		}
		local destDepot = myDepotTownList.GetValue(town2);
		if (!destDepot) {
			destDepot = buildDepot(destDock);
			if (destDepot != -1)
				myDepotTownList.SetValue(town2, destDepot);
		}

		if (!sourceDepot && !destDepot) {
			Warning("Couldn't build any depot in the water; aborting");
			if (buildDest)
				AITile.DemolishTile(destDock);
			if (buildSource)
				AITile.DemolishTile(sourceDock);
			return false;
		}

		if (sourceDepot == null)
			sourceDepot = destDepot;

		local prod1 = AITown.GetLastMonthProduction(town1, cargo) * (100 - AITown.GetLastMonthTransportedPercentage(town1, cargo));
		// TODO use path for placing bouys and include them in the path
		local ret =  connectDocks(sourceDock, destDock, sourceDepot, cargo, prod1);
		if (!ret) {
			Warning("Failure... cleaning up...");
			if (buildDest)
				AITile.DemolishTile(destDock);
			if (buildSource)
				AITile.DemolishTile(sourceDock);
			return false;
		} else {
			Debug("Route completed by boat");
			if (buildDest)
				addDock(town2, destDock);
			if (buildSource)
				addDock(town1, sourceDock);
			return true;
		}
	}

	/**
	 * Connects industry to given town or industry and starts transporting given cargo
	 */
	function connectIndustry(supplier, destination, cargoID, destIsTown) {
		if (getEngine(cargoID) == -1) {
			Warning("Can't connect industries as there's no suitable boat available");
			return false;
		}
		local vehList = AIVehicleList();
		vehList.Valuate(AIVehicle.GetVehicleType);
		vehList.KeepValue(AIVehicle.VT_WATER);
		if (vehList.Count() > maxShips) {
			Warning("Reached maximum amount of ships!");
			return false;
		}

		local sourceDock = -1;
		local sourceTiles;

		if (AIIndustry.IsBuiltOnWater(supplier) && AIIndustry.HasDock(supplier)) {
			Debug("Industry on water and already has dock!");
			sourceDock = AIIndustry.GetDockLocation(supplier);
			sourceTiles = AITileList_IndustryProducing(supplier, dockRadius);
		} else {
			sourceTiles = AITileList_IndustryProducing(supplier, dockRadius);
			sourceTiles.Valuate(AITile.IsCoastTile);
			sourceTiles.KeepValue(1);

			if (sourceTiles.Count() == 0) {
				Debug("Industry not near the coast");
				return false;
			}
			sourceTiles.Valuate(AITile.IsBuildable);
			sourceTiles.KeepValue(1);
			if (sourceTiles.Count() == 0) {
				Debug("No room for a dock near supplier");
				return false;
			}

			Debug("Industry is close to water: " + sourceTiles.Count());
		}

		local destTiles;

		if (destIsTown) {
			destTiles = AITileList();
			SafeAddRectangle(destTiles, AITown.GetLocation(destination), dockRadius * 2);
			destTiles.Valuate(AITile.GetCargoAcceptance, ::PASSENGER_ID, 1, 1, dockRadius);
			destTiles.KeepAboveValue(7);
			destTiles.Valuate(AITile.GetCargoProduction, ::PASSENGER_ID, 1, 1, dockRadius);
			destTiles.RemoveValue(0);
		} else {
			destTiles = AITileList_IndustryAccepting(destination, dockRadius);
		}

		destTiles.Valuate(AITile.IsCoastTile);
		destTiles.KeepValue(1);

		if (destTiles.Count() == 0) {
			Debug("Destination is not near the coast");
			return false;
		}

		destTiles.Valuate(AITile.IsBuildable);
		destTiles.KeepValue(1);

		if (destTiles.Count() == 0) {
			Debug("Destination has no room for a dock");
			return false;
		}

		Debug("Destination is close to water; checking connection..." + destTiles.Count());
		if (!connectedCheck(sourceTiles, destTiles)) {
			Debug("There's no direct connection between source and destination by water");
			return false;
		}
		Debug("There is a connection between source and destination by water!");

		local destDock = buildDock(destTiles);
		if (destDock == -1) {
			Debug("Building a destination dock failed...");
			return false;
		}

		if (sourceDock == -1) {
			sourceDock = buildDock(sourceTiles);
			if (sourceDock == -1) {
				Debug("Building a source dock failed...");
				AITile.DemolishTile(destDock);
				return false;
			}
		}

		local sourceDepot = buildDepot(sourceDock);
		local destDepot = buildDepot(destDock);
		if (!sourceDepot && !destDepot) {
			Warning("Couldn't build any depot in the water; aborting");
			AITile.DemolishTile(destDock);
			AITile.DemolishTile(sourceDock);
			return false;
		}

		if (sourceDepot == null)
			sourceDepot = destDepot;

		// TODO: calculate number of boats
		local prod = AIIndustry.GetLastMonthProduction(supplier, cargoID) - AIIndustry.GetLastMonthTransported(supplier, cargoID);
		return connectDocks(sourceDock, destDock, sourceDepot, cargoID, prod);
	}
}

