/**
 *    This file is part of OtviAI.
 *
 *    OtviAI is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    OtviAI is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with OtviAI.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2008-2013 Otto Visser.
 * Suggestions/comments and bugs can go to: otviAI@otvi.nl
 */

require("util.nut");
require("transport.nut");
require("RoadFinder/RoadFinder.nut");

class OtviRoad extends OtviTransport {
	myRoadDepotTownList = null;
	myDepotsTileList = null;
	myLorryTileList = null;
	myStationsTileList = null;

	passengerBusEngine = -1;
	mailLorryEngine = -1;

	expandMoneyMin = -1

	LORRY_RADIUS = -1;
	BUS_RADIUS = -1;
	
	lastMail1 = -1;
	lastMail2 = -1;
	lastMailVehicle = -1;
	lastTown1 = -1;
	lastTown2 = -1;
	lastPassengerVehicle = -1;
	lastMaxBusses = 0;

	lastMaxTrucks = 0;
	lastSupplier = -1;
	lastTruck = -1;
	lastCargo = -1;

	maxVehicles = -1;
	roadFinder = null;

	function constructor() {
		Debug("Initializing road transportation...");
		roadFinder = RoadFinder();

		myLorryTileList = AIList();
		myRoadDepotTownList = AITownList();
		myDepotsTileList = AIList();
		myStationsTileList = AIList();

		roadFinder.cost.tile = 1;
		roadFinder.cost.max_cost = 30000;		// TODO: still an estimate
		roadFinder.cost.no_existing_road = 6;	// The cost that is added to _cost_tile if no road exists yet.
		roadFinder.cost.turn = 10;				// The cost that is added to _cost_tile if the direction changes.
		roadFinder.cost.slope =	15;				// The extra cost if a road tile is sloped.
		roadFinder.cost.bridge_per_tile = 20;	// The cost per tile of a new bridge, this is added to _cost_tile.
		roadFinder.cost.tunnel_per_tile = 20;	// The cost per tile of a new tunnel, this is added to _cost_tile.
		roadFinder.cost.coast =	20;				// The extra cost for a coast tile.
		roadFinder.cost.max_bridge_length = 10;	// The maximum length of a bridge that will be build.
		roadFinder.cost.max_tunnel_length = 10;	// The maximum length of a tunnel that will be build.
		
		AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_ROAD);

		LORRY_RADIUS = AIStation.GetCoverageRadius(AIStation.STATION_TRUCK_STOP);
		BUS_RADIUS = AIStation.GetCoverageRadius(AIStation.STATION_BUS_STOP);

		maxVehicles = AIGameSettings.GetValue("vehicle.max_roadveh");
		Debug("Max road vehicles: " + maxVehicles);
	}

	function getName() {
		return "road";
	}

	function isOperational() {
		if (passengerBusEngine == -1 && mailLorryEngine == -1)
			return false;

		if (maxVehicles == 0)
			return false;

		return true;
	}

	function canEnhance() {
		local vehList = AIVehicleList();
		vehList.Valuate(AIVehicle.GetVehicleType);
		vehList.KeepValue(AIVehicle.VT_ROAD);
		if (vehList.Count() >= maxVehicles) {
			//Warning("Reached maximum amount of vehicles!");
			return false;
		}
		if (lastMaxTrucks > 0) {
			if(AIStation.GetCargoWaiting(lastSupplier, lastCargo) > 3 * AIVehicle.GetCapacity(lastTruck, lastCargo)) {
				return true;
			}
		}
		if (lastMaxBusses > 0) {
			local pass1 = AIStation.GetStationID(lastTown1);
			local pass2 = AIStation.GetStationID(lastTown2);
			if(min(AIStation.GetCargoWaiting(pass1, ::PASSENGER_ID), AIStation.GetCargoWaiting(pass2, ::PASSENGER_ID)) > 4 * AIVehicle.GetCapacity(lastPassengerVehicle, ::PASSENGER_ID) && lastMaxBusses)
				return true;
			local lorry1 = AIStation.GetStationID(lastMail1);
			local lorry2 = AIStation.GetStationID(lastMail2);
			if(min(AIStation.GetCargoWaiting(lorry1, ::MAIL_ID), AIStation.GetCargoWaiting(lorry2, ::MAIL_ID)) > 3 * AIVehicle.GetCapacity(lastMailVehicle, ::MAIL_ID))
				return true;
		}
		return false;
	}

	function getEnhanceCost() {
		return expandMoneyMin;
	}

	function enhance() {
		local res = false;
		if (lastMaxTrucks > 0) {
			if(AIStation.GetCargoWaiting(lastSupplier, lastCargo) > 3 * AIVehicle.GetCapacity(lastTruck, lastCargo)) {
				Debug("Adding trucks at " + AIStation.GetName(lastSupplier));
				local vehicle = AIVehicle.CloneVehicle(myStationsTileList.GetValue(AIStation.GetLocation(lastSupplier)), lastTruck, false);
				if (AIVehicle.IsValidVehicle(vehicle)) {
					AIVehicle.StartStopVehicle(vehicle);
					lastMaxTrucks--;
					lastTruck = vehicle;
					res = true;
				}
			}
		}
		if(lastMail1 != -1 && lastMail2 != -1) {
			// add mail trucks where there is plenty cargo
			local lorry1 = AIStation.GetStationID(lastMail1);
			local lorry2 = AIStation.GetStationID(lastMail2);
			if(min(AIStation.GetCargoWaiting(lorry1, ::MAIL_ID), AIStation.GetCargoWaiting(lorry2, ::MAIL_ID)) > 3 * AIVehicle.GetCapacity(lastMailVehicle, ::MAIL_ID)) {
				Debug("Adding mail trucks at " + AIStation.GetName(lorry1) + " to " + AIStation.GetName(lorry2));
				res = connectMailStations(lastMail1, lastMail2) || res;
			}
		}
		if(lastTown1 != -1 && lastTown2 != -1) {
			// add busses where there is plenty cargo
			local pass1 = AIStation.GetStationID(lastTown1);
			local pass2 = AIStation.GetStationID(lastTown2);
			if(min(AIStation.GetCargoWaiting(pass1, ::PASSENGER_ID), AIStation.GetCargoWaiting(pass2, ::PASSENGER_ID)) > 4 * AIVehicle.GetCapacity(lastPassengerVehicle, ::PASSENGER_ID) && lastMaxBusses) {
				Debug("Adding busses at " + AIStation.GetName(pass1) + " to " + AIStation.GetName(pass2));
				if(connectPassengerStations(lastTown1, lastTown2)) {
					lastMaxBusses--;
					res = true || res;
				}
			}
		}
		return res;
	}

	function getExpansionCost(cargo, distance = 100) {
		if (cargo == ::PASSENGER_ID)
			return 4 * AIEngine.GetPrice(passengerBusEngine) + 2 * AIEngine.GetPrice(mailLorryEngine) + 2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_BUS_STOP) + 2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_TRUCK_STOP) + 2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_DEPOT) + distance * 2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_ROAD);
		else
			return 5 * AIEngine.GetPrice(getEngine(cargo)) + 2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_TRUCK_STOP) + 2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_DEPOT) + distance * 2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_ROAD);
	}

	function getTownTransportCost(startLoc, endLoc) {
		if (passengerBusEngine == -1)  {
			Warning("Road transport not yet available");
			return 255;
		}
		local vehList = AIVehicleList();
		vehList.Valuate(AIVehicle.GetVehicleType);
		vehList.KeepValue(AIVehicle.VT_ROAD);
		if (vehList.Count() > (maxVehicles - 5)) {
			Warning("Reached maximum amount of vehicles!");
			return 254;
		}

		local ret = 1;
		if (::RONDJE_ALERT)
			ret = 3;

		local dist = AIMap.DistanceManhattan(AITown.GetLocation(startLoc), AITown.GetLocation(endLoc));
		//Debug("Bus from " + AITown.GetName(startLoc) + " to " + AITown.GetName(endLoc) + " has a distance of " + dist);
		if (dist > 200) {
			return (ret * 8);
		}
		if (dist > 100) {
			return (ret * 4);
		}
		return (2 * ret);
	}
		
	function getCargoTransportCost(startLoc, endLoc, cargoID) {
		if (getEngine(cargoID) == -1) {
			Warning("There is no road vehicle available for " + AICargo.GetCargoLabel(cargoID));
			return 255;
		}
		local vehList = AIVehicleList();
		vehList.Valuate(AIVehicle.GetVehicleType);
		vehList.KeepValue(AIVehicle.VT_ROAD);
		if (vehList.Count() > (maxVehicles - 5)) {
			Warning("Reached maximum amount of vehicles!");
			return 254;
		}

		local ret = 1;
		if (::RONDJE_ALERT)
			ret = 5;

		local dist = AIMap.DistanceManhattan(startLoc, endLoc);
		if (dist > 200)
			return (ret * 8);
		if (dist > 100)
			return (ret * 4);
		else
			return (ret * 2);
	}
		
	function getEngine(cargoID, distance = 100) {
		// TODO cache result and update all engines on update, not just pass + mail
		local engine_list = AIEngineList(AIVehicle.VT_ROAD);
		engine_list.Valuate(AIEngine.CanRefitCargo, cargoID);
		engine_list.KeepValue(1);

		engine_list.Valuate(AIEngine.GetRoadType);
		engine_list.KeepValue(AIRoad.ROADTYPE_ROAD);

		// TODO: sort with valuator based on combination of breakdownLevel, speed and capacity
		/*if (::breakdownLevel == 0)
			engine_list.Valuate(AIEngine.GetCapacity);
		else
			engine_list.Valuate(AIEngine.GetReliability);
*/
		engine_list.Valuate(OtviTransport.profitValuator, cargoID, distance);
		engine_list.KeepAboveValue(0);

		engine_list.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);
		if (engine_list.IsEmpty()) {
			Warning("No suitable/profitable " + AICargo.GetCargoLabel(cargoID) + " road engines available (yet?)!!");
			return -1;
		}
		// TODO if value < 0: return -2 
		
		local engine = engine_list.Begin();
		while (AIEngine.GetMaximumOrderDistance(engine) != 0 && AIEngine.GetMaximumOrderDistance(engine) < (distance + 20) && !(engine_list.IsEnd())) {
			Warning("Picked next engine due to order distance limitations on " + AIEngine.GetName(engine));
			engine = engine_list.Next();
		}
		if (AIEngine.IsValidEngine(engine)) {
			Debug("Picked road engine for cargo " + AICargo.GetCargoLabel(cargoID) + ": " + AIEngine.GetName(engine));
			return engine;
		}
		return -1;
	}

	function updateEngines() {
		passengerBusEngine = getEngine(::PASSENGER_ID);
		mailLorryEngine = getEngine(::MAIL_ID);
		expandMoneyMin = min(AIEngine.GetPrice(passengerBusEngine), AIEngine.GetPrice(mailLorryEngine));
	}

	/**
	 * 2 passenger stations build, make road and then place busses
	 */
	function connectTowns(starts, goals) {
		local distance = AIMap.DistanceManhattan(starts[0], goals[0]);
		roadFinder.cost.max_cost = 30 * distance + 300;	// TODO based on hillyness?
		roadFinder.InitializePath(starts, goals);
		local path = false;
		local i = 0;
		local err;
		local iterations = 300 * distance; // if we pass this; something is wrong
		//local iterations = -1;	// TODO limit to reasonable amount, elevate if fails
		// TODO check whether higher values helps finding faster
		if (iterations > (150 * 200)) {
			Warning("Limiting to roughly 200 days");
			iterations = 150 * 200;
		}
		Debug("pathfinding for max " + roadFinder.cost.max_cost + " cost and iterations: " + iterations);
		local startTick = GetTick();
		while (path == false && iterations > 0) {
			path = roadFinder.FindPath(150);
			iterations -= 150;
		}
		Debug("Total ticks passed with " + iterations + " left: " + (GetTick() - startTick));

		if (path == false) {
			Error("!! No path found, too little iterations...");
			roadFinder.reset();
			return false;
		}
		if (path == null) {
			roadFinder.reset();
			Error("!! Impossible to create route, too expensive");
			return false;
		}
		Debug("path found for cost: " + path.GetCost() + ", building");

		while (path != null) {
			local par = path.GetParent();
			if (par != null) {
				if (AIMap.DistanceManhattan(path.GetTile(), par.GetTile()) == 1 ) {
					arrangeBalance(2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_ROAD));
					if (!AIRoad.BuildRoad(path.GetTile(), par.GetTile())) {
						switch(AIError.GetLastError()) {
							case AIError.ERR_VEHICLE_IN_THE_WAY:
								Warning("Building road failed temporary: vehicle in the way...");
								local tries = 10;
								// try again till the stupid vehicle moved?
								while(!AIRoad.BuildRoad(path.GetTile(), par.GetTile()) && tries--)
									::INSTANCE.Sleep(5);
								break;
							case AIRoad.ERR_ROAD_WORKS_IN_PROGRESS:
								Warning("Building road failed: road works in progress");
								// good, someone else is building this, skip tile
								break;
							case AIError.ERR_ALREADY_BUILT:
								// Debug("Building road failed: already built");
								// even better; someone else already built this; silent ignore
								break;
							case AIError.ERR_LAND_SLOPED_WRONG:
								Error("Building road failed: wrong land slope");
								// not good; redo from start or terraform?
								return connectTowns(starts, goals);
							case AIError.ERR_AREA_NOT_CLEAR:
								Warning("Building road failed: not clear, demolishing tile...");
								AISign.BuildSign(path.GetTile(), "I'm on your land, demolishing your stuffs");
								// demolish and retry?
								if (AITile.DemolishTile(path.GetTile())) {
									AIRoad.BuildRoad(path.GetTile(), par.GetTile());
								} else {	// happens when other ai builds after calc
									return connectTowns(starts, goals);
								}
								break;
							case AIRoad.ERR_ROAD_ONE_WAY_ROADS_CANNOT_HAVE_JUNCTIONS:
								Error("Building road failed: no junctions on one way roads allowed");
								// can't happen? Not building those
								return false;
							case AIError.ERR_NOT_ENOUGH_CASH:
								Warning("Panic: Out of cash!!");
								local count = 10;
								local res = false;
								arrangeBalance(5 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_ROAD));
								while(!(res = AIRoad.BuildRoad(path.GetTile(), par.GetTile())) && count-- > 0)
									::INSTANCE.Sleep(100);

								if (!res)
									return false;

								break;
							default:
								Warning("Unknown error during road building " + AIError.GetLastErrorString());
								break;	// no clue what this is; silent ignore for now
						}
					}
				} else {
					/* Build a bridge or tunnel. */
					if (!AIBridge.IsBridgeTile(path.GetTile()) && !AITunnel.IsTunnelTile(path.GetTile())) {
						/* If it was a road tile, demolish it first. Do this to work around expended roadbits. */
						if (AIRoad.IsRoadTile(path.GetTile())) AITile.DemolishTile(path.GetTile());
						if (AITunnel.GetOtherTunnelEnd(path.GetTile()) == par.GetTile()) {
							arrangeBalance(50000);
							if (!AITunnel.BuildTunnel(AIVehicle.VT_ROAD, path.GetTile())) {
								Error("Couldn't build tunnel");
								return connectTowns(starts, goals);
							}
						} else {
							local bridge_list = AIBridgeList_Length(AIMap.DistanceManhattan(path.GetTile(), par.GetTile()) + 1);
							bridge_list.Valuate(AIBridge.GetMaxSpeed);
							bridge_list.Sort(AIList.SORT_BY_VALUE, false);
							local bridge = bridge_list.Begin();
							arrangeBalance(AIBridge.GetPrice(bridge, AIMap.DistanceManhattan(path.GetTile(), par.GetTile()) + 1));
							if (!AIBridge.BuildBridge(AIVehicle.VT_ROAD, bridge, path.GetTile(), par.GetTile())) {
								Error("Couldn't build bridge");
								return connectTowns(starts, goals);
							}
						}
					}
				}
			}
			path = par;
		}

		/*while (path != null) {
			if (path.GetParent() != null) {
				local par = path.GetParent().GetTile();
				if (!AIRoad.AreRoadTilesConnected(path.GetTile(), par) && (!AIBridge.IsBridgeTile(path.GetTile()) || AIBridge.GetOtherBridgeEnd(path.GetTile()) != par)) {
					AIRoad.BuildRoad(path.GetTile(), par);
				}
			}
			path = path.GetParent();
		}*/
		Debug("path done");
		return true;
	}

	// TODO: AIVehicle.SetName
	function connectPassengerStations(station1, station2, cargo = 0) {
		if (cargo == 0)
			cargo = ::PASSENGER_ID;
		else {
			if (AICargo.HasCargoClass(cargo, AICargo.CC_PASSENGERS)) {
				Warning("Cargo is not for normal passengers: " + cargo);
			} else {
				Warning("Cargo is not for people: " + cargo);
				return true;
			}
		}

		local price = AIEngine.GetPrice(passengerBusEngine);
		arrangeBalance(price);
		local bus1 = AIVehicle.BuildVehicle(myStationsTileList.GetValue(station1), passengerBusEngine);
		if (!AIVehicle.IsValidVehicle(bus1))
			return false;
		if(!AIVehicle.RefitVehicle(bus1, cargo)) {
			Error("Error refitting bus: " + AIError.GetLastErrorString());
			return false;
		}

		Debug("Built bus: " + bus1);
		lastPassengerVehicle = bus1;
		local res = AIOrder.AppendOrder(bus1, station1, AIOrder.OF_FULL_LOAD | AIOrder.OF_NON_STOP_INTERMEDIATE);
		if (::breakdownLevel != 0)
			AIOrder.AppendOrder(bus1, station1, AIOrder.OF_GOTO_NEAREST_DEPOT | AIOrder.OF_SERVICE_IF_NEEDED);
		res = res && AIOrder.AppendOrder(bus1, station2, AIOrder.OF_FULL_LOAD | AIOrder.OF_NON_STOP_INTERMEDIATE);
		if (::breakdownLevel != 0)
			AIOrder.AppendOrder(bus1, station2, AIOrder.OF_GOTO_NEAREST_DEPOT | AIOrder.OF_SERVICE_IF_NEEDED);
		if (res)
			AIVehicle.StartStopVehicle(bus1);
		else
			Error("Couldn't give bus proper orders: " + AIError.GetLastErrorString());

		arrangeBalance(price);
		local bus2 = AIVehicle.BuildVehicle(myStationsTileList.GetValue(station2), passengerBusEngine);
		if (!AIVehicle.IsValidVehicle(bus2))
			return true; 	// We at least built one working: so true
		if(!AIVehicle.RefitVehicle(bus2, cargo)) {
			Error("Error refitting bus: " + AIError.GetLastErrorString());
			return true; // We at least built one working: so true
		}
		Debug("Built bus: " + bus2);
		res = AIOrder.AppendOrder(bus2, station2, AIOrder.OF_FULL_LOAD | AIOrder.OF_NON_STOP_INTERMEDIATE);
		if (::breakdownLevel != 0)
			AIOrder.AppendOrder(bus2, station2, AIOrder.OF_GOTO_NEAREST_DEPOT | AIOrder.OF_SERVICE_IF_NEEDED);
		res = res && AIOrder.AppendOrder(bus2, station1, AIOrder.OF_FULL_LOAD | AIOrder.OF_NON_STOP_INTERMEDIATE);
		if (::breakdownLevel != 0)
			AIOrder.AppendOrder(bus2, station1, AIOrder.OF_GOTO_NEAREST_DEPOT | AIOrder.OF_SERVICE_IF_NEEDED);
		if (res)
			AIVehicle.StartStopVehicle(bus2);
		else
			Error("Couldn't give bus proper orders: " + AIError.GetLastErrorString());

		return true;
	}
	
	// TODO: AIVehicle.SetName
	function connectMailStations(lorry1, lorry2) {
		if (!(AIStation.IsValidStation(AIStation.GetStationID(lorry1)) && (AIStation.IsValidStation(AIStation.GetStationID(lorry2))))) {
			Error("Trying to connect non valid mail stations");
			return false;
		}

		local price = AIEngine.GetPrice(mailLorryEngine);
		arrangeBalance(price);
		local mail1 = AIVehicle.BuildVehicle(myLorryTileList.GetValue(lorry1), mailLorryEngine);
		if(!AIVehicle.IsValidVehicle(mail1))
			return false;
		if(!AIVehicle.RefitVehicle(mail1, ::MAIL_ID)) {
			Error("Error refitting lorry: " + AIError.GetLastErrorString());
			return false;
		}

		Debug("Built a mail truck: " + mail1);
		lastMailVehicle = mail1;
		local res = AIOrder.AppendOrder(mail1, lorry1, AIOrder.OF_FULL_LOAD | AIOrder.OF_NON_STOP_INTERMEDIATE);
		if (::breakdownLevel != 0)
			AIOrder.AppendOrder(mail1, lorry1, AIOrder.OF_GOTO_NEAREST_DEPOT | AIOrder.OF_SERVICE_IF_NEEDED);
		res = res && AIOrder.AppendOrder(mail1, lorry2, AIOrder.OF_FULL_LOAD | AIOrder.OF_NON_STOP_INTERMEDIATE);
		if (::breakdownLevel != 0)
			AIOrder.AppendOrder(mail1, lorry2, AIOrder.OF_GOTO_NEAREST_DEPOT | AIOrder.OF_SERVICE_IF_NEEDED);
		if (res)
			AIVehicle.StartStopVehicle(mail1);
		else
			Error("Couldn't give mail truck proper orders: " + AIError.GetLastErrorString());

		arrangeBalance(price);
		local mail2 = AIVehicle.BuildVehicle(myLorryTileList.GetValue(lorry2), mailLorryEngine);
		if(!AIVehicle.IsValidVehicle(mail2))
			return true; 	// We at least built one working: so true
		if(!AIVehicle.RefitVehicle(mail2, ::MAIL_ID)) {
			Error("Error refitting lorry: " + AIError.GetLastErrorString());
			return true; 	// We at least built one working: so true
		}

		Debug("Built a mail truck: " + mail2);
		res = AIOrder.AppendOrder(mail2, lorry2, AIOrder.OF_FULL_LOAD | AIOrder.OF_NON_STOP_INTERMEDIATE);
		if (::breakdownLevel != 0)
			AIOrder.AppendOrder(mail2, lorry2, AIOrder.OF_GOTO_NEAREST_DEPOT | AIOrder.OF_SERVICE_IF_NEEDED);
		res = res && AIOrder.AppendOrder(mail2, lorry1, AIOrder.OF_FULL_LOAD | AIOrder.OF_NON_STOP_INTERMEDIATE);
		if (::breakdownLevel != 0)
			AIOrder.AppendOrder(mail2, lorry1, AIOrder.OF_GOTO_NEAREST_DEPOT | AIOrder.OF_SERVICE_IF_NEEDED);
		if (res)
			AIVehicle.StartStopVehicle(mail2);
		else
			Error("Couldn't give mail truck proper orders: " + AIError.GetLastErrorString());

		return true;
	}
	
	// TODO: AIVehicle.SetName
	function connectIndustryStations(station1, station2, cargoID) {	
		// TODO base engine count not just on distance, but also on production
		local distance = AITile.GetDistanceManhattanToTile(station1, station2);
		local engine = getEngine(cargoID, distance);
		if (engine == -1) {
			Warning("Can't build trucks!");
			return false;
		}
		// based on tiles of 686km and that loading takes a day:
		// TODO: take production into account!!! perhaps build bigger station if lots of cargo?
		// also: size of vehicle and actual road distance
		//local count = ((distance * 686 * 2) / (AIEngine.GetMaxSpeed(engine) * 24)) + 1;
		local count = (distance / ((AIEngine.GetMaxSpeed(engine) / 10) + 1));	// tweaked to 10. May need to be raised to 12...
		//if (count > 20) {
		//	Warning("Reducing truck count from " + count + " to 20 until new calculation works");
		//	count = 20;
		//}
		local ret = false;

		Debug("Want to build this many trucks: " + count);
		for (local i = 0; i < count; i++) {
			arrangeBalance(AIEngine.GetPrice(engine));

			local vehicle = AIVehicle.BuildVehicle(myStationsTileList.GetValue(station1), engine);
			if (!AIVehicle.IsValidVehicle(vehicle))
				Warning("Could not build truck: " + AIError.GetLastErrorString());
			else {
				if(!AIVehicle.RefitVehicle(vehicle, cargoID)) {
					Error("Error refitting vehicle: " + AIError.GetLastErrorString());
					return ret;
				}
				Debug("Built truck " + vehicle + " which is valid: " + AIVehicle.IsValidVehicle(vehicle));
				local res = AIOrder.AppendOrder(vehicle, station1, AIOrder.OF_FULL_LOAD | AIOrder.OF_NON_STOP_INTERMEDIATE);
				if (::breakdownLevel != 0)
					AIOrder.AppendOrder(vehicle, station1, AIOrder.OF_GOTO_NEAREST_DEPOT | AIOrder.OF_SERVICE_IF_NEEDED);
				res = res && AIOrder.AppendOrder(vehicle, station2, AIOrder.OF_NONE | AIOrder.OF_NON_STOP_INTERMEDIATE);
				if (::breakdownLevel != 0)
					AIOrder.AppendOrder(vehicle, station2, AIOrder.OF_GOTO_NEAREST_DEPOT | AIOrder.OF_SERVICE_IF_NEEDED);
				if (res)
					AIVehicle.StartStopVehicle(vehicle);
				else {
					Error("Couldn't give lorry proper orders: " + AIError.GetLastErrorString());
					return ret;
				}
				ret = true;

				lastTruck = vehicle;
				lastSupplier = AIStation.GetStationID(station1);
				lastCargo = cargoID;
				lastMaxTrucks = count - i - 1;
			}
		}
		return ret;
	}

	static function getRoadTileOption(t) {
		local adjacent = AITileList();
		adjacent.AddTile(t - AIMap.GetTileIndex(1,0));
		adjacent.AddTile(t - AIMap.GetTileIndex(0,1));
		adjacent.AddTile(t - AIMap.GetTileIndex(-1,0));
		adjacent.AddTile(t - AIMap.GetTileIndex(0,-1));
		adjacent.Valuate(AITile.IsBuildable);
		adjacent.KeepValue(1);
		if (adjacent.Count())
			return adjacent.Begin();
		else
			return null;
	}

	static function getRoadTile(t) {
		local adjacent = AITileList();
		adjacent.AddTile(t - AIMap.GetTileIndex(1,0));
		adjacent.AddTile(t - AIMap.GetTileIndex(0,1));
		adjacent.AddTile(t - AIMap.GetTileIndex(-1,0));
		adjacent.AddTile(t - AIMap.GetTileIndex(0,-1));
		adjacent.Valuate(AIRoad.IsRoadTile);
		adjacent.KeepValue(1);
		adjacent.Valuate(AIRoad.IsRoadDepotTile);
		adjacent.KeepValue(0);
		adjacent.Valuate(AIRoad.IsRoadStationTile);
		adjacent.KeepValue(0);
		if (adjacent.Count())
			return adjacent.Begin();
		else
			return null;
	}

	/**
	 * hack to support basic sqrt function for other functions
	 * TODO: consider using lookup table for speed reasons (as suggested by Truebrain)
	 * TODO: useless now?
	 */
/*	function sqrt(i) {
		if (i == 0)
			return 0;   // Avoid divide by zero
		local n = (i / 2) + 1;       // Initial estimate, never low
		local n1 = (n + (i / n)) / 2;
		while (n1 < n) {
			n = n1;
			n1 = (n + (i / n)) / 2;
		}
		return n;
	}*/

	function buildNonInlinePassengerStation(town) {
		Debug("Building non inline bus stop...");
		// Find empty square as close to town centre as possible
		local curRange = 1;
		local maxRange = sqrt(AITown.GetPopulation(town)/100) + 4; // search larger area
		local area = AITileList();
		local townLoc = AITown.GetLocation(town);

		arrangeBalance(2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_BUS_STOP) + 2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_ROAD));

		while (curRange < maxRange) {
			//Debug("looking with range " + curRange);
			SafeAddRectangle(area, townLoc, curRange, curRange - 1);
			area.Valuate(AITile.IsBuildable);
			area.KeepValue(1);
			if (area.Count()) {
				//Debug("Found " + area.Count() + " possible options");
				foreach (tile, val in area) {
					local opening = getRoadTile(tile);
					if(opening) {
						// Debug("tile " + t + " has opening " + opening);
						if (AIRoad.BuildRoad(opening, tile) && AIRoad.BuildRoadStation(tile, opening, AIRoad.ROADVEHTYPE_BUS, AIStation.STATION_JOIN_ADJACENT)) {
							//AISign.BuildSign(t, ("bus " + AITown.GetName(town)));
							return tile;
						}
						Error("Error building passenger station " + AIError.GetLastErrorString());
						if (AIError.GetLastError() == AIError.ERR_LOCAL_AUTHORITY_REFUSES) {
							Warning("Pissed off authorities; planting tree...");
							AITile.DemolishTile(tile);
							AITile.PlantTree(tile);
						} else {
							AITile.DemolishTile(tile);
						}
						//AISign.BuildSign(t, "" + AIRoad.GetNeighbourRoadCount(t));
					} //else {
						// Debug("tile " + t + " has no opening");
					//}
				}
				curRange++;
			} else {
				curRange++;
			}
		}
		Debug("No possible station location found");
		return false;
	}
	
	function buildPassengerStation(town) {
		Debug("Building bus station in " + AITown.GetName(town));
		// Find empty square as close to town centre as possible
		local curRange = 1;
		local maxRange = sqrt(AITown.GetPopulation(town)/100) + 2; //TODO check value correctness 
		local area = AITileList();
		local townLoc = AITown.GetLocation(town);
		
		arrangeBalance(2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_BUS_STOP) + AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_ROAD));

		while (curRange < maxRange) {
			//Debug("looking with range " + curRange);
			SafeAddRectangle(area, townLoc, curRange, curRange - 1);
			area.Valuate(AIRoad.IsRoadTile);
			area.KeepValue(1);
			area.Valuate(AIRoad.IsDriveThroughRoadStationTile);
			area.KeepValue(0);
			area.Valuate(AIRoad.GetNeighbourRoadCount);
			area.KeepValue(2);	// entrance and exit; allow 1 as well?
			if (area.Count()) {
				//Debug("Found " + area.Count() + " possible options");
				foreach (tile, val in area) {
					local opening = getRoadTile(tile);
					if(opening) {
						// Debug("tile " + t + " has opening " + opening);
						if (/*AIRoad.BuildRoad(opening, t) &&*/ AIRoad.BuildDriveThroughRoadStation(tile, opening, AIRoad.ROADVEHTYPE_BUS, AIStation.STATION_JOIN_ADJACENT)) {
							//AISign.BuildSign(t, ("bus " + AITown.GetName(town)));
							return tile;
						}
						if (AIError.GetLastError() == AIRoad.ERR_ROAD_CANNOT_BUILD_ON_TOWN_ROAD)
							return buildNonInlinePassengerStation(town);
						// silently ignore error related to trying to build in a corner:
						if (AIError.GetLastError() != AIRoad.ERR_ROAD_DRIVE_THROUGH_WRONG_DIRECTION)
							Warning("Error building bus stop: " + AIError.GetLastErrorString());
						if (AIError.GetLastError() == AIError.ERR_LOCAL_AUTHORITY_REFUSES)
							return false;	// it's not going to work
						//AISign.BuildSign(t, AIError.GetLastErrorString());

					} //else {
						// Debug("tile " + t + " has no opening");
					//}
				}
				curRange++;
			} else {
				curRange++;
			}
		}
		Warning("No possible bus stop location found in " + AITown.GetName(town));
		return false;
	}
	
	function buildMailStation(town) {
		if (mailLorryEngine == -1) {
			Warning("No mail engine available, not building mail stations");
			return false;
		}
		return buildTownLorryStation(AITown.GetLocation(town), ::MAIL_ID, true);
	}
		
	function buildNonInlineLorryStation(location, cargo, isSource) {
		Debug("Building lorry station on new road...");
		// Find empty square as close to tile as possible
		local curRange = 1;
		local maxRange = LORRY_RADIUS * 2; //TODO check value correctness
		local area = AITileList();
		local area2 = AITileList();
 
		// We got the north most tile; I wanna search around it's center
		location = location + AIMap.GetTileIndex(1, 1);
		
		arrangeBalance(2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_BUS_STOP) + 2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_ROAD));

		// TODO use industry AITileLists
		while (curRange < maxRange) {
			//AILog.Info("looking with range " + curRange);
			SafeAddRectangle(area, location, curRange, curRange - 1);
			area.Valuate(AITile.IsBuildable);
			area.KeepValue(1);
			if (!isSource || cargo == ::MAIL_ID) {
				area.Valuate(AITile.GetCargoAcceptance, cargo, 1, 1, LORRY_RADIUS);
				area.KeepAboveValue(7);
			}
			if (isSource || cargo == ::MAIL_ID) {
				area.Valuate(AITile.GetCargoProduction, cargo, 1, 1, LORRY_RADIUS);
				area.RemoveValue(0);
			}
			area.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);
			SafeAddRectangle(area2, location, curRange + 1, curRange);
			area2.Valuate(AITile.IsBuildable);
			area2.KeepValue(1);
			if (area.Count() && area2.Count()) {
				//AILog.Info("Found " + area.Count() + " possible options");
				foreach (tile, val in area) {
					local opening = area2.Begin();
					while ((AIMap.DistanceManhattan(tile, opening) != 1 || opening == tile) && !area2.IsEnd()) {
						opening = area2.Next();
					}
					if (opening) {
						// AILog.Info("tile " + t + " has opening " + opening);
						if (AIRoad.BuildRoad(opening, tile) && AIRoad.BuildDriveThroughRoadStation(tile, opening, AIRoad.ROADVEHTYPE_TRUCK, AIStation.STATION_JOIN_ADJACENT)) {
							//AISign.BuildSign(t, ("Mail " + AITown.GetName(town)));
							return tile;
						}
						//AILog.Info("tile " + t + " has wrong slope!!!");
					} //else {
						// AILog.Info("tile " + t + " has no opening");
					//}
				}
				curRange++;
			} else {
				curRange++;
			}
		}
		Warning("No possible lorry station location found!!!");
		return false;
	}

	function buildTCStation(tc, cargo, isSource) {
		Debug("Building trade center station...");
		// Find empty square as close to industry as possible
		local targetLoc = tc + AIMap.GetTileIndex(3, 3);
		local area = AITileList();
		SafeAddRectangle(area, tc, LORRY_RADIUS);
		area.Valuate(AITile.IsBuildable);
		area.KeepValue(1);
		area.Valuate(AITile.GetSlope);
		area.KeepValue(AITile.SLOPE_FLAT);
		
		if (isSource) {
			//area.Valuate(AITile.GetCargoProduction, cargo, 1, 1, LORRY_RADIUS);
			area.Valuate(AITile.GetDistanceManhattanToTile, targetLoc);
			area.Sort(AIList.SORT_BY_VALUE, AIList.SORT_ASCENDING);
		} else {
			area.Valuate(AITile.GetCargoAcceptance, cargo, 1, 1, LORRY_RADIUS);
			area.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);
		}

		arrangeBalance(2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_BUS_STOP) + 2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_ROAD));

		if (area.Count()) {
			foreach (tile, val in area) {
				local opening = getRoadTile(tile);
				local buildNewRoad = false;
				if(opening == null) {
					opening = getRoadTileOption(tile);
					buildNewRoad = true;
				}
				if (opening) {
					if (AIRoad.BuildRoad(opening, tile) && AIRoad.BuildDriveThroughRoadStation(tile, opening, AIRoad.ROADVEHTYPE_TRUCK, AIStation.STATION_JOIN_ADJACENT)) {
						return tile;
					}
					// silently ignore error related to trying to build in a corner:
					if (AIError.GetLastError() != AIRoad.ERR_ROAD_DRIVE_THROUGH_WRONG_DIRECTION)
						Warning("Error building truck stop: " + AIError.GetLastErrorString());
					if (AIError.GetLastError() == AIError.ERR_LOCAL_AUTHORITY_REFUSES) {
						Warning("Pissed off authorities; planting tree...");
						AITile.DemolishTile(tile);
						AITile.PlantTree(tile);
						if (buildNewRoad) {
							AITile.DemolishTile(opening);
							AITile.PlantTree(opening);
						}
					} else {
						AITile.DemolishTile(tile);
					}
				}
			}
		}

		return null;
	}

	function buildIndustryLorryStation(industry, cargo, isSource) {
		Debug("Building cargo lorry station...");
		// Find empty square as close to industry as possible
		local area;
		
		if (isSource) {
			local targetLoc = AIIndustry.GetLocation(industry) + AIMap.GetTileIndex(0, 2);
			area = AITileList_IndustryProducing(industry, LORRY_RADIUS);
			area.Valuate(AITile.IsBuildable);
			area.KeepValue(1);
			area.Valuate(AITile.GetSlope);
			area.KeepValue(AITile.SLOPE_FLAT);
			//area.Valuate(AITile.GetCargoProduction, cargo, 1, 1, LORRY_RADIUS);
			area.Valuate(AITile.GetDistanceManhattanToTile, targetLoc);
			area.Sort(AIList.SORT_BY_VALUE, AIList.SORT_ASCENDING);
		} else {
			area = AITileList_IndustryAccepting(industry, LORRY_RADIUS);
			area.Valuate(AITile.IsBuildable);
			area.KeepValue(1);
			area.Valuate(AITile.GetSlope);
			area.KeepValue(AITile.SLOPE_FLAT);
			area.Valuate(AITile.GetCargoAcceptance, cargo, 1, 1, LORRY_RADIUS);
			area.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);
		}

		arrangeBalance(3 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_BUS_STOP) + 2 * 3 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_ROAD));

		if (area.Count()) {
			foreach (tile, val in area) {
				local opening = getRoadTile(tile);
				local buildNewRoad = false;
				if(opening == null) {
					opening = getRoadTileOption(tile);
					buildNewRoad = true;
				}
				if (opening) {
					if (AIRoad.BuildRoad(opening, tile) && AIRoad.BuildDriveThroughRoadStation(tile, opening, AIRoad.ROADVEHTYPE_TRUCK, AIStation.STATION_JOIN_ADJACENT)) {
						return tile;
					}
					// silently ignore error related to trying to build in a corner:
					if (AIError.GetLastError() != AIRoad.ERR_ROAD_DRIVE_THROUGH_WRONG_DIRECTION)
						Warning("Error building truck stop: " + AIError.GetLastErrorString());
					if (AIError.GetLastError() == AIError.ERR_LOCAL_AUTHORITY_REFUSES) {
						Warning("Pissed off authorities; planting tree...");
						AITile.DemolishTile(tile);
						AITile.PlantTree(tile);
						if (buildNewRoad) {
							AITile.DemolishTile(opening);
							AITile.PlantTree(opening);
						}
					} else {
						AITile.DemolishTile(tile);
					}
				}
			}
		}

		return null;
	}
	
	function buildTownLorryStation(location, cargo, isSource) {
		Debug("Building town lorry station...");
		// Find empty square as close to town centre as possible
		local curRange = 1;
		local maxRange = LORRY_RADIUS * 2; //TODO check value correctness
		local area = AITileList();
		
		arrangeBalance(2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_BUS_STOP) + AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_ROAD));

		while (curRange < maxRange) {
			//Debug("looking with range " + curRange);
			SafeAddRectangle(area, location, curRange, curRange - 1);
			area.Valuate(AITile.IsBuildable);
			area.KeepValue(1);
			if (!isSource || cargo == ::MAIL_ID) {
				area.Valuate(AITile.GetCargoAcceptance, cargo, 1, 1, LORRY_RADIUS);
				area.KeepAboveValue(7);
			}
			if (isSource || cargo == ::MAIL_ID) {
				area.Valuate(AITile.GetCargoProduction, cargo, 1, 1, LORRY_RADIUS);
				area.RemoveValue(0);
			}
			area.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);
			if (area.Count()) {
				//Debug("Found " + area.Count() + " possible options");
				foreach (tile, val in area) {
					//Debug("option has value: " + area.GetValue(t));
					local opening = getRoadTile(tile);
					if(opening) {
						// Debug("tile " + t + " has opening " + opening);
						if (AIRoad.BuildRoad(opening, tile) && AIRoad.BuildDriveThroughRoadStation(tile, opening, AIRoad.ROADVEHTYPE_TRUCK, AIStation.STATION_JOIN_ADJACENT)) {
							//AISign.BuildSign(t, ("Mail " + AITown.GetName(town)));
							return tile;
						}
						if (AIError.GetLastError() == AIRoad.ERR_ROAD_CANNOT_BUILD_ON_TOWN_ROAD)
							return buildNonInlineLorryStation(location, cargo, isSource);
						// silently ignore error related to trying to build in a corner:
						if (AIError.GetLastError() != AIRoad.ERR_ROAD_DRIVE_THROUGH_WRONG_DIRECTION)
							Warning("Error building town lorry station: " + AIError.GetLastErrorString());
						if (AIError.GetLastError() == AIError.ERR_LOCAL_AUTHORITY_REFUSES)
							return false;
						//Debug("tile " + t + " has wrong slope!!!");
					} //else {
						// Debug("tile " + t + " has no opening");
					//}
				}
				curRange++;
			} else {
				curRange++;
			}
		}
		Warning("No existing road to abuse found... trying to build elsewhere...");
		return buildNonInlineLorryStation(location, cargo, isSource);
	}
	
	function buildDepot(station) {
		Debug("Building road depot...");
		// Find empty square as close to station as possible
		local curRange = 1;
		
		arrangeBalance(2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_DEPOT) + AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_ROAD));

		while (curRange < 10) {
			local area = AITileList();
			//AILog.Info("looking with range " + curRange);
			SafeAddRectangle(area, station, curRange, curRange - 1);
			area.Valuate(AITile.IsBuildable);
			area.KeepValue(1);
			if (area.Count()) {
				//AILog.Info("Found " + area.Count() + " possible options");
				foreach (tile, val in area) {
					local opening = getRoadTile(tile);
					if(opening) {
						//AILog.Info("tile " + t + " has opening " + opening);
						if (AIRoad.BuildRoad(tile, opening) && AIRoad.BuildRoadDepot(tile, opening)) {
							//AISign.BuildSign(t, ("depot"));
							return tile;
						}
						//AILog.Info("tile " + t + " has wrong slope!!!");
					} //else {
						//AILog.Info("tile " + t + " has no opening");
					//}
				}
				curRange++; // the found options had no road connections; enlarge search area
			} else {
				curRange++;
			}
		}
		AILog.Info("No possible depot location found");
		return null;
	}
	
	function buildTownDepot(town) {
		Debug("Building town road depot in " + AITown.GetName(town));
		// Find empty square as close to center as possible
		local curRange = 1;
		local maxRange = sqrt(AITown.GetPopulation(town)/100) + 5; //TODO check value correctness
		local area = AITileList();
		local townLoc = AITown.GetLocation(town);
		
		arrangeBalance(2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_DEPOT) + AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_ROAD));

		while (curRange < maxRange) {
			//Debug("looking with range " + curRange);
			SafeAddRectangle(area, townLoc, curRange, curRange - 1);
			area.Valuate(AITile.IsBuildable);
			area.KeepValue(1);
			if (area.Count()) {
				//Debug("Found " + area.Count() + " possible options");
				foreach (tile, val in area) {
					local opening = getRoadTile(tile);
					if(opening) {
						//Debug("tile " + t + " has opening " + opening);
						if (AIRoad.BuildRoad(tile, opening) && AIRoad.BuildRoadDepot(tile, opening)) {
							//AISign.BuildSign(t, ("depot"));
							return tile;
						}
						//Debug("tile " + t + " has wrong slope!!!");
					} //else {
						//Debug("tile " + t + " has no opening");
					//}
				}
				curRange++; // the found options had no road connections; enlarge search area
			} else {
				curRange++;
			}
		}
		Debug("No possible depot location found");
		return null;
	}
	
	// TODO FIXME duplicate code with industry
	// tODO support passengers
	function connectTrade(tradeCenter, dest, cargo, TCIsSender, destIsTown) {
		if (getEngine(cargo) == -1) {
			Warning("Can't connect trade center as there's no suitable vehicle available");
			return false;
		}
		local vehList = AIVehicleList();
		vehList.Valuate(AIVehicle.GetVehicleType);
		vehList.KeepValue(AIVehicle.VT_ROAD);
		if (vehList.Count() > (maxVehicles - 5)) {
			Warning("Reached maximum amount of vehicles!");
			return false;
		}
		AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_ROAD);

		local station1 = buildTCStation(tradeCenter, cargo, true);
		if (!station1)
			return false; // bugger, no cleanup needed
		local loc2;
		local station2;
		if (destIsTown) {
			loc2 = AITown.GetLocation(dest);
			station2 = buildTownLorryStation(loc2, cargo, false);
		} else {
			loc2 = AIIndustry.GetLocation(dest);
			station2 = buildIndustryLorryStation(dest, cargo, false);
		}

		if (!station2)
			return false; // bugger TODO: cleanup station 1

		Debug("Route length is " + AITile.GetDistanceManhattanToTile(station1, station2));
		local starts = [station1];
		local goals = [station2];
		if(!connectTowns(starts, goals))
			return false;

		local depot1 = buildDepot(station1);
		if (!depot1)
			return false;

		myStationsTileList.AddItem(station1, depot1);
		local depot2 = null;

		if(destIsTown) {
			depot2 = myRoadDepotTownList.GetValue(dest);
			if (!depot2) {
				depot2 = buildTownDepot(dest);
				if(!depot2)
					return false;
				myRoadDepotTownList.SetValue(dest, depot2);
			}
		} else {
			depot2 = buildDepot(station2);
			if (!depot2)
				return false;
		}

		myStationsTileList.AddItem(station2, depot2);

		connectTowns([station1], [depot1]);
		connectTowns([station2], [depot2]);
		return connectIndustryStations(station1, station2, cargo);
	}

	function connectIndustry(source, dest, cargo, destIsTown) {
		if (getEngine(cargo) == -1) {
			Warning("Can't connect industries as there's no suitable truck available");
			return false;
		}
		local vehList = AIVehicleList();
		vehList.Valuate(AIVehicle.GetVehicleType);
		vehList.KeepValue(AIVehicle.VT_ROAD);
		if (vehList.Count() > (maxVehicles - 5)) {
			Warning("Reached maximum amount of vehicles!");
			return false;
		}
		AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_ROAD);

		local station1 = buildIndustryLorryStation(source, cargo, true);
		if (!station1)
			return false; // bugger, but no cleanup needed

		local build1 = true;
		local loc2;
		local station2;
		if (destIsTown) {
			loc2 = AITown.GetLocation(dest);
			station2 = buildTownLorryStation(loc2, cargo, false);
		} else {
			loc2 = AIIndustry.GetLocation(dest);
			station2 = buildIndustryLorryStation(dest, cargo, false);
		}

		if (!station2) {
			if (build1) {
				AITile.DemolishTile(station1);
				Warning("Cleaning up station after failure to connect...");
			}
			return false;
		}

		local build2 = true;

		Debug("Route between 2 industries is " + AITile.GetDistanceManhattanToTile(station1, station2));
		local starts = [station1];
		local goals = [station2];
		if(!connectTowns(starts, goals))
			return false;

		local depot1 = buildDepot(station1);
		if (!depot1)
			return false;

		myStationsTileList.AddItem(station1, depot1);
		local depot2 = null;

		if(destIsTown) {
			depot2 = myRoadDepotTownList.GetValue(dest);
			if (!depot2) {
				depot2 = buildTownDepot(dest);
				if(!depot2)
					return false;
				myRoadDepotTownList.SetValue(dest, depot2);
			}
		} else {
			depot2 = buildDepot(station2);
			if (!depot2)
				return false;
		}

		myStationsTileList.AddItem(station2, depot2);

		connectTowns([station1], [depot1]);
		connectTowns([station2], [depot2]);
		if (!connectIndustryStations(station1, station2, cargo)) {
			if (build1) {
				AITile.DemolishTile(station1);
				Warning("Cleaning up station after failure to connect...");
			}
			if (build2) {
				AIRoad.RemoveRoadStation(station2);	// TODO determine whether to remove road as well
				Warning("Cleaning up station after failure to connect...");
			}
			return false;
		}
		return true;
	}

	function connect2Towns(biggestTown, connectTown, cargo = 0) {
		if (cargo == 0)
			cargo = ::PASSENGER_ID;

		local vehList = AIVehicleList();
		vehList.Valuate(AIVehicle.GetVehicleType);
		vehList.KeepValue(AIVehicle.VT_ROAD);
		if (vehList.Count() > (maxVehicles - 5)) {
			Warning("Reached maximum amount of vehicles!");
			return false;
		}

		if (getEngine(cargo, AIMap.DistanceManhattan(AITown.GetLocation(biggestTown), AITown.GetLocation(connectTown))) == -1) {
			Warning("Can't connect towns as there's no suitable road vehicle available!");
			return false;
		}

		AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_ROAD);

		local station1 = buildPassengerStation(biggestTown);
		if (!station1) {
			Warning("!!! Building station failed !!!");
			return false;
		}
		local buildPass1 = true;
		local buildMail1 = false;
		local mail1 = null;
		local depot1 = myRoadDepotTownList.GetValue(biggestTown);
		if (!depot1) {
			depot1 = buildTownDepot(biggestTown);
			if(!depot1) {
				// to prevent it from choosing this "bad" town again
				myRoadDepotTownList.SetValue(biggestTown, -1);
				return false;
			}
			myRoadDepotTownList.SetValue(biggestTown, depot1);
			mail1 = buildMailStation(biggestTown);
			if (mail1) { 
				buildMail1 = true;
				myLorryTileList.AddItem(mail1, depot1);
				myDepotsTileList.AddItem(depot1, mail1);
			}
		} else {
			mail1 = myDepotsTileList.GetValue(myRoadDepotTownList.GetValue(biggestTown));
			if (!mail1) {
				mail1 = buildMailStation(biggestTown);
				if (mail1) { 
					buildMail1 = true;
					myLorryTileList.AddItem(mail1, depot1);
					myDepotsTileList.AddItem(depot1, mail1);
				}
			}
		}
		myStationsTileList.AddItem(station1, depot1);

		local station2 = buildPassengerStation(connectTown);
		if (!station2) {
			Warning("!!! Building station failed !!!");
			if (buildPass1) {
				Warning("Removing station");
				AIRoad.RemoveRoadStation(station1); // TODO determine whether to remove road...
			}
			if (buildMail1) {
				Warning("Removing station");
				AIRoad.RemoveRoadStation(mail1);
			}
			return false;
		}

		local buildPass2 = true;
		local buildMail2 = false;

		local mail2 = null;

		local depot2 = myRoadDepotTownList.GetValue(connectTown);
		if (!depot2) {
			depot2 = buildTownDepot(connectTown);
			if(!depot2)
				return false;
			myRoadDepotTownList.SetValue(connectTown, depot2);
			if(mail1)
				mail2 = buildMailStation(connectTown);
			if (mail2) { 
				buildMail2 = true;
				myLorryTileList.AddItem(mail2, depot2);
				myDepotsTileList.AddItem(depot2, mail2);
			}
		} else {
			mail2 = myDepotsTileList.GetValue(myRoadDepotTownList.GetValue(connectTown));
			if (!mail2 && mail1) {
				if(mail1)
					mail2 = buildMailStation(connectTown);
				if (mail2) { 
					buildMail2 = true;
					myLorryTileList.AddItem(mail2, depot2);
					myDepotsTileList.AddItem(depot2, mail2);
				}
			}
		}
		myStationsTileList.AddItem(station2, depot2);

		local distance = AITile.GetDistanceManhattanToTile(station1, station2);
		Debug("Route between 2 cities is " + distance);
		local opening1 = AIRoad.GetRoadDepotFrontTile(depot1);
		local opening2 = AIRoad.GetRoadDepotFrontTile(depot2);
		local starts = mail1 ? [opening1, mail1, station1] : [opening1, station1];
		local goals = mail2 ? [opening2, mail2, station2] : [opening2, station2];
		if(!connectTowns(starts, goals)) {
			Warning("Failed to connect; cleaning up...");
			if (buildPass1) {
				Warning("Removing station");
				AIRoad.RemoveRoadStation(station1); // TODO determine whether to remove road...
			}
			if (buildMail1) {
				Warning("Removing station");
				AIRoad.RemoveRoadStation(mail1);
			}
			if (buildPass2) {
				Warning("Removing station");
				AIRoad.RemoveRoadStation(station2); // TODO determine whether to remove road...
			}
			if (buildMail2) {
				Warning("Removing station");
				AIRoad.RemoveRoadStation(mail2);
			}
			return false;
		}
		if (mail1 && mail2) {
			// make sure the mail stations are connected to the road build for the passengers
			connectTowns([mail1], [opening1]);
			connectTowns([mail2], [opening2]);
			// sending out mail truck first, otherwise it might get stopped at a drive through station by our own bus
			lastMail1 = mail1;
			lastMail2 = mail2;
			connectMailStations(mail1, mail2);
		}
		// make sure depots are reachable as well
		connectTowns([station1], [opening1]);
		connectTowns([station2], [opening2]);

		//we picked BIG towns, lets try to build more busses in 1 go
		local pop1 = AITown.GetPopulation(biggestTown);
		local pop2 = AITown.GetPopulation(connectTown);
		// TODO: see whether this value max sense:
		local divider = AIEngine.GetCapacity(passengerBusEngine) * 5;
		local maxBussesPop = (pop1 < pop2 ? pop1 : pop2) / divider;	// built a bus for every 'divider' passengers in the smallest town
		local maxBussesDist = (distance / ((AIEngine.GetMaxSpeed(passengerBusEngine) / 10) + 1));	// tweaked to 10. May need to be raised to 12...
		local maxBusses = maxBussesPop < maxBussesDist ? maxBussesPop : maxBussesDist;
		maxBusses++;	// plus 2 busses for both stops
		Debug("Want to deploy " + (maxBusses * 2) + " busses for this route (pop/dist:) " + ((maxBussesPop + 1) * 2) + "/" + ((maxBussesDist + 1) * 2));
		while(maxBusses--)
			connectPassengerStations(station1, station2, cargo);
		if(maxBusses > 0) {
			lastMaxBusses = maxBusses;
			lastTown1 = station1;
			lastTown2 = station2;
		}

		return true;
	}
}

