/**
 *    This file is part of OtviAI.
 *
 *    OtviAI is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    OtviAI is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with OtviAI.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2008-2013 Otto Visser.
 * Suggestions/comments and bugs can go to: otviAI@otvi.nl
 */

require("util.nut");
require("transport.nut");
require("RoadFinder/RoadFinder.nut");

class OtviTram extends OtviTransport {
	myRoadDepotTownList = null;
	myDepotsTileList = null;
	myLorryTileList = null;
	myStationsTileList = null;

	passengerBusEngine = -1;
	mailLorryEngine = -1;

	expandMoneyMin = -1

	LORRY_RADIUS = -1;
	BUS_RADIUS = -1;
	
	lastMail1 = -1;
	lastMail2 = -1;
	lastMailVehicle = -1;
	lastTown1 = -1;
	lastTown2 = -1;
	lastPassengerVehicle = -1;
	lastMaxBusses = 0;

	maxVehicles = -1;
	roadFinder = null;

	function constructor() {
		Debug("Initializing tram transportation...");
		roadFinder = RoadFinder();

		myLorryTileList = AIList();
		myRoadDepotTownList = AITownList();
		myDepotsTileList = AIList();
		myStationsTileList = AIList();

		roadFinder.cost.tile = 1;
		roadFinder.cost.max_cost = 30000;		// TODO: still an estimate
		roadFinder.cost.no_existing_road = 5;	// The cost that is added to _cost_tile if no road exists yet.
		roadFinder.cost.turn = 10;				// The cost that is added to _cost_tile if the direction changes.
		roadFinder.cost.slope =	15;				// The extra cost if a road tile is sloped.
		roadFinder.cost.bridge_per_tile = 20;	// The cost per tile of a new bridge, this is added to _cost_tile.
		roadFinder.cost.tunnel_per_tile = 20;	// The cost per tile of a new tunnel, this is added to _cost_tile.
		roadFinder.cost.coast =	20;				// The extra cost for a coast tile.
		roadFinder.cost.max_bridge_length = 10;	// The maximum length of a bridge that will be build.
		roadFinder.cost.max_tunnel_length = 10;	// The maximum length of a tunnel that will be build.
		
		AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_TRAM);

		LORRY_RADIUS = AIStation.GetCoverageRadius(AIStation.STATION_TRUCK_STOP);
		BUS_RADIUS = AIStation.GetCoverageRadius(AIStation.STATION_BUS_STOP);
		maxVehicles = AIGameSettings.GetValue("vehicle.max_roadveh");
	}

	function getName() {
		return "tram";
	}

	function isOperational() {
		if (passengerBusEngine == -1 && mailLorryEngine == -1)
			return false;
		return true;
	}

	function canEnhance() {
		local vehList = AIVehicleList();
		vehList.Valuate(AIVehicle.GetVehicleType);
		vehList.KeepValue(AIVehicle.VT_ROAD);
		if (vehList.Count() >= maxVehicles) {
			//Warning("Reached maximum amount of vehicles!");
			return false;
		}
		if (lastMaxBusses > 0) {
			local pass1 = AIStation.GetStationID(lastTown1);
			local pass2 = AIStation.GetStationID(lastTown2);
			if(min(AIStation.GetCargoWaiting(pass1, ::PASSENGER_ID), AIStation.GetCargoWaiting(pass2, ::PASSENGER_ID)) > 4 * AIVehicle.GetCapacity(lastPassengerVehicle, ::PASSENGER_ID) && lastMaxBusses)
				return true;
			local lorry1 = AIStation.GetStationID(lastMail1);
			local lorry2 = AIStation.GetStationID(lastMail2);
			if(min(AIStation.GetCargoWaiting(lorry1, ::MAIL_ID), AIStation.GetCargoWaiting(lorry2, ::MAIL_ID)) > 3 * AIVehicle.GetCapacity(lastMailVehicle, ::MAIL_ID))
				return true;
		}
		return false;
	}

	function getEnhanceCost() {
		return expandMoneyMin;
	}

	function enhance() {
		local res = false;
		if(lastMail1 != -1 && lastMail2 != -1) {
			// add mail trams where there is plenty cargo
			local lorry1 = AIStation.GetStationID(lastMail1);
			local lorry2 = AIStation.GetStationID(lastMail2);
			if(min(AIStation.GetCargoWaiting(lorry1, ::MAIL_ID), AIStation.GetCargoWaiting(lorry2, ::MAIL_ID)) > 3 * AIVehicle.GetCapacity(lastMailVehicle, ::MAIL_ID)) {
				Debug("Adding mail trams at " + AIStation.GetName(lorry1) + " to " + AIStation.GetName(lorry2));
				res = connectMailStations(lastMail1, lastMail2);
			}
		}
		if(lastTown1 != -1 && lastTown2 != -1) {
			// add trams where there is plenty cargo
			local pass1 = AIStation.GetStationID(lastTown1);
			local pass2 = AIStation.GetStationID(lastTown2);
			if(min(AIStation.GetCargoWaiting(pass1, ::PASSENGER_ID), AIStation.GetCargoWaiting(pass2, ::PASSENGER_ID)) > 4 * AIVehicle.GetCapacity(lastPassengerVehicle, ::PASSENGER_ID) && lastMaxBusses) {
				Debug("Adding trams at " + AIStation.GetName(pass1) + " to " + AIStation.GetName(pass2));
				if(connectPassengerStations(lastTown1, lastTown2)) {
					lastMaxBusses--;
					res = true;
				}
			}
		}
		return res;
	}

	function getExpansionCost(cargo, distance = 100) {
		if (cargo == ::PASSENGER_ID)
			return 2 * AIEngine.GetPrice(passengerBusEngine) + 2 * AIEngine.GetPrice(mailLorryEngine) + 2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_BUS_STOP) + 2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_TRUCK_STOP) + 2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_DEPOT) + distance * 2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_ROAD);
		else
			return 4 * AIEngine.GetPrice(getEngine(cargo)) + 2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_TRUCK_STOP) + 2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_DEPOT) + distance * 2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_ROAD);
	}

	function getTownTransportCost(startLoc, endLoc) {
		if (passengerBusEngine == -1)  {
			Warning("Tram transport not yet available");
			return 255;
		}
		local vehList = AIVehicleList();
		vehList.Valuate(AIVehicle.GetVehicleType);
		vehList.KeepValue(AIVehicle.VT_ROAD);
		if (vehList.Count() > (maxVehicles - 5)) {
			Warning("Reached maximum amount of vehicles!");
			return 254;
		}

		local dist = AIMap.DistanceManhattan(AITown.GetLocation(startLoc), AITown.GetLocation(endLoc));
		//Debug("Tram from " + AITown.GetName(startLoc) + " to " + AITown.GetName(endLoc) + " has a distance of " + dist);
		if (dist > 200)
			return 10;

		if (dist > 100)
			return 5;

		if (dist < 42)
			return 1;

		return 3;
	}
		
	function getIndustryTransportCost(start, endLoc, cargoID) {
		if (getEngine(cargoID) == -1) {
			Warning("There is no tram available for " + AICargo.GetCargoLabel(cargoID));
			return 255;
		}
		local vehList = AIVehicleList();
		vehList.Valuate(AIVehicle.GetVehicleType);
		vehList.KeepValue(AIVehicle.VT_ROAD);
		if (vehList.Count() > (maxVehicles - 5)) {
			Warning("Reached maximum amount of vehicles!");
			return 254;
		}

		local dist = AIMap.DistanceManhattan(AITown.GetLocation(startLoc), AITown.GetLocation(endLoc));
		//Debug("Tram from " + AITown.GetName(startLoc) + " to " + AITown.GetName(endLoc) + " has a distance of " + dist);
		if (dist > 200)
			return 10;

		if (dist > 100)
			return 5;

		if (dist < 42)
			return 1;

		return 3;
	}
		
	function getEngine(cargoID, distance = 100) {
		local engine_list = AIEngineList(AIVehicle.VT_ROAD);
		engine_list.Valuate(AIEngine.CanRefitCargo, cargoID);
		engine_list.KeepValue(1);

		engine_list.Valuate(AIEngine.GetRoadType);
		engine_list.KeepValue(AIRoad.ROADTYPE_TRAM);

		// TODO: sort with valuator based on combination of breakdownLevel, speed and capacity
		/*if (::breakdownLevel == 0)
			engine_list.Valuate(AIEngine.GetCapacity);
		else
			engine_list.Valuate(AIEngine.GetReliability);
*/
		engine_list.Valuate(OtviTransport.profitValuator, cargoID, distance);	// TODO production, sub?
		engine_list.KeepAboveValue(0);

		engine_list.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);
		if (engine_list.IsEmpty()) {
			Warning("No suitable/profitable " + AICargo.GetCargoLabel(cargoID) + " tram available (yet?)!!");
			return -1;
		}
		
		local engine = engine_list.Begin();
		while (AIEngine.GetMaximumOrderDistance(engine) != 0 && AIEngine.GetMaximumOrderDistance(engine) < (distance + 20) && !(engine_list.IsEnd())) {
			Warning("Picked next engine due to order distance limitations on " + AIEngine.GetName(engine));
			engine = engine_list.Next();
		}

		if (AIEngine.IsValidEngine(engine)) {
			Debug("Picked tram engine for cargo " + AICargo.GetCargoLabel(cargoID) + ": " + AIEngine.GetName(engine));
			return engine;
		}
		return -1;
	}

	function updateEngines() {
		passengerBusEngine = getEngine(::PASSENGER_ID);
		mailLorryEngine = getEngine(::MAIL_ID);
		expandMoneyMin = 2 * AIEngine.GetPrice(passengerBusEngine);
		expandMoneyMin = (2 * AIEngine.GetPrice(mailLorryEngine)) < expandMoneyMin ? expandMoneyMin : 2 * AIEngine.GetPrice(mailLorryEngine);
	}

	/**
	 * 2 passenger stations build, make road and then place busses
	 */
	function connectTowns(starts, goals) {
		local distance = AIMap.DistanceManhattan(starts[0], goals[0]); 
		roadFinder.cost.max_cost = 15 * distance + 150;
		roadFinder.InitializePath(starts, goals);
		local path = false;
		local i = 0;
		local err;
		local iterations = 300 * distance; // if we pass this; something is wrong
		
		if (iterations > 200 * 150) {
			Warning("Limiting calculations to roughly 200 days");
			iterations = 200 * 150;
		}
		//local iterations = -1;	// TODO limit to reasonable amount, elevate if fails
		// TODO check whether higher values helps finding faster
		Debug("pathfinding for max " + roadFinder.cost.max_cost + " cost and iterations: " + iterations);
		local startTick = GetTick();
		while (path == false && iterations > 0) {
			path = roadFinder.FindPath(iterations);
			iterations -= 150;
		}
		Debug("Total ticks passed with " + iterations + " left: " + (GetTick() - startTick));

		if(!path) {
			if (path == false) {
				Error("!! No path found, too little iterations...");	// impossible atm
				roadFinder.reset();
				return false;
			} else {
				Warning("!! No path found, too expensive, retrying...");
				roadFinder.reset();
				roadFinder.cost.max_cost = 30 * distance + 300;
				Debug("pathfinding for max " + roadFinder.cost.max_cost + " cost and iterations: " + iterations);
				roadFinder.InitializePath(starts, goals);
				path = roadFinder.FindPath(iterations);
			}
		}
		if(!path) {
			roadFinder.reset();
			Error("Impossible to create route, way too expensive");
			return false;
		}
		Debug("path found for cost: " + path.GetCost() + ", building");

		arrangeBalance(2 * distance * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_ROAD));

		while (path != null) {
			local par = path.GetParent();
			if (par != null) {
				if (AIMap.DistanceManhattan(path.GetTile(), par.GetTile()) == 1 ) {
					if (!AIRoad.BuildRoad(path.GetTile(), par.GetTile())) {
						switch(AIError.GetLastError()) {
							case AIError.ERR_VEHICLE_IN_THE_WAY:
								Warning("Building road failed temporary: vehicle in the way...");
								// try again till the stupid vehicle moved?
								while(!AIRoad.BuildRoad(path.GetTile(), par.GetTile()));
								break;
							case AIRoad.ERR_ROAD_WORKS_IN_PROGRESS:
								Warning("Building road failed: road works in progress");
								// good, someone else is building this, skip tile
								break;
							case AIError.ERR_ALREADY_BUILT:
								// Debug("Building road failed: already built");
								// even better; someone else already built this; silent ignore
								break;
							case AIError.ERR_LAND_SLOPED_WRONG:
								Error("Building road failed: wrong land slope");
								// not good; redo from start or terraform?
								return connectTowns(starts, goals);
							case AIError.ERR_AREA_NOT_CLEAR:
								Warning("Building road failed: not clear, demolishing tile...");
								AISign.BuildSign(path.GetTile(), "I'm on your land, demolishing your stuffs");
								// demolish and retry?
								if (AITile.DemolishTile(path.GetTile())) {
									AIRoad.BuildRoad(path.GetTile(), par.GetTile());
								} else {	// happens when other ai builds after calc
									return connectTowns(starts, goals);
								}
								break;
							case AIRoad.ERR_ROAD_ONE_WAY_ROADS_CANNOT_HAVE_JUNCTIONS:
								Error("Building road failed: no junctions on one way roads allowed");
								// can't happen? Not building those
								return false;
							case AIError.ERR_NOT_ENOUGH_CASH:
								Warning("Panic: Out of cash!!");
								local count = 10;
								local res = false;
								arrangeBalance(5 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_ROAD));
								while(!(res = AIRoad.BuildRoad(path.GetTile(), par.GetTile())) && count-- > 0)
									::INSTANCE.Sleep(100);

								if (!res)
									return false;

								break;
							default:
								Warning("Unknown error during road building " + AIError.GetLastErrorString());
								break;	// no clue what this is; silent ignore for now
						}
					}
				} else {
					/* Build a bridge or tunnel. */
					if (!AIBridge.IsBridgeTile(path.GetTile()) && !AITunnel.IsTunnelTile(path.GetTile())) {
						/* If it was a road tile, demolish it first. Do this to work around expended roadbits. */
						if (AIRoad.IsRoadTile(path.GetTile())) AITile.DemolishTile(path.GetTile());
						if (AITunnel.GetOtherTunnelEnd(path.GetTile()) == par.GetTile()) {
							if (!AITunnel.BuildTunnel(AIVehicle.VT_ROAD, path.GetTile())) {
								/* An error occured while building a tunnel. TODO: handle it. */
								// TODO expected to fail once per tunnel atm
								// AISign.BuildSign(path.GetTile(), ("Tunnel fail"));
								Error("Couldn't build tunnel (normal once per tunnel atm");
							}
						} else {
							local bridge_list = AIBridgeList_Length(AIMap.DistanceManhattan(path.GetTile(), par.GetTile()) + 1);
							bridge_list.Valuate(AIBridge.GetMaxSpeed);
							bridge_list.Sort(AIList.SORT_BY_VALUE, false);
							if (!AIBridge.BuildBridge(AIVehicle.VT_ROAD, bridge_list.Begin(), path.GetTile(), par.GetTile())) {
								/* An error occured while building a bridge. TODO: handle it. */
								// TODO expected to fail once per bridge atm
								// AISign.BuildSign(path.GetTile(), ("Bridge fail"));
								Error("Couldn't build bridge (normal once per bridge atm)");
							}
						}
					}
				}
			}
			path = par;
		}

		/*while (path != null) {
			if (path.GetParent() != null) {
				local par = path.GetParent().GetTile();
				if (!AIRoad.AreRoadTilesConnected(path.GetTile(), par) && (!AIBridge.IsBridgeTile(path.GetTile()) || AIBridge.GetOtherBridgeEnd(path.GetTile()) != par)) {
					AIRoad.BuildRoad(path.GetTile(), par);
				}
			}
			path = path.GetParent();
		}*/
		Debug("path done");
		return true;
	}

	function connectPassengerStations(station1, station2, cargo = 0) {		
		if (cargo == 0)
			cargo = ::PASSENGER_ID;
		else {
			if (AICargo.HasCargoClass(cargo, AICargo.CC_PASSENGERS)) {
				Warning("Cargo is not for normal passengers: " + cargo);
			} else {
				Warning("Cargo is not for people: " + cargo);
				return true;
			}
		}

		local price = AIEngine.GetPrice(passengerBusEngine);
		arrangeBalance(price);
		local bus1 = AIVehicle.BuildVehicle(myStationsTileList.GetValue(station1), passengerBusEngine);
		if (!AIVehicle.IsValidVehicle(bus1))
			return false;
		if(!AIVehicle.RefitVehicle(bus1, cargo)) {
			Error("Error refitting tram: " + AIError.GetLastErrorString());
			return false;
		}

		Debug("Built tram: " + bus1);
		lastPassengerVehicle = bus1;
		AIOrder.AppendOrder(bus1, station1, AIOrder.OF_FULL_LOAD | AIOrder.OF_NON_STOP_INTERMEDIATE);
		if (::breakdownLevel != 0)
			AIOrder.AppendOrder(bus1, station1, AIOrder.OF_GOTO_NEAREST_DEPOT | AIOrder.OF_SERVICE_IF_NEEDED);
		AIOrder.AppendOrder(bus1, station2, AIOrder.OF_FULL_LOAD | AIOrder.OF_NON_STOP_INTERMEDIATE);
		if (::breakdownLevel != 0)
			AIOrder.AppendOrder(bus1, station2, AIOrder.OF_GOTO_NEAREST_DEPOT | AIOrder.OF_SERVICE_IF_NEEDED);
		AIVehicle.StartStopVehicle(bus1);

		arrangeBalance(price);
		local bus2 = AIVehicle.BuildVehicle(myStationsTileList.GetValue(station2), passengerBusEngine);
		if (!AIVehicle.IsValidVehicle(bus2))
			return false; 	// TODO change to return 1 for 1 success
		if(!AIVehicle.RefitVehicle(bus2, cargo)) {
			Error("Error refitting tram: " + AIError.GetLastErrorString());
			return false; // TODO
		}
		Debug("Built tram: " + bus2);
		AIOrder.AppendOrder(bus2, station2, AIOrder.OF_FULL_LOAD | AIOrder.OF_NON_STOP_INTERMEDIATE);
		if (::breakdownLevel != 0)
			AIOrder.AppendOrder(bus2, station2, AIOrder.OF_GOTO_NEAREST_DEPOT | AIOrder.OF_SERVICE_IF_NEEDED);
		AIOrder.AppendOrder(bus2, station1, AIOrder.OF_FULL_LOAD | AIOrder.OF_NON_STOP_INTERMEDIATE);
		if (::breakdownLevel != 0)
			AIOrder.AppendOrder(bus2, station1, AIOrder.OF_GOTO_NEAREST_DEPOT | AIOrder.OF_SERVICE_IF_NEEDED);
		AIVehicle.StartStopVehicle(bus2);

		return true;	// TODO change to 2
	}
	
	function connectMailStations(lorry1, lorry2) {
		local price = AIEngine.GetPrice(mailLorryEngine);
		arrangeBalance(price);
		local mail1 = AIVehicle.BuildVehicle(myLorryTileList.GetValue(lorry1), mailLorryEngine);
		if(!AIVehicle.IsValidVehicle(mail1))
			return false;
		if(!AIVehicle.RefitVehicle(mail1, ::MAIL_ID)) {
			Error("Error refitting tram: " + AIError.GetLastErrorString());
			return false;
		}

		Debug("Built a mail tram: " + mail1 + " at location " + lorry1);
		lastMailVehicle = mail1;
		AIOrder.AppendOrder(mail1, lorry1, AIOrder.OF_FULL_LOAD | AIOrder.OF_NON_STOP_INTERMEDIATE);
		if (::breakdownLevel != 0)
			AIOrder.AppendOrder(mail1, lorry1, AIOrder.OF_GOTO_NEAREST_DEPOT | AIOrder.OF_SERVICE_IF_NEEDED);
		AIOrder.AppendOrder(mail1, lorry2, AIOrder.OF_FULL_LOAD | AIOrder.OF_NON_STOP_INTERMEDIATE);
		if (::breakdownLevel != 0)
			AIOrder.AppendOrder(mail1, lorry2, AIOrder.OF_GOTO_NEAREST_DEPOT | AIOrder.OF_SERVICE_IF_NEEDED);
		AIVehicle.StartStopVehicle(mail1);

		arrangeBalance(price);
		local mail2 = AIVehicle.BuildVehicle(myLorryTileList.GetValue(lorry2), mailLorryEngine);
		if(!AIVehicle.IsValidVehicle(mail2))
			return false;
		if(!AIVehicle.RefitVehicle(mail2, ::MAIL_ID)) {
			Error("Error refitting tram: " + AIError.GetLastErrorString());
			return false;
		}

		Debug("Built a mail tram: " + mail2 + " at location " + lorry2);
		AIOrder.AppendOrder(mail2, lorry2, AIOrder.OF_FULL_LOAD | AIOrder.OF_NON_STOP_INTERMEDIATE);
		if (::breakdownLevel != 0)
			AIOrder.AppendOrder(mail2, lorry2, AIOrder.OF_GOTO_NEAREST_DEPOT | AIOrder.OF_SERVICE_IF_NEEDED);
		AIOrder.AppendOrder(mail2, lorry1, AIOrder.OF_FULL_LOAD | AIOrder.OF_NON_STOP_INTERMEDIATE);
		if (::breakdownLevel != 0)
			AIOrder.AppendOrder(mail2, lorry1, AIOrder.OF_GOTO_NEAREST_DEPOT | AIOrder.OF_SERVICE_IF_NEEDED);
		AIVehicle.StartStopVehicle(mail2);

		return true;
	}
	
	static function getRoadTile(t) {
		local adjacent = AITileList();
		adjacent.AddTile(t - AIMap.GetTileIndex(1,0));
		adjacent.AddTile(t - AIMap.GetTileIndex(0,1));
		adjacent.AddTile(t - AIMap.GetTileIndex(-1,0));
		adjacent.AddTile(t - AIMap.GetTileIndex(0,-1));
		adjacent.Valuate(AIRoad.IsRoadTile);
		adjacent.KeepValue(1);
		adjacent.Valuate(AIRoad.IsRoadDepotTile);
		adjacent.KeepValue(0);
		adjacent.Valuate(AIRoad.IsRoadStationTile);
		adjacent.KeepValue(0);
		if (adjacent.Count())
			return adjacent.Begin();
		else
			return null;
	}

	/**
	 * hack to support basic sqrt function for other functions
	 * TODO: consider using lookup table for speed reasons (as suggested by Truebrain)
	 * TODO: useless now?
	 */
/*	function sqrt(i) {
		if (i == 0)
			return 0;   // Avoid divide by zero
		local n = (i / 2) + 1;       // Initial estimate, never low
		local n1 = (n + (i / n)) / 2;
		while (n1 < n) {
			n = n1;
			n1 = (n + (i / n)) / 2;
		}
		return n;
	}*/

	function buildPassengerStation(town) {
		Debug("Building tram stop...");
		// Find empty square as close to town centre as possible
		local spotFound = false;
		local curRange = 1;
		local maxRange = sqrt(AITown.GetPopulation(town)/100) + 2; //TODO check value correctness 
		local area = AITileList();
		local townLoc = AITown.GetLocation(town);
		
		arrangeBalance(2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_BUS_STOP) + 2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_ROAD));

		while (curRange < maxRange) {
			//Debug("looking with range " + curRange);
			SafeAddRectangle(area, townLoc, curRange, curRange - 1);

			AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_ROAD);	// switch for neighbourcount function
			area.Valuate(AIRoad.IsRoadTile);
			area.KeepValue(1);
			//Debug("Town road tiles count (road mode): " + area.Count());
			area.Valuate(AIRoad.IsDriveThroughRoadStationTile);
			area.KeepValue(0);
			//Debug("Town road tiles count without station (tram mode): " + area.Count());
			area.Valuate(AIRoad.GetNeighbourRoadCount);
			area.KeepValue(2);	// entrance and exit; allow 1 as well?
			//Debug("Town road tiles count with 2 neighbours (tram mode): " + area.Count());
			area.Valuate(AITile.GetCargoAcceptance, ::PASSENGER_ID, 1, 1, BUS_RADIUS);
			area.KeepAboveValue(7);
			area.Valuate(AITile.GetCargoProduction, ::PASSENGER_ID, 1, 1, BUS_RADIUS);
			area.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);

			AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_TRAM);	// switch back
			if (area.Count()) {
				Debug("Found " + area.Count() + " possible options");
				for (local t = area.Begin(); !area.IsEnd(); t = area.Next()) {
					local opening = getRoadTile(t);
					if(opening) {
						//Debug("tile " + t + " has opening " + opening);
						if (AIRoad.BuildDriveThroughRoadStation(t, opening, AIRoad.ROADVEHTYPE_BUS, AIStation.STATION_JOIN_ADJACENT)) {
							AIRoad.BuildRoad(t, AIRoad.GetRoadStationFrontTile(t));
							AIRoad.BuildRoad(t, AIRoad.GetDriveThroughBackTile(t));
							//AISign.BuildSign(t, ("bus " + AITown.GetName(town)));
							return t;
						}
						Warning("Error building tram stop: " + AIError.GetLastErrorString());
						if (AIError.GetLastError() == AIError.ERR_LOCAL_AUTHORITY_REFUSES) 
							return false;
						//AISign.BuildSign(t, AIError.GetLastErrorString());

					}/* else {
						Debug("tile " + t + " has no opening");
					}*/
				}
				curRange++;
			} else {
				curRange++;
			}
		}
		Warning("No possible tram stop location found");
		return false;
	}
	
	function buildMailStation(town) {
		if (mailLorryEngine == -1) {
			Warning("No mail engine available, not building mail stations");
			return null;
		}

		arrangeBalance(2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_TRUCK_STOP) + 2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_ROAD));

		Debug("Building tram mail station...");
		// Find empty square as close to town centre as possible
		local spotFound = false;
		local curRange = 1;
		local maxRange = sqrt(AITown.GetPopulation(town)/100) + 2; //TODO check value correctness
		local area = AITileList();
		local townLoc = AITown.GetLocation(town);
		
		while (curRange < maxRange) {
			//Debug("looking with range " + curRange);
			SafeAddRectangle(area, townLoc, curRange, curRange - 1);
			AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_ROAD);	// switch for neighbourcount function
			area.Valuate(AIRoad.IsRoadTile);
			area.KeepValue(1);
			area.Valuate(AIRoad.IsDriveThroughRoadStationTile);
			area.KeepValue(0);
			area.Valuate(AIRoad.GetNeighbourRoadCount);
			area.KeepValue(2);	// entrance and exit; allow 1 as well?
			area.Valuate(AITile.GetCargoAcceptance, ::MAIL_ID, 1, 1, LORRY_RADIUS);
			area.KeepAboveValue(7);
			area.Valuate(AITile.GetCargoProduction, ::MAIL_ID, 1, 1, LORRY_RADIUS);
			area.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);
			AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_TRAM);	// switch back
			if (area.Count()) {
				//Debug("Found " + area.Count() + " possible options");
				for (local t = area.Begin(); !area.IsEnd(); t = area.Next()) {
					//Debug("option has value: " + area.GetValue(t));
					local opening = getRoadTile(t);
					if(opening) {
						// Debug("tile " + t + " has opening " + opening);
						if (AIRoad.BuildDriveThroughRoadStation(t, opening, AIRoad.ROADVEHTYPE_TRUCK, AIStation.STATION_JOIN_ADJACENT)) {
							AIRoad.BuildRoad(t, AIRoad.GetRoadStationFrontTile(t));
							AIRoad.BuildRoad(t, AIRoad.GetDriveThroughBackTile(t));
							//AISign.BuildSign(t, ("Mail " + AITown.GetName(town)));
							return t;
						}
						//Debug("tile " + t + " has wrong slope!!!");
					} //else {
						// Debug("tile " + t + " has no opening");
					//}
				}
				curRange++;
			} else {
				curRange++;
			}
		}
		Debug("No possible mail location found");
		return null;
	}
	
	function buildDepot(station) {
		Debug("Building depot...");
		// Find empty square as close to station as possible
		local spotFound = false;
		local curRange = 1;
		local area = AITileList();
		
		arrangeBalance(2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_DEPOT) + AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_ROAD));

		while (curRange < 10) {
			//AILog.Info("looking with range " + curRange);
			area.AddRectangle(station - AIMap.GetTileIndex(curRange, curRange), station + AIMap.GetTileIndex(curRange, curRange));
			area.Valuate(AITile.IsBuildable);
			area.KeepValue(1);
			if (area.Count()) {
				//AILog.Info("Found " + area.Count() + " possible options");
				for (local t = area.Begin(); !area.IsEnd(); t = area.Next()) {
					local opening = getRoadTile(t);
					if(opening) {
						//AILog.Info("tile " + t + " has opening " + opening);
						if (AIRoad.BuildRoad(t, opening) && AIRoad.BuildRoadDepot(t, opening)) {
							//AISign.BuildSign(t, ("depot"));
							return t;
						}
						//AILog.Info("tile " + t + " has wrong slope!!!");
					} //else {
						//AILog.Info("tile " + t + " has no opening");
					//}
				}
				curRange++; // the found options had no road connections; enlarge search area
			} else {
				curRange++;
			}
		}
		AILog.Info("No possible depot location found");
		return null;
	}
	
	function buildTownDepot(station, town) {
		Debug("Building town depot...");
		// Find empty square as close to station as possible
		local spotFound = false;
		local curRange = 1;
		local maxRange = sqrt(AITown.GetPopulation(town)/100) + 4; //TODO check value correctness
		local area = AITileList();
		local townLoc = AITown.GetLocation(town);
		
		arrangeBalance(2 * AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_DEPOT) + AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_ROAD));

		while (curRange < maxRange) {
			//Debug("looking with range " + curRange);
			SafeAddRectangle(area, townLoc, curRange, curRange - 1);
			area.Valuate(AITile.IsBuildable);
			area.KeepValue(1);
			if (area.Count()) {
				//Debug("Found " + area.Count() + " possible options");
				for (local t = area.Begin(); !area.IsEnd(); t = area.Next()) {
					local opening = getRoadTile(t);
					if(opening) {
						//Debug("tile " + t + " has opening " + opening);
						if (AIRoad.BuildRoadDepot(t, opening)) {
							AIRoad.BuildRoad(opening, t);
							//AISign.BuildSign(t, ("depot"));
							return t;
						}
						//Debug("tile " + t + " has wrong slope!!!");
					} //else {
						//Debug("tile " + t + " has no opening");
					//}
				}
				curRange++; // the found options had no road connections; enlarge search area
			} else {
				curRange++;
			}
		}
		Debug("No possible depot location found");
		return null;
	}

	function connect2Towns(biggestTown, connectTown, cargo = 0) {
		local vehList = AIVehicleList();
		vehList.Valuate(AIVehicle.GetVehicleType);
		vehList.KeepValue(AIVehicle.VT_ROAD);
		if (vehList.Count() > (maxVehicles - 5)) {
			Warning("Reached maximum amount of vehicles!");
			return false;
		}
		AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_TRAM);

		local station1 = buildPassengerStation(biggestTown);
		local mail1;
		if (!station1) {
			Warning("!!! Building station failed !!!");
			return false;
		}
		local depot1 = myRoadDepotTownList.GetValue(biggestTown);
		if (!depot1) {
			mail1 = buildMailStation(biggestTown);
			depot1 = buildTownDepot(station1, biggestTown);
			if(!depot1) {
				// to prevent it from choosing this "bad" town again
				myRoadDepotTownList.SetValue(biggestTown, -1);
				return false;
			}
			myRoadDepotTownList.SetValue(biggestTown, depot1);
			if (mail1) { 
				myLorryTileList.AddItem(mail1, depot1);
				myDepotsTileList.AddItem(depot1, mail1);
			}
		} else {
			mail1 = myDepotsTileList.GetValue(myRoadDepotTownList.GetValue(biggestTown));
			if (!mail1) {
				mail1 = buildMailStation(biggestTown);
				if (mail1) { 
					myLorryTileList.AddItem(mail1, depot1);
					myDepotsTileList.AddItem(depot1, mail1);
				}
			}
		}
		myStationsTileList.AddItem(station1, depot1);

		local station2 = buildPassengerStation(connectTown);
		if (!station2) {
			Warning("!!! Building station failed !!!");
			return false;
		}

		local mail2;

		local depot2 = myRoadDepotTownList.GetValue(connectTown);
		if (!depot2) {
			if(mail1)
				mail2 = buildMailStation(connectTown);
			depot2 = buildTownDepot(station2, connectTown);
			if(!depot2)
				return false;
			myRoadDepotTownList.SetValue(connectTown, depot2);
			if (mail2) { 
				myLorryTileList.AddItem(mail2, depot2);
				myDepotsTileList.AddItem(depot2, mail2);
			}
		} else {
			mail2 = myDepotsTileList.GetValue(myRoadDepotTownList.GetValue(connectTown));
			if (!mail2 && mail1) {
				if(mail1)
					mail2 = buildMailStation(connectTown);
				if (mail2) { 
					myLorryTileList.AddItem(mail2, depot2);
					myDepotsTileList.AddItem(depot2, mail2);
				}
			}
		}
		myStationsTileList.AddItem(station2, depot2);

		local distance = AITile.GetDistanceManhattanToTile(station1, station2);
		Debug("Route between 2 cities is " + distance);
		local opening1 = AIRoad.GetRoadDepotFrontTile(depot1);
		local opening2 = AIRoad.GetRoadDepotFrontTile(depot2);

		// connect tram stops to depot
		if (!connectTowns([opening1], [station1]))
			return false;
		if (!connectTowns([opening2], [station2]))
			return false;

		local starts = mail1 ? [opening1, mail1, station1] : [opening1, station1];
		local goals = mail2 ? [opening2, mail2, station2] : [opening2, station2];
		if(!connectTowns(starts, goals)) {
			return false;
		}
		if (mail1 && mail2) {
			// make sure the mail stations are connected to the depots
			connectTowns([opening1], [mail1]);
			connectTowns([opening2], [mail2]);
			// sending out mail tram first, otherwise it might get stuck behind another tram
			lastMail1 = mail1;
			lastMail2 = mail2;
			connectMailStations(mail1, mail2);
		}

		//we picked BIG towns, lets try to build more busses in 1 go
		local pop1 = AITown.GetPopulation(biggestTown);
		local pop2 = AITown.GetPopulation(connectTown);
		// TODO: see whether this value max sense:
		local divider = AIEngine.GetCapacity(passengerBusEngine) * 5;
		local maxBussesPop = (pop1 < pop2 ? pop1 : pop2) / divider;	// built a bus for every hundred passengers in the smallest town
		local maxBussesDist = (distance / (AIEngine.GetMaxSpeed(passengerBusEngine) / 10));	// tweaked to 10. May need to be raised to 12...
		local maxBusses = maxBussesPop < maxBussesDist ? maxBussesPop : maxBussesDist;
		maxBusses++;	// plus 2 busses for both stops
		Debug("Want to deploy " + (maxBusses * 2) + " trams for this route (pop/dist:) " + ((maxBussesPop + 1) * 2) + "/" + ((maxBussesDist + 1) * 2));
		while(maxBusses--)
			connectPassengerStations(station1, station2, cargo);
		if(maxBusses) {
			lastMaxBusses = maxBusses;
			lastTown1 = station1;
			lastTown2 = station2;
		}

		return true;
	}
}

