/*
	MogulAI - an artificial intelligence for OpenTTD
	Copyright (C) 2009 - 2010 Kazantsev Lev (Dezmond_snz)

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License along
	with this program; if not, write to the Free Software Foundation, Inc.,
	51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

require ("Util.nut");

class PathBuilder
{
	static RESULT_NONE = 0;
	static RESULT_DONE = 1;
	static RESULT_ERROR = 2;
	
	_path = null;
	_idx = -1;
	_vehicleInTheWayWaitTime = -1;
	
	constructor(path)
	{
		_path = path;
		_vehicleInTheWayWaitTime = -1;
		_idx = 0;
	}
	
	function BuildRoad()
	{
		local result = PathBuilder.RESULT_NONE;
		
		do{
			result = this.BuildRoadStep();
		} while (result == PathBuilder.RESULT_NONE);
		
		return result;
	}
	function BuildRoadStep()
	{
		local cur = _path[_idx];
		local next = _path[_idx+1];
		
		if (!FundsManager.CheckMoneyConstruction( getCost(cur._tile, next._tile, next._reachType), false ))
			return PathBuilder.RESULT_NONE;
		
		if (next._reachType == ReachableType.Normal)
		{
			if (!AIRoad.BuildRoad(cur._tile, next._tile))
			{
				local err = AIError.GetLastError();
	      		if (err == AIError.ERR_VEHICLE_IN_THE_WAY)
	      		{
	      			if (_vehicleInTheWayWaitTime == -1)
	      				_vehicleInTheWayWaitTime = 200; // wait some time...
	      			else if (_vehicleInTheWayWaitTime == 0) // time is out!
	      			{
	      				LogError("Cannot build road 'cuz there is vehicles in the way!");
	      				_vehicleInTheWayWaitTime = -1;
	      				return PathBuilder.RESULT_ERROR;
	      			}
	      			
	      			_vehicleInTheWayWaitTime--;
	      			AIController.Sleep(1);
      				return PathBuilder.RESULT_NONE;
	      		} // if (err == AIError.ERR_VEHICLE_IN_THE_WAY)
	      		else if (
	      						err == AIRoad.ERR_ROAD_WORKS_IN_PROGRESS ||
								err == AIRoad.ERR_ROAD_ONE_WAY_ROADS_CANNOT_HAVE_JUNCTIONS ||
								err == AIError.ERR_AREA_NOT_CLEAR ||
			      				err == AIError.ERR_FLAT_LAND_REQUIRED ||
	    		  				err == AIError.ERR_LAND_SLOPED_WRONG ||
	    		  				err == AIError.ERR_SITE_UNSUITABLE
	    		  			)
	    		{
	    			LogError("Cannot build road: "+ AIError.GetLastErrorString());
	    			LogWarning(Util.TileToString(cur._tile)+" -> "+Util.TileToString(next._tile));
	      			return PathBuilder.RESULT_ERROR;
				} // else if (
			} // if (!AIRoad.BuildRoad(_path._tile, par._tile))
		} // if (_path._reachType == ReachableType.Normal)
		else if (next._reachType == ReachableType.Bridge)
		{
			if (!AIBridge.IsBridgeTile(cur._tile))
			{
				local list = AIBridgeList_Length(AIMap.DistanceManhattan(cur._tile, next._tile)+1);
				if (list.Count() == 0)
				{
					LogError ("No suitable bridge types exists!");
					return PathBuilder.RESULT_ERROR;
				}
				list.Valuate(AIBridge.GetMaxSpeed);
				list.Sort(AIAbstractList.SORT_BY_VALUE, AIAbstractList.SORT_DESCENDING);
				
				if (!AITile.IsBuildable(cur._tile))		AITile.DemolishTile(cur._tile);
				if (!AITile.IsBuildable(next._tile))	AITile.DemolishTile(next._tile);
				if (!AIBridge.BuildBridge(AIVehicle.VT_ROAD, list.Begin(), cur._tile, next._tile))
				{
					LogError("Cannot build bridge: "+ AIError.GetLastErrorString());
	    			LogWarning(Util.TileToString(cur._tile)+" -> "+Util.TileToString(next._tile));
					return PathBuilder.RESULT_ERROR;
				}
			}
		}
		else if (next._reachType == ReachableType.Tunnel)
		{
			if (!AITunnel.IsTunnelTile(cur._tile))
			{
				if (!AITile.IsBuildable(cur._tile))		AITile.DemolishTile(cur._tile);
				if (!AITile.IsBuildable(next._tile))	AITile.DemolishTile(next._tile);
				if (!AITunnel.BuildTunnel(AIVehicle.VT_ROAD, cur._tile))
				{
					LogError("Cannot build bridge: "+ AIError.GetLastErrorString());
	    			LogWarning(Util.TileToString(cur._tile)+" -> "+Util.TileToString(next._tile));
					return PathBuilder.RESULT_ERROR;
				}
			}
		}
		
		_idx++;
		if (_idx+1 >= _path.len())
			return PathBuilder.RESULT_DONE;
		return PathBuilder.RESULT_NONE;
	}
	
	function CheckPath()
	{
		local foo = AITestMode();
		local cost = AIAccounting();
		
		local path = _path;
		for (local i=0; i< path.len()-1; i++)
		{
			local cur = path[i];
			local next = path[i+1];
			
			if (next._reachType == ReachableType.Normal)
			{
				if (!AIRoad.BuildRoad(cur._tile, next._tile))
				{
					local err = AIError.GetLastError();
					if (
		      				err == AIRoad.ERR_ROAD_WORKS_IN_PROGRESS ||
							err == AIRoad.ERR_ROAD_ONE_WAY_ROADS_CANNOT_HAVE_JUNCTIONS ||
							err == AIError.ERR_AREA_NOT_CLEAR ||
				    		err == AIError.ERR_FLAT_LAND_REQUIRED ||
		    				err == AIError.ERR_LAND_SLOPED_WRONG ||
		    				err == AIError.ERR_SITE_UNSUITABLE
		    			)
		      			return false;
				}
				else if (next._reachType == ReachableType.Bridge)
				{
					if (AIBridge.IsBridgeTile(cur._tile) && AIBridge.GetOtherBridgeEnd(cur._tile) != next._tile)
						return false;
				}
				else if (next._reachType == ReachableType.Tunnel)
				{
					if (AITunnel.GetOtherTunnelEnd(cur._tile) != next._tile)
						return false;
					if (!AITunnel.BuildTunnel(AIVehicle.VT_ROAD, cur._tile))
						return false;
				}
			}
		}
		return cost.GetCosts();
	}
	
	function getCost(start, end, reachType)
	{
		local foo = AITestMode();
		local cost = AIAccounting();
		if (reachType == ReachableType.Normal)
			AIRoad.BuildRoad(start, end);
		else if (reachType == ReachableType.Tunnel)
			AITunnel.BuildTunnel(AIVehicle.VT_ROAD, start);
		else if (reachType == ReachableType.Bridge)
		{
			local list = AIBridgeList_Length(AIMap.DistanceManhattan(start, end)+1);
			if (list.Count() > 0)
			{
				list.Valuate(AIBridge.GetMaxSpeed);
				list.Sort(AIAbstractList.SORT_BY_VALUE, AIAbstractList.SORT_DESCENDING);
				AIBridge.BuildBridge(AIVehicle.VT_ROAD, list.Begin() ,start, end);
			}
		}
		return cost.GetCosts();		
	}
	
	function LogInfo(msg)
	{
		AILog.Info("PathBuilder: "+msg);
	}
	function LogWarning(msg)
	{
		AILog.Warning("PathBuilder: "+msg);
	}
	function LogError(msg)
	{
		AILog.Error("PathBuilder: "+msg);
	}
	
	function LogDebug(msg)
	{
		if (DEBUG)
			AILog.Info("(DBG) PathBuilder: "+msg);
	}
	
}