/*
	MogulAI - an artificial intelligence for OpenTTD
	Copyright (C) 2009 - 2010 Kazantsev Lev (Dezmond_snz)

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License along
	with this program; if not, write to the Free Software Foundation, Inc.,
	51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

require("Util.nut");

const CARGO_CHECK_MAX = 10;
const MAX_QUEUE_TIME=8;
const DEBUG = 1;

class Route
{
	_source = null;
	_dest = null;
	_sourceStation=null;
	_destStation=null;
	_depotTile = null;
	_cargo = null;
	_vehicleList = null;
	_vehicleToSellList = null;
	
	_cargoCheckCount = 0;
	_stationCount = 1;
	
	_isClosed = false;
	_newEngine = null;
	_timeEngineCheck = 0;
	
	_queuedVehicles = {};
	_checkQueueTime = 0;
	_enoughMoney = true;
	
	constructor(source, dest, sourceStation, destStation, cargo, depot, stationCount = 1, newEngine = null)
	{
		_source = source;
		_dest = dest;
		_sourceStation = sourceStation;
		_destStation = destStation;
		_depotTile = depot;
		_cargo = cargo;
		_cargoCheckCount = 0;
		_stationCount=stationCount;
		_isClosed = false;
		_newEngine = newEngine;
		_timeEngineCheck = AIController.GetTick();
		UpdateVehicleList();
		_vehicleToSellList = AIList();
		_queuedVehicles = {};
		_checkQueueTime = AIDate.GetCurrentDate() + 30.0 * (AITile.GetDistanceManhattanToTile(_sourceStation, _destStation) * 2 / (AIEngine.GetMaxSpeed(AIVehicle.GetEngineType(_vehicleList.Begin())))) + 16;
		_enoughMoney = true;
	}
	
	function UpdateVehicleList()
	{
		local vehicleList = AIVehicleList_Station(AIStation.GetStationID(_sourceStation));
		
		vehicleList.Valuate(Route._check, _destStation);
		vehicleList.KeepValue(1);
		
		_vehicleList = AIList();
		_vehicleList.AddList(vehicleList);
	}
	
	function AddVehicles(count)
	{
		UpdateVehicleList();
		if (_vehicleList.Count() <= 0)
		{
			LogWarning("Vehicles count is less or equal zero!!!");
			return false;
		}
		
		if (FundsManager.CheckMoneyForEngine(AIVehicle.GetEngineType(_vehicleList.Begin()), count, false))
		{
			LogInfo("Adding vehicles to route "+AICargo.GetCargoLabel(_cargo)+".");
			for (local i=count; i>0; i--)
			{
				local dupl = AIVehicle.CloneVehicle(_depotTile, _vehicleList.Begin(), true);
				AIVehicle.StartStopVehicle(dupl);
				_vehicleList.AddItem(dupl, 0);
			}
			CheckIfNeedUpgradeStations();
			_cargoCheckCount = 0;
			_enoughMoney = true;
			return true;
		}
		else
		{
			if (_enoughMoney)
			{
				LogInfo("Not enough funds for new vehicle.");
				_enoughMoney = false;
			}
			return false;
		}
	}
	
	function CheckCargoWaiting()
	{
		local station = AIStation.GetStationID(_sourceStation);
		local capacity = AIVehicle.GetCapacity(_vehicleList.Begin(), _cargo);
		
		if (AIStation.GetCargoWaiting(station, _cargo)*2 > capacity)
		{
			//LogInfo("Cargo is waiting at station ("+(_cargoCheckCount+1)+").");
			_cargoCheckCount++;
		}
		else if (_cargoCheckCount > 0)
		{
			_cargoCheckCount--;
		}
		
		if (_cargoCheckCount >= CARGO_CHECK_MAX)
		{
			local vehicleList = AIVehicleList();
			local vehicleCount = vehicleList.Count();
			local vehicleMax = AIGameSettings.GetValue("vehicle.max_roadveh");
			if (vehicleCount >= vehicleMax)
			{
				_cargoCheckCount = 0;
				return false;
			}
			
			if (_enoughMoney)
				LogInfo("Route needs more vehicles!");
			return true;
		}
		return false;
	}
	
	function CheckQueuedVehicles()
	{
		if (_checkQueueTime > AIDate.GetCurrentDate())
			return;
		
		UpdateVehicleList();
		local vehList = AIList();
		vehList.AddList(_vehicleList);
		
		vehList.RemoveList(_vehicleToSellList);
		
		vehList.Valuate(AIVehicle.GetState);
		vehList.KeepValue(AIVehicle.VS_RUNNING); // only vehicles that are running

		vehList.Valuate(AIVehicle.GetLocation);
		
		local srcList = AITileList_StationType(AIStation.GetStationID(_sourceStation), AIStation.STATION_TRUCK_STOP);
		foreach(st,_ in srcList) // remove vehicles staying at source station
		{
			vehList.RemoveValue(st);
		}
		local dstList = AITileList_StationType(AIStation.GetStationID(_destStation), AIStation.STATION_TRUCK_STOP);
		foreach(st,_ in dstList) // remove vehicles staying at dest station
		{
			vehList.RemoveValue(st);
		}
		
		foreach(veh,_ in vehList) // now in list only running vehicles out of stations
		{
			local time = MAX_QUEUE_TIME;
			if (_queuedVehicles.rawin(veh))
				time = _queuedVehicles[veh];
			
			if (AIVehicle.GetCurrentSpeed(veh) != 0) // vehicle have some speed again!
			{
				if (!_queuedVehicles.rawin(veh)) // this vehicle not at monitoring
					continue;
				time++;
			}
			else // vehicle stays somewhere outside station
			{
				//LogDebug(AIVehicle.GetName(veh)+" at monitoring! "+time);
				time--;
			}
			
			if (time == 0)
			{
				LogInfo("It seems "+AIVehicle.GetName(veh)+" is stuck in queue!");
				_vehicleToSellList.AddItem(veh, 0);
				_queuedVehicles.rawdelete(veh);
			}
			else if (time == MAX_QUEUE_TIME)
			{
				_queuedVehicles.rawdelete(veh);
			}
			else
				_queuedVehicles.rawset(veh, time);
		}
	}
	
	function CheckIfNeedUpgradeStations()
	{
		UpdateVehicleList();
		
		local vehCount = _vehicleList.Count();
		local dist = AITile.GetDistanceManhattanToTile(_sourceStation, _destStation);
		local speed = AIEngine.GetMaxSpeed(AIVehicle.GetEngineType(_vehicleList.Begin()));
		
		local way = (dist*30/speed)*2+8;
		
		local singleCount = way/4;
		
		if (singleCount*_stationCount >= vehCount)
			return true;
		
		LogInfo ("Upgrading station...");
		if (UpgradeStation(_sourceStation, true))
			_stationCount++;
		UpgradeStation(_destStation, false);
		return true;
	}
	
	function CheckIsRouteValid()
	{
		UpdateVehicleList();
		if (_vehicleList.Count() == 0)
		{
			if (_vehicleToSellList.Count() == 0)
				LogWarning("No vehicles anymore.");
			return false;
		}
		
		local station = AIStation.GetStationID(_sourceStation);
		local tileList = AITileList_StationType(station, AIStation.STATION_ANY);
		
		if (tileList.Count() == 0)
		{
			LogWarning("Can't find tiles of source station!");
			return false;
		}
		
		tileList.Valuate(AITile.GetCargoProduction, _cargo, 1, 1, AIStation.GetCoverageRadius(AIStation.STATION_TRUCK_STOP));
		tileList.KeepAboveValue(0);
		
		if (tileList.Count() == 0)
		{
			LogInfo("Source station ain't produce "+AICargo.GetCargoLabel(_cargo)+" anymore!");
			return false;
		}
		
		local station = AIStation.GetStationID(_destStation);
		local tileList = AITileList_StationType(station, AIStation.STATION_ANY);
		
		if (tileList.Count() == 0)
		{
			LogWarning("Can't find tiles of source station!");
			return false;
		}
		
		tileList.Valuate(AITile.GetCargoAcceptance, _cargo, 1, 1, AIStation.GetCoverageRadius(AIStation.STATION_TRUCK_STOP));
		tileList.KeepAboveValue(7);
		
		if (tileList.Count() == 0)
		{
			LogInfo("Dest station ain't accept "+AICargo.GetCargoLabel(_cargo)+" anymore!");
			return false;
		}
		
		return true;
	}
	
	function UpgradeStation(station, isSource)
	{
		local stFront = AIRoad.GetRoadStationFrontTile(station);
		
		for (local i=2; i < 10; i+=2)
		{
			local tileList = Util.AddRectTilesToList(station, i, i);
			tileList.Valuate(AIMap.DistanceManhattan, station);
			tileList.KeepAboveValue(i-2);
			
			tileList.Valuate(AITile.IsBuildable);
			tileList.KeepValue(1);
			
			tileList.Valuate(AIRoad.IsRoadTile);
			tileList.KeepValue(0);
			
			if (tileList.Count() > 0)
			{
				Util.MixListRandomly(tileList);
				foreach (tile,_ in tileList)
				{
					local front = buildPath(stFront, tile);
					if (front == null) continue;
					
					local errMsg = Util.BuildStationAt(tile, front, AIStation.GetStationID(station));
					if (errMsg == null)
						return true;
					else
						LogWarning("Error building station: "+errMsg);
				}
			}
		}
		
		LogWarning("No free space near found!");
		return false;
	}
	
	function CloseRoute()
	{
		UpdateVehicleList();
		
		if (_vehicleList.Count() > 0)
		{
			LogInfo("Closing route...");
			foreach(veh,_ in _vehicleList)
				if (!_vehicleToSellList.HasItem(veh))
					_vehicleToSellList.AddItem(veh,0);
			
			UpdateVehicleList();
			return false;
		}
		else if (_vehicleToSellList.Count() == 0)
		{
			LogInfo("All vehicles sold. Removing stations...");
			DestroyStation(_sourceStation);
			DestroyStation(_destStation);
			_isClosed = true;
			LogInfo("Route Closed.");
			return true;
		}
		else
			return false;
	}
	
	function DestroyStation(tile)
	{
		local station = AIStation.GetStationID(tile);
		if (!AIStation.IsValidStation(station))
		{
			LogWarning("DestroyStation: no station at tile "+tile+"!");
			return true;
		}
		
		local tileList = AITileList_StationType(station, AIStation.STATION_ANY);
		
		if (tileList.Count() == 0)
		{
			LogWarning("Can't find tiles of source station!");
			return false;
		}
		
		foreach(t,_ in tileList)
		{
			if (!AIRoad.RemoveRoadStation(tile))
				LogWarning("Cannot destroy station at tile "+t+"!");
		}
		return true;
	}
	
	function CheckVehiclesReplace()
	{
		if (_timeEngineCheck > AIController.GetTick())	return false;
		_timeEngineCheck = AIController.GetTick()+100;
		
		UpdateVehicleList();
		if (_vehicleList.Count() == 0)
			return false;
		
		if (_newEngine != null) // already replacing
		{
			local oldList = AIList();
			oldList.AddList(_vehicleList);
			
			oldList.Valuate(AIVehicle.GetEngineType);
			oldList.RemoveValue(_newEngine);
			
			if (oldList.Count() == 0)
			{
				_newEngine = null;
				LogInfo("Finished replacing vehicles.");
				return false;
			}
			else
			{
				local group = AIVehicle.GetGroupID(_vehicleList.Begin());
				
				local curEngine = AIVehicle.GetEngineType(oldList.Begin());
				local newEngine = AIGroup.GetEngineReplacement(group, curEngine);
				
				if (!AIEngine.IsValidEngine(newEngine) || newEngine != _newEngine)
					AIGroup.SetAutoReplace(group, curEngine, _newEngine);
				
				return true;
			}
		}
		else
		{
			local engine = Util.FindBestEngine(AIRoad.ROADTYPE_ROAD, _cargo);
			if (engine == null)
				return false;
			
			local curEngine = AIVehicle.GetEngineType(_vehicleList.Begin());
			if (engine != curEngine)
			{
				LogInfo("Better engine exists for this route.");
				_newEngine = engine;
				local group = AIVehicle.GetGroupID(_vehicleList.Begin());
				AIGroup.SetAutoReplace(group,curEngine,_newEngine);
				return true;
			}
			return false;
		}
	}
	/*function ProcessVehicleReplacment()
	{
		local oldList = AIList();
		UpdateVehicleList();
		oldList.AddList(_vehicleList);
		
		oldList.Valuate(AIVehicle.GetEngineType);
		oldList.RemoveValue(_newEngine);
		
		if (oldList.Count() == 0)
		{
			LogInfo("Finished replacing vehicles.");
			_newEngine = null;
			return true;
		}
		
		foreach (veh,_ in oldList)
		{
			if (AIVehicle.IsStoppedInDepot(veh))
				ReplaceVehicle(veh, _newEngine);
			else if(!AIOrder.IsGotoDepotOrder(veh, AIOrder.ORDER_CURRENT) || (AIOrder.GetOrderFlags(veh, AIOrder.ORDER_CURRENT) & AIOrder.AIOF_STOP_IN_DEPOT) != AIOrder.AIOF_STOP_IN_DEPOT)
				Util.SendVehicleToDepot(veh);
		}
		
		return false;
	}
	function ReplaceVehicle(vehicle, newEngine)
	{
		if (!FundsManager.CheckMoneyForEngine(newEngine, 1, false))
			return false;
		
		local group = AIVehicle.GetGroupID(vehicle);
		
		if (!AIVehicle.SellVehicle(vehicle))
		{
			LogError("Cannot sell "+AIVehicle.GetName(vehicle)+": "+AIError.GetLastErrorString());
			return false;
		}
		
		local newVehicle = AIVehicle.BuildVehicle(_depotTile, newEngine);
		if (!newVehicle)
		{
			LogError("Cannot buy new vehicle: "+AIError.GetLastErrorString());
			return false;
		}
		
		if (AIVehicle.GetCapacity(newVehicle, _cargo) == 0)
			AIVehicle.RefitVehicle(newVehicle, _cargo);
		
		AIOrder.AppendOrder(newVehicle, _sourceStation, AIOrder.AIOF_NON_STOP_INTERMEDIATE|AIOrder.AIOF_FULL_LOAD_ANY);
		AIOrder.AppendOrder(newVehicle, _destStation, AIOrder.AIOF_NON_STOP_INTERMEDIATE|AIOrder.AIOF_NO_LOAD);
		
		AIGroup.MoveVehicle(group, newVehicle);
		AIVehicle.StartStopVehicle(newVehicle);
		
		return true;
	}*/
	function SellVehicles()
	{
		local soldList = AIList();
		foreach (veh,_ in _vehicleToSellList)
		{
			if (!AIVehicle.IsValidVehicle(veh))
			{
				soldList.AddItem(veh,0);
				continue;
			}
			
			if (AIVehicle.IsStoppedInDepot(veh))
			{
				local name = AIVehicle.GetName(veh);
				if (AIVehicle.SellVehicle(veh))
					soldList.AddItem(veh,0);
				else
					LogWarning("Can't sell vehicle "+name+": "+AIError.GetLastErrorString());
			}
			else if(!(AIOrder.IsGotoDepotOrder(veh, AIOrder.ORDER_CURRENT) && (AIOrder.GetOrderFlags(veh, AIOrder.ORDER_CURRENT) & AIOrder.AIOF_STOP_IN_DEPOT) != 0))
				Util.SendVehicleToDepotForever(veh);
		}
		_vehicleToSellList.RemoveList(soldList);
		if (_vehicleToSellList.Count() == 0)
			return true;
		return false;
	}
	
	function IsVehicleThere(vehicle)
	{
		UpdateVehicleList();
		
		if (_vehicleList.HasItem(vehicle))
			return true;
		return false;
	}
	
	function VehicleUnprofitable(vehicle)
	{
		if (!_vehicleToSellList.HasItem(vehicle))
		{
			LogWarning(AIVehicle.GetName(vehicle)+" has negative profit!");
			_vehicleToSellList.AddItem(vehicle, 0);
		}
	}
}

function Route::_check(vehicle, dest)
{
	for (local i=0; i<AIOrder.GetOrderCount(vehicle); i++)
	{
		if (AIOrder.IsGotoStationOrder(vehicle, i) && AIOrder.GetOrderDestination(vehicle, i) == dest)
			return true;
	}
	return false;
}

function Route::buildPath(src, dest)
{
	local pf = RoadPathFinder(src, dest, 200);
	for (local i=0;i<25; i++)
		if(pf.Itterate(2)) break; // should be enough
	
	local path = pf.GetCurrentPaths();
	if (path == null)
		return null;
	
	local last = path[path.len()-1]._parentNode;
	
	local pb = PathBuilder(path);
	if (pb.BuildRoad() == PathBuilder.RESULT_ERROR)
		return null;
	
	if (last == null)
		return null;
	return last._tile;
}

function Route::GetName()
{
	return AICargo.GetCargoLabel(_cargo)+": "+Util.TileToString(_sourceStation)+" -> "+Util.TileToString(_destStation);
}
function Route::GetSourceIndustry()
{
	return _source;
}
function Route::GetDestinationIndustry()
{
	return _dest;
}
function Route:: GetCargo()
{
	return _cargo;
}
function Route::IsClosed()
{
	return _isClosed;
}
function Route::GetRaw()
{
	local raw = [];
	raw.append(_source);			// [0]
	raw.append(_dest);				// [1]
	raw.append(_sourceStation);	// [2]
	raw.append(_destStation);	// [3]
	raw.append(_cargo);				// [4]
	raw.append(_depotTile);		// [5]
	raw.append(_stationCount);//[6]
	raw.append(_newEngine);	// [7]
	return raw;
}
function Route::FromRaw(version, raw)
{
	local source = raw[0];
	local dest = raw[1];
	local sourceTile = raw[2];
	local destTile = raw[3];
	local cargo = raw[4];
	local depotTile = raw[5];
	local stationCount = raw[6];
	local newEngine = null;
	if (version > 5380)
		newEngine = raw[7];
	
	return Route(source, dest, sourceTile, destTile, cargo, depotTile, stationCount, newEngine);
}

function Route::LogInfo(msg)
{
	AILog.Info("Route "+GetName()+": "+msg);
}
function Route::LogWarning(msg)
{
	AILog.Warning("Route "+GetName()+": "+msg);
}
function Route::LogError(msg)
{
	AILog.Error("Route "+GetName()+": "+msg);
}

function Route::LogDebug(msg)
{
	if (DEBUG)
		AILog.Warning("(DBG) Route "+GetName()+": "+msg);
}