/*
	MogulAI - an artificial intelligence for OpenTTD
	Copyright (C) 2009 - 2010 Kazantsev Lev (Dezmond_snz)

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License along
	with this program; if not, write to the Free Software Foundation, Inc.,
	51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

require("Route.nut");
require("RouteManager.nut");
require("RouteBuildManager.nut");
require("FundsManager.nut");
require("DamnAuthorityManager.nut");
require("Util.nut");

const DEBUG = 0;

class MogulAI extends AIController
{
	Instance = null;
	_loadData = null;
	_loadVersion = null;
	
	_routeBuilder = null;
	_routeManager = null;
	_damnAuthorityManager = null;
	
	_revision = 0;
	_loaded = false;
	
	constructor()
	{
		Instance = this;
		_damnAuthorityManager = DamnAuthorityManager();
		_routeManager = RouteManager();
		_routeBuilder = RouteBuildManager(_routeManager, _damnAuthorityManager);
	}
	
	function Start()
	{
		if (AIGameSettings.IsDisabledVehicleType(AIVehicle.VT_ROAD))
		{
			LogError("I don't think i can play while you deny use road vehicles for me! Fire!");
			local var = 7654765/0;
		}
		LogInfo("MogulAI Started.");
		
		AICompany.SetAutoRenewStatus(true);
		AICompany.SetAutoRenewMonths(6);
		AICompany.SetAutoRenewMoney(50000);
		
		local v = AIController.GetVersion();
		_revision = v & 0x3ffff;
		local isRelease = ((v >> 19) & 0x1) == 1;
		local build = (v >> 20) & 0xf;
		local minor = (v >> 24) & 0xf;
		local major = (v >> 28) & 0xf;
		LogDebug("Version of OpenTTD: "+major+"."+minor+"."+build+", rev "+_revision+(isRelease?"(Release)":"")+" raw "+v);
		
		if (!_loaded) SetCompanyName();
		else LoadInternal(_loadVersion, _loadData);
		
		while(true)
		{
			FundsManager.CheckMoneyGeneral();
			CheckEvents();
			_routeBuilder.MakeStep();
			_routeManager.MakeStep();
			_damnAuthorityManager.ProcessDamnAuthorities();
			//this.Sleep(1);
		}
	}
	
	function GetItemAtIndex(list, idx)
	{
		if (idx >= list.Count() || idx < 0)
			return null;
		local item = list.Begin();
		for (local i=0; i< idx; i++)
			item = list.Next();
		return item;
	}
	
	function  SetCompanyName()
	{
	 	if (_revision >= 17016) AICompany.SetPresidentGender(AICompany.GENDER_MALE);
		if(!AICompany.SetName("Moguls Trans Corp ltd."))
		{
	 		local i = 2;
	 		while(!AICompany.SetName("Moguls Trans Corp #" + i+" ltd."))
	 		{
	 			i = i + 1;
	 			if(i > 255)	break;
	 		}
	 		
	 		if (i == 2)
		 		AICompany.SetPresidentName("Tore Santori The "+i+"nd.");
 			else if (i == 3)
		 		AICompany.SetPresidentName("Tore Santori The "+i+"rd.");
		 	else
		 		AICompany.SetPresidentName("Tore Santori The "+i+"th.");
		 	return;
	 	}
	 	AICompany.SetPresidentName("Tore Santori");
	}
	
	function CheckEvents()
	{
		while (AIEventController.IsEventWaiting())
		{
			local e = AIEventController.GetNextEvent();
			switch (e.GetEventType())
			{
			case AIEvent.AI_ET_ENGINE_PREVIEW:
				AIEventEnginePreview.Convert(e).AcceptPreview();
				break;
			case AIEvent.AI_ET_VEHICLE_UNPROFITABLE:
				_routeManager.NotifyVehicleUnprofitable(AIEventVehicleUnprofitable.Convert(e).GetVehicleID());
				break;
			default:
				break;
			}
		}
	}
	
	function Save()
	{
		LogInfo("Saving...");
		local table = {};
		
		table.rawset("routemanager", this._routeManager.Save());
		table.rawset("routebuilder", this._routeBuilder.Save());
		return table;
	}
	function Load(version, data)
	{
		_loadData = data;
		_loadVersion = version;
		_loaded = true;
	}
	function LoadInternal(version, data)
	{
		LogInfo("Loading...");
		
		if (data.rawin("routemanager"))
			this._routeManager.Load(version, data.rawget("routemanager"));
		
		if (data.rawin("routebuilder"))
			this._routeBuilder.Load(version, data.rawget("routebuilder"));
		
		LogInfo("Loaded game.");
	}
	
	function LogInfo(msg)
	{
		AILog.Info("MogulAI: "+msg);
	}
	function LogWarning(msg)
	{
		AILog.Warning("MogulAI: "+msg);
	}
	function LogError(msg)
	{
		AILog.Error("MogulAI: "+msg);
	}
	
	function LogDebug(msg)
	{
		if (DEBUG)
			AILog.Info("(DBG) MogulAI: "+msg);
	}
}
