/*
 * This file is part of SCPClient_NoCarGoal, which is an AI Library for OpenTTD
 * Copyright (C) 2013  Leif Linse
 *
 * SCPClient_NoCarGoal is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License
 *
 * SCPClient_NoCarGoal is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with SCPClient.NoCarGoal; If not, see <http://www.gnu.org/licenses/> or
 * write to the Free Software Foundation, Inc., 51 Franklin Street, 
 * Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

class SCPClient_NoCarGoal {

	/* Private members variables */
	static COMMAND_SET = "NoCarGoal";

	_scp = null;      // SCPLib instance
	_goals = null;    // goal array
	_end_date = null; // End date of NoCarGoal game
	_nocargoal_game = null; // Is this a NoCarGoal game? (true/null)
	_n_uncompleted_goals = null; // # of uncompleted goal cargos
	_last_cargo_data_date = null; // date when the last cargo update was sent

	/**
	 * Library constructor
	 * @param scp_ptr Instance of SCPLib. If null is passed, the library
	 * will act as if no NoCarGoal GS has been detected.
	 */
	constructor(scp_ptr)
	{
		this._goals = null;
		this._end_date = null;
		this._nocargoal_game = null;
		this._n_uncompleted_goals = null;

		this._scp = scp_ptr;

		this.RegisterCommands();
		this.AskForGoals();
	}

	/* Public methods */

	/**
	 * Is this a NoCarGoal game?
	 * @return True, if the game has a GS that respond to NoCarGoal
	 * commands, otherwise false.
	 * @note If you call this method to early, it will return false even if
	 * the game has the NoCarGoal with SCP activated because a round-trip
	 * has not yet been completed.
	 */
	function IsNoCarGoalGame();

	/**
	 * Get a list of goal cargos.
	 * @param filter_completed If true, only non-completed goals will be
	 * included in the list
	 * @return An AIList with the goal cargos as items. The list value
	 * contains the remaining cargo to transport that was last sent from the
	 * GS. If no NoCarGoal goals exist, an AIList with zero items is 
	 * returned.
	 */
	function GetGoalCargoList(filter_completed = false);

	/**
	 * Get an array with all goal data for each cargo. Note that the result
	 * is the last data received from NoCarGoal. The update frequency of
	 * these data is about once a month.
	 * @return An array with a table at each array index or null. The table
	 * will have these keys:
	 *  - cargo          = cargo id of cargo to transport
	 *  - transported    = amount of cargo already transported
	 *  - goal           = target amount of cargo to transport
	 */
	function GetGoalCargoArray();

	/**
	 * Check if a given cargo is a goal cargo
	 * @param filter_completed If true, the method will only return true if
	 * the goal has not been met for the given cargo.
	 * @return True if given cargo is one of the goal cargos.
	 */
	function IsGoalCargo(cargo_id, filter_completed = false);

	/**
	 * Filter an AIList of cargos to only contain goal cargos. If no goals
	 * have been given by the GS yet, no filter will be applied.
	 * @param filter_completed If true, only non-completed goals will be
	 * included in the list
	 * @return None (the passed list will get filtered)
	 */
	function FilterCargoList(list, filter_completed = false, disable_when_all_completed = true);

	/**
	 * Get number of uncompleted goals
	 * @return the number of goal cargos that has not yet met the goal.
	 * if no NoCarGoal GS has been identified, 0 is returned.
	 */
	function GetNUncompletedGoals();

	/**
	 * Get the end date of NoCarGoal game
	 * @return null or the end date of NoCarGoal game
	 */
	function GetEndDate();

	/**
	 * Get the send date of the last received cargo goal update. (NoCarGoal
	 * will stamp its updates with the send date, and this is the date
	 * returned by this method.)
	 * @return A date or null (if no cargo data has been received)
	 */
	function GetLastCargoUpdateDate();
}

/**** Private methods: ****/

function SCPClient_NoCarGoal::RegisterCommands()
{
	if (this._scp == null) return;

	local self = this;

	// AI -> GS commands:
	this._scp.AddCommand("CurrentGoal", COMMAND_SET, self, SCPClient_NoCarGoal.ReceivedCurrentGoalCommand);
	this._scp.AddCommand("Setting", COMMAND_SET, self, SCPClient_NoCarGoal.ReceivedSettingCommand);

	// GS -> AI commands:
	//this._scp.AddCommand("GoalCompleted", COMMAND_SET, self, SCPClient_NoCarGoal.ReceivedGoalCompletedCommand);
}

function SCPClient_NoCarGoal::AskForGoals()
{
	if (this._scp == null) return;

	this.GetCurrentGoal(AICompany.ResolveCompanyID(AICompany.COMPANY_SELF));

	// Tell GS to send monthly goal updates
	this._scp.QueryServer("Setting", COMMAND_SET, "ai_monthly_report", 1);
}

/**** Public API methods: ****/

function SCPClient_NoCarGoal::IsNoCarGoalGame()
{
	if (this._scp == null) return false;

	return this._nocargoal_game == true;
}

function SCPClient_NoCarGoal::GetGoalCargoList(filter_completed = false)
{
	if (this._goals == null) return AIList();

	local list = AIList();
	for (local g = 0; g < this._goals.len(); g++)
	{
		local to_transport = this._goals[g].goal - this._goals[g].transported;
		if (!filter_completed || to_transport > 0) {
			list.AddItem(this._goals[g].cargo, max(to_transport, 0));
		}
	}
}

function SCPClient_NoCarGoal::GetGoalCargoArray()
{
	return this._goals;
}

function SCPClient_NoCarGoal::IsGoalCargo(cargo_id, filter_completed = false)
{
	if (this._goals == null) return false;

	for (local g = 0; g < this._goals.len(); g++) {
		if (this._goals[g].cargo == cargo_id) {
			if (!filter_completed) return true;
			local to_transport = this._goals[g].goal - this._goals[g].transported;
			return to_transport > 0;
		}
	}

	return false;
}

function SCPClient_NoCarGoal::FilterCargoList(list, filter_completed = false, disable_when_all_completed = true)
{
	if (this._goals == null) return;

	//AILog.Info("Filter cargo list:");
	foreach(cargo, _ in list) {
		if (!SCPClient_NoCarGoal.IsGoalCargo(cargo, filter_completed)) {
			//AILog.Info("  Remove: " + AICargo.GetCargoLabel(cargo));
			list.RemoveItem(cargo);
		}
		//else AILog.Info("  Keep: " + AICargo.GetCargoLabel(cargo));
	}
}

function SCPClient_NoCarGoal::GetNUncompletedGoals()
{
	if(this._n_uncompleted_goals == null) return 0;
	return this._n_uncompleted_goals;
}

function SCPClient_NoCarGoal::GetEndDate()
{
	return this._end_date;
}

function SCPClient_NoCarGoal::GetLastCargoUpdateDate()
{
	return this._last_cargo_data_date;
}

/** SCP in/out methods (private to the library) */

/*****************************************************************
 *                                                               *
 *   Outgoing Commands - commands that we can send to GSes       *
 *                                                               *
 *****************************************************************/

// Call one of these methods to send a command to a GS.

function SCPClient_NoCarGoal::GetCurrentGoal(company_id)
{
	this._scp.QueryServer("CurrentGoal", COMMAND_SET, [company_id]);
}

/*****************************************************************
 *                                                               *
 *   Incoming Commands - commands that we can get from AIs       *
 *                                                               *
 *****************************************************************/

// These methods are called by the SCP library when we call this._scp.Check() and there is
// a received incoming message.

// Use 'self' instead of 'this'.

function SCPClient_NoCarGoal::ReceivedCurrentGoalCommand(message, self)
{
	self._nocargoal_game = true;

	// check that the response gives the goal of our company and not some other company
	if (message.GetIntData(0) != null && AICompany.IsMine(message.GetIntData(0))) {
		local sent_date = message.GetIntData(1);
		local time_left = message.GetIntData(2);
		if (sent_date != null && time_left != null && (self._end_date == null || time_left > 0)) {
			self._end_date = sent_date + time_left;
		}
		self._last_cargo_data_date = sent_date;

		// Verify that none of the goal data is null.
		local valid_goals = true;
		for (local i = 0; i < 9; i++) {
			valid_goals = valid_goals && message.GetIntData(i) != null;
		}

		// Only update self._goals if the new data do not contain null fields.
		if (valid_goals) {
			self._goals = [];
			self._n_uncompleted_goals = 0;
			local s = "";
			local head_data_len = 3;
			for (local g = 0; g < 3; g++) {
				local goal = {
					cargo = message.GetIntData(head_data_len + 3 * g),
					goal = message.GetIntData(head_data_len + 3 * g + 1),
					transported = message.GetIntData(head_data_len + 3 * g + 2),
				};

				if (goal.goal - goal.transported > 0) {
					self._n_uncompleted_goals++;
				}

				self._goals.append(goal);

				if (s != "") s += ", ";
				s += goal.cargo + ":" + AICargo.GetCargoLabel(goal.cargo);
			}

			//AILog.Info("Received cargos: " + s);
		}
	}
}

function SCPClient_NoCarGoal::ReceivedSettingCommand(message, self)
{
	self._nocargoal_game = true;

	local setting = message.GetStringData(0);
	if (setting != null) {
		local value = message.GetIntData(1);
	}
}
