/*
 * This file is part of SuperLib, which is an AI Library for OpenTTD
 * Copyright (C) 2010  Leif Linse
 *
 * SuperLib is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License
 *
 * SuperLib is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with SuperLib; If not, see <http://www.gnu.org/licenses/> or
 * write to the Free Software Foundation, Inc., 51 Franklin Street, 
 * Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

class _SuperLib_Airport
{
	//////////////////////////////////////////////////////////////////////
	//                                                                  //
	//  Airport types                                                   //
	//                                                                  //
	//////////////////////////////////////////////////////////////////////

	/*
	 * Returns an AIList with all airport types
	 */
	static function GetAirportTypeList();

	/*
	 * Returns the last available airport that the given engine can use
	 */
	static function GetLastAvailableAirportType(engine_id);

	/*
	 * Returns true if the plane type can (safely) use the airport type,
	 * else false.
	 * Large aircrafts on small airports yields false.
	 */
	static function IsPlaneTypeAcceptedByAirportType(plane_type, airport_type);

	/*
	 * Same as above, but with reversed argument order.
	 */
	static function IsAirportTypeAcceptingPlaneType(airport_type, plane_type);
	
	/* Checks if there is at least one airport that the given plan type can land (safely) on */
	static function AreThereAirportsForPlaneType(plane_type);

	//////////////////////////////////////////////////////////////////////
	//                                                                  //
	//  Airport information                                             //
	//                                                                  //
	//////////////////////////////////////////////////////////////////////

	/* 
	 * Get the north tile of the airport of a given station.
	 * Returns an invalid tile if there is no airport for the given station
	 */
	static function GetAirportTile(station_id);
	
	/*
	 * Get the tile of one of the hangars of the airport of the given station.
	 * Returns an invalid tile on failure.
	 */
	static function GetHangarTile(station_id);

	/*
	 * Check if a given station has a small airport
	 */
	static function IsSmallAirport(station_id);

	//////////////////////////////////////////////////////////////////////
	//                                                                  //
	//  Aircraft information                                            //
	//                                                                  //
	//////////////////////////////////////////////////////////////////////
	/*
	 * Returns an AIList of all aircrafts in all the hangar(s) of the airport
	 * of a given station. 
	 * Returns an empty AIList if there are no aircrafts. 
	 */
	static function GetAircraftsInHangar(station_id);

	/*
	 * Returns the number of non-stoped aircrafts in airport depot
	 */
	static function GetNumNonStopedAircraftsInAirportDepot(station_id);

	/*
	 * Returns a the number of aircrafts that are in the landing queue.
	 * That is aircrafts that are circulating and waiting to land.
	 */
	static function GetNumAircraftsInAirportQueue(station_id);
}

/*static*/ function _SuperLib_Airport::GetAirportTypeList()
{
	local list = AIList();
	for(local i = 0; i < AIAirport.AT_INVALID; ++i)
	{
		if(AIAirport.IsAirportInformationAvailable(i))
			list.AddItem(i, i);
	}

	return list;
}

/*static*/ function _SuperLib_Airport::GetLastAvailableAirportType(engine_id)
{
	local airport_types = _SuperLib_Airport.GetAirportTypeList();

	// Only keep airports that accept the given engine type
	airport_types.Valuate(_SuperLib_Airport.IsAirportTypeAcceptingPlaneType, AIEngine.GetPlaneType(engine_id));
	airport_types.KeepValue(1);

	// Only keep buildable airports
	airport_types.Valuate(AIAirport.IsValidAirportType);
	airport_types.KeepValue(1);

	if (airport_types.IsEmpty())
		return -1;

	// Return the airport with highest index
	airport_types.Sort(AIAbstractList.SORT_BY_ITEM, AIAbstractList.SORT_DESCENDING);
	return airport_types.Begin();
}

/*static*/ function _SuperLib_Airport::IsAirportTypeAcceptingPlaneType(airport_type, plane_type)
{
	return _SuperLib_Airport.IsPlaneTypeAcceptedByAirportType(plane_type, airport_type);
}

/*static*/ function _SuperLib_Airport::AreThereAirportsForPlaneType(plane_type)
{
	local list = _SuperLib_Airport.GetAirportTypeList();

	// Only keep buildable airports
	list.Valuate(AIAirport.IsValidAirportType);
	list.KeepValue(1);

	list.Valuate(_SuperLib_Airport.IsAirportTypeAcceptingPlaneType, plane_type);
	list.KeepValue(1);

	return !list.IsEmpty();
}

/*static*/ function _SuperLib_Airport::IsPlaneTypeAcceptedByAirportType(plane_type, airport_type)
{
	// Use full listing of hard coded airport types so that we don't assume anything about
	// possible future added airports.
	local large_ap = airport_type == AIAirport.AT_LARGE ||
				airport_type == AIAirport.AT_METROPOLITAN ||
				airport_type == AIAirport.AT_INTERNATIONAL ||
				airport_type == AIAirport.AT_INTERCON;

	local small_ap = airport_type == AIAirport.AT_SMALL ||
				airport_type == AIAirport.AT_COMMUTER;

	local heli_ap = airport_type == AIAirport.AT_HELIPORT ||
				airport_type == AIAirport.AT_HELISTATION ||
				airport_type == AIAirport.AT_HELIDEPOT;

	switch(plane_type)
	{
		case AIAirport.PT_BIG_PLANE:
			return large_ap;

		case AIAirport.PT_SMALL_PLANE:
			return large_ap || small_ap;

		case AIAirport.PT_HELICOPTER:
			return heli_ap || large_ap || small_ap; // All currently (2010-06-26) hardcoded airport types can take helicopters

	}

	_SuperLib_Log.Error("SuperLib::Airport::IsPlaneTypeAcceptedByAirportType: Unknown plane type (" + plane_type + ").", _SuperLib_Log.LVL_INFO);
	return false; // Should not happen
}

/*static*/ function _SuperLib_Airport::GetAirportTile(station_id)
{
	local airport_tiles = AITileList_StationType(station_id, AIStation.STATION_AIRPORT);
	if(airport_tiles.IsEmpty())
		return -1;

	airport_tiles.Valuate(Utils_Valuator.ItemValuator);
	airport_tiles.KeepBottom(1);

	return airport_tiles.Begin();
}

/*static*/ function _SuperLib_Airport::GetHangarTile(station_id)
{
	local hangar_tile = AIAirport.GetHangarOfAirport(_SuperLib_Airport.GetAirportTile(station_id));
	return hangar_tile;
}

// Loops through all hangars in an airport and retrive the aircrafts in them
/*static*/ function _SuperLib_Airport::GetAircraftsInHangar(station_id)
{
	local hangar_aircraft_list = AIList();

	// Loop through all hangars of the airport
	local airport_tiles = AITileList_StationType(station_id, AIStation.STATION_AIRPORT);
	airport_tiles.Valuate(AIAirport.IsHangarTile);
	airport_tiles.KeepValue(1);

	for(local hangar_tile = airport_tiles.Begin(); airport_tiles.HasNext(); hangar_tile = airport_tiles.Next())
	{
		// Get a list of all vehicles in current hangar
		local vehicle_list = AIVehicleList_Detpot(hangar_tile);

		// Add those vehicles to the list of all airplanes in hangars of this airport
		hangar_aircraft_list.AddList(vehicle_list);
	}

	return hangar_aircraft_list;
}

/*static*/ function _SuperLib_Airport::IsSmallAirport(station_id)
{
	local airport_type = AIAirport.GetAirportType(_SuperLib_Airport.GetAirportTile(station_id));
	return airport_type == AIAirport.AT_SMALL || airport_type == AIAirport.AT_COMMUTER;
}

// returns the station location for current order. If current order is the hangar, return the airport location.
function _SuperLib_Airport_Private_GetCurrentOrderDestinationStationSignLocation(vehicle_id)
{
	local loc = _SuperLib_Order.GetCurrentOrderDestination(vehicle_id);

	// change loc tile to the station sign location
	loc = AIStation.GetLocation(AIStation.GetStationID(loc));

	return loc;
}

/*static*/ function _SuperLib_Airport::GetNumNonStopedAircraftsInAirportDepot(station_id)
{
	local hangar_vehicles = _SuperLib_Airport.GetAircraftsInHangar(station_id);

	hangar_vehicles.Valuate(AIVehicle.IsStoppedInDepot);
	hangar_vehicles.KeepValue(0);

	return hangar_vehicles.Count();
}

/*static*/ function _SuperLib_Airport::GetNumAircraftsInAirportQueue(station_id)
{
	// aircrafts
	local station_vehicle_list = AIVehicleList_Station(station_id);
	station_vehicle_list.Valuate(AIVehicle.GetVehicleType);
	station_vehicle_list.KeepValue(AIVehicle.VT_AIR);

	// get airport tile
	local airport_tile = _SuperLib_Airport.GetAirportTile(station_id);

	// get airport holding rectangle
	local holding_rect = AITileList();	

	switch(AIAirport.GetAirportType(airport_tile))
	{
		case AIAirport.AT_SMALL:
			holding_rect.AddRectangle(Tile.GetTileRelative(airport_tile, -2, 0), Tile.GetTileRelative(airport_tile, 17, 12));
			holding_rect.RemoveRectangle(Tile.GetTileRelative(airport_tile, 1, 1), Tile.GetTileRelative(airport_tile, 3, 2)); // remove non-holding airport tiles
			break;
		
		case AIAirport.AT_COMMUTER:
			holding_rect.AddRectangle(Tile.GetTileRelative(airport_tile, -2, 0), Tile.GetTileRelative(airport_tile, 14, 9));
			holding_rect.RemoveRectangle(Tile.GetTileRelative(airport_tile, 1, 1), Tile.GetTileRelative(airport_tile, 4, 3)); // remove non-holding airport tiles
			break;

		case AIAirport.AT_LARGE:
			holding_rect.AddRectangle(Tile.GetTileRelative(airport_tile, 9, 0), Tile.GetTileRelative(airport_tile, 17, 5));
			break;

		case AIAirport.AT_METROPOLITAN:
			holding_rect.AddRectangle(Tile.GetTileRelative(airport_tile, -2, 0), Tile.GetTileRelative(airport_tile, 17, 12));
			holding_rect.RemoveRectangle(Tile.GetTileRelative(airport_tile, 1, 1), Tile.GetTileRelative(airport_tile, 5, 5)); // remove non-holding airport tiles
			break;

		case AIAirport.AT_INTERNATIONAL:
			holding_rect.AddRectangle(Tile.GetTileRelative(airport_tile, -2, 0), Tile.GetTileRelative(airport_tile, 19, 13));
			holding_rect.RemoveRectangle(Tile.GetTileRelative(airport_tile, 1, 1), Tile.GetTileRelative(airport_tile, 6, 6)); // remove non-holding airport tiles
			break;

		case AIAirport.AT_INTERCON:
			holding_rect.AddRectangle(Tile.GetTileRelative(airport_tile, -13, -11), Tile.GetTileRelative(airport_tile, 19, 21));
			holding_rect.RemoveRectangle(Tile.GetTileRelative(airport_tile, 0, 0), Tile.GetTileRelative(airport_tile, 8, 10)); // remove non-holding airport tiles
			break;

		default:
			// Unknown airport type -> crash the AI
			KABOOOOOM_unknown_airport_type();
	}

	station_vehicle_list.Valuate(SuperLib.Vehicle.VehicleIsWithinTileList, holding_rect);
	station_vehicle_list.KeepValue(1);

	// Only keep vehicles that are on the way to the airport
	station_vehicle_list.Valuate(_SuperLib_Airport_Private_GetCurrentOrderDestinationStationSignLocation);
	station_vehicle_list.KeepValue(AIStation.GetLocation(station_id));

	// remove vehicles that are in a depot (the right depot of international would else give false positive
	station_vehicle_list.Valuate(AIVehicle.IsInDepot);
	station_vehicle_list.KeepValue(0);

	return station_vehicle_list.Count();
}
