/*
 * This file is part of SuperLib, which is an AI Library for OpenTTD
 * Copyright (C) 2010  Leif Linse
 *
 * SuperLib is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License
 *
 * SuperLib is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with SuperLib; If not, see <http://www.gnu.org/licenses/> or
 * write to the Free Software Foundation, Inc., 51 Franklin Street, 
 * Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

class _SuperLib_Vehicle
{
	//////////////////////////////////////////////////////////////////////
	//                                                                  //
	//  Where are the vehicles?                                         //
	//                                                                  //
	//////////////////////////////////////////////////////////////////////

	static function IsVehicleAtStation(vehicle_id, station_id);

	static function HasTileListVehiclesOnIt(tile_rect, vehicle_type);
	static function VehicleIsWithinTileList(vehicle_id, tile_list);
	static function GetVehiclesAtTile(tile);


	static function GetVehicleCargoType(vehicle_id);
}

/*static*/ function _SuperLib_Vehicle::IsVehicleAtStation(vehicle_id, station_id)
{
	if(AIVehicle.GetState(vehicle_id) != AIVehicle.VS_AT_STATION)
		return false;

	local veh_tile = AIVehicle.GetLocation(vehicle_id);

	// Check if the tile that the vehicle is at is a station
	// with the right station id
	return AIStation.GetStationID(veh_tile) == station_id;
}

/*static*/ function _SuperLib_Vehicle::HasTileListVehiclesOnIt(tile_rect, vehicle_type)
{
	local all_veh = AIVehicleList();
	all_veh.Valuate(AIVehicle.GetVehicleType);
	all_veh.KeepValue(vehicle_type);

	all_veh.Valuate(_SuperLib_Vehicle.VehicleIsWithinTileList, tile_rect);
	all_veh.KeepValue(1);
	
	return !all_veh.IsEmpty();
}

/*static*/ function _SuperLib_Vehicle::VehicleIsWithinTileList(vehicle_id, tile_list)
{
	return tile_list.HasItem(AIVehicle.GetLocation(vehicle_id));
}

/*static*/ function _SuperLib_Vehicle::GetVehiclesAtTile(tile)
{
	local vehicles = AIVehicleList();
	vehicles.Valuate(AIVehicle.GetLocation);
	vehicles.KeepValue(tile);

	return vehicles;
}

/*static*/ function _SuperLib_Vehicle::GetVehicleCargoType(vehicle_id)
{
	// Go through all cargos and check the capacity for each
	// cargo.
	local max_cargo = -1;
	local max_cap = -1;

	local cargos = AICargoList();
	foreach(cargo, _ in cargos)
	{
		local cap = AIVehicle.GetCapacity(vehicle_id, cargo);
		if(cap > max_cap)
		{
			max_cap = cap;
			max_cargo = cargo;
		}
	}

	// Return the cargo which the vehicle has highest capacity
	// for.
	return max_cargo;
}
