
class ICStationNode extends OptNodeBaseClass
{
	town_mgr = null;
	station_stats = null;
	best_engine_small = null;
	best_engine_large = null;
	
	constructor(station_statistics, town_mgr)
	{
		this.station_stats = station_statistics;
		this.town_mgr = town_mgr;

		// cache the best engine to save some computation time
		this.best_engine_small = Builder.GetEngine(1000, AIVehicle.VT_AIR, true);
		this.best_engine_large = Builder.GetEngine(1000, AIVehicle.VT_AIR, false);
	}

	function GetArchValue(other_node);
	function GetNodeId();
}

function ICStationNode::GetArchValue(other_node)
{
	// Make two lists, one of the airports (ap) and one of the top left tiles of the two airports
	local ap = [this.station_stats.station_id, other_node.station_stats.station_id];
	local ap_tiles = [Station.GetAirportTile(ap[0]), Station.GetAirportTile(ap[1])];
	
	// Check if any of the two airport has a small airport
	local ap_ai_list = AIList();
	ap_ai_list.AddItem(ap[0], 0);
	ap_ai_list.AddItem(ap[1], 0);
	local has_small_airport = Builder.HasSmallAirport(ap_ai_list);

	// best engine for small/large airport have been pre-cached so now we can just get that value
	local best_engine = has_small_airport? this.best_engine_small: this.best_engine_large;

	// Get the distance between the two airports
	local dist_manhattan = AIMap.DistanceManhattan(ap_tiles[0], ap_tiles[1]);

	// Estimate the travel time between the airports using engine: best_engine
	local travel_time = Engine.GetStationToStationAircraftTravelTime(best_engine, ap[0], ap[1]);

	local income = AICargo.GetCargoIncome(Helper.GetPAXCargo(), dist_manhattan, travel_time);

	// Todo: Add a population + pop equity factor
	// or just use the num pax generated last month. 
	local pax_gen = 10;

	return income;
}

function ICStationNode::GetNodeId()
{
	return station_stats.station_id;
}


//// Air Manager ////////////////////////////////////////////////////

class AirManager
{
	static function OptimizeAirportPairs();
	static function GetTransportDemand(station1_stats, station2_stats);
}

function AirManager::OptimizeAirportPairs()
{
	AILog.Info("-- Optimize Airport Pairs --");

	local a = 3;
	local pair_optimizer = PairOptimizer();

	local ic_stations = Station.GetAllInterCityStations();
	for(local station_id = ic_stations.Begin(); ic_stations.HasNext(); station_id = ic_stations.Next())
	{
		local town_mgr = towns.GetTownManager(AIStation.GetNearestTown(station_id));
		town_mgr.RemoveInvalidStations();
		town_mgr.AddStationsFromMap();
		local station_stats = town_mgr.GetStationStatistics(station_id);

		if(station_stats != null)
			pair_optimizer.AddNode(ICStationNode(station_stats, town_mgr));
	}

	local solution = pair_optimizer.MakeTabuSearch();
	local changed_pairs = [];

	local i = 1;
	foreach(pair in solution)
	{
		local station1_node = pair.GetNodeObj(0);
		local station2_node = pair.GetNodeObj(1);

		if(station1_node != null)
			Helper.SetSign(AIStation.GetLocation(station1_node.station_stats.station_id) - 1, "Pair " + i, true);
		if(station2_node != null)
			Helper.SetSign(AIStation.GetLocation(station2_node.station_stats.station_id) - 1, "Pair " + i, true);

		i++;

		if(station1_node == null || station2_node == null)
			continue;

		// check if the pair already has a connection (thus is unchanged)
		local station_list = AIList();
		station_list.AddItem(station1_node.station_stats.station_id, 0);
		station_list.AddItem(station2_node.station_stats.station_id, 0);
		
		if(Station.AreStationsConnectedByVehicles(station_list, AIVehicle.VT_AIR))
			continue;

		// Add pairs that are not part of the current solution to a specific list
		changed_pairs.append(pair);
	}

	
	// Now the task is to rearange the aircrafts servicing changed_pairs so 
	// that they form the connections in changed_pairs.
	AirManager.RearrangeConnections(changed_pairs);

	AILog.Info("Done optimizing airport pairs.");
}

function AirManager::RearrangeConnections(changed_pairs)
{
	AILog.Info("There is " + changed_pairs.len() + " changed pairs.");

	// 1. All aircrafts of all stations form a pool of aircrafts
	// 2. (in memory) assign the aircrafts in a smart way to the different new pairs
	// 3. Make a quick change by changing the shared orders of each fleet to form the connections (in case there is a save/load in the middle of the process)
	// 4. Move aircraft by aircraft according to the in-memory plan. If possible avoid temporarily loosing any connection as those could be lost in a save/load.

	// Re-arrange the data a bit.
	// Make a new list of pairs with associated data for
	// storing the transport demand and scores for later usage.
	local pair_list = [];
	foreach(pair in changed_pairs)
	{
		local station_list = [];
		station_list.append( (pair.GetNodeObj(0)).station_stats );
		station_list.append( (pair.GetNodeObj(1)).station_stats );
		
		pair_list.append(
			{ station_list = station_list
			  transport_demand = AirManager.GetTransportDemand(station_list[0], station_list[1])
			  score = 0 } );
	}

	// Make a list of all stations
	local all_station_list = AIList();
	foreach(pair in pair_list)
	{
		all_station_list.AddItem(pair.station_list[0].station_id, 0);
		all_station_list.AddItem(pair.station_list[1].station_id, 0);
	}	
	AILog.Info("all_station_list.Count(): " + all_station_list.Count());

	foreach(station_id, _ in all_station_list)
	{
		AILog.Info("station: " + AIStation.GetName(station_id));
	}

	// Make the aircraft pool
	local aircraft_pool = Station.GetAllAirVehiclesOfStationList(all_station_list);
	AILog.Info("aircraft pool size: " + aircraft_pool.Count());

	// Sort the list so that the big planes appear first
	aircraft_pool.Valuate(Vehicle.GetAirVehiclePlaneType);
	if(AIAirport.PT_SMALL_PLANE > AIAirport.PT_BIG_PLANE)
	{
		aircraft_pool.Sort(AIAbstractList.SORT_BY_VALUE, AIAbstractList.SORT_ASCENDING);
	}
	else
	{
		aircraft_pool.Sort(AIAbstractList.SORT_BY_VALUE, AIAbstractList.SORT_DESCENDING);
	}

	AILog.Info("Preparing an assignment of airplanes to new airport pairs...");
	local aircraft_assignment_list = [];
	local aircraft_sell_list = [];
	foreach(aircraft_id, plane_type in aircraft_pool)
	{
		local big_plane = plane_type == AIAirport.PT_BIG_PLANE || plane_type == AIAirport.PT_INVALID;

		local pair_scores = AIList();
		local i = 0;
		local max_score = 0;
		local best_pair = -1;
		foreach(pair in pair_list)
		{
			local small_airport = Station.IsSmallAirport(pair.station_list[0].station_id) || Station.IsSmallAirport(pair.station_list[1].station_id);

			if(small_airport && big_plane)
			{
				pair.score = - 100;
			}
			else
			{
				local transport_demand = pair.transport_demand;
				local cap = AIVehicle.GetCapacity(aircraft_id, Helper.GetPAXCargo());
				local speed = AIEngine.GetMaxSpeed(AIVehicle.GetEngineType(aircraft_id)); // TODO: replace with an estimation of travel time that take into account accerelation etc. too as well as plane speed setting
				local mean_ap_usage = (pair.station_list[0].usage.aircraft.percent_usage + pair.station_list[1].usage.aircraft.percent_usage) / 2;

				pair.score = cap * speed - mean_ap_usage * 10;

				// Todo relate to transport demand
			}

			if(pair.score > max_score)
			{
				max_score = pair.score;
				best_pair = i;
			}

			i++;
		}

		if(best_pair == -1)
		{
			// Aircraft is not wanted -> sell it
			aircraft_sell_list.append(aircraft_id);
			AILog.Info("aircraft " + AIVehicle.GetName(aircraft_id) + " is not wanted by any pair.");
		}
		else
		{
			// Store the best assignment
			aircraft_assignment_list.append( { aircraft_id=aircraft_id,
					pair_index=best_pair } );
			AILog.Info("aircraft " + AIVehicle.GetName(aircraft_id) + " has been assigned to pair " + best_pair);
		}
	}

	AILog.Info("Breaking before making swap of shared orders groups");
	Helper.BreakPoint(AIMap.GetTileIndex(30, 30));

	// Make a fast swap of shared order groups to form the pairs in the map structure.
	AILog.Info("Making a fast swap of shared order groups");
	{
		// Get the shared vehicle groups
		local used_station_list = [];
		foreach(_ in all_station_list)
		{
			used_station_list.append(false);
		}
		
		AILog.Info("Finding shared groups..");
		local shared_groups = [];
		local j = 0;
		local all_station_list_copy = AIList(); // Because of foreach uses Next() behind the screen a copy of the list is required for the inner loop over all stations
		all_station_list_copy.AddList(all_station_list);
		foreach(j_station_id, _ in all_station_list)
		{
			AILog.Info("station " + AIStation.GetName(j_station_id));

			// Skip stations that have been used (are included in the orders of any of the found shared groups)
			if(used_station_list[j] == true)
			{
				AILog.Info("  already used");
				j++;
				continue;
			}

			local ic_vehicles = AIVehicleList_Station(j_station_id);
			ic_vehicles.Valuate(AIVehicle.GetVehicleType);
			ic_vehicles.KeepValue(AIVehicle.VT_AIR);

			if(ic_vehicles.IsEmpty())
			{
				AILog.Info("  has no visiting vehicles");
				j++;
				continue;
			}

			shared_groups.append(ic_vehicles);

			// Mark the visited stations as used
			AILog.Info("  mark visited stations as used");
			local visited_stations = Order.GetStationListFromOrders(ic_vehicles.Begin());
			foreach(visited_station_id, _ in visited_stations)
			{
				local k = 0;
				foreach(k_station_id, _ in all_station_list_copy)
				{
					if(k_station_id == visited_station_id)
					{
						AILog.Info("    mark " + AIStation.GetName(k_station_id) + " as used");
						used_station_list[k] = true;
					}

					k++;
				}
			}

			j++;
		}

		AILog.Info("Found " + shared_groups.len() + " shared groups.");

		// Now shared_groups contain a list of groups of vehicles that correspond to the old pairs.
		for(local j = 0; j < pair_list.len(); j++)
		{
			// Convert the squirrel list of station statistics into a AIList with just station ids. 
			// This list is used as argument to set the new orders.
			local ai_list_stations = Station.StationStatsSquirrelList_To_StationIdAIList(pair_list[j].station_list);

			AILog.Info("New pair: " + AIStation.GetName(pair_list[j].station_list[0].station_id) + " & " + AIStation.GetName(pair_list[j].station_list[1].station_id));
			if(j >= shared_groups.len())
			{
				// There is to few shared groups
				
				// Look for an aircraft in the pool that share its orders with at least one other aircraft
				local found_aircraft_id = -1;
				foreach(aircraft_id, plane_type in aircraft_pool)
				{
					local shared_orders_group = AIVehicleList_SharedOrders(aircraft_id);
					if(shared_orders_group.Count() > 1)
					{
						found_aircraft_id = shared_orders_group.Begin();
						break;
					}
				}

				if(found_aircraft_id != -1 && AIVehicle.IsValidVehicle(found_aircraft_id))
				{
					AILog.Info("Breaking before attempting use an aircraft to bind a single pair. (" + AIVehicle.GetName(found_aircraft_id) + ")");
					Helper.BreakPoint(AIMap.GetTileIndex(30, 30));

					AIOrder.UnshareOrders(found_aircraft_id);
					Builder.UpdateInterCityOrders(found_aircraft_id, ai_list_stations);
				}
				else
				{
					AILog.Warning("There is to few aircrafts to form all the new pairs. This is really strange.");
				}
			}
			else
			{
				local veh_id = shared_groups[j].Begin();

				AILog.Info("Breaking before attempting use a shared orders group to bind a single pair ( containing veh " + AIVehicle.GetName(veh_id) + ")");
				Helper.BreakPoint(AIMap.GetTileIndex(30, 30));

				// Change the orders so they match the station list for the current pair
				Builder.UpdateInterCityOrders(veh_id, ai_list_stations); 
			}
		}
	}

/*
	// Go through the aircraft_assignment_list and re-assign all the aircrafts and then go through aircraft_sell_list and sell un-wanted aircrafts.
	AILog.Info("Using the prepaired assignment, tell each airplane which airport pair it should serve");
	{
		foreach(assignment in aircraft_assignment_list)
		{
			// A bit of crash security check doesn't harm.
			if(assignment.pair_index < 0 || assignment.pair_index >= pair_list.len())
			{
				AILog.Error("Aircraft assigned to new pair that does not exist. Selling the aircraft instead");
				aircraft_sell_list.append(assignment.aircraft_id);
				continue;
			}

			local pair = pair_list[assignment.pair_index];
			local aircraft_id = assignment.aircraft_id;

			if(Order.HasStationInOrders(aircraft_id, pair.station_list[0].station_id) && Order.HasStationInOrders(aircraft_id, pair.station_list[1].station_id))
			{
				// Aircraft already visit the correct pair
				continue;
			}

			AIOrder.UnshareOrders(aircraft_id);
			local shared_orders_group_to_join = AIVehicleList_Station(pair.station_list[0].station_id);
			if(shared_orders_group_to_join.IsEmpty())
			{
				// There are currently no aircraft serving the target pair, so make new orders for the aircraft
				local ai_list_station_ids = Station.StationStatsSquirrelList_To_StationIdAIList(pair.station_list); 
				Builder.UpdateInterCityOrders(aircraft_id, ai_list_station_ids);
			}
			else
			{
				// Join the group
				AIOrder.ShareOrders(aircraft_id, shared_orders_group_to_join.Begin());
			}
		}

		foreach(aircraft_id in aircraft_sell_list)
		{
			Vehicle.SendVehicleToDepotForSelling(aircraft_id);
		}

	}
*/

}

function AirManager::GetTransportDemand(station1_stats, station2_stats)
{
	local list = [];
	list.append(station1_stats);
	list.append(station2_stats);

	local max_demand = 0;
	local second_demand = 0;
	foreach(station_stats in list)
	{
		local town_id = AITile.GetClosestTown(AIStation.GetLocation(station_stats.station_id));
		local pop = AITown.GetPopulation(town_id);

		local demand = pop / 10 /* <-- small factor */ + (station_stats.pax_waiting * station_stats.usage.aircraft.percent_usage) / 100; 

		if(demand > max_demand)
		{
			second_demand = max_demand;
			max_demand = demand;
		}
	}

	local transport_distance = AIMap.DistanceManhattan(AIStation.GetLocation(station1_stats.station_id), AIStation.GetLocation(station2_stats.station_id));
	// The idea is that you need the amount of capacity that is required to transport the max_demand back and forth.
	local transport_demand = max_demand * transport_distance * 2; // - (max_demand - second_demand) / 3 * transport_distance;

	return transport_demand;
}
