/*
 * This file is part of PAXLink, which is an AI for OpenTTD
 * Copyright (C) 2008, 2009  Leif Linse
 *
 * PAXLink is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License
 *
 * PAXLink is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with PAXLink; If not, see <http://www.gnu.org/licenses/> or
 * write to the Free Software Foundation, Inc., 51 Franklin Street, 
 * Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

//////////////////////////////////////////////////////
//                                                  //
//   CLASS: Builder                                 //
//                                                  //
//////////////////////////////////////////////////////
class Builder {

	constructor()
	{
	}

	static function HasTileRectVehiclesOnIt(tile_rect, vehicle_type);

	static function DoesAirportAcceptPAX(station_id);
	static function WillAirportAcceptPAX(top_left_tile, airport_type);
	static function HasSmallAirport(station_list);
	static function HasAirportRectAtLeastOneBusStopInTown(top_left_tile, width, height, town_id);
	static function FindInterCityStationPlacement(town_id, station_w, station_h, airport_type);

	static function CostToFlattern(top_left_tile, width, height);
	static function IsTileRectBuildableAndFlat(top_left_tile, width, height);
	static function MakeTownGrid(town_id, size);

	static function ShouldHaveFeederMode(town_id, inter_city_station);
	static function ConnectAllBusStationsAndDepotsByRoad(town_mgr);

	static function UpdateInterCityOrders(vehicle_id, inter_city_stations);
}


function GetBuildGridSize(town_id)
{
	local pop = AITown.GetPopulation(town_id);

	local size = 1 + pop / 2000;
	size = size.tointeger();

	//AILog.Info("Suggests grid size " + size);

	return size;
}

function Builder::MakeTownGrid(town_id, size)
{
	local town_center = AITown.GetLocation(town_id);

	local town_center_x = AIMap.GetTileX(town_center);
	local town_center_y = AIMap.GetTileY(town_center);

	local ret = AIList();

	local x, y;
	for(x = town_center_x - 7 * size + 3; x <= town_center_x + 7 * size - 3; x += 7)
	{
		for(y = town_center_y - 7 * size + 3; y <= town_center_y + 7 * size - 3; y += 7)
		{
			local tile = AIMap.GetTileIndex(x, y);

			// only add tile if it is closest to town_id town
			if(AITile.GetClosestTown(tile) == town_id)
			{
				ret.AddItem(tile, 0);
			}
		}
	}

	return ret;
}

function IsTileOnTownRoadGrid(tile_id)
{
	local n = AIGameSettings.GetValue("economy.town_layout").tointeger();
	local dx = abs(tile_id % AIMap.GetMapSizeY());
	local dy = abs(tile_id / AIMap.GetMapSizeY());
	local on_road_grid = (n == 2 || n == 3) &&     // Town has 2x2 or 3x3 road grid layout
		(dx % (n+1) == 0 || dy % (n+1) == 0);      // and tile is on the road grid

	return on_road_grid;
}

function GetNeighbours4(tile_id)
{
	local list = AITileList();

	if(!AIMap.IsValidTile(tile_id))
		return list;

	local tile_x = AIMap.GetTileX(tile_id);
	local tile_y = AIMap.GetTileY(tile_id);

	list.AddTile(Helper.GetTileRelative(tile_id, -1, 0));
	list.AddTile(Helper.GetTileRelative(tile_id, 0, -1));
	list.AddTile(Helper.GetTileRelative(tile_id, 1, 0));
	list.AddTile(Helper.GetTileRelative(tile_id, 0, 1));

	return list;
}

function GetNeighbours8(tile_id)
{
	local list = AITileList();

	if(!AIMap.IsValidTile(tile_id))
		return list;

	local tile_x = AIMap.GetTileX(tile_id);
	local tile_y = AIMap.GetTileY(tile_id);

	list.AddTile(Helper.GetTileRelative(tile_id, -1, 0));
	list.AddTile(Helper.GetTileRelative(tile_id, 0, -1));
	list.AddTile(Helper.GetTileRelative(tile_id, 1, 0));
	list.AddTile(Helper.GetTileRelative(tile_id, 0, 1));

	list.AddTile(Helper.GetTileRelative(tile_id, -1, -1));
	list.AddTile(Helper.GetTileRelative(tile_id, 1, -1));
	list.AddTile(Helper.GetTileRelative(tile_id, 1, 1));
	list.AddTile(Helper.GetTileRelative(tile_id, -1, 1));

	return list;
}

function GridTileNeighbourScoreValuator(neighbour_tile_id, grid_tile_id)
{
	local score = 0;

	if(AITile.IsBuildable(neighbour_tile_id))
		score += 100;

	if(AIRoad.GetNeighbourRoadCount(neighbour_tile_id) != 0)
		score += 200;

	local diff_x = AIMap.GetTileX(grid_tile_id) - AIMap.GetTileX(neighbour_tile_id);
	local diff_y = AIMap.GetTileY(grid_tile_id) - AIMap.GetTileY(neighbour_tile_id);
	local opposite_tile_id = grid_tile_id + AIMap.GetTileIndex(diff_x, diff_y);

	// don't give bonus to end of road.
	if(AIRoad.IsRoadTile(opposite_tile_id) && AIRoad.AreRoadTilesConnected(opposite_tile_id, grid_tile_id))
		score -= 200;

	// prefer tiles close to grid_tile_id
	score -= AIMap.DistanceManhattan(grid_tile_id, neighbour_tile_id) * 10

	// Give a big penalty to tiles that is on the town grid
	if(AIController.GetSetting("avoid_town_road_grid") == 1 && IsTileOnTownRoadGrid(neighbour_tile_id))
		score -= 400;

	return score;
}

function Builder::CostToClearTiles(tile_list)
{
	local test_mode = AITestMode();
	local account = AIAccounting();

	// Execute DemolishTile for all tiles
	Utils_Valuator.Valuate(tile_list, AITile.DemolishTile);

	return account.GetCosts();
}

function CanConnectToRoad(road_tile, adjacent_tile_to_connect)
{
	// If road_tile don't have road type "road" (ie it is only a tram track), then we can't connect to it
	if(!AIRoad.HasRoadType(road_tile, AIRoad.ROADTYPE_ROAD))
		return false;

	local neighbours = GetNeighbours4(road_tile);
	
	neighbours.RemoveTile(adjacent_tile_to_connect);

	// This function requires that road_tile is connected with at least one other road tile.
	neighbours.Valuate(AIRoad.IsRoadTile);
	if(Helper.ListValueSum(neighbours) < 1)
		return false;
	
	for(local neighbour_tile_id = neighbours.Begin(); neighbours.HasNext(); neighbour_tile_id = neighbours.Next())
	{
		if(AIRoad.IsRoadTile(neighbour_tile_id))
		{
			local ret = AIRoad.CanBuildConnectedRoadPartsHere(road_tile, neighbour_tile_id, adjacent_tile_to_connect);
			if(ret == 0 || ret == -1)
				return false;
		}
	}

	return true;
}

function Builder::BuildRoadStation(tile_id, depot)
{
	if(AITile.IsWaterTile(tile_id))
		return false;

	// Don't even try if the tile has a steep slope
	local slope = AITile.GetSlope(tile_id);
	if(slope != 0 && AITile.IsSteepSlope(slope))
		return false;

	if(AIController.GetSetting("forbid_town_road_grid") == 1 && IsTileOnTownRoadGrid(tile_id))
		return false;

	local neighbours = GetNeighbours4(tile_id);
	neighbours.Valuate(AIRoad.IsRoadTile);
	neighbours.KeepValue(1);
	neighbours.Valuate(AIRoad.HasRoadType, AIRoad.ROADTYPE_ROAD);
	neighbours.KeepValue(1);

	/*Helper.ClearAllSigns();
	for(local tile = neighbours.Begin(); neighbours.HasNext(); tile = neighbours.Next())
	{
		//Helper.SetSign(tile, "n");
		AISign.BuildSign(tile, "n");
	}*/

	//MyValuate(neighbours, CanConnectToRoad, tile_id);
	neighbours.Valuate(CanConnectToRoad, tile_id);
	neighbours.KeepValue(1);

	neighbours.Valuate(Helper.GetMaxHeightOfTile);
	neighbours.KeepValue(Helper.GetMaxHeightOfTile(tile_id));

	// Return false if none of the neighbours has road
	if(neighbours.Count() == 0)
		return false;

	local front = neighbours.Begin();

	
	if(!CanConnectToRoad(front, tile_id))
		return false;

	if(!AITile.IsBuildable(tile_id) && !AIRoad.IsRoadTile(tile_id) && !AICompany.IsMine(AITile.GetOwner(tile_id)))
	{
		AITile.DemolishTile(tile_id);
	}

	// Build road
	// Keep trying until we get another error than vehicle in the way
	while(!AIRoad.BuildRoad(tile_id, front) && AIError.GetLastError() == AIError.ERR_VEHICLE_IN_THE_WAY);

	if(!AIRoad.AreRoadTilesConnected(tile_id, front))
		return false;

	//Helper.SetSign(tile_id, "tile");
	//Helper.SetSign(front, "front");

	// Build station/depot
	// Keep trying until we get another error than vehicle in the way
	local ret = false; // for some strange reason this variable declaration can't be inside the loop
	do
	{
		if(depot)
			ret = AIRoad.BuildRoadDepot(tile_id, front);
		else
			ret = AIRoad.BuildRoadStation(tile_id, front, AIRoad.ROADVEHTYPE_BUS, AIStation.STATION_NEW);
	}
	while(!ret && AIError.GetLastError() == AIError.ERR_VEHICLE_IN_THE_WAY);

	return ret;
}

function Builder::BuildInCityStationCloseTo(tile_id, town_id)
{
	local depot = false

	//if(AITile.IsBuildable(tile_id))
	if(BuildRoadStation(tile_id, depot))
	{
		return tile_id;
	}
	else
	{
		AILog.Info("BulidRoadStation error: " + AIError.GetLastErrorString());
		if(AIError.GetLastError() == AIError.ERR_LOCAL_AUTHORITY_REFUSES)
			return AIMap.GetTileIndex(-1, -1);

		local neighbours = GetNeighbours8(tile_id);

		// only accept neighbours that are in the town
		neighbours.Valuate(AITile.GetClosestTown);
		neighbours.KeepValue(town_id);

		//MyValuate(neighbours, GridTileNeighbourScoreValuator, tile_id);
		//Helper.MyClassValuate(neighbours, this.GridTileNeighbourScoreValuator, this, tile_id);
		neighbours.Valuate(GridTileNeighbourScoreValuator, tile_id);
		neighbours.Sort(AIAbstractList.SORT_BY_VALUE, false);

		if(neighbours.Count() > 0)
		{
			// Try with first neighbour, but if that fails, take next one and try.
			local build_at = neighbours.Begin();
			while(neighbours.HasNext())
			{
				if(BuildRoadStation(build_at, depot))
				{
					AILog.Info("Built road station.");
					return build_at;
				}
				else if(AIError.GetLastError() == AIError.ERR_LOCAL_AUTHORITY_REFUSES)
				{
					// Don't retry if the reason for failure was that the local authority refused building
					return AIMap.GetTileIndex(-1, -1);
				}

				build_at = neighbours.Next();
			}
		}
	}

	//AILog.Info("Returning tile -1, -1");
	return AIMap.GetTileIndex(-1, -1);
}

//// BEGIN OF INTER-CITY helpers

function Builder::WillAirportAcceptPAX(top_left_tile, airport_type)
{
	local coverage_radius = AIAirport.GetAirportCoverageRadius(airport_type);

	//Helper.SetSign(top_left_tile, "ap tile " + coverage_radius + " type: " + airport_type, true);

	local ap_w = AIAirport.GetAirportWidth(airport_type);
	local ap_h = AIAirport.GetAirportHeight(airport_type);

	local acc = AITile.GetCargoAcceptance(top_left_tile, Helper.GetPAXCargo(), ap_w, ap_h, coverage_radius);
	//Helper.SetSign(Helper.GetTileRelative(top_left_tile, ap_w - 1, ap_h - 1), "ap bl tile " + acc + " w " + ap_w + " h " + ap_h, true);


	return AITile.GetCargoAcceptance(top_left_tile, Helper.GetPAXCargo(), ap_w, ap_h, coverage_radius) >= 8;
}

function Builder::DoesAirportAcceptPAX(station_id)
{
	local airport_tile = Station.GetAirportTile(station_id);
	local airport_type = AIAirport.GetAirportType(airport_tile);
	
	return Builder.WillAirportAcceptPAX(airport_tile, airport_type);
}

function Builder::IsTileRectBuildableAndFlat(top_left_tile, width, height)
{
	local tiles = AITileList();

	// First use API function to check rect as that is faster than doing it self in squirrel.
	if(!AITile.IsBuildableRectangle(top_left_tile, width, height))
		return false;

	// Then only if it is buildable, check that it is flat also.
	tiles.AddRectangle(top_left_tile, Helper.GetTileRelative(top_left_tile, width - 1, height - 1));

	// remember how many tiles there are from the beginning
	local count_before = tiles.Count();

	// remove non-flat tiles
	tiles.Valuate(AITile.GetSlope);
	tiles.KeepValue(0);

	// if all tiles are remaining, then all tiles are flat
	return count_before == tiles.Count();
}

function Builder::CostToFlattern_EachInList(list, width, height)
{
	local top_left_tile;
	for(top_left_tile = list.Begin(); list.HasNext(); top_left_tile = list.Next())
	{
		list.SetValue(top_left_tile, Builder.CostToFlattern(top_left_tile, width, height));
	}
}
function Builder::CostToFlattern(top_left_tile, width, height)
{
	if(!AITile.IsBuildableRectangle(top_left_tile, width, height))
		return -1; // not buildable

	if(Builder.IsTileRectBuildableAndFlat(top_left_tile, width, height))
		return 0; // zero cost

	local level_cost = 0;
	{{
		local test = AITestMode();
		local account = AIAccounting();
		
		if(!AITile.LevelTiles(top_left_tile, Helper.GetTileRelative(top_left_tile, width, height)))
			return -1;

		level_cost = account.GetCosts();
	}}

	return level_cost;

	return 0;
}

function Builder::FlatternRect(top_left_tile, width, height)
{
	local tiles = AITileList();

	tiles.AddRectangle(top_left_tile, Helper.GetTileRelative(top_left_tile, width - 1, height - 1));

	AILog.Info("Flattern rect");
	if(AIController.GetSetting("debug_signs") == 1)
	{
		//Helper.ClearAllSigns();
		for(local i = tiles.Begin(); tiles.HasNext(); i = tiles.Next())
		{
			//AISign.BuildSign(i, "flat");
		}
	}

	return AITile.LevelTiles(top_left_tile, Helper.GetTileRelative(top_left_tile, width, height));
}

function Builder::HasTileRectVehiclesOnIt(tile_rect, vehicle_type)
{
	local all_veh = AIVehicleList();
	all_veh.Valuate(AIVehicle.GetVehicleType);
	all_veh.KeepValue(vehicle_type);

	all_veh.Valuate(StationStatistics.VehicleIsWithinTileList, tile_rect);
	all_veh.KeepValue(1);
	
	return !all_veh.IsEmpty();
}

function Builder::HasAirportRectAtLeastOneBusStopInTown(top_left_tile, width, height, town_id)
{
	// loop through all bus stop tiles and check if closest town has desired town id
	for(local i = 0; i < width; i++)
	{
		local bus_stop_tile = Helper.GetTileRelative(top_left_tile, i, height - 2);
		if(AITile.GetClosestTown(bus_stop_tile) == town_id)
			return true; // until a bus stop is found that is OK, then return true
	}

	// All bus stops are in wrong town
	return false;
}

//function WillAirportAccept

function Builder::FindInterCityStationPlacement(town_id, station_w, station_h, airport_type, maximum_noise)
{
	local town_tiles = AITileList();
	local town_tile = AITown.GetLocation(town_id);
	local radius = 20 + Helper.Max(0, AITown.GetPopulation(town_id) / 1000);
	town_tiles.AddRectangle(Helper.GetTileRelative(town_tile, -radius, -radius), Helper.GetTileRelative(town_tile, radius, radius));

	/*for(local i = town_tiles.Begin(); town_tiles.HasNext(); i = town_tiles.Next())
	{
		Helper.SetSign(i, " ");
	}*/

	town_tiles.Valuate(AITile.GetClosestTown);
	town_tiles.KeepValue(town_id);

	town_tiles.Valuate(Builder.WillAirportAcceptPAX, airport_type);
	town_tiles.KeepValue(1);

	town_tiles.Valuate(Builder.HasAirportRectAtLeastOneBusStopInTown, station_w, station_h, town_id);
	town_tiles.RemoveValue(0);

	if(maximum_noise != -1)
	{
		town_tiles.Valuate(AIAirport.GetNoiseLevelIncrease, airport_type);
		town_tiles.KeepBelowValue(maximum_noise + 1);
	}

	Utils_Valuator.Valuate(town_tiles, Builder.CostToFlattern, station_w, station_h);
	town_tiles.RemoveValue(-1);
	town_tiles.KeepBelowValue(AICompany.GetBankBalance(AICompany.COMPANY_SELF) / 10);

	//town_tiles.Valuate(Builder.IsTileRectBuildableAndFlat, station_w, station_h);
	//town_tiles.KeepValue(1);

	if(AIController.GetSetting("debug_signs") == 1)
	{
		for(local i = town_tiles.Begin(); town_tiles.HasNext(); i = town_tiles.Next())
		{
			//AISign.BuildSign(i, "bf");
		}
	}


	//Helper.SetSign(town_tiles.Begin(), "here");

	if(town_tiles.IsEmpty())
		return -1;

	//town_tiles.Valuate(AIMap.DistanceManhattan, town_tile);
	town_tiles.Sort(AIAbstractList.SORT_BY_VALUE, true); // smallest first

	return town_tiles.Begin();
}

function GetBestAirportAvailable(max_x, max_y)
{
	local airports = [ AIAirport.AT_INTERCON, AIAirport.AT_INTERNATIONAL, AIAirport.AT_METROPOLITAN, AIAirport.AT_LARGE, AIAirport.AT_COMMUTER, AIAirport.AT_SMALL ]

	foreach(airport in airports)
	{
		if(AIAirport.IsValidAirportType(airport) && AIAirport.GetAirportWidth(airport) <= max_x && AIAirport.GetAirportHeight(airport) <= max_y)
			return airport;
	}

	return AIAirport.AT_INVALID;
}

//// END OF INTER-CITY

function StationTileXValuator(station_id)
{
	return AIMap.GetTileX(AIStation.GetLocation(station_id));
}
function StationTileYValuator(station_id)
{
	return AIMap.GetTileY(AIStation.GetLocation(station_id));
}

function GetStationsRectangle(stations_list, grow_border_by = 0)
{
	//Helper.ClearAllSigns();

	local min_x, min_y, max_x, max_y;
	stations_list.Valuate(StationTileXValuator);
	stations_list.Sort(AIAbstractList.SORT_BY_VALUE, true); // lowest value first
	min_x = stations_list.GetValue(stations_list.Begin());
	stations_list.Valuate(StationTileXValuator);
	stations_list.Sort(AIAbstractList.SORT_BY_VALUE, false); // highest value first
	max_x = stations_list.GetValue(stations_list.Begin());

	stations_list.Valuate(StationTileYValuator);
	stations_list.Sort(AIAbstractList.SORT_BY_VALUE, true); // lowest value first
	min_y = stations_list.GetValue(stations_list.Begin());
	stations_list.Sort(AIAbstractList.SORT_BY_VALUE, false); // highest value first
	max_y = stations_list.GetValue(stations_list.Begin());
	
	local rect_from = AIMap.GetTileIndex(min_x - grow_border_by, min_y - grow_border_by);
	local rect_to = AIMap.GetTileIndex(max_x + grow_border_by, max_y + grow_border_by);

	//AISign.BuildSign(rect_from, "rect from");
	//AISign.BuildSign(rect_to, "rect to");

	local tiles = AITileList();
	tiles.AddRectangle(rect_to, rect_from);

	return tiles;
}

function BuildDepotAtTileScore_EachInList(list)
{
	local item;
	for(item = list.Begin(); list.HasNext(); item = list.Next())
	{
		list.SetValue(item, BuildDepotAtTileScoreValuator(item));
	}
}

function BuildDepotAtTileScoreValuator(tile_id)
{
	local score = 0;

	local road_neighbour_count = AIRoad.GetNeighbourRoadCount(tile_id);

	if(AIRoad.IsRoadTile(tile_id))
	{
		// give score if tile has road, but only one neighbor has that, as that means it is a road stump
		if(road_neighbour_count == 1)
			score += 1000;
	}
	else
	{
		// better build it along an existing road, than hijacking a road stump
		if(road_neighbour_count > 0)
			score += 2000;
	}

	if(AITile.GetSlope(tile_id) == 0)
		score += 500;

	if(AITile.IsBuildable(tile_id))
	{
		score += 100;
	}
	else
	{
		local testmode = AITestMode();
		local acount = AIAccounting();

		AITile.DemolishTile(tile_id);
		
		score -= acount.GetCosts();
	}

	return score;
}

function GetDepotPlacementArea(town_id)
{
	local town_tiles = AITileList();
	local stations = Station.GetAllFeederStationsOfTown(town_id);
	town_tiles = GetStationsRectangle(stations, 2);

	town_tiles.Valuate(AITile.GetClosestTown);
	town_tiles.KeepValue(town_id);

	return town_tiles;
}

function FindDepot(town_id)
{
	local depot_list = AIDepotList(AITile.TRANSPORT_ROAD);
	depot_list.Valuate(AITile.GetClosestTown);
	depot_list.KeepValue(town_id);

	if(depot_list.IsEmpty())
	{
		return AIMap.GetTileIndex(-1, -1);
	}

	return depot_list.Begin();
}

function GetRoadStationCatchedCargo(tile_id)
{
	return AITile.GetCargoAcceptance(tile_id, Helper.GetPAXCargo(), 1, 1, 3);
}

function Builder::ConstructTownStations(town_mgr, money_needed_to_construct)
{
	local town_id = town_mgr.town_id; // town_id used to be the in-parameter to this function and is thus used a lot.

	//Helper.ClearAllSigns();
	AILog.Info("building in " + AITown.GetName(town_id));
	//AISign.BuildSign(AITown.GetLocation(town_id), "Build stations");

	// Ask the town manager how many feeder bus stops we want to have in the town
	local wanted_number_of_bus_stops = town_mgr.GetNumberOfFeederStationsNeeded();
	AILog.Info("Want to have " + wanted_number_of_bus_stops + " bus stops in town " + AITown.GetName(town_id));

	//// First go through old stations ////
	local old_stations_list = Station.GetAllFeederStationsOfTown(town_id);

	// Go through all old stations and remove stations that version 7 or earlier built so it they faced a road it could not be connected to.
	for(local old_station_id = old_stations_list.Begin(); old_stations_list.HasNext(); old_station_id = old_stations_list.Next())
	{
		local station_tile = AIStation.GetLocation(old_station_id);
		local front_tile = AIRoad.GetRoadStationFrontTile(station_tile);
		if(!CanConnectToRoad(front_tile, station_tile))
		{
			AILog.Info("Removing station that is not connected to road");
			AIRoad.RemoveRoadStation(station_tile);
			Station.DemolishStation(old_station_id);
		}
	}

	old_stations_list = Station.GetAllFeederStationsOfTown(town_id);

	// Check if we need to remove stations to full fill the number of stations we want to have in the town
	local ignore = 0;
	while(old_stations_list.Count() - ignore > wanted_number_of_bus_stops)
	{
		AILog.Info("There are more old stations than wanted bus stops (" + old_stations_list.Count() + " old stations; want to have " + wanted_number_of_bus_stops + " stations)");
		local remove_station = -1;
		local min_usage = 100000;
		for(local old_station_id = old_stations_list.Begin(); old_stations_list.HasNext(); old_station_id = old_stations_list.Next())
		{
			local statistics = town_mgr.GetStationStatistics(old_station_id)
			if(statistics)
			{
				local usage = statistics.usage.bus.percent_usage;
				if(usage < min_usage)
				{
					min_usage = usage;
					remove_station = old_station_id;
				}
			}
		}
		
		if(!Station.DemolishStation(remove_station));
			ignore++;

		old_stations_list = Station.GetAllFeederStationsOfTown(town_id);
	}
	
	if(AICompany.GetBankBalance(AICompany.COMPANY_SELF) <= money_needed_to_construct)
	{
		return false;
	}

	//// Second possible build new grid stations in town ////
	local new_stations_list = AIList();

	// only build new stations when we want more stations
	local num_stations_to_build = wanted_number_of_bus_stops - old_stations_list.Count();
	if(num_stations_to_build > 0) 
	{
		// TODO: Improve by going through all tiles and give them a score and then try with the highest score first so that
		// the cost to clear in terms of town rating and cost can be weighted to the cargo available there etc. 

		local num_stations_built = 0;
		local tile_list = MakeTownGrid(town_id, GetBuildGridSize(town_id));
		tile_list.Valuate(AIMap.DistanceManhattan, AITown.GetLocation(town_id))
		tile_list.Sort(AIAbstractList.SORT_BY_VALUE, true); // lowest first -> start by trying locations closest to city center
		local tile = tile_list.Begin();
		local i = 0;
		while(num_stations_to_build > num_stations_built && tile_list.HasNext())
		{
			//AISign.BuildSign(tile, " grid " + i);

			local cargo = GetRoadStationCatchedCargo(tile);

			// If an old station is on grid-tile or one of the neighbours, then don't build a station
			old_stations_list.Valuate(AIStation.GetDistanceSquareToTile, tile);
			old_stations_list.Sort(AIAbstractList.SORT_BY_VALUE, true); // lowest value first
			if(tile_list.Count() > 1 &&  // don't construct anything if only one grid point has enough cargo
						(old_stations_list.IsEmpty() || old_stations_list.GetValue(old_stations_list.Begin()) > 2))
			{
				if(cargo >= 30) 
				{
					local ret = BuildInCityStationCloseTo(tile, town_id);
					if(AIMap.IsValidTile(ret))
					{
						new_stations_list.AddItem(ret, 0);
						num_stations_built++;

						local station_id = AIStation.GetStationID(ret);
						local new_name = AITown.GetName(town_id) + " Feeder Station " + station_id;
						AILog.Info("Set station name to \"" + new_name + "\"");
						AIStation.SetName(station_id, new_name);
					}
				}
			}
			else if(tile_list.Count() <= 1 || (!old_stations_list.IsEmpty() && cargo < 15))
			{
				local station_tile = AIStation.GetLocation(old_stations_list.Begin());
				AIRoad.RemoveRoadStation(station_tile);
			}

			i++;
			tile = tile_list.Next();
		}
	}

	// If number of stations in town doesn't meet the requirement for the current operation mode, return false
	local total_stations_list = Station.GetAllFeederStationsOfTown(town_id);
	local inter_city_station = Station.FindInterCityStation(town_id);
	local feeder_mode = ShouldHaveFeederMode(town_id, inter_city_station);
	if(!feeder_mode && total_stations_list.Count() < 2)
	{
		return false;
	}
	else if(feeder_mode && total_stations_list.Count() < 1)
	{
		return false;
	}



	//// Third: build depot ////

	// Check if the town has a depot
	local depot_placement_tiles = GetDepotPlacementArea(town_id);
	local depot = FindDepot(town_id);

	if(AIMap.IsValidTile(depot))
	{
		local depot_tile = depot_placement_tiles.Begin();
		AILog.Info("depot exist on tile " + depot_tile);
	}
	else
	{
		AILog.Info("building new depot");
	
		// get the boundary box of the stations in town

		// locate tile within depot_placement_tiles to place depot
		depot_placement_tiles.Valuate(AIRoad.GetNeighbourRoadCount);
		//depot_placement_tiles.KeepAboveValue(0);
		depot_placement_tiles.Valuate(AITile.IsWaterTile);
		//depot_placement_tiles.KeepValue(0);
		depot_placement_tiles.Valuate(AITile.IsStationTile);
		//depot_placement_tiles.KeepValue(0);
		//depot_placement_tiles.Valuate(AITile.IsSteepSlope);
		//depot_placement_tiles.KeepValue(0);
		depot_placement_tiles.Valuate(AITile.GetOwner);
		depot_placement_tiles.RemoveValue(AICompany.ResolveCompanyID(AICompany.COMPANY_SELF));

		Utils_Valuator.Valuate(depot_placement_tiles, BuildDepotAtTileScoreValuator);
		//depot_placement_tiles.Valuate(BuildDepotAtTileScoreValuator);
		BuildDepotAtTileScore_EachInList(depot_placement_tiles);
		depot_placement_tiles.Sort(AIAbstractList.SORT_BY_VALUE, false); // highest value first
		
		local depot_tile = depot_placement_tiles.Begin();
		while(depot_placement_tiles.HasNext())
		{
			//AILog.Info("Trying to build depot at tile " + depot_tile);
			//AISign.BuildSign(depot_tile, "build depot");

			local depot = true;
			if(BuildRoadStation(depot_tile, depot))
				break;			

			depot_tile = depot_placement_tiles.Next();
		}

		if(!AIRoad.IsRoadDepotTile(depot_tile))
			AILog.Error("Failed to build depot!");

		// depot built likely is equal to no buses yet in town.
		BuyBusesForTown(town_id)
	}

	//// Fourth decide if inter city station should be built ////

	/*if(AIStation.IsValidStation(inter_city_station))
	{
		Helper.SetSign(AITown.GetLocation(town_id), "Has INTER CITY");
	}*/
	if(!AIStation.IsValidStation(inter_city_station) && AITown.GetPopulation(town_id) > 700 &&
		AITown.GetRating(town_id, AICompany.COMPANY_SELF) >= AITown.TOWN_RATING_POOR) // Require at least poor rating before building an airport
	{

		Helper.SetSign(AITown.GetLocation(town_id), "trying to build IC");
		AILog.Info("Trying to build INTER-CITY station");

		local best_airport_available = GetBestAirportAvailable(8, 8);
		local ap_w = AIAirport.GetAirportWidth(best_airport_available);
		local ap_h = AIAirport.GetAirportHeight(best_airport_available);

		AILog.Info("best available airport = " + best_airport_available);

		ap_h += 2; // for bus stops

		local max_noise_add = AITown.GetAllowedNoise(town_id);
		
		// Find somewhere to build the airport.
		// since level rect function do not guarantee to work in exec mode even if it work in test-mode
		// it need to be an iterative process to level land for the airport.
		local i = 0;
		local inter_city_placement = Builder.FindInterCityStationPlacement(town_id, ap_w, ap_h, best_airport_available, max_noise_add);
		do
		{
			inter_city_placement = Builder.FindInterCityStationPlacement(town_id, ap_w, ap_h, best_airport_available, max_noise_add);

			FlatternRect(inter_city_placement, ap_w, ap_h);
	
			// Abort after 5 tries
			if(++i > 5)
				break;
		}
		while(AIMap.IsValidTile(inter_city_placement) && !Builder.IsTileRectBuildableAndFlat(inter_city_placement, ap_w, ap_h))

		if(AIMap.IsValidTile(inter_city_placement))
		{
			local ap_x = AIMap.GetTileX(inter_city_placement);
			local ap_y = AIMap.GetTileY(inter_city_placement);

			//Helper.ClearAllSigns();
			//AISign.BuildSign(inter_city_placement, "INTER-CITY");
			
			// Build airport
			local airport_status = AIAirport.BuildAirport(inter_city_placement, best_airport_available, AIStation.STATION_NEW);

			if(!AIAirport.IsAirportTile(inter_city_placement))
			{
				AILog.Info("Build airport error msg: " + AIError.GetLastErrorString());
			}
			else
			{
				// So that the bus station maintenance code below find the airport, and builds bus stations
				inter_city_station = AIStation.GetStationID(inter_city_placement);

				// update the possibility to add an airport to the town now that we have built a new airport for the town
				town_mgr.ForceRecheckIfAirportCanBeAdded();
			}
		}

	}

	if(AIStation.IsValidStation(inter_city_station))
	{
		AILog.Info("Maintaining IC airport");

		local ic_location = AIStation.GetLocation(inter_city_station);
		if(!Builder.DoesAirportAcceptPAX(inter_city_station))
		{
			local vehicles = AIVehicleList_Station(inter_city_station);

			local cost_to_demolish = Station.CostToDemolishStation(inter_city_station);
			local can_afford_demolishing = AICompany.GetBankBalance(AICompany.COMPANY_SELF) * 2 > cost_to_demolish * 3;

			if(vehicles.IsEmpty() && can_afford_demolishing)
			{
				AILog.Warning("IC station does not accept PAX and all vehicles have been sold => destroy the IC station in " + AITown.GetName(town_id));

				// Demolish airport
				Station.DemolishStation(inter_city_station);
			}
		}

		// Only continue maintaining the IC station if it has not been removed :-)
		inter_city_station = AIStation.GetStationID(ic_location);
		if(AIStation.IsValidStation(inter_city_station))
		{
			AILog.Info("Maintaining bus stations of IC airport");
			// Town has IC station, check if it is working properly
			// This code also run, when the airport was just built, to build the bus stations etc. (to save on code lines to maintain)

			local station_tile = Station.GetAirportTile(inter_city_station);
			local airport_type = AIAirport.GetAirportType(station_tile);

			local ap_x = AIMap.GetTileX(station_tile);
			local ap_y = AIMap.GetTileY(station_tile);

			local ap_w = AIAirport.GetAirportWidth(airport_type);
			local ap_h = AIAirport.GetAirportHeight(airport_type);

			ap_h += 2;

			// set station name
			local ic_station_id = AIStation.GetStationID(station_tile);
			local new_name = AITown.GetName(town_id) + " IC Airport";
			AILog.Info("Set station name to \"" + new_name + "\"");
			AIStation.SetName(ic_station_id, new_name);

			// Build bus stations
			AIRail.SetCurrentRailType(AIRailTypeList().Begin());
			local bus_y = ap_y + ap_h - 2;
			for(local bus_x = ap_x; bus_x < ap_x + ap_w; bus_x++)
			{
				local bus_tile = AIMap.GetTileIndex(bus_x, bus_y);
				local bus_front_tile = AIMap.GetTileIndex(bus_x, bus_y + 1);

				// skip tiles that already has bus station
				if(AIRoad.IsRoadStationTile(bus_tile))
					continue;

				local closest_town = AITile.GetClosestTown(bus_tile);
				local retry = AITown.GetRating(closest_town, AICompany.COMPANY_SELF) > AITown.TOWN_RATING_VERY_POOR; 
				if(retry) // Only remove rail on one tile to check if town has become more friendly towards us
				{
					if(AIRail.IsRailTile(bus_tile));
					{
						AIRail.RemoveRailTrack(bus_tile, AIRail.RAILTRACK_NE_SW);
						AIRail.RemoveRailTrack(bus_tile, AIRail.RAILTRACK_NW_SE);
					}
				}

				AIRoad.BuildRoad(bus_tile, bus_front_tile)
				if(!AIRoad.BuildRoadStation(bus_tile, bus_front_tile, AIRoad.ROADVEHTYPE_BUS, inter_city_station))
				{
					if(!AIRail.IsRailTile(bus_tile))
					{
						AITile.DemolishTile(bus_tile);
						AIRail.BuildRailTrack(bus_tile, AIRail.RAILTRACK_NE_SW);
						AIRail.BuildRailTrack(bus_tile, AIRail.RAILTRACK_NW_SE);
					}
					Helper.SetSign(bus_tile, "reserve");
				}
				else
				{
					Helper.SetSign(bus_tile, "");
				}
			}

			// Build road infront of bus stations that connect all stations
			local road_from = AIMap.GetTileIndex(ap_x, bus_y + 1);
			local road_to = AIMap.GetTileIndex(ap_x + ap_w - 1, bus_y + 1);
			AIRoad.BuildRoad(road_from, road_to);

			// Connect inter-city station with town
			RoadBuilder.ConnectTiles(road_from, AITown.GetLocation(town_id), 5);

			// Change buses to feeder-mode
			local stations_list = Station.GetAllFeederStationsOfTown(town_id); // re-generate grid stations list

			local inter_city_station = Station.FindInterCityStation(town_id);
			local vehicles = Station.GetAllVehiclesOfStationList(total_stations_list)
		}
	}

	AILog.Info("Updating intra city orders");
	UpdateIntraCityOrdersForTown(town_id);
	AILog.Info("Intra city orders updated");

	return true;
}

function Builder::ConnectAllBusStationsAndDepotsByRoad(town_mgr)
{
	// don't do anything if the town manager doesn't know of any stations in the town
	if(town_mgr.station_list.len() == 0)
		return;

	// if there is a ic_station, then try to connect all bus stations to that station.
	local connect_all_to = -1;
	local ic_station_id = Station.FindInterCityStation(town_mgr.town_id);
	if(AIStation.IsValidStation(ic_station_id))
	{
		AILog.Info("connect to IC station");
		connect_all_to = ic_station_id;
	}
	else
	{
		AILog.Info("connect to feeder station ");
		foreach(station in town_mgr.station_list)
		{
			if(Station.IsFeederStation(station.station_id))
			{
				connect_all_to = station.station_id;
				break;
			}
		}
	}

	// it could happen that none of the town_mgr stations are valid. Then exit the function
	if(!Station.IsNonEmptyStation(connect_all_to))
		return;
		
	// connect all stations & depots to this tile:
	local to_tile = Station.GetRoadConnectTileOfStation(connect_all_to);
	if(!AIMap.IsValidTile(to_tile))
		return;

	local num_tries = 2;
	local num_days_per_try = 10;
	
	// go through all stations in the town and connect them to to_tile.
	foreach(station in town_mgr.station_list)
	{
		if(station.station_id == connect_all_to)
			continue;

		local from_tile = Station.GetRoadConnectTileOfStation(station.station_id);

		RoadBuilder.ConnectTiles(from_tile, to_tile, num_tries, num_days_per_try);
	}
	
	// go through all depots in the town and connect them to to_tile
	local depots = AIDepotList(AITile.TRANSPORT_ROAD);
	depots.Valuate(AITile.GetClosestTown);
	depots.KeepValue(town_mgr.town_id);
	for(local depot_tile = depots.Begin(); depots.HasNext(); depot_tile = depots.Next())
	{
		local from_tile = AIRoad.GetRoadDepotFrontTile(depot_tile);
		RoadBuilder.ConnectTiles(from_tile, to_tile, num_tries, num_days_per_try);
	}
}

function GetBusEngineScoreValuator(engine_id, needed_capacity)
{
	local score = 0;

	// if can afford engine
	if(AIEngine.GetPrice(engine_id) < AICompany.GetBankBalance(AICompany.COMPANY_SELF))
		score += 1000;

	// If we need 20 capacity, then only those 20 should share the costs.
	local tot_pax_cap = AIEngine.GetCapacity(engine_id);
	local count_pax = Helper.Min(tot_pax_cap, needed_capacity)

	local price = AIEngine.GetPrice(engine_id);
	local running_cost = AIEngine.GetRunningCost(engine_id);

	// Handle the case when engine capacity is zero or 
	// needed_capacity is zero, so that division by zero does not occur.
	local price_per_pax = 1000; 
	local running_cost_per_pax = 1000;
	if(count_pax > 0) 
	{
		price_per_pax = price / count_pax;
		running_cost_per_pax = running_cost / count_pax;
	}



	score -= running_cost_per_pax * 3;

	score -= price_per_pax

	// some hacks to valuate capacity and speed more if needed capacity is really high
	local pax_factor = 2;
	local speed_factor = 2;
	if(needed_capacity > 200)
		pax_factor = 4;
	if(needed_capacity > 800)
	{
		speed_factor = 4;
		pax_factor = 8;
	}

	//pax_factor *= 2;
	speed_factor *= 2;

	score += tot_pax_cap * pax_factor; // while only the needed pax cap shares the cost, a bigger bus get a score bonus even if the extra space isn't needed for now.

	score += AIEngine.GetMaxSpeed(engine_id) * speed_factor;

	score += AIBase.RandRange(100);

	if(AIEngine.GetReliability(engine_id) > 50)
		score += AIEngine.GetReliability(engine_id) * 2;
	
	return score;
}

function Builder::GetEngine(needed_capacity, vehicle_type, only_small_aircrafts = false)
{
	local engine_list = AIEngineList(vehicle_type);

	// Only keep non-articulated
	if(vehicle_type == AIVehicle.VT_ROAD)
	{
		// filter out articulated vehicles
		engine_list.Valuate(AIEngine.IsArticulated)
		engine_list.KeepValue(0);

		// filter out trams
		engine_list.Valuate(AIEngine.GetRoadType);
		engine_list.KeepValue(AIRoad.ROADTYPE_ROAD);
	}
	else if(vehicle_type == AIVehicle.VT_AIR)
	{
		if(only_small_aircrafts)
		{
			engine_list.Valuate(AIEngine.GetPlaneType);
			engine_list.RemoveValue(AIAirport.PT_BIG_PLANE);
		}
	}

	// Don't buy zero-price vehicles. These are often there for eye candy etc.
	engine_list.Valuate(AIEngine.GetPrice);
	engine_list.RemoveValue(0);
	engine_list.Valuate(AIEngine.GetRunningCost);
	engine_list.RemoveValue(0);

	
	//engine_list.Valuate(AIEngine.GetCargoType);
	//engine_list.KeepValue(Helper.GetPAXCargo());
	engine_list.Valuate(AIEngine.CanRefitCargo, Helper.GetPAXCargo());
	engine_list.KeepValue(1);

	/*for(local i = engine_list.Begin(); engine_list.HasNext(); i = engine_list.Next())
	{
		AILog.Info("Engine name: " + AIEngine.GetName(i));
	}*/

	//MyValuate(engine_list, GetBusEngineScoreValuator, needed_capacity);
	engine_list.Valuate(GetBusEngineScoreValuator, needed_capacity);
	engine_list.Sort(AIAbstractList.SORT_BY_VALUE, false); // highest first
	engine_list.KeepTop(1);
	return engine_list.Begin();
}

// Intra (in city) transport
function Builder::UpdateIntraCityOrders_LocalMode(vehicle_id, town_id, town_stations)
{
	AILog.Info("bus only stations: " + town_stations.Count());

	local order_list = OrderList();
	for(local station_id = town_stations.Begin(); town_stations.HasNext(); station_id = town_stations.Next())
	{
		AILog.Info("stop at station_id " + station_id);
		if(AIStation.IsValidStation(station_id))
			AILog.Info("  which is a valid station");
		order_list.AddStop(station_id, AIOrder.AIOF_NON_STOP_INTERMEDIATE);
	}

	order_list.ApplyToVehicle(vehicle_id);
}

function Builder::UpdateIntraCityOrders_FeederMode(vehicle_id, town_id, bus_only_stations, airport_station)
{
	AILog.Info("UpdateIntraCityOrders_FeederMode, vehicle: " + AIVehicle.GetName(vehicle_id));

	local order_list = OrderList();

	AILog.Info("bus only stations: " + bus_only_stations.Count());

	for(local station_id = bus_only_stations.Begin(); bus_only_stations.HasNext(); station_id = bus_only_stations.Next())
	{
		//Helper.SetSign(AIStation.GetLocation(station_id), "add to feeder mode");
		AILog.Info("stop at station_id " + station_id);
		if(AIStation.IsValidStation(station_id))
			AILog.Info("  which is a valid station");
		order_list.AddStop(station_id, AIOrder.AIOF_NON_STOP_INTERMEDIATE | AIOrder.AIOF_NO_UNLOAD);
	}
	order_list.AddStop(airport_station, AIOrder.AIOF_NON_STOP_INTERMEDIATE | AIOrder.AIOF_NO_LOAD | AIOrder.AIOF_TRANSFER);
	order_list.SkipToLastWhenFull(AIController.GetSetting("use_conditional_orders") == 1);

	order_list.ApplyToVehicle(vehicle_id);
}

/* Just use 
 * local inter_city_station = Station.FindInterCityStation(town_id);
 * and then send inter_city_station as second parameter
 * 
 * If the function return true, then inter_city_station is valid too
 */
function Builder::ShouldHaveFeederMode(town_id, inter_city_station)
{
	// There need to be at least 2 airports AND the inter_city_station need to be valid AND have at least one bus stop AND accept passengers
	if(Station.GetAllInterCityStations().Count() < 2 || 
			!AIStation.IsValidStation(inter_city_station) || 
			!AIStation.HasStationType(inter_city_station, AIStation.STATION_BUS_STOP) ||
			!Builder.DoesAirportAcceptPAX(inter_city_station))
		return false;

	// but the inter city station in question should also be serviced by at least one airplane
	local station_list = AIList();
	station_list.AddItem(inter_city_station, 0);
	local vehicle_list = Station.GetAllAirVehiclesOfStationList(station_list);

	return !vehicle_list.IsEmpty();
}

function Builder::UpdateIntraCityOrdersForTown(town_id)
{
	local inter_city_station = Station.FindInterCityStation(town_id);

	local stations_list = Station.GetAllFeederStationsOfTown(town_id);
	local grid_list = MakeTownGrid(town_id, GetBuildGridSize(town_id));
	local station_vehicles_list = Station.GetAllRoadVehiclesOfStationList(stations_list);

	if(ShouldHaveFeederMode(town_id, inter_city_station))
	{
		// inter-city-mode -> feed the airport
		AILog.Warning("UpdateIntraCityOrdersForTown: Town " + AITown.GetName(town_id) + " is in INTER-CITY mode");

		if(!station_vehicles_list.IsEmpty())
		{
			AILog.Info("updating orders - feeder mode");
			UpdateIntraCityOrders_FeederMode(station_vehicles_list.Begin(), town_id, stations_list, inter_city_station);
		}
	}
	else
	{
		if(!station_vehicles_list.IsEmpty())
		{
			AILog.Info("updating orders - local mode");
			UpdateIntraCityOrders_LocalMode(station_vehicles_list.Begin(), town_id, stations_list);
		}
	}
}

// Inter city
function Builder::UpdateInterCityOrders(vehicle_id, inter_city_stations)
{
	local order_list = OrderList();
	foreach(station_id, _ in inter_city_stations)
	{
		order_list.AddStop(station_id, AIOrder.AIOF_NONE);
	}

	order_list.ApplyToVehicle(vehicle_id);
}

function VehicleRunningCostPerCapacityValuator(vehicle_id, needed_capacity)
{
	local capacity = Helper.Min(needed_capacity, AIVehicle.GetCapacity(vehicle_id, Helper.GetPAXCargo())); // don't let capacity we don't need pay for the running cost
	local running_cost = AIVehicle.GetRunningCost(vehicle_id);

	if(capacity == 0)
		return running_cost * 1000;

	return running_cost / capacity;
}

function BuyVehicles(existing_capacity, target_capacity, engine, depot, existing_vehicles_list, order_data)
{
	local needed_capacity = target_capacity - existing_capacity;
	local engine_capacity = AIEngine.GetCapacity(engine);
	local vehicle_type = AIEngine.GetVehicleType(engine);

	//Helper.SetSign(depot, "buy depot");


	if(AICompany.GetBankBalance(AICompany.COMPANY_SELF) < AIEngine.GetPrice(engine) + 10000)
		return;

	// Buy one engine
	local vehicle = AIVehicle.BuildVehicle(depot, engine);
	if(!AIVehicle.IsValidVehicle(vehicle))
	{
		AILog.Error("failed to buy vehicle");
	    AILog.Error("error: " + AIError.GetLastErrorString());
		return false;
	}
	AIVehicle.RefitVehicle(vehicle, Helper.GetPAXCargo()); // since GetEngine can return engines that by default don't carry PAX (but can be refitted to PAX), we need to refit bough vehicles to PAX.
	//AIVehicle.SetName(vehicle, "Re: " + AIError.GetLastErrorString());
	

	AIVehicle.StartStopVehicle(vehicle);

	// Make orders
	if(!existing_vehicles_list.IsEmpty())
	{
		AILog.Info("Share orders. New vehicle " + AIVehicle.GetName(vehicle) + " shares from " + AIVehicle.GetName(existing_vehicles_list.Begin()));
		AIOrder.ShareOrders(vehicle, existing_vehicles_list.Begin());
	}
	else
	{
		// different vehicle types (buses vs airplanes) has different oder functions
		if(vehicle_type == AIVehicle.VT_ROAD)
		{
			AILog.Info("First vehicle (road) so make new orders. feeder_mode = " + order_data.feeder_mode? "true":"false");
			if(order_data.feeder_mode)
				UpdateIntraCityOrders_FeederMode(vehicle, order_data.town_id, order_data.stations_list, order_data.inter_city_station);
			else
				UpdateIntraCityOrders_LocalMode(vehicle, order_data.town_id, order_data.stations_list);
		}
		else if(vehicle_type == AIVehicle.VT_AIR)
		{
			AILog.Info("First vehicle (air) so make new orders.");

			UpdateInterCityOrders(vehicle, order_data.station_list);
		}
	}

	// Clone the first built vehicle until needed capacity is reached or cloning fails
	needed_capacity -= engine_capacity;
	AILog.Info("After the first bought vehicle the remaining needed capacity is " + needed_capacity);
	AILog.Info("Bank balance: " + AICompany.GetBankBalance(AICompany.COMPANY_SELF));
	AILog.Info("Bank balance minimum limit: " + (AIEngine.GetPrice(engine) * 2));
	while(needed_capacity > 0 && 
			AICompany.GetBankBalance(AICompany.COMPANY_SELF) > AIEngine.GetPrice(engine) * 2) // a little buffer of money is good to have
	{
		AILog.Info("Clone vehicle");
		local share_orders = true;
		local clone_vehicle = AIVehicle.CloneVehicle(depot, vehicle, share_orders);

		if(!AIVehicle.IsValidVehicle(clone_vehicle))
		{
			AILog.Error("failed to clone vehicle");
			return false;
		}

		AIVehicle.StartStopVehicle(clone_vehicle);
		needed_capacity -= engine_capacity;

		AILog.Info("After the last cloned vehicle the remaining needed capacity is " + needed_capacity);
		AILog.Info("Bank balance: " + AICompany.GetBankBalance(AICompany.COMPANY_SELF));
		AILog.Info("Bank balance minimum limit: " + (AIEngine.GetPrice(engine) * 2));

		existing_vehicles_list.AddItem(clone_vehicle, 0); // add vehicle to list, in case caller have use for it
	}

	AILog.Info("Done buying vehicles. Remaining needed capacity is: " + needed_capacity);

	return true;
}

function SellVehicles(existing_capacity, target_capacity, existing_vehicles_list)
{
	local needed_capacity = target_capacity - existing_capacity;

	local keep_vehicle_list = AIList();
	local sell_vehicle_list = AIList();
	sell_vehicle_list.AddList(existing_vehicles_list);
	local kept_capacity = 0;
	while(kept_capacity < target_capacity)
	{

		// sort vehicles after their running cost / pax. But as when buying vehicles, if we only need 20, and the bus has 30 seats, only
		// let the needed 20 seats share the cost.
		sell_vehicle_list.Valuate(VehicleRunningCostPerCapacityValuator, target_capacity - kept_capacity);

		sell_vehicle_list.Sort(AIAbstractList.SORT_BY_VALUE, true); // lowest value first
		local keep_vehicle = sell_vehicle_list.Begin();
		local keep_vehicle_capacity = AIVehicle.GetCapacity(keep_vehicle, Helper.GetPAXCargo());

		// Don't keep the selected vehicle if it will make the resulting capacity will get higher than the target capacity
		if(kept_capacity + keep_vehicle_capacity > target_capacity && 
			(target_capacity == 0 || (existing_vehicles_list.Count() - sell_vehicle_list.Count() >= 1)) // make sure there is at least one vehicle remaining if target_capacity is zero
			)
		{
			break;
		}

		sell_vehicle_list.RemoveTop(1);
		
		keep_vehicle_list.AddItem(keep_vehicle, 0);
		kept_capacity += AIVehicle.GetCapacity(keep_vehicle, Helper.GetPAXCargo());
		AILog.Info("Keep vehicle " + AIVehicle.GetName(keep_vehicle) + ". kept capacity: " + kept_capacity + "; target cap: " + target_capacity);
	}

	if(sell_vehicle_list.IsEmpty())
		AILog.Warning("Warning: sell vehicles list is Empty!");

	// now sell the vehicles in sell_vehicle_list
	for(local vehicle_id = sell_vehicle_list.Begin(); sell_vehicle_list.HasNext(); vehicle_id = sell_vehicle_list.Next())
	{
		Vehicle.SendVehicleToDepotForSelling(vehicle_id);
	}
}

function Builder::BuyBusesForTown(town_id)
{
	local town_manager = towns.GetTownManager(town_id);

	local stations_list = Station.GetAllFeederStationsOfTown(town_id);
	local grid_list = MakeTownGrid(town_id, GetBuildGridSize(town_id));

	AILog.Info("Buying new buses for town " + AITown.GetName(town_id) + " that has " + stations_list.Count() + " stations");

	if(stations_list.Count() < 1) 
	{
		return false;
	}

	// Check if feeder-mode or just transporting around passengers in the town
	// This affects some parts below
	local inter_city_station = Station.FindInterCityStation(town_id);
	local feeder_mode = ShouldHaveFeederMode(town_id, inter_city_station);

	local station_vehicles_list = Station.GetAllRoadVehiclesOfStationList(stations_list);
	local town_pop = AITown.GetPopulation(town_id);

	local num_vehicles_per_station = station_vehicles_list.Count() / stations_list.Count();

	// Decide how many buses are needed
	station_vehicles_list.Valuate(AIVehicle.GetCapacity, Helper.GetPAXCargo());
	local existing_capacity = Helper.ListValueSum(station_vehicles_list);
	local target_capacity = existing_capacity;

	if((!feeder_mode && stations_list.Count() < 2) || (feeder_mode && stations_list.Count() < 1))
	{
		target_capacity = 0;
	}
	else if(station_vehicles_list.IsEmpty())
	{
		// if there are no buses, base it on town population
		target_capacity = town_pop / 20;
		AILog.Info("There are no buses in the town => buy buses for population / 20");
	}
	else
	{
		// Allow some time from last bus was bough until performance is checked again
		local min_age = 90;

		station_vehicles_list.Valuate(AIVehicle.GetAge);
		local youngest_vehicle_age = Helper.GetListMinValue(station_vehicles_list);

		if(youngest_vehicle_age > min_age)
		{
			// check how well the service work now and adjust the capacity if needed
			stations_list.Valuate(AIStation.GetCargoWaiting, Helper.GetPAXCargo());
			local max_waiting = Helper.GetListMaxValue(stations_list);
			local tot_waiting = Helper.ListValueSum(stations_list);
			local mean_waiting = tot_waiting / stations_list.Count();

			if(max_waiting > 150 || mean_waiting > 60)
			{
				AILog.Info("Too many people are wating for a bus => buy more buses");
				AILog.Info("mean_waiting = " + mean_waiting);
				local new_cap = Helper.Min(Helper.Min(50, Helper.Max(0, mean_waiting - 125) / 2), 400);	
				AILog.Info("new_cap = " + new_cap);
				target_capacity = existing_capacity + new_cap;
				AILog.Info("target_capacity = " + target_capacity);
			}
			else if(mean_waiting < 20 /*&& station_vehicles_list.Count() > 1 FIXME: uncomment */)
			{
				AILog.Info("Too few people are waiting for a bus => sell buses");
				local new_cap = Helper.Max(Helper.Min(50, Helper.Max(0, mean_waiting - 125) / 2), 400);	
				target_capacity = Helper.Min(existing_capacity - 50, (existing_capacity * 3 / 4) );
				AILog.Info("target_capacity = " + target_capacity);
			}
		}
		else
		{
			AILog.Info("Youngest bus to young (" + youngest_vehicle_age + " (" + min_age + ")) for performance check in town " + AITown.GetName(town_id));
		}
	}

	// Don't buy toooo many buses if the stations can't handle more.
	local max_bus_station_usage = town_manager.GetMaxBusStationUsage();
	Helper.SetSign(AITown.GetLocation(town_id), "max u: " + max_bus_station_usage);
	if(target_capacity != 0 && max_bus_station_usage > 250)
	{
		// remove the vehicle with least amount of capacity
		station_vehicles_list.Valuate(AIVehicle.GetCapacity, Helper.GetPAXCargo());
		station_vehicles_list.Sort(AIAbstractList.SORT_BY_VALUE, true); // lowest first
		target_capacity = existing_capacity - station_vehicles_list.GetValue(station_vehicles_list.Begin()) - 1;
	}
	else
	if(target_capacity != 0 && max_bus_station_usage > 180)
	{
		target_capacity = existing_capacity;
	}

	// In feeder mode,
	if(target_capacity != 0 && feeder_mode)
	{
		local ic_station_waiting = AIStation.GetCargoWaiting(inter_city_station, Helper.GetPAXCargo());
		local use_break_points = Helper.HasSign(AITown.GetName(town_id) + "_debug");

		AILog.Info("Waiting PAX at IC station = " + ic_station_waiting);
		AILog.Info("target_capacity = " + target_capacity);
		if(use_break_points)
			Helper.BreakPoint(AIStation.GetLocation(inter_city_station) + 1);


		// Get the capacity of the biggest aircraft servicing the airport
		local air_vehicles = AIVehicleList_Station(inter_city_station);
		air_vehicles.Valuate(AIVehicle.GetCapacity, Helper.GetPAXCargo());
		air_vehicles.KeepTop(1);
		local max_air_capacity = air_vehicles.GetValue(air_vehicles.Begin());

		// base the range of allowed number of waiting passengers on the size of the biggest aircraft
		local max_allowed_waiting__stop_buy = max_air_capacity * 2;
		local max_allowed_waiting__reduce = max_air_capacity * 4;
		local min_wanted_waiting = max_air_capacity / 2;

		// don't buy more buses when there is more than 200 waiting -> instead spend the money on planes
		if(ic_station_waiting > max_allowed_waiting__stop_buy && existing_capacity > 0)
			target_capacity = Helper.Min(target_capacity, existing_capacity);

		AILog.Info("target_capacity = " + target_capacity);
		if(use_break_points)
			Helper.BreakPoint(AIStation.GetLocation(inter_city_station) + 1);
		
		// and reduce the number of buses if there is more than 700 waiting
		if(ic_station_waiting > max_allowed_waiting__reduce)
			target_capacity = Helper.Min(target_capacity, Helper.Max(existing_capacity - 50, existing_capacity / 2));

		AILog.Info("target_capacity = " + target_capacity);
		if(use_break_points)
			Helper.BreakPoint(AIStation.GetLocation(inter_city_station) + 1);

		// Don't sell the last bus
		if(target_capacity <= 0)
			target_capacity = 1;

		AILog.Info("target_capacity = " + target_capacity);
	}

	local needed_capacity = target_capacity - existing_capacity;
	if(needed_capacity > 0) 
	{
		AILog.Info("Need more buses because town has capacity: " + existing_capacity + " but want to have capacity: " + target_capacity);

		// bus engine
		local bus_engine = GetEngine(needed_capacity, AIVehicle.VT_ROAD);
		if(!AIEngine.IsValidEngine(bus_engine))
		{
			AILog.Error("can't find valid bus engine to build for town " + AITown.GetName(town_id));
			return false;
		}

		AILog.Info("Selected bus engine: " + AIEngine.GetName(bus_engine));

		// Find a depot
		local depot_placement_tiles = GetDepotPlacementArea(town_id);
		local depot = FindDepot(town_id);

		if(!AIMap.IsValidTile(depot))
		{
			AILog.Error("can't find depot in town " + AITown.GetName(town_id));
			return false;
		}

		// order data
		local orders_data = { town_id = town_id, feeder_mode = feeder_mode, stations_list = stations_list, inter_city_station = inter_city_station };

		return BuyVehicles(existing_capacity, target_capacity, bus_engine, depot, station_vehicles_list, orders_data);
	}
	else if(needed_capacity < 0)
	{
		AILog.Info("Want to reduce capacity from: " + existing_capacity + " to target capacity: " + target_capacity);

		SellVehicles(existing_capacity, target_capacity, station_vehicles_list);
	}
	else
	{
		AILog.Info("has enough capacity: " + existing_capacity + " for target capacity: " + target_capacity);
	}

	return true;
}


function Builder::HasSmallAirport(station_list)
{
	for(local station_id = station_list.Begin(); station_list.HasNext(); station_id = station_list.Next())
	{
		if(Station.IsSmallAirport(station_id))
			return true;
	}

	return false;
}


function Builder::ManageInterCityVehiclesOfInterCityConnection(station_list)
{
	AILog.Info("Builder::ManageInterCityVehiclesOfInterCityConnection");

	if(station_list.IsEmpty())
	{
		AILog.Info("Builder::ManageInterCityVehiclesOfInterCityConnection: Connection is empty");
		return false;
	}

	// Get max usage (in percent => 100 * queue length)
	local max_usage = 0;
	for(local i = station_list.Begin(); station_list.HasNext(); i = station_list.Next())
	{
		AILog.Info("Station: " + i + ":" + AIStation.GetName(i));

		local station = towns.GetStationStatistics(i);
		if(station != null && station.usage.aircraft.percent_usage > max_usage)
			max_usage = station.usage.aircraft.percent_usage;
	}

	// Get vehicle list
	local all_vehicles_list = Station.GetAllAirVehiclesOfStationList(station_list);
	AILog.Info("Builder::ManageInterCityVehiclesOfInterCityConnection: has " + all_vehicles_list.Count() + " vehicles and " + station_list.Count() + " stations");

	// Check if there is a station that don't accept PAX
	for(local ap_station = station_list.Begin(); station_list.HasNext(); ap_station = station_list.Next())
	{
		if(!Builder.DoesAirportAcceptPAX(ap_station))
		{
			// One or more of the stations does not accept PAX
			// -> Sell all vehicles

			Helper.SetSign(AIStation.GetLocation(ap_station) +1, "no accept");
			AILog.Info("Airport " + AIStation.GetName(ap_station) + " does not accept PAX. => Sell all planes for connection with this airport");
			
			Utils_Valuator.Valuate(all_vehicles_list, Vehicle.SendVehicleToDepotForSelling);
			all_vehicles_list.Clear();

			local town_id = AITile.GetClosestTown(AIStation.GetLocation(ap_station));
			UpdateIntraCityOrdersForTown(town_id);
			AILog.Info("buses in town have been updated to run in local mode");

			return true;
		}
		else
		{
			Helper.SetSign(AIStation.GetLocation(ap_station) +1, "accept");
		}
	}
/*	Utils_Valuator.Valuate(station_list, DoesAirportAcceptPAX);
	AILog.Info("Num stations that accept PAX: " + Helper.ListValueSum(station_list) + " | Num stations: " + station_list.Count());
	if(Helper.ListValueSum(station_list) != station_list.Count())
	{
		// One or more of the stations does not accept PAX
		// -> Sell all vehicles
		
		Utils_Valuator.Valuate(all_vehicles_list, Vehicle.SendVehicleToDepotForSelling);
		all_vehicles_list.Clear();

		return true;
	}*/


	// Get age of youngest plane
	all_vehicles_list.Valuate(AIVehicle.GetAge);
	all_vehicles_list.Sort(AIAbstractList.SORT_BY_VALUE, true); // youngest first
	local youngest_airplane = all_vehicles_list.GetValue(all_vehicles_list.Begin());

	if(all_vehicles_list.IsEmpty() || youngest_airplane > 30)
	{
		// Get existing capacity
		all_vehicles_list.Valuate(AIVehicle.GetCapacity, Helper.GetPAXCargo());
		local existing_capacity = Helper.ListValueSum(all_vehicles_list);
		local target_capacity = existing_capacity;

		// Get number of waiting passengers
		station_list.Valuate(AIStation.GetCargoWaiting, Helper.GetPAXCargo());
		local sum_waiting = Helper.ListValueSum(station_list);
		local mean_waiting = sum_waiting / station_list.Count();
		station_list.Sort(AIAbstractList.SORT_BY_VALUE, false); // max waiting first
		local max_waiting_station_id = station_list.Begin();
		local max_waiting = station_list.GetValue(station_list.Begin());

		local small_airport = Builder.HasSmallAirport(station_list);

		// Decide target capacity
		if(all_vehicles_list.IsEmpty())
		{
			// find an engine, where we pay for up to 100 seats, and set target capacity to the capacity of that engine
			local engine = GetEngine(100, AIVehicle.VT_AIR, small_airport);
			target_capacity = AIEngine.GetCapacity(engine);
		}
		else if(mean_waiting > 100 || max_waiting > 200)
		{
			target_capacity = existing_capacity + 50 + Helper.Max(0, (mean_waiting - 100)*2 / 3);
		}
		else if(mean_waiting < 10)
		{
			target_capacity = Helper.Max(existing_capacity - 70, existing_capacity / 2);
		}

		// don't allow a higher usage than 400
		if(max_usage > 400)
		{
			target_capacity = Helper.Min(target_capacity, existing_capacity);
		}
		if(max_usage > 800 && target_capacity >= existing_capacity)
		{
			// take the capacity of some random aircraft
			local some_aircraft_capacity = AIVehicle.GetCapacity(all_vehicles_list.Begin(), Helper.GetPAXCargo())

			// reduce the capacity with that amount
			target_capacity = target_capacity - some_aircraft_capacity;
		}

		if(target_capacity < 10)
			target_capacity = 10;

		local needed_capacity = target_capacity - existing_capacity;
		if(needed_capacity > 0) 
		{
			AILog.Info("Need more airplanes, because connection has capacity: " + existing_capacity + " but want to have capacity: " + target_capacity);

			// airplane engine
			local airplane_engine = GetEngine(needed_capacity, AIVehicle.VT_AIR, small_airport);

			// find a hangar
			local hangar_tile = Station.GetHangarTile(station_list.Begin());
			if(!AIAirport.IsHangarTile(hangar_tile))
			{
				AILog.Warning("Failed to find hangar tile of station " + AIStation.GetName(station_list.Begin()));
				return false;
			}

			// order data
			local orders_data = { station_list = station_list };

			return BuyVehicles(existing_capacity, target_capacity, airplane_engine, hangar_tile, all_vehicles_list, orders_data);
		}
		else if(needed_capacity < 0)
		{
			AILog.Info("Want to reduce capacity from: " + existing_capacity + " to target capacity: " + target_capacity);
			AILog.Info("have " + all_vehicles_list.Count() + " vehicles");

			SellVehicles(existing_capacity, target_capacity, all_vehicles_list);
		}
		else
		{
			AILog.Info("has enough capacity: " + existing_capacity + " for target capacity: " + target_capacity);
		}

	} // <- end of buy/sell airplanes

	all_vehicles_list = Station.GetAllAirVehiclesOfStationList(station_list);
	if(!all_vehicles_list.IsEmpty())
	{
		UpdateInterCityOrders(all_vehicles_list.Begin(), station_list);
	}
}

