
class OptNodeBaseClass
{
	function GetArchValue(other_node);

	// Return an id that can be compared using != and ==
	function GetNodeId();
};

class OptPair
{
	node = null;
	cached_value = null;

//public:
	constructor(node1, node2)
	{
		this.node = [node1, node2];
		cached_value = null;
	}

	function GetPairValue()
	{
		// Return value 0 if any of the nodes are null
		if(node[0] == null || node[1] == null)
			return 0;

		// Calculate the value if there is no cached value
		if(this.cached_value == null)
			this.cached_value = node[0].GetArchValue(node[1]);

		return this.cached_value;
	}

	function SetNodeObj(node_index, new_node_obj)
	{
		if(node[node_index] != new_node_obj)
		{
			node[node_index] = new_node_obj;

			// Clear the cached value of the pair if one of the nodes is changed
			this.cached_value = null;
		}
	}
	function GetNodeObj(node_index)
	{
		return node[node_index];
	}
};

class TabuList
{
	list = null;
	tabu_len = 0;

	constructor(pair_length)
	{
		list = [];
		tabu_len = pair_length * 2;
	}

	function Add(node0, node1)
	{
		if(node0 != null)
			list.append(node0.GetNodeId());

		if(node1 != null)
			list.append(node1.GetNodeId());

		while(list.len() > tabu_len)
		{
			list.remove(0);
		}
	}

	function IsTabu(node0, node1)
	{
		foreach(node_id in list)
		{
			if( (node0 && node_id == node0.GetNodeId()) ||
					(node1 && node_id == node1.GetNodeId()) )
				return true;
		}

		return false;
	}
}

class PairOptimizer
{
//private:
	pairs = null;
	un_paired_nodes = null;

//public:
	constructor()
	{
		pairs = [];
		un_paired_nodes = [];
	}

	function AddNode(node);
	function AddPair(node1, node2);
	function MakeTabuSearch();

//private:
	function MakeBaseSolution(); // adds all un-paired nodes to new pairs

	function SwapNodes(pair_a, pair_b, node_number_in_pair_a, node_number_in_pair_b);
	function EvaluateSwap(pair_a, pair_b, node_number_in_pair_a, node_number_in_pair_b);

	function MakeSolutionCopy();
	static function GetSolutionValue(solution);
};


function PairOptimizer::AddNode(node)
{
	un_paired_nodes.append(node);
}

function PairOptimizer::AddPair(node0, node1)
{
	pairs.append(OptPair(node0, node1));
}

function PairOptimizer::MakeBaseSolution()
{
	// Make sure all unpaired nodes are added to the list of pairs.

	if(un_paired_nodes.len() > 0)
	{
		local i = 0;
		local un_paired_node = un_paired_nodes[i];

		// Go through all pairs and see if there is any pairs that have a null node
		foreach(pair in pairs)
		{
			local exit_foreach = false;

			// if there are null nodes, then add the un-paired nodes there
			for(local j = 0; j < 2; j++)
			{
				if(pair.node[j] == null)
				{
					pair.node[j] = un_paired_node;
					if(++i >= un_paired_nodes.len())
					{
						exit_foreach = true;
						break;
					}
					un_paired_node = un_paired_nodes[i];
				}
			}
					
			if(exit_foreach)
				break;
		}

		// if there are more unpaired nodes left after null spots have been filled, then add new pairs
		while(i < un_paired_nodes.len())
		{
			local node0 = un_paired_nodes[i++];
			local node1 = i < un_paired_nodes.len()? un_paired_nodes[i++]: null;
			local pair = OptPair(node0, node1);
			pairs.append(pair);
		}

		// clear the list of unpaired nodes
		un_paired_nodes = [];
	}
}

function PairOptimizer::MakeTabuSearch()
{
	MakeBaseSolution();

	if(pairs.len() <= 1)
		return pairs;
	
	local max_iterations = 100;
	local tabu_length = Helper.Min(7, pairs.len() / 2);

	local tabu_list = TabuList(tabu_length);

	local current_solution_value = GetSolutionValue(pairs);

	local best_swap_pair_a = null;
	local best_swap_pair_b = null;
	local best_swap_pair_a_node_number = null;
	local best_swap_pair_b_node_number = null;
	local best_swap_value = 0;

	local best_solution = [];
	local best_solution_value = null;

	for(local k = 0; k < max_iterations; k++)
	{
		// Evaluate all solutions in the neighborhood
		best_swap_value = null;
		for(local i = 0; i < pairs.len() - 1; i++)
		{
			for(local j = i + 1; j < pairs.len(); j++)
			{
				local pair_a = pairs[i];
				local pair_b = pairs[j];

				for(local m = 0; m < 2; m++)
				{
					for(local n = 0; n < 2; n++)
					{
						local pair_a_node_number = m;
						local pair_b_node_number = m;

						if(!tabu_list.IsTabu(pair_a.GetNodeObj(pair_a_node_number), pair_b.GetNodeObj(pair_b_node_number)))
						{
							local value = EvaluateSwap(pair_a, pair_b, pair_a_node_number, pair_b_node_number);
							if(best_swap_value == null || value > best_swap_value)
							{
								best_swap_pair_a = pair_a;
								best_swap_pair_b = pair_b;
								best_swap_pair_a_node_number = pair_a_node_number;
								best_swap_pair_b_node_number = pair_b_node_number;
								best_swap_value = value;
							}
						}
					}
				}
			}
		}

		// Make sure at least one non-tabu neighbor was found
		if(best_swap_value == null)
		{
			AILog.Error("PairOptimizer: No neighbors found");
			return best_solution;
		}
		
		// Swap nodes of best allowed move
		SwapNodes(best_swap_pair_a, best_swap_pair_b, best_swap_pair_a_node_number, best_swap_pair_b_node_number);

		// Update tabu list
		tabu_list.Add(best_swap_pair_a.GetNodeObj(best_swap_pair_a_node_number), best_swap_pair_b.GetNodeObj(best_swap_pair_b_node_number));
		
		// Update current solution value
		current_solution_value += best_swap_value;

		// Check if global optimum was found
		if(current_solution_value > best_solution_value)
		{
			best_solution = MakeSolutionCopy();
		}
	}
		
	return best_solution;
}

function PairOptimizer::SwapNodes(pair_a, pair_b, node_number_in_pair_a, node_number_in_pair_b)
{
	local old_a = pair_a.GetNodeObj(node_number_in_pair_a);

	pair_a.SetNodeObj(node_number_in_pair_a, pair_b.GetNodeObj(node_number_in_pair_b));
	pair_b.SetNodeObj(node_number_in_pair_b, old_a);
}

function PairOptimizer::EvaluateSwap(pair_a, pair_b, node_number_in_pair_a, node_number_in_pair_b)
{
	// Make a set of pairs to try the swap with
	// Using new temporary pairs the cached value 
	// of pair_a and pair_b will not be erased.
	local new1 = OptPair(pair_a.GetNodeObj(0), pair_a.GetNodeObj(1));
	local new2 = OptPair(pair_b.GetNodeObj(0), pair_b.GetNodeObj(1));
	SwapNodes(new1, new2, node_number_in_pair_a, node_number_in_pair_b);
	
	// Calculate the old and new value
	local old_value = pair_a.GetPairValue() + pair_b.GetPairValue();
	local new_value = new1.GetPairValue() + new2.GetPairValue();

	return new_value - old_value;
}

function PairOptimizer::MakeSolutionCopy()
{
	local solution_copy = [];

	foreach(pair in pairs)
	{
		local pair_copy = OptPair(pair.GetNodeObj(0), pair.GetNodeObj(1));
		solution_copy.append(pair_copy);
	}

	return solution_copy;
}

function PairOptimizer::GetSolutionValue(solution)
{
	local tot_value = 0;
	foreach(pair in solution)
	{
		tot_value += pair.GetPairValue();
	}

	return tot_value;
}
