/*
 * This file is part of PAXLink, which is an AI for OpenTTD
 * Copyright (C) 2008, 2009  Leif Linse
 *
 * PAXLink is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License
 *
 * PAXLink is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with PAXLink; If not, see <http://www.gnu.org/licenses/> or
 * write to the Free Software Foundation, Inc., 51 Franklin Street, 
 * Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

import("pathfinder.road", "RPF", 3);

class RoadBuilder {

	static function ConnectTiles(tile1, tile2, call_count);

}

function IsStraight(tile1, tile2)
{
	return (AIMap.GetTileX(tile1) == AIMap.GetTileX(tile2) ||
			AIMap.GetTileY(tile1) == AIMap.GetTileY(tile2));
}

// max_days is per try
function RoadBuilder::ConnectTiles(tile1, tile2, call_count, max_days = -1) {
	local pathfinder = RPF();
	local start_day = AIDate.GetCurrentDate();

	pathfinder.InitializePath([tile1], [tile2]);

	local path = false;
	while (path == false) {
		path = pathfinder.FindPath(100);
		AIController.Sleep(1);

		if(max_days != -1 && AIDate.GetCurrentDate() - start_day > max_days)
			return false;
	}

	if(path == null) {
		AILog.Info("Failed to find path");
		return false;
	}
	AILog.Info("Path found, now start building!");

	//AISign.BuildSign(path.GetTile(), "Start building");

	while (path != null) {
		local par = path.GetParent();
		if (par != null) {
			local last_node = path.GetTile();
			if (AIMap.DistanceManhattan(path.GetTile(), par.GetTile()) == 1 ) {
				if (AIRoad.AreRoadTilesConnected(path.GetTile(), par.GetTile())) {
					// there is already a road here, don't do anything
				} else {

					/* Look for longest straight road and build it as one build command */
					local straight_begin = path;
					local straight_end = par;
					local prev = straight_end.GetParent();
					while(prev != null && 
							IsStraight(straight_begin.GetTile(), prev.GetTile()) &&
							AIMap.DistanceManhattan(straight_end.GetTile(), prev.GetTile()) == 1)
					{
						straight_end = prev;
						prev = straight_end.GetParent();
					}

					/* update the looping vars. (path is set to par in the end of the main loop) */
					par = straight_end;

					//AISign.BuildSign(path.GetTile(), "path");
					//AISign.BuildSign(par.GetTile(), "par");

					//AISign.BuildSign(straight_begin.GetTile(), "begin");
					//AISign.BuildSign(straight_end.GetTile(), "end");

					// Try to build a stretch for maximum of 10 days before doing a re-pathfind.
					// Building will fail when there is a vehicle in the way for example
					local day_start = AIDate.GetCurrentDate();
					local max_try_days = 10;
					while (!AIRoad.BuildRoad(straight_begin.GetTile(), straight_end.GetTile())) {
						AILog.Error(AIError.GetLastErrorString());
						AIController.Sleep(5);

						if(day_start + max_try_days < AIDate.GetCurrentDate()) {
							// Try PF again
							if (call_count > 1 || !RoadBuilder.ConnectTiles(par.GetTile(), path.GetTile(), call_count+1, max_days))
								return false;
						}
					}
				}
			} else {
				/* Build a bridge or tunnel. */
				if (!AIBridge.IsBridgeTile(path.GetTile()) && !AITunnel.IsTunnelTile(path.GetTile())) {
					/* If it was a road tile, demolish it first. Do this to work around expended roadbits. */
					if (AIRoad.IsRoadTile(path.GetTile())) AITile.DemolishTile(path.GetTile());
					if (AITunnel.GetOtherTunnelEnd(path.GetTile()) == par.GetTile()) {
						if (!AITunnel.BuildTunnel(AIVehicle.VT_ROAD, path.GetTile())) {
							/* An error occured while building a tunnel. TODO: handle it. */
							AILog.Error("Build tunnel error: " + AIError.GetLastErrorString());
						}
					} else {
						local bridge_list = AIBridgeList_Length(AIMap.DistanceManhattan(path.GetTile(), par.GetTile()) +1);
						bridge_list.Valuate(AIBridge.GetMaxSpeed);
						if (!AIBridge.BuildBridge(AIVehicle.VT_ROAD, bridge_list.Begin(), path.GetTile(), par.GetTile())) {
							/* An error occured while building a bridge. TODO: handle it. */
							AILog.Error("Build bridge error: " + AIError.GetLastErrorString());
						}
					}
				}
			}
		}
		path = par;
	}

	//AISign.BuildSign(tile1, "Done");

	return true;
}
