
class Station {

	static function StationStatsSquirrelList_To_StationIdAIList(squirrel_list_of_station_stats);

	// PAXLink strategy related functions
	static function VehicleIsWithinTileList(vehicle_id, tile_list);
	static function IsFeederStation(station_id);
	static function IsNonEmptyStation(station_id); // checks that the stations have at least one transport mode


	// Other functions
	static function CostToDemolishStation(station_id);
	static function DemolishStation(station_id);

	static function FindInterCityStation_EachInList(list);
	static function FindInterCityStation(town_id);
	static function GetAllInterCityStations();
	static function GetAllFeederStationsOfTown(town_id);
	static function GetAllFeederStations(town_id);

	// retrieve information about vehicles related to stations
	static function GetAllVehiclesOfStationList(station_list);
	static function GetAllRoadVehiclesOfStationList(station_list);
	static function GetAllAirVehiclesOfStationList(station_list);

	static function AreStationsConnectedByVehicles(station_list, vehicle_type);

	// Bus related functions
	static function GetRoadConnectTileOfStation(station_id); // gets the tile which road to somewhere else should be connected to

	// Airport related functions
	static function GetAirportTile(station_id);
	static function GetHangarTile(station_id);
	static function GetAircraftsInHangar(station_id);
	static function IsSmallAirport(station_id);
};

function Station::StationStatsSquirrelList_To_StationIdAIList(squirrel_list_of_station_stats)
{
	local ai_list_station_ids = AIList();
	foreach(station in squirrel_list_of_station_stats)
	{
		ai_list_station_ids.AddItem(station.station_id, 0);
	}

	return ai_list_station_ids;
}

function Station::IsFeederStation(station_id)
{
	return AIStation.IsValidStation(station_id) &&
			AIStation.HasStationType(station_id, AIStation.STATION_BUS_STOP) &&
			!AIStation.HasStationType(station_id, AIStation.STATION_TRAIN) &&
			!AIStation.HasStationType(station_id, AIStation.STATION_TRUCK_STOP) &&
			!AIStation.HasStationType(station_id, AIStation.STATION_AIRPORT) &&
			!AIStation.HasStationType(station_id, AIStation.STATION_DOCK);
}

function Station::IsICStation(station_id)
{
	return AIStation.IsValidStation(station_id) &&
			AIStation.HasStationType(station_id, AIStation.STATION_AIRPORT);
}

function Station::IsNonEmptyStation(station_id)
{
	return AIStation.IsValidStation(station_id) && 
			(
				AIStation.HasStationType(station_id, AIStation.STATION_BUS_STOP) ||
				AIStation.HasStationType(station_id, AIStation.STATION_TRAIN) ||
				AIStation.HasStationType(station_id, AIStation.STATION_TRUCK_STOP) ||
				AIStation.HasStationType(station_id, AIStation.STATION_AIRPORT) ||
				AIStation.HasStationType(station_id, AIStation.STATION_DOCK) 
			);
}

function Station::GetAllInterCityStations()
{
	local station_list = AIStationList(AIStation.STATION_AIRPORT);
	return station_list;
}

function Station::GetAllFeederStationsOfTown(town_id)
{
	local stations = AIStationList(AIStation.STATION_BUS_STOP);
	stations.Valuate(AIStation.GetNearestTown);
	stations.KeepValue(town_id);

	// Only keep feeder stations
	stations.Valuate(Station.IsFeederStation)
	stations.KeepValue(1);

	return stations;
}

function Station::GetAllFeederStations(town_id)
{
	local stations = AIStationList(AIStation.STATION_BUS_STOP);

	// Only keep feeder stations
	stations.Valuate(Station.IsFeederStation)
	stations.KeepValue(1);

	return stations;
}

function Station::CostToDemolishStation(station_id)
{
	local station_tiles = AITileList_StationType(station_id, AIStation.STATION_ANY);
	local cost_to_clear = Builder.CostToClearTiles(station_tiles);
	return cost_to_clear;
}

function Station::DemolishStation(station_id)
{
	local station_tiles = AITileList_StationType(station_id, AIStation.STATION_ANY);
	local succeeded = true;

	for(local tile = station_tiles.Begin(); station_tiles.HasNext(); tile = station_tiles.Next())
	{
		if(AIRoad.IsRoadStationTile(tile))
		{
			local front = AIRoad.GetRoadStationFrontTile(tile);

			// Delay demolishing if there is a vehicle on a road stop
			local tile_rect = AITileList();
			tile_rect.AddRectangle(tile, front);
			if(Builder.HasTileRectVehiclesOnIt(tile_rect, AIVehicle.VT_ROAD))
			{
				AILog.Warning("bus stop has vehicles on it (or own vehicle in front) => delay destruction of station");

				main_instance.Sleep(10);
			}

			local day_start = AIDate.GetCurrentDate();
			while(!AIRoad.RemoveRoadStation(tile) && AIError.GetLastError() == AIError.ERR_VEHICLE_IN_THE_WAY && AIDate.GetCurrentDate() - day_start < 20);
			// Only remove the road, if the road station was removed
			day_start = AIDate.GetCurrentDate();
			if(!AIRoad.IsRoadStationTile(tile))
				while(!AIRoad.RemoveRoad(tile, front) && AIError.GetLastError() == AIError.ERR_VEHICLE_IN_THE_WAY && AIDate.GetCurrentDate() - day_start < 5);
			AILog.Info("Whiles done");
		}
		else if(AIAirport.IsAirportTile(tile))
		{
			// Remove the potential rail tracks used to reserve tiles for bus stops
			local ap_top_left = Station.GetAirportTile(AIStation.GetStationID(tile));
			local ap_type = AIAirport.GetAirportType(ap_top_left);

			local ap_x = AIMap.GetTileX(ap_top_left);
			local ap_y = AIMap.GetTileY(ap_top_left);

			local ap_w = AIAirport.GetAirportWidth(ap_type);
			local ap_h = AIAirport.GetAirportHeight(ap_type);

			local bus_y = ap_y + ap_h;
			for(local bus_x = ap_x; bus_x < ap_x + ap_w; bus_x++)
			{
				local bus_tile = AIMap.GetTileIndex(bus_x, bus_y);
				local bus_front_tile = AIMap.GetTileIndex(bus_x, bus_y + 1);
			
				if(AIRail.IsRailTile(bus_tile));
				{
					Helper.SetSign(bus_tile, "clear rail");
					AIRail.RemoveRailTrack(bus_tile, AIRail.RAILTRACK_NE_SW);
					AIRail.RemoveRailTrack(bus_tile, AIRail.RAILTRACK_NW_SE);
				}
			}

			// Demolish the airport
			AITile.DemolishTile(tile);
		}
		else
			AITile.DemolishTile(tile);

		// In order for this function to return true no parts of the station should be remaining
		succeeded = succeeded && AIStation.IsValidStation(AIStation.GetStationID(tile));
	}

	return succeeded;
}

// Workaround until there is a class valuator as this is called from main.
function Station::FindInterCityStation_EachInList(list)
{
	local item;
	for(item = list.Begin(); list.HasNext(); item = list.Next())
	{
		list.SetValue(item, Station.FindInterCityStation(item));
	}
}

function Station::FindInterCityStation(town_id)
{
	local station_list = Station.GetAllInterCityStations();

	station_list.Valuate(AIStation.GetNearestTown);
	station_list.KeepValue(town_id);

	if(station_list.IsEmpty())
		return -1;

	local station = station_list.Begin();
	if(AITile.GetClosestTown(AIStation.GetLocation(station)) != town_id)
	{
		Helper.SetSign(GetLocation(town_id) + AIMap.GetTileIndex(2, 2), "  ERROR: station.GetNearestTown != tile.GetClosestTown");
		AILog.Error("@@@@@@@@@@@@ ERROR ERROR ERROR - station.GetNearestTown != tile.GetClosestTown for intercity station in town " + AITown.GetName(town_id))
	}

	return station_list.Begin();
}

//// Vehicle query functions ////
function Station::GetAllVehiclesOfStationList(station_list)
{
	local all_vehicles_list = AIList();
	local station = station_list.Begin();
	while(station_list.HasNext())
	{
		local vehicles = AIVehicleList_Station(station);
		all_vehicles_list.AddList(vehicles);

		station = station_list.Next();
	}

	return all_vehicles_list;
}

function Station::GetAllRoadVehiclesOfStationList(station_list)
{
	local all_vehicles_list = Station.GetAllVehiclesOfStationList(station_list);
	all_vehicles_list.Valuate(AIVehicle.GetVehicleType);
	all_vehicles_list.KeepValue(AIVehicle.VT_ROAD);

	return all_vehicles_list;
}

function Station::GetAllAirVehiclesOfStationList(station_list)
{
	local all_vehicles_list = Station.GetAllVehiclesOfStationList(station_list);
	all_vehicles_list.Valuate(AIVehicle.GetVehicleType);
	all_vehicles_list.KeepValue(AIVehicle.VT_AIR);

	return all_vehicles_list;
}

function Station::AreStationsConnectedByVehicles(station_list, vehicle_type)
{
	local station_list_length = station_list.Count();
	if(station_list_length == 0)
		return false;

	if(station_list_length == 1)
		return true;

	local common_vehicles = AIList();
	common_vehicles.AddList(AIVehicleList_Station(station_list.Begin()));
	common_vehicles.Valuate(AIVehicle.GetVehicleType);
	common_vehicles.KeepValue(vehicle_type);

	// If any of the following stations are connected with the first one, then they all must be visited by at least one common vehicle that also 
	// visits the first station
	local station_id = station_list.Begin();
	station_id = station_list.Next(); // start with the second station in the list
	for(; station_list.HasNext(); station_id = station_list.Next())
	{
		// Keep vehicles that are visited by station 1 and each following station
		local vehicles = AIVehicleList_Station(station_id);
		vehicles.Valuate(AIVehicle.GetVehicleType);
		vehicles.KeepValue(vehicle_type);
		common_vehicles.KeepList(vehicles);
	}
	
	// All stations are connected if they have at least one vehicle that visit them all
	return !common_vehicles.IsEmpty();
}

//// Bus/Road functions ////
function Station::GetRoadConnectTileOfStation(station_id)
{
	// All stations that PAXLink builds have all its road stations inter-connected by road between its front tiles, so just get one of the
	// road station tiles; then get the front tile of that tile.
	local tile_list = AITileList_StationType(station_id, AIStation.STATION_BUS_STOP || AIStation.STATION_TRUCK_STOP);
	if(tile_list.Count() == 0)
		return -1;

	local road_station_tile = tile_list.Begin();
	local front = AIRoad.GetRoadStationFrontTile(road_station_tile);

	return front;
}


//// Airport functions ////
function Station::GetAirportTile(station_id)
{
	local airport_tiles = AITileList_StationType(station_id, AIStation.STATION_AIRPORT);
	if(airport_tiles.IsEmpty())
		return -1;

	airport_tiles.Valuate(Utils_Valuator.ItemValuator);
	airport_tiles.KeepBottom(1);

	return airport_tiles.Begin();
}

function Station::GetHangarTile(station_id)
{
	local hangar_tile = AIAirport.GetHangarOfAirport(Station.GetAirportTile(station_id));
	return hangar_tile;
}

// Loops through all hangars in an airport and check how many vehicles it has in them
function Station::GetAircraftsInHangar(station_id)
{
	local tot_num = 0;
	local hangar_aircraft_list = AIList();

	// Loop through all hangars of the airport
	local airport_tiles = AITileList_StationType(station_id, AIStation.STATION_AIRPORT);
	airport_tiles.Valuate(AIAirport.IsHangarTile);
	airport_tiles.KeepValue(1);

	for(local hangar_tile = airport_tiles.Begin(); airport_tiles.HasNext(); hangar_tile = airport_tiles.Next())
	{

		// Get a list of all vehicles in current hangar
		local vehicle_list = AIVehicleList_Station(station_id);
		vehicle_list.Valuate(AIVehicle.GetLocation);
		vehicle_list.KeepValue(hangar_tile);

		Helper.SetSign(hangar_tile, "hangar " + vehicle_list.Count());

		// Add those vehicles to the list of all airplanes in hangars of this airport
		hangar_aircraft_list.AddList(vehicle_list);
	}

	hangar_aircraft_list.Valuate(AIVehicle.IsStoppedInDepot);
	hangar_aircraft_list.KeepValue(0);

	Helper.SetSign(AIStation.GetLocation(station_id) - 2, "hang tot " + hangar_aircraft_list.Count());
	return hangar_aircraft_list;
}

function Station::IsSmallAirport(station_id)
{
	local airport_type = AIAirport.GetAirportType(Station.GetAirportTile(station_id));
	return airport_type == AIAirport.AT_SMALL || airport_type == AIAirport.AT_COMMUTER;
}
