/*require("globals.nut");
require("helper.nut");
require("orders.nut");
require("builder.nut");*/

function GetClosestTownOfStation(station_id)
{
	return AITile.GetClosestTown(AIStation.GetLocation(station_id));
}

//////////////////////////////////////////////////////
//                                                  //
//   CLASS: TownList                                //
//                                                  //
//////////////////////////////////////////////////////

class TownList {
	list = null;

	constructor() {
		list = [];
		
		CreateTownList();
	}

	function CreateTownList();
	function GetTownManager(town_id);
	function GetStationStatistics(station_id);
	function UpdateTownStatistics();
}

function TownList::CreateTownList()
{
	AILog.Info("Adding towns to town list");

	local all_towns = AITownList();
	for(local town_id = all_towns.Begin(); all_towns.HasNext(); town_id = all_towns.Next())
	{
		//AILog.Info("Adding town to town list");
		local town_manager = TownManager(town_id);
		this.list.push(town_manager);

		Helper.SetSign(AITown.GetLocation(town_id) + 1, "town " + town_id);
		AILog.Info("added " + AITown.GetName(town_id));
	}
}

function TownList::UpdateTownStatistics(num)
{
	foreach(_, town_manager in this.list)
	{
		//AILog.Info("reading statistics data for town");
		town_manager.ReadStatisticsData();
		Helper.SetSign(AITown.GetLocation(town_manager.town_id) + 2, "stat " + num);
	}
}

function TownList::GetTownManager(town_id)
{
	foreach(_, town_manager in this.list)
	{
		if(town_manager.town_id == town_id)
			return town_manager;
	}

	AILog.Info("num town managers " + this.list.len());
	AILog.Info("unknown town id " + town_id);
	KABOOOOM_unknown_town_id();
	return null;
}

function TownList::GetStationStatistics(station_id)
{
	if(!AIStation.IsValidStation(station_id))
	{
		AILog.Error("TownList::GetStationStatistics(station_id): Bad station id: " + station_id);
		return null;
	}
	local town_id = GetClosestTownOfStation(station_id);
	local town_manager = GetTownManager(town_id);
	
	return town_manager.GetStationStatistics(station_id);
}


//////////////////////////////////////////////////////
//                                                  //
//   CLASS: StationStatistics                       //
//                                                  //
//////////////////////////////////////////////////////


// Member class of TownManager
class StationStatistics {
	station_id = null;
	pax_waiting = null;
	rating = null;

	usage = null;

	created_date = null;
	
	location_tile_id = null; // Used only to validate that the station have not disappeared and another station somewhere else have been built with the same station id

	constructor(station_id)
	{
		this.station_id = station_id;
		pax_waiting = -1;
		rating = -1;
		usage = { bus =    {station_type = AIStation.STATION_BUS_STOP,   mode = AIVehicle.VT_ROAD, percent_usage = -1}, 
				truck =    {station_type = AIStation.STATION_TRUCK_STOP, mode = AIVehicle.VT_ROAD, percent_usage = -1},
				aircraft = {station_type = AIStation.STATION_AIRPORT,    mode = AIVehicle.VT_AIR,  percent_usage = -1},
				train =    {station_type = AIStation.STATION_TRAIN,      mode = AIVehicle.VT_RAIL, percent_usage = -1} };

		created_date = AIDate.GetCurrentDate();
		this.location_tile_id = AIStation.GetLocation(station_id);
	}
};

function StationStatistics::VehicleIsWithinTileList(vehicle_id, tile_list)
{
	return tile_list.HasItem(AIVehicle.GetLocation(vehicle_id));
}

// returns the station location for current order. If current order is the hangar, return the airport location.
function StationStatistics::GetCurrentOrderDestinationStationSignLocation(vehicle_id)
{
	local loc = Order.GetCurrentOrderDestination(vehicle_id);

	// change loc tile to the station sign location
	loc = AIStation.GetLocation(AIStation.GetStationID(loc));

	return loc;
}

function StationStatistics::GetNumAircraftsInAirportQueue(station_id)
{
	// aircrafts
	local station_vehicle_list = AIVehicleList_Station(station_id);
	station_vehicle_list.Valuate(AIVehicle.GetVehicleType);
	station_vehicle_list.KeepValue(AIVehicle.VT_AIR);

	// get airport tile
	local station_tile_list = AITileList_StationType(station_id, AIStation.STATION_AIRPORT);
	station_tile_list.Valuate(Helper.DummyValuator);
	station_tile_list.Sort(AIAbstractList.SORT_BY_VALUE, true); // smallest first

	local airport_tile = station_tile_list.Begin();
	//local airport_tile = Station.GetAirportTile(station_id);

	// get airport holding rectangle
	local holding_rect = AITileList();	

	switch(AIAirport.GetAirportType(airport_tile))
	{
		case AIAirport.AT_SMALL:
			holding_rect.AddRectangle(Helper.GetTileRelative(airport_tile, -2, 0), Helper.GetTileRelative(airport_tile, 17, 12));
			holding_rect.RemoveRectangle(Helper.GetTileRelative(airport_tile, 1, 1), Helper.GetTileRelative(airport_tile, 3, 2)); // remove non-holding airport tiles
			break;
		
		case AIAirport.AT_COMMUTER:
			holding_rect.AddRectangle(Helper.GetTileRelative(airport_tile, -2, 0), Helper.GetTileRelative(airport_tile, 14, 9));
			holding_rect.RemoveRectangle(Helper.GetTileRelative(airport_tile, 1, 1), Helper.GetTileRelative(airport_tile, 4, 3)); // remove non-holding airport tiles
			break;

		case AIAirport.AT_LARGE:
			holding_rect.AddRectangle(Helper.GetTileRelative(airport_tile, 9, 0), Helper.GetTileRelative(airport_tile, 17, 5));
			break;

		case AIAirport.AT_METROPOLITAN:
			holding_rect.AddRectangle(Helper.GetTileRelative(airport_tile, -2, 0), Helper.GetTileRelative(airport_tile, 17, 12));
			holding_rect.RemoveRectangle(Helper.GetTileRelative(airport_tile, 1, 1), Helper.GetTileRelative(airport_tile, 5, 5)); // remove non-holding airport tiles
			break;

		case AIAirport.AT_INTERNATIONAL:
			holding_rect.AddRectangle(Helper.GetTileRelative(airport_tile, -2, 0), Helper.GetTileRelative(airport_tile, 19, 13));
			holding_rect.RemoveRectangle(Helper.GetTileRelative(airport_tile, 1, 1), Helper.GetTileRelative(airport_tile, 6, 6)); // remove non-holding airport tiles
			break;

		case AIAirport.AT_INTERCON:
			holding_rect.AddRectangle(Helper.GetTileRelative(airport_tile, -13, -11), Helper.GetTileRelative(airport_tile, 19, 21));
			holding_rect.RemoveRectangle(Helper.GetTileRelative(airport_tile, 0, 0), Helper.GetTileRelative(airport_tile, 8, 10)); // remove non-holding airport tiles
			break;

		default:
			// Unknown airport type -> crash the AI
			KABOOOOOM_unknown_airport_type();
	}

	station_vehicle_list.Valuate(VehicleIsWithinTileList, holding_rect);
	station_vehicle_list.KeepValue(1);

	// Only keep vehicles that are on the way to the airport
	station_vehicle_list.Valuate(GetCurrentOrderDestinationStationSignLocation);
	station_vehicle_list.KeepValue(AIStation.GetLocation(station_id));

	// remove vehicles that are in a depot (the right depot of international would else give false positive
	station_vehicle_list.Valuate(AIVehicle.IsInDepot);
	station_vehicle_list.KeepValue(0);

	return station_vehicle_list.Count();
}

function StationStatistics::GetNumNonStopedAircraftsInAirportDepot(station_id)
{
	local hangar_vehicles = Station.GetAircraftsInHangar(station_id);

	hangar_vehicles.Valuate(AIVehicle.IsStoppedInDepot);
	hangar_vehicles.KeepValue(0);

	return hangar_vehicles.Count();
}

function StationStatistics::ReadStatisticsData()
{
	if(!AIStation.IsValidStation(this.station_id))
		return;


	//AILog.Info("ReadStatisticsData for station " + AIStation.GetName(this.station_id));

	local alpha = 2; // weight of old value (new value will have weight 1)

	local currently_waiting = AIStation.GetCargoWaiting(this.station_id, Helper.GetPAXCargo());
	if(this.pax_waiting == -1)
		this.pax_waiting = currently_waiting;
	else
		this.pax_waiting = (this.pax_waiting * alpha + currently_waiting) / (alpha + 1);

	//AILog.Info("pax_waiting = " + this.pax_waiting);

	local current_rating = AIStation.GetCargoRating(this.station_id, Helper.GetPAXCargo());
	if(this.rating == -1)
		this.rating = current_rating;
	else
		this.rating = (this.rating * alpha + current_rating) / (alpha + 1);

	//AILog.Info("rating = " + this.rating);

	// Get an estimate on the vehicle load of the station, by counting the number of vehicles on station tiles
	foreach(_, transp_mode_usage in this.usage)
	{
		if(AIStation.HasStationType(this.station_id, transp_mode_usage.station_type))
		{

			if(transp_mode_usage.station_type == AIStation.STATION_AIRPORT)
			{
				local hangar_num = GetNumNonStopedAircraftsInAirportDepot(this.station_id);
				local holding_num = GetNumAircraftsInAirportQueue(this.station_id);
				local new_percent_usage = (holding_num + hangar_num)* 100;
				if(transp_mode_usage.percent_usage == -1)
					transp_mode_usage.percent_usage = new_percent_usage;
				else
					transp_mode_usage.percent_usage = (transp_mode_usage.percent_usage * alpha + new_percent_usage) / (alpha + 1);

				Helper.SetSign(AIStation.GetLocation(station_id) + 2, "hold: " + holding_num + " han: " + hangar_num);
				Helper.SetSign(AIStation.GetLocation(station_id) + 3, "queue: " + (new_percent_usage / 100));
			}
			else
			{
				local station_tile_list = AITileList_StationType(this.station_id, transp_mode_usage.station_type);
				local station_vehicle_list = AIVehicleList_Station(this.station_id);
				station_vehicle_list.Valuate(AIVehicle.GetVehicleType);
				station_vehicle_list.KeepValue(transp_mode_usage.mode);

				local num_vehicles_on_station = 0;

				for(local vehicle = station_vehicle_list.Begin(); station_vehicle_list.HasNext(); vehicle = station_vehicle_list.Next())
				{
					for(local tile = station_tile_list.Begin(); station_tile_list.HasNext(); tile = station_tile_list.Next())
					{
						if(AIVehicle.GetLocation(vehicle) == tile)
						{
							num_vehicles_on_station++;
						}
					}
				}

				local new_percent_usage = num_vehicles_on_station * 100 / station_tile_list.Count();
				Helper.SetSign(station_tile_list.Begin() - 1, "curr usage: " + new_percent_usage);
				//Helper.SetSign(station_tile_list.Begin() - 3, "prev value: " + transp_mode_usage.percent_usage);

				if(transp_mode_usage.percent_usage == -1)
				{
					transp_mode_usage.percent_usage = new_percent_usage;
					//Helper.SetSign(station_tile_list.Begin() - 2, "no prev value");
				}
				else
				{
				//	transp_mode_usage.percent_usage = (transp_mode_usage.percent_usage * alpha + new_percent_usage) / (alpha + 1);
					local top = transp_mode_usage.percent_usage * alpha + new_percent_usage;
					local bot = (alpha + 1);
					transp_mode_usage.percent_usage = top / bot;

					//Helper.SetSign(station_tile_list.Begin() - 2, "t: " + top + " b: " + bot + " np: " + transp_mode_usage.percent_usage);
				}
			}
		}
		else
		{
			transp_mode_usage.percent_usage = -1;
		}
	}

	//AILog.Info("bus usage = " + this.usage.bus.percent_usage);

	Helper.SetSign(AIStation.GetLocation(station_id), "w" + pax_waiting + " r" + rating + " u" + usage.bus.percent_usage + " a" + usage.aircraft.percent_usage);
}

function StationStatistics::RemoveStationSigns() 
{
	Helper.SetSign(this.location_tile_id, "");
	Helper.SetSign(this.location_tile_id - 1, "");
	Helper.SetSign(this.location_tile_id + 1, "");
}

//////////////////////////////////////////////////////
//                                                  //
//   CLASS: TownManager                             //
//                                                  //
//////////////////////////////////////////////////////

class TownManager {
	town_id = null;
	station_list = null;
	expandability = null;

	can_add_airport = false; // Always use CanAirportBeAdded function which re-checks only every Y days, since checking is expansive
	last_can_add_airport_check = null;

	constructor(town_id)
	{
		this.town_id = town_id;
		station_list = []; // empty array
		expandability = 0;
		can_add_airport = null;
		last_can_add_airport_check = 0;

		RecreateFromMap();
	}

	function GetStationStatistics(station_id);
	function GetMaxBusStationUsage();

	function RecreateFromMap();
	function AddStationsFromMap();

	function GetExpandability();
	function CanAirportBeAdded(); // caches the result and re-checks only every Y days, since checking is expansive

	function ReadStatisticsData();
	function ForceRecheckIfAirportCanBeAdded(); // force re-check of if an airport can be added in the town

	// strategies
	function GetNumberOfFeederStations();
	function GetNumberOfFeederStationsNeeded();
};

function TownManager::GetStationStatistics(station_id)
{
	foreach(_, station in this.station_list)
	{
		if(station.station_id == station_id)
			return station;
	}

	return null;
}

function TownManager::RecreateFromMap()
{
	AddStationsFromMap();
}

function TownManager::AddStationsFromMap()
{
	local stations = Helper.GetStationsInTown(this.town_id);

	for(local station_id = stations.Begin(); stations.HasNext(); station_id = stations.Next())
	{
		// Skip station if already added to station_list
		local skip = false;
		foreach(_, station in this.station_list)
		{
			if(station.station_id == station_id)
			{
				skip = true;
				break;
			}
		}

		if(!skip)
		{
			// Add new station to list
			local station_info = StationStatistics(station_id);
			this.station_list.push(station_info);
		}
	}
}


function TownManager::RemoveInvalidStations()
{
	// Loop through all station data and remove those entries where the station id is invalid

	// Restart the foreach when a station have been removed so that we don't do anything dangerous.
	local removed_station = false;
	do
	{
		removed_station = false;

		foreach(idx, station in this.station_list)
		{
			local remove = false;
			if(!Station.IsNonEmptyStation(station.station_id))
			{
				AILog.Info("TownManager::RemoveInvalidStations: invalid or station without transport modes: {data index: " + idx + ", station id: " + station.station_id + "}");
				remove = true;
			}
			else if(station.location_tile_id != AIStation.GetLocation(station.station_id))
			{
				AILog.Info("TownManager::RemoveInvalidStations: station id: " + station.station_id + " have been found on the wrong location. It has probably been destroyed and a new station elsewhere has got the same station id. => Remove from this town manager.");
				remove = true;
			}

			if(remove)
			{
				station.RemoveStationSigns();
				this.station_list.remove(idx);
				removed_station = true;

				break;
			}
		}
	}
	while(removed_station)
}

function TownManager::ReadStatisticsData()
{
	RemoveInvalidStations();
	AddStationsFromMap();

	foreach(_, station in this.station_list)
	{
		station.ReadStatisticsData();
	}

	// Keep an average of the expandability of the town.
	local current_expandability = GetExpandability();
	local alpha = 5;
	this.expandability = (current_expandability + alpha * this.expandability) / (alpha + 1);
}

function TownManager::GetMaxBusStationUsage()
{
	local max_usage = 0;
	foreach(_, station in this.station_list)
	{
		if(Station.IsFeederStation(station.station_id))
		{
			if(station.usage.bus.percent_usage > max_usage)
				max_usage = station.usage.bus.percent_usage;
		}
	}

	return max_usage;
}

function TownManager::GetExpandability()
{
	local pax_produced = AITown.GetLastMonthProduction(this.town_id, Helper.GetPAXCargo());
	local pax_transported = AITown.GetLastMonthTransported(this.town_id, Helper.GetPAXCargo());
	local pax_percent_transported = AITown.GetLastMonthTransportedPercentage(this.town_id, Helper.GetPAXCargo());

	local not_transported = pax_produced - pax_transported;

	Helper.SetSign(AITown.GetLocation(this.town_id) + 4, "p:" + pax_produced + " t:" + pax_transported + " %t:" + pax_percent_transported + " nt:" + not_transported);

	local expandability = not_transported; 
	return expandability;
}

function TownManager::GetNumberOfFeederStations()
{
	local num_feeder_stations = 0;
	foreach(_, station in this.station_list)
	{
		if(Station.IsFeederStation(station.station_id))
			num_feeder_stations++;
	}

	return num_feeder_stations;
}

function TownManager::GetNumberOfFeederStationsNeeded()
{
	local num_existing = GetNumberOfFeederStations();
	AILog.Info("num existing: " + num_existing);

	if(Builder.ShouldHaveFeederMode(town_id, Station.FindInterCityStation(town_id)))
	{
		AILog.Info("feeder mode");
		if(num_existing == 0)
		{
			return 1;
		}
		else
		{
			// Collect some status about the feeder bus stations in town that can be used to determine the new amount of stations we want to have
			local max_usage = 0;
			local min_usage = 10000;
			local max_waiting = 0;
			local min_waiting = 10000;
			local min_constructed_date = -1;
			local num_used_stations = 0;
			foreach(_, station in this.station_list)
			{
				AILog.Info("Station " + AIStation.GetName(station.station_id));
				if(Station.IsFeederStation(station.station_id))
				{
					// not all of these are used, but code have not been tuned yet, so remove unused stats after a while
					if(station.usage.bus.percent_usage > max_usage)
						max_usage = station.usage.bus.percent_usage;
					if(station.usage.bus.percent_usage < min_usage)
						min_usage = station.usage.bus.percent_usage;
					if(station.pax_waiting > max_waiting)
						max_waiting = station.pax_waiting;
					if(station.pax_waiting < min_waiting)
						min_waiting = station.pax_waiting;
					if(min_constructed_date == -1 || station.created_date < min_constructed_date)
						min_constructed_date = station.created_date;

					AILog.Info("    is feeder station with usage " + station.usage.bus.percent_usage + " and cargo rating: " + AIStation.GetCargoRating(station.station_id, Helper.GetPAXCargo()));

					if(station.usage.bus.percent_usage > 5 && AIStation.GetCargoRating(station.station_id, Helper.GetPAXCargo()) > 50)
					{
						num_used_stations++;
						AILog.Info("    used++");
					}
					else if(station.created_date + 45 > AIDate.GetCurrentDate())
					{
						// report very young stations as used even if they aren't.
						num_used_stations++;
					}
				}
			}

			local unused = num_existing - num_used_stations;
			AILog.Info(AITown.GetName(town_id) + " has " + num_used_stations + " used and " + unused + " unused stations");

			if(unused == 0)
			{
				// If all stations are used, check if an expansion is needed
				// Expand if the existing stations don't supply enough passengers
				// to the buses
				if(max_waiting < 30 && 
						AIDate.GetCurrentDate() - min_constructed_date > 70 ) // don't build an extra station if the youngest is less than 70 days old
					return num_existing + 1;

				return num_existing;
			}

			// if less than all stations have been used, keep only the used stations as long as at least one station is used.
			if(num_used_stations >= 1)
			{
				// don't reduce if there is a new station younger than 70 days
				if(AIDate.GetCurrentDate() - min_constructed_date > 70)
					return num_existing;

				return num_used_stations;
			}

			// at minimum keep one station.
			return 1;
		}
	}
	else
	{
		AILog.Info("not feeder mode -> 2");
		// If the city is not in feeder mode it needs two stations in order to have a bus service.
		return 2;
	}
	
}

function TownManager::CanAirportBeAdded()
{
	if(can_add_airport == null || AIDate.GetCurrentDate() - last_can_add_airport_check > 365)
	{
		ForceRecheckIfAirportCanBeAdded();
		last_can_add_airport_check = AIDate.GetCurrentDate();
	}
	
	return can_add_airport;
}

function TownManager::ForceRecheckIfAirportCanBeAdded()
{
	local best_airport_available = GetBestAirportAvailable(8, 8);
	local ap_w = AIAirport.GetAirportWidth(best_airport_available);
	local ap_h = AIAirport.GetAirportHeight(best_airport_available);

	local max_noise_add = AITown.GetAllowedNoise(this.town_id);
	local ap_location = Builder.FindInterCityStationPlacement(this.town_id, ap_w, ap_h, best_airport_available, max_noise_add);
	if(AIMap.IsValidTile(ap_location))
		this.can_add_airport = true;
	else
		this.can_add_airport = false;
}
