/*
 * This file is part of StreetTraffic, which is an AI for OpenTTD
 * Copyright (C) 2009 Frank Juedes
 *
 * StreetTraffic is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License
 *
 * StreetTraffic is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with StreetTraffic; If not, see <http://www.gnu.org/licenses/> or
 * write to the Free Software Foundation, Inc., 51 Franklin Street, 
 * Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

/*
 * Much of the code of this AI has been taken from the PAXLink AI,
 * and from TownCars (written by Zuu / Leif Linse). 
 */


require("helper.nut");

// --------------------------------------------------------------------
// main AI class
// --------------------------------------------------------------------
class StreetTraffic extends AIController {
  loaded_from_save = null; // flag for load/save
  MyDepots         = null; // will contain a list of all depot/town tile-id's

  constructor() {
    loaded_from_save = false;
    MyDepots = AIList();
  } // constructor

  function Start();
  function Stop() { Helper.ClearAllSigns(); } 

  function Save();
  function Load(version, data);

} // StreetTraffic


// --------------------------------------------------------------------
// private function InitCompany: Initialize the company
// --------------------------------------------------------------------
function StreetTraffic::InitCompany() {
  // Set company name
  if (!AICompany.SetName("StreetTraffic")) {
    local i = 2;
    while (!AICompany.SetName("StreetTraffic #" + i)) {
      i += 1;
      if (i > 255) break;
    } // while
  } // if
  
  // Set predident's name
  AICompany.SetPresidentName("Frank/2")

  // Get the maximum amount of money
  AICompany.SetLoanAmount(AICompany.GetMaxLoanAmount());

  // Enable automatic renewal of vehicles
  AICompany.SetAutoRenewStatus(true);
  AICompany.SetAutoRenewMonths(1);

  // write greetings int the log
  AILog.Info("Hi my name is " + AICompany.GetName(AICompany.COMPANY_SELF) + ". I will generate a lot of road-traffic.");
  AILog.Info("--------------------------------------------------");
} // InitCompany

// --------------------------------------------------------------------
// private function MyValuate:
// --------------------------------------------------------------------
function StreetTraffic::MyValuate(list, valuator, ...) {
  assert(typeof(list) == "instance");
  assert(typeof(valuator) == "function");
   
  local args = [this, null];
   
  for(local c = 0; c < vargc; c++) {
    args.append(vargv[c]);
  } // for

  foreach (item, _ in list) {
    args[1] = item;
    local value = valuator.acall(args);
    if (typeof(value) == "bool") {
       value = value ? 1 : 0;
    } else if (typeof(value) != "integer") {
       throw("Invalid return type from valuator");
    } // if
    list.SetValue(item, value);
  } // foreach
} // MyValuate

// --------------------------------------------------------------------
// private function GetTileRelative
// --------------------------------------------------------------------
function GetTileRelative(relative_to_tile, delta_x, delta_y) {
  local tile_x = AIMap.GetTileX(relative_to_tile);
  local tile_y = AIMap.GetTileY(relative_to_tile);

  local new_x = tile_x + delta_x;
  local new_y = tile_y + delta_y;

  return AIMap.GetTileIndex(new_x, new_y);
} // GetTileRelative

// --------------------------------------------------------------------
// private function GetNeighbours4:
// --------------------------------------------------------------------
function StreetTraffic::GetNeighbours4(tile_id) {
  local list = AITileList();

  if (!AIMap.IsValidTile(tile_id)) return list;

  local tile_x = AIMap.GetTileX(tile_id);
  local tile_y = AIMap.GetTileY(tile_id);

  list.AddTile(GetTileRelative(tile_id, -1, 0));
  list.AddTile(GetTileRelative(tile_id, 0, -1));
  list.AddTile(GetTileRelative(tile_id, 1, 0));
  list.AddTile(GetTileRelative(tile_id, 0, 1));

  return list;
} // GetNeighbours

// --------------------------------------------------------------------
// private function CanConnectToRoad:
// --------------------------------------------------------------------
function CanConnectToRoad(road_tile, adjacent_tile_to_connect) {
  local neighbours = GetNeighbours4(road_tile);
	
  neighbours.RemoveTile(adjacent_tile_to_connect);

  // This function requires that road_tile is connected with at least one other road tile.
  neighbours.Valuate(AIRoad.IsRoadTile);
  if (Helper.ListValueSum(neighbours) < 1) return false;
	
  for(local neighbour_tile_id = neighbours.Begin(); neighbours.HasNext(); neighbour_tile_id = neighbours.Next()) {
    if (AIRoad.IsRoadTile(neighbour_tile_id)) {
      local ret = AIRoad.CanBuildConnectedRoadPartsHere(road_tile, neighbour_tile_id, adjacent_tile_to_connect);
      if (ret == 0 || ret == -1) return false;
    } // if
  } // for
  return true;
} // CanConnectToRoad

// --------------------------------------------------------------------
// private function BuildRoadStation:
// --------------------------------------------------------------------
function StreetTraffic::BuildRoadStation(tile_id, depot) {
  //AISign.BuildSign(tile_id, "Bld stn");
  AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_ROAD);

  if (AITile.IsWaterTile(tile_id)) return false;

  // Don't even try if the tile has a steep slope
  local slope = AITile.GetSlope(tile_id);
  if (slope != 0 && AITile.IsSteepSlope(slope)) return false;

  local neighbours = GetNeighbours4(tile_id);
  neighbours.Valuate(AIRoad.IsRoadTile);
  neighbours.KeepValue(1);

  MyValuate(neighbours, CanConnectToRoad, tile_id);
  neighbours.KeepValue(1);

  //MyValuate(neighbours, AITile.IsSteepSlope); // steep sloped neighbours can't be used
  //neighbours.KeepValue(0);

  neighbours.Valuate(Helper.GetMaxHeightOfTile);
  neighbours.KeepValue(Helper.GetMaxHeightOfTile(tile_id));

  // Return false if none of the neighbours has road
  if (neighbours.Count() == 0) return false;

  local front = neighbours.Begin();

  if (!CanConnectToRoad(front, tile_id)) return false;

  if (!AITile.IsBuildable(tile_id) && !AIRoad.IsRoadTile(tile_id) && !AICompany.IsMine(AITile.GetOwner(tile_id))) {
    AITile.DemolishTile(tile_id);
  } // if

  // Build road
  // Keep trying until we get another error than vehicle in the way
  while(!AIRoad.BuildRoad(tile_id, front) && AIError.GetLastError() == AIError.ERR_VEHICLE_IN_THE_WAY); 
	
  if (!AIRoad.AreRoadTilesConnected(tile_id, front)) return false;

  Helper.SetSign(tile_id, "tile");
  Helper.SetSign(front, "front");

  // Build station/depot
  // Keep trying until we get another error than vehicle in the way
  local ret = false; // for some strange reason this variable declaration can't be inside the loop
  do {
    if (depot) ret = AIRoad.BuildRoadDepot(tile_id, front);
    else ret = AIRoad.BuildRoadStation(tile_id, front, AIRoad.ROADVEHTYPE_BUS, AIStation.STATION_JOIN_NEW);
  } // do

  while(!ret && AIError.GetLastError() == AIError.ERR_VEHICLE_IN_THE_WAY);

  return ret;
} // BuildStation

// --------------------------------------------------------------------
// private function BuildDepotAtTileScoreValidator:
// --------------------------------------------------------------------
function BuildDepotAtTileScoreValuator(tile_id) {
  local score = 0;
  local road_neighbour_count = AIRoad.GetNeighbourRoadCount(tile_id);

  if (AIRoad.IsRoadTile(tile_id)) {
    // give score if tile has road, but only one neighbor has that, as that means it is a road stump
    if (road_neighbour_count == 1) score += 1000;
  } else {
    // better build it along an existing road, than hijacking a road stump
    if (road_neighbour_count > 0) score += 2000;
  } // if

  if (AITile.GetSlope(tile_id) == 0) score += 500;

  if (AITile.IsBuildable(tile_id)) {
    score += 100;
  } else {
    local testmode = AITestMode();
    local acount = AIAccounting();
    AITile.DemolishTile(tile_id);
    score -= acount.GetCosts();
  } // if
  return score;
} // BuildDeportAtTileScoreValuator

// --------------------------------------------------------------------
// private function BuildDepot:
// --------------------------------------------------------------------
function StreetTraffic::BuildDepot(town_id) {
  local depot_placement_tiles = AITileList();
  local town_loc = AITown.GetLocation(town_id);
  depot_placement_tiles.AddRectangle(town_loc - AIMap.GetTileIndex(15, 15), town_loc + AIMap.GetTileIndex(15, 15));

  // Only consider tiles that are buildable
  depot_placement_tiles.Valuate(AITile.IsBuildable);
  depot_placement_tiles.KeepValue(1);

  // search from town center and outwards
  depot_placement_tiles.Valuate(AIMap.DistanceManhattan, town_loc);
  depot_placement_tiles.Sort(AIAbstractList.SORT_BY_VALUE, true); // highest value last

  // try with all tiles until a working one is found
  local depot_tile = depot_placement_tiles.Begin();
  while(depot_placement_tiles.HasNext()) {
    //Helper.SetSign(depot_tile, "tile");
    //AILog.Info("Trying to build depot at tile " + depot_tile);
    //AISign.BuildSign(depot_tile, "build depot");

    local depot = true;
    if (BuildRoadStation(depot_tile, depot)) break;			
    depot_tile = depot_placement_tiles.Next();
  } // while

  if (!AIRoad.IsRoadDepotTile(depot_tile)) return -1;

  // depot built likely is equal to no buses yet in town.
  return depot_tile;
} // BuildDepot

// --------------------------------------------------------------------
// private function BuildCars
// --------------------------------------------------------------------
function StreetTraffic::BuildCars(town_id, car_count, depot_tile, car_sleep_time) {
  local engine_list = AIEngineList(AIVehicle.VT_ROAD);
  //AILog.Info("Number of road vehicle types: " + engine_list.Count());
        
  // Apply filter conditions as set in the configuration
  if (AIController.GetSetting("build_only_zero_price_rvs") == 1) {
    engine_list.Valuate(AIEngine.GetPrice);
    engine_list.KeepValue(0);
    //AILog.Info("Filter zero_price_rvs applied, remaining number of road vehicle types: " + engine_list.Count());
  } // if

  if (AIController.GetSetting("build_only_zero_running_cost_rvs") == 1) {
    engine_list.Valuate(AIEngine.GetRunningCost);
    engine_list.KeepValue(0);
    //AILog.Info("Filter zero_running_cost_rvs applied, remaining number of road vehicle types: " + engine_list.Count());
  } // if

  if (AIController.GetSetting("build_only_zero_capacity_rvs") == 1) {
    engine_list.Valuate(AIEngine.GetCapacity);
    engine_list.KeepValue(0);
    //AILog.Info("Filter zero_capacity_rvs applied, remaining number of road vehicle types: " + engine_list.Count());
  } // if

  if (AIController.GetSetting("build_only_if_cargo_type_is_pax") == 1) {
    engine_list.Valuate(AIEngine.GetCargoType);
    engine_list.KeepValue(Helper.GetPAXCargo());
    //AILog.Info("Filter cargo_type_is_pax applied, remaining number of road vehicle types: " + engine_list.Count());
  } // if

  //AILog.Info("Final number of road vehicle types: " + engine_list.Count());
  if (engine_list.Count() == 0) {
    AILog.Error("There is currently no vehicle that the AI is allowed to build.");
  } // if
    
  local engine_id = engine_list.Begin();
  for (local i = 0; i < car_count; i++) {
    local vehicle_id = AIVehicle.BuildVehicle(depot_tile, engine_id);
    if (!AIVehicle.IsValidVehicle(vehicle_id)) {
      AILog.Error("failed to buy vehicles.");
      return false;
    } // if

    AIVehicle.StartStopVehicle(vehicle_id);

    if (engine_list.HasNext()) {
      engine_id = engine_list.Next();
    } else {
      engine_id = engine_list.Begin();
    } // if
    this.Sleep(car_sleep_time);
  } // for
  return true;
} // BuildCars

// --------------------------------------------------------------------
// Get the number of vehicles owned by me
// --------------------------------------------------------------------
function GetCurrVehCount() {
  local MyVehicles = AIVehicleList();
  return MyVehicles.Count();
} // GetCurrVehCount

// --------------------------------------------------------------------
// Start AI:
// - Create one depot in each town
// - Endless: Build vehicles in each town if number of vehicles less than maximum
//  --------------------------------------------------------------------
function StreetTraffic::Start() {
  InitCompany();

  // Get the settings for this AI 
  local max_num_towns = AIController.GetSetting("num_towns_max");
  local max_num_cars = Helper.Min(AIController.GetSetting("num_cars_total"),
                                  AIGameSettings.GetValue("max_roadveh")) ;
  local num_cars_per_town = AIController.GetSetting("num_cars_per_town");
  local car_sleep_time = AIController.GetSetting("car_sleep_time");
  local town_sleep_time = AIController.GetSetting("town_sleep_time");
  local eol_sleep_time  = AIController.GetSetting("eol_sleep_time");

  if (!loaded_from_save) {
    //
    // Create one depot in each town until the maximum of towns is reached
    //
    local Counter  = 0;
    local TownList = AITownList();
    TownList.Valuate(AITown.GetPopulation);
    TownList.Sort(AIAbstractList.SORT_BY_VALUE, false); // highest value first
    foreach (TownId,dummy in TownList) { 
      AILog.Info("Building depot in " + AITown.GetName(TownId) + ", population " + AITown.GetPopulation(TownId));
      local depot_tile = BuildDepot(TownId);
      if (AIMap.IsValidTile(depot_tile)) {
        MyDepots.AddItem(TownId,depot_tile);
        Counter += 1;
        if (Counter >= max_num_towns) break;
      } else {
        AILog.Warning("Failed to build depot: No cars will be build in this town!");
      } // if
    } // foreach
    AILog.Info("--------------------------------------------------");
  } // if

  //
  // Build and maintain cars
  //
  while (true) { 
    AILog.Info("I am allowed to own " + max_num_cars + " cars, currently i own " + GetCurrVehCount() + " cars.");
    AILog.Info("--------------------------------------------------");
    if (GetCurrVehCount() < max_num_cars) {

      local non_used_towns = AITownList();
      non_used_towns.Valuate(AITown.GetPopulation);
      non_used_towns.Sort(AIAbstractList.SORT_BY_VALUE, false); // highest value first

      while (GetCurrVehCount() < max_num_cars && non_used_towns.Count() > 0) { // Build cars in one town
        local town_id = non_used_towns.Begin();
        non_used_towns.RemoveTop(1);
        if (MyDepots.HasItem(town_id)) {
          local depot_tile = MyDepots.GetValue(town_id);
          local num_cars_for_this_town = AITown.GetPopulation(town_id) / 100 * num_cars_per_town;

          if (num_cars_for_this_town > 0) {
            local num_cars_to_buy = Helper.Min(num_cars_for_this_town, max_num_cars-GetCurrVehCount());
            AILog.Info("Now working in " + AITown.GetName(town_id) + ". Building " + num_cars_to_buy + " cars here.");
            if (!BuildCars(depot_tile, num_cars_to_buy, depot_tile, car_sleep_time)) {
              AILog.Warning("Building cars failed, aborting...");
            } // if
          } // if

          AILog.Info("Finished with " + AITown.GetName(town_id) + " - sleeping for " + town_sleep_time + " ticks.");
          AILog.Info(" ");
          this.Sleep(town_sleep_time);
        } // if
      } // while

      if (GetCurrVehCount() < max_num_cars) {
        AILog.Info("I have built cars in every town. Now sleeping for " + eol_sleep_time + " ticks, then start again.");
      } // if
    } else {
        AILog.Info("I have built the maximum allowed number of cars. Now sleeping for " + eol_sleep_time + " ticks, then start again.");
    } // if 
    AILog.Info("--------------------------------------------------");
    this.Sleep(eol_sleep_time);
  } // while
} // Start

// --------------------------------------------------------------------
// public function Save(): Save AI specific data
// --------------------------------------------------------------------
function StreetTraffic::Save() {
  local SaveData  = {};
  local DepotList = [];

  AILog.Info("Saving Depot-List");
  foreach (TownId, DepotId in MyDepots) { 
    local Name = "Town" + TownId;
    SaveData.rawset(Name,DepotId);
  } // foreach

  return SaveData;
} // Save

// --------------------------------------------------------------------
// public function Load: Load AI specific data
// --------------------------------------------------------------------
function StreetTraffic::Load(version, data) {
  loaded_from_save = true;
  AILog.Info("Previously saved with AI version " + version);
  local TownList = AITownList();
  foreach (TownId,dummy in TownList) {
    local Name = "Town" + TownId;
    if (data.rawin(Name)) {
      local DepotId = data.rawget(Name);
      MyDepots.AddItem(TownId,DepotId);
    } // foreach
  } // if
} // Load

