/*
 * This file is part of FastPTPAI, which is an AI for OpenTTD
 *
 * FastPTPAI is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License
 *
 * FastPTPAI is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with FastPTPAI; If not, see <http://www.gnu.org/licenses/> or
 * write to the Free Software Foundation, Inc., 51 Franklin Street, 
 * Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */
import("util.superlib","SuperLib",36);
import("pathfinder.rail", "RailPathFinder", 1);;

require("pathfinder.nut");

require("valuators.nut");
require("utils.nut");

//require("town.nut");
//require("pathfinder.nut");
//require("convoy.nut");

Result		<- SuperLib.Result;
Log			<- SuperLib.Log;
Helper		<- SuperLib.Helper;
ScoreList	<- SuperLib.ScoreList;
Money		<- SuperLib.Money;
Tile		<- SuperLib.Tile;
Direction	<- SuperLib.Direction;
Engine		<- SuperLib.Engine;
Vehicle		<- SuperLib.Vehicle;
SLStation	<- SuperLib.Station;
Airport		<- SuperLib.Airport;
SLIndustry	<- SuperLib.Industry;
Town		<- SuperLib.Town;
Order		<- SuperLib.Order;
OrderList	<- SuperLib.OrderList;
Road		<- SuperLib.Road;
RoadBuilder	<- SuperLib.RoadBuilder;

//AIAbstractList <- AIList; // For Rail Path Finder

aiInstance	<- null;

class FastPTPAI extends AIController {
	blacklist = null; // List of industries that dont work with the AI
	engineBlacklist = null; // List of train engines that dont work with the AI
	constructor() { blacklist = []; engineBlacklist = []; }
	function Start();
	function Save();
	function Load(version,data);
	function BuildHQ();
	function BuildMoneymaker();
}

function FastPTPAI::Start()
{
	AIController.Sleep(5);
	aiInstance = this;
	AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_ROAD);
	Log.Info("FastPTPAI Started",Log.LVL_SUB_DECISIONS);
	if(AIGameSettings.IsDisabledVehicleType(AIVehicle.VT_RAIL)) {
		Log.Error("This is a rail only AI!  I cannot play unless rail is enabled!");
		Log.Error("If you do not know how to fix this issue go to Advanced Settings -> Competitors -> Computer Players -> Enable Rail");
		Log.Error("Once enabled, go ahead and click the 'Restart AI' button on this dialog");
		Log.Error("Until then, I will crash because I cannot play!");
		FATAL_CantSupportVehicles();
	}
	// Name the company
	if(!AICompany.SetName("Fast Transport Inc.")) {
		if(!AICompany.SetName("Speedy Transport Inc.")) {
			if(!AICompany.SetName("Another Fast Transport Inc.")) {
				local i = 1;
				while(!AICompany.SetName("Fast Impersonator #" + i)) {
					i = i + 1;
				}
			}
		}
	}
	BuildHQ();
	AIRail.SetCurrentRailType(AIRailTypeList().Begin());
	local townlist;
	local event;
	while(true)
	{
		// Manage events...
		while(AIEventController.IsEventWaiting()) {
			event = AIEventController.GetNextEvent();
			switch(event.GetEventType()) {
				case AIEvent.ET_ENGINE_PREVIEW:
					local preview = AIEventEnginePreview.Convert(event);
					preview.AcceptPreview();
					Log.Info("Accepted preview for vehicle " + preview.GetName());
					break;
			}
		}
		// now for the sweet stuff!
		Money.MaxLoan();
		for(local i = 0;i < 3;i++) {
			if(!BuildMoneymaker()) {
				Log.Info("Trying again...", Log.LVL_SUB_DECISIONS);
				BuildMoneymaker(); // Might work if we try a second time
			}
		}
		Money.MakeMaximumPayback();
		
		AIController.Sleep(10);
	}
}

function FastPTPAI::BuildMoneymaker() {
	WaitForMoney(50000, -1);
	Log.Info("Creating moneymaker route...", Log.LVL_SUB_DECISIONS);

	local cargos = AICargoList();
	cargos.Valuate(AICargo.HasCargoClass, AICargo.CC_BULK);
	cargos.KeepValue(1);
	// Go for the most expensive cargo
	//cargos.Valuate(AICargo.GetCargoIncome, 50, 30);
	// Go for a random cargo
	cargos.Valuate(AIBase.RandItem);
	cargos.Sort(AIList.SORT_BY_VALUE, false);
	if(cargos.IsEmpty()) {
		Log.Warning("No cargos to work with!");
		return null;
	}
	local cargo = cargos.Begin();
	Log.Info("Route cargo: " + AICargo.GetCargoLabel(cargo));
	
	local industries = AIIndustryList_CargoProducing(cargo);
	industries.Valuate(AIIndustry.GetAmountOfStationsAround);
	industries.KeepValue(0);
	industries.RemoveList(Helper.SquirrelListToAIList(blacklist));
	industries.Valuate(AIIndustry.GetLastMonthProduction, cargo);
	industries.Sort(AIList.SORT_BY_VALUE, false);
	if(industries.IsEmpty()) {
		Log.Warning("Could not find good industry!");
		return false;
	}
	local from = industries.Begin();
	
	industries = AIIndustryList_CargoAccepting(cargo);
	//industries.Valuate(AIIndustry.GetAmountOfStationsAround);
	//industries.KeepValue(0);
	//industries.RemoveList(Helper.SquirrelListToAIList(blacklist));
	industries.Valuate(AIIndustry.GetDistanceManhattanToTile, AIIndustry.GetLocation(from));
	industries.KeepBetweenValue(75, 150);
	industries.Sort(AIList.SORT_BY_VALUE, false);
	if(industries.IsEmpty()) {
		Log.Warning("Could not find good industry near " + Tile.GetTileString(AIIndustry.GetLocation(from)));
		blacklist.append(from);
		return false;
	}
	local to = industries.Begin();
	
	Log.Info("Route industries: " + AIIndustry.GetName(from) + " => " + AIIndustry.GetName(to), Log.LVL_SUB_DECISIONS);
	
	
	local tiles = Tile.MakeTileRectAroundTile(AIIndustry.GetLocation(from), 15);
	tiles.Valuate(AITile.IsBuildableRectangle, 7, 2);
	tiles.KeepValue(1);
	tiles.Valuate(AITile.GetDistanceManhattanToTile, AIIndustry.GetLocation(from));
	tiles.RemoveAboveValue(4);
	tiles.Valuate(GetDistanceFromWater);
	//tiles.KeepAboveValue(Helper.Max(Station.STATION_RAILMAP.len(), Station.STATION_RAILMAP[0].len()) + 2);
	tiles.Sort(AIList.SORT_BY_VALUE, false);
	if(tiles.IsEmpty()) {
		Log.Error("Cannot not build station near industry!");
		blacklist.append(from);
		return false;
	}
	local tile = tiles.Begin();
	// Build station
	Tile.FlatternRect(tile, 5, 1);
	if(!AIRail.BuildRailStation(tile, AIRail.RAILTRACK_NE_SW, 1, 5, AIStation.STATION_NEW)) {
		Log.Error("Could not build station: " + AIError.GetLastErrorString());
		blacklist.append(from);
		return false;
	}
	local fromid = AIStation.GetStationID(tile);
	
	tiles = Tile.MakeTileRectAroundTile(AIIndustry.GetLocation(to), 15);
	tiles.Valuate(AITile.IsBuildableRectangle, 7, 2);
	tiles.KeepValue(1);
	tiles.Valuate(AITile.GetDistanceManhattanToTile, AIIndustry.GetLocation(to));
	tiles.RemoveAboveValue(4);
	tiles.Valuate(GetDistanceFromWater);
	tiles.Sort(AIList.SORT_BY_VALUE, false);
	if(tiles.IsEmpty()) {
		Log.Error("Cannot not build station near industry!");
		blacklist.append(from);
		return false;
	}
	tile = tiles.Begin();
	Tile.FlatternRect(tile, 5, 1);
	if(!AIRail.BuildRailStation(tile, AIRail.RAILTRACK_NE_SW, 1, 5, AIStation.STATION_NEW)) {
		Log.Error("Could not build station!");
		blacklist.append(to);
		return false;
	}
	local toid = AIStation.GetStationID(tile);
	
	// Connect;
	Log.Info("Connecting industries...", Log.LVL_SUB_DECISIONS);
	local pf = Pathfinder([Direction.GetTileInDirection(AIStation.GetLocation(fromid), Direction.DIR_SW, 5),Direction.GetTileInDirection(AIStation.GetLocation(fromid), Direction.DIR_SW, 4)], [Direction.GetTileInDirection(AIStation.GetLocation(toid), Direction.DIR_SW, 5),Direction.GetTileInDirection(AIStation.GetLocation(toid), Direction.DIR_SW, 4)], false, [], 2.0);
	Log.Info("Finding Path...");
	local ret = pf.FindPath();
	if(!ret) {
		Log.Warning("Failed to pathfind rail!");
		blacklist.append(from);
		return false;
	}
	
	Log.Info("Building Path...");
	ret = pf.BuildPath();
	if(ret != "success") {
		Log.Warning("Failed to build rail!");
		return false;
	}
	Log.Info("Success!", Log.LVL_SUB_DECISIONS);
	
	// Find a good spot to build depot
	local tiles = Tile.MakeTileRectAroundTile(AIStation.GetLocation(fromid), 12);
	tiles.Valuate(IsAlongPath, pf);
	tiles.KeepValue(1);
	tiles.Valuate(AITile.IsBuildable);
	tiles.KeepValue(1);
	tiles.Valuate(AIMap.DistanceManhattan, AIStation.GetLocation(fromid));
	tiles.Sort(AIList.SORT_BY_VALUE, false);
	local tile = tiles.Begin();
	Log.Info("Building Depot for " + AIStation.GetName(fromid), Log.LVL_SUB_DECISIONS);
	local built = false;
	local front = null;
	while(!built) {
	// Find the tile where the railroad is at
		local ts = Tile.GetNeighbours4MainDir(tile);
		ts.Valuate(AIRail.IsRailTile);
		ts.KeepValue(1);
		// Now build
		front = ts.Begin();
		if(!AIRail.BuildRailDepot(tile, front)) {
			tile = tiles.Next();
			if(!tiles.IsEnd()) {}
			else {
				Log.Warning("Could not build depot!");
				break;
			}
		}
		else built = true;
	}
	// Connect the depot to the railway, and we are done!
	local dir = Direction.GetDirectionToAdjacentTile(tile, front);
	// Remove signal
	if(AIRail.GetSignalType(front, Direction.GetAdjacentTileInDirection(front, Direction.TurnDirClockwise90Deg(dir, 1))) != AIRail.SIGNALTYPE_NONE) {
		AIRail.RemoveSignal(front, Direction.GetAdjacentTileInDirection(front, Direction.TurnDirClockwise90Deg(dir, 1)));
	}
	if(AIRail.GetSignalType(front, Direction.GetAdjacentTileInDirection(front, Direction.TurnDirAntiClockwise90Deg(dir, 1))) != AIRail.SIGNALTYPE_NONE) {
		AIRail.RemoveSignal(front, Direction.GetAdjacentTileInDirection(front, Direction.TurnDirAntiClockwise90Deg(dir, 1)));
	}
	// Connect to railway
	AIRail.BuildRailTrack(front, ConvertDirectionToRail(Direction.TurnDirClockwise45Deg(Direction.OppositeDir(dir), 1)));
	AIRail.BuildRailTrack(front,ConvertDirectionToRail(Direction.TurnDirAntiClockwise45Deg(Direction.OppositeDir(dir), 1)));
	
	local depot = tile;
	
	// Find a good spot to build
	local tiles = Tile.MakeTileRectAroundTile(AIStation.GetLocation(toid), 12);
	tiles.Valuate(IsAlongPath, pf);
	tiles.KeepValue(1);
	tiles.Valuate(AITile.IsBuildable);
	tiles.KeepValue(1);
	tiles.Valuate(AIMap.DistanceManhattan, AIStation.GetLocation(toid));
	tiles.Sort(AIList.SORT_BY_VALUE, false);
	local tile = tiles.Begin();
	Log.Info("Building Depot for " + AIStation.GetName(toid), Log.LVL_SUB_DECISIONS);
	local built = false;
	local front = null;
	while(!built) {
	// Find the tile where the railroad is at
		local ts = Tile.GetNeighbours4MainDir(tile);
		ts.Valuate(AIRail.IsRailTile);
		ts.KeepValue(1);
		// Now build
		front = ts.Begin();
		if(!AIRail.BuildRailDepot(tile, front)) {
			tile = tiles.Next();
			if(!tiles.IsEnd()) {}
			else {
				Log.Warning("Could not build depot!");
				break;
			}
		}
		else built = true;
	}
	// Connect the depot to the railway, and we are done!
	local dir = Direction.GetDirectionToAdjacentTile(tile, front);
	// Remove signal
	if(AIRail.GetSignalType(front, Direction.GetAdjacentTileInDirection(front, Direction.TurnDirClockwise90Deg(dir, 1))) != AIRail.SIGNALTYPE_NONE) {
		AIRail.RemoveSignal(front, Direction.GetAdjacentTileInDirection(front, Direction.TurnDirClockwise90Deg(dir, 1)));
	}
	if(AIRail.GetSignalType(front, Direction.GetAdjacentTileInDirection(front, Direction.TurnDirAntiClockwise90Deg(dir, 1))) != AIRail.SIGNALTYPE_NONE) {
		AIRail.RemoveSignal(front, Direction.GetAdjacentTileInDirection(front, Direction.TurnDirAntiClockwise90Deg(dir, 1)));
	}
	// Connect to railway
	AIRail.BuildRailTrack(front, ConvertDirectionToRail(Direction.TurnDirClockwise45Deg(Direction.OppositeDir(dir), 1)));
	AIRail.BuildRailTrack(front,ConvertDirectionToRail(Direction.TurnDirAntiClockwise45Deg(Direction.OppositeDir(dir), 1)));
	local redo = true;
	local train = null;
	while(redo) {
		redo = false;
		// Briefly choose engine
		local engines = AIEngineList(AIVehicle.VT_RAIL);
		engines.Valuate(AIEngine.IsWagon);
		engines.KeepValue(0); // No wagons
		engines.Valuate(AIEngine.CanRunOnRail, AIRailTypeList().Begin());
		engines.KeepValue(1); // Must be able to run on our rail type
		engines.RemoveList(Helper.SquirrelListToAIList(engineBlacklist));
	
		engines.Valuate(AIEngine.GetMaxSpeed);
		engines.Sort(AIList.SORT_BY_VALUE, false);
		if(engines.IsEmpty()) {
			Log.Warning("No good engines!");
			return false;
		}
		local engine = engines.Begin();
	
		// Briefly choose wagon
		local wagons = AIEngineList(AIVehicle.VT_RAIL);
		wagons.Valuate(AIEngine.IsWagon);
		wagons.KeepValue(1); // Only wagons
		wagons.Valuate(AIEngine.CanRefitCargo, cargo);
		wagons.KeepValue(1);
		wagons.Valuate(AIEngine.GetCapacity);
		wagons.Sort(AIList.SORT_BY_VALUE, false);
		if(wagons.IsEmpty()) {
			Log.Warning("No good wagons for cargo: " + AICargo.GetCargoLabel(cargo));
			return false;
		}
		local wid = wagons.Begin();
		
		WaitForMoney(AIEngine.GetPrice(engine) + AIEngine.GetPrice(wid) * 9 + 1000, -1);

		// Build train
		train = AIVehicle.BuildVehicle(depot, engine);
		if(!AIVehicle.IsValidVehicle(train)) {
			Log.Warning("Could not build train engine: " + AIError.GetLastErrorString());
			return false;
		}
		for(local i = 0;i < 8;i++) {
			local wagon = AIVehicle.BuildVehicle(depot, wid);
			//local wagon = AIVehicle.BuildVehicle(depot, wid);
			if(!AIVehicle.IsValidVehicle(wagon)) {
				Log.Warning("Could not build wagon!");
				return false;
			}
			if(!AIVehicle.RefitVehicle(wagon, cargo)) {
				Log.Warning("Could not refit wagon to " + AICargo.GetCargoLabel(cargo));
				return false;
			}
			if(!AIVehicle.MoveWagon(wagon, 0, train, 0)) {
				Log.Warning("Could not move wagon to trian engine!");
				Log.Warning("Blacklisting engine: " + AIEngine.GetName(engine));
				engineBlacklist.append(engine);
				// No return since we keep trying until an engine works
				redo = true;
			}
		}
	}
	// Add orders
	AIOrder.AppendOrder(train, AIStation.GetLocation(fromid), AIOrder.OF_FULL_LOAD_ANY | AIOrder.OF_NON_STOP_INTERMEDIATE);
	AIOrder.AppendOrder(train, AIStation.GetLocation(toid), AIOrder.OF_NON_STOP_INTERMEDIATE);
	
	if(!AIVehicle.StartStopVehicle(train)) {
		Log.Warning("Could not start train!");
		return false;
	}
	return true;
}

function FastPTPAI::BuildHQ() {
	local center;
	local built = false;
	local tries = 0;
	Log.Info("Building HQ...",Log.LVL_SUB_DECISIONS);
	// Build the HQ
	local townlist = AITownList();
	townlist.Valuate(AITown.GetPopulation);
	townlist.Sort(AIList.SORT_BY_VALUE, false);
	foreach(hqtown, _ in townlist) {
		Log.Info("Attempting to build HQ in " + AITown.GetName(hqtown),Log.LVL_SUB_DECISIONS);
		local skip = false;
		local tilelist = Tile.GetTownTiles(hqtown);
		tilelist.Valuate(AITile.GetClosestTown);
		tilelist.KeepValue(hqtown);
		// Scan for any company HQs and go in a different city if one is there
		for(local i = 0;i < 16;i++) {
			if(AICompany.ResolveCompanyID(i) == AICompany.COMPANY_INVALID) continue;
			local tile = AICompany.GetCompanyHQ(i);
			if(tile != AIMap.TILE_INVALID) {
				if(AITile.GetClosestTown(tile) == hqtown) {
					Log.Info("Found another player's HQ in town " + AITown.GetName(hqtown));
					skip = true;
					break;
				}
			}
		}
		if(skip) continue;
		tilelist.Valuate(AITile.IsBuildableRectangle,2,2);
		tilelist.KeepValue(1);
		tilelist.Valuate(AIMap.DistanceManhattan,AITown.GetLocation(hqtown));
		tilelist.Sort(AIList.SORT_BY_VALUE, true);
		if(!AICompany.BuildCompanyHQ(tilelist.Begin())) {
			Log.Warning("First attempt to build HQ failed.");
		}
		else {
			built = true;
			center = hqtown;
		}
		while(!built && tries <= 5) {
			if(!AICompany.BuildCompanyHQ(tilelist.Next())) {
				Log.Warning("Could not build HQ!");
				tries++;
			}
			else {
				built = true;
				center = hqtown;
				break;
			}
		}
		if(built) break;
		if(!center) {
			Log.Warning("Completely failed to build HQ :(");
		}
	}
}

function FastPTPAI::Save()
{
	local ptp = {};
	return ptp;
}

function FastPTPAI::Load(version,data)
{
}
